#include "processor.h"

#include <drive/backend/billing/exports/yt_nodes.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/history_iterator/history_iterator.h>

#include <rtline/util/algorithm/ptr.h>

template <class TEvent>
class TReportItem {
private:
    const TEvent& Event;
    const NDrive::NBilling::TFiscalItem& BillingAccount;

    TAtomicSharedPtr<TFullCompiledRiding> CompiledSession;

public:
    TReportItem(const TEvent& ev, const NDrive::NBilling::TFiscalItem& bAccount)
        : Event(ev)
        , BillingAccount(bAccount)
    {
    }

    NJson::TJsonValue GetReport(const NDrive::IServer* server) const {
        auto compiledRide = CompiledSession.Get();
        auto& ev = Event;
        const NDrive::NBilling::TFiscalItem& i = BillingAccount;

        const TString& sessionId = ev.GetSessionId();
        ui64 sum = i.GetSum() > 0 ? i.GetSum() : ev.GetBill();
        ui64 promocodeSum = (i.GetType() == NDrive::NBilling::EAccount::Bonus) ? sum : 0;
        ui64 totalSum = (i.GetType() == NDrive::NBilling::EAccount::Bonus) ? 0 : sum;

        NJson::TJsonValue session;
        session["account_id"] = i.GetAccountId();
        session["account_name"] = i.GetUniqueName();
        session["drive_user_id"] = ev.GetUserId() ? ev.GetUserId() : ev.GetHistoryUserId();
        session["sequential_id"] = ev.GetHistoryEventId();
        session["service_order_id"] = sessionId;
        session["orig_transaction_id"] = sessionId;
        session["transaction_id"] = i.GetTransactionId();

        session["dt"] = ToString(ev.GetHistoryInstant());
        session["timestamp"] = ev.GetHistoryInstant().Seconds();

        session["bill"] = ev.GetBill();
        session["commission_sum"] = "0.0";
        session["promocode_sum"] = NBillingExports::GetStringSum(promocodeSum, false);
        session["total_sum"] = NBillingExports::GetStringSum(totalSum, false);
        session["transaction_currency"] = "RUB";
        session["type"] = NBillingExports::GetOrderType(ev.GetBillingType());
        session["user_discount"] = 0;
        if constexpr (std::is_same<TEvent, TObjectEvent<TCompiledBill>>::value) {
            session["final_bill"] = ev.GetFinal();
        }

        auto api = Yensured(server)->GetDriveAPI();
        auto accountDescription = Yensured(api)->GetBillingManager().GetAccountsManager().GetDescriptionByName(i.GetUniqueName());
        if (accountDescription) {
            session["parent_id"] = accountDescription->GetParentId();
        }

        if (compiledRide) {
            session["real_order_id"] = compiledRide->GetSessionId();
            session["started_at"] = ToString<TInstant>(compiledRide->GetStartInstant());
            session["finished_at"] = ToString<TInstant>(compiledRide->GetFinishInstant());
            session["duration"] = compiledRide->GetDuration().Seconds();
            session["tariff"] = compiledRide->GetOfferName();
            if (auto snapshot = compiledRide->GetSnapshotsDiffPtr()) {
                if (auto start = snapshot->GetStartPtr()) {
                    auto coordinate = start->GetCoord();
                    auto tags = api->GetTagsInPoint(coordinate);
                    session["start_point_coordinates"] = NJson::ToJson(coordinate);
                    session["start_point_tags"] = NJson::ToJson(tags);
                }
                if (auto finish = snapshot->GetLastPtr()) {
                    auto coordinate = finish->GetCoord();
                    auto tags = api->GetTagsInPoint(coordinate);
                    session["finish_point_coordinates"] = NJson::ToJson(coordinate);
                    session["finish_point_tags"] = NJson::ToJson(tags);
                }
                if (auto mileage = snapshot->GetMileagePtr()) {
                    session["total_mileage"] = *mileage;
                } else {
                    session["total_mileage"] = 0;
                }
            }

            const TString& objectId = compiledRide->TCompiledRiding::GetObjectId();
            session["car_id"] = objectId;
            auto car = server->GetDriveAPI()->GetCarsData()->GetObject(objectId, TInstant::Zero());
            if (car) {
                session["number"] = car->GetNumber();
                session["model"] = car->GetModel();
                auto models = server->GetDriveAPI()->GetModelsData()->GetCached({ car->GetModel() });
                auto model = models.GetResultPtr(car->GetModel());
                if (model) {
                    session["model_name"] = model->GetName();
                }
            }
        }
        return session;
    }

    const TString& GetRealSessionId() const {
        return Event.GetRealSessionId() ? Event.GetRealSessionId() : Event.GetSessionId();
    }

    template <class T>
    void SetCompiledSession(T&& value) {
        CompiledSession = std::forward<T>(value);
    }
};

template <class TProcessor>
template <class T>
void TCompiledEntityHistory<TProcessor>::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const T& manager) {
    TJsonReport& r = g.MutableReport();

    const TCgiParameters& cgi = TBase::Context->GetCgiParameters();
    auto billingTypes = MakeSet(TBase::template GetValues<EBillingType>(cgi, "billing_type", false));
    auto cursor = TBase::template GetValue<ui64>(cgi, "cursor", false).GetOrElse(0);
    auto limit = TBase::template GetValue<size_t>(cgi, "limit", false).GetOrElse(100);
    auto since = TBase::GetTimestamp(cgi, "since", false);
    auto until = TBase::GetTimestamp(cgi, "until", TInstant::Max());
    auto sessionIds = TBase::GetStrings(cgi, "session_id", false);
    auto accountNames = TBase::GetStrings(cgi, "account_name", false);

    if (billingTypes.empty()) {
        billingTypes.insert(EBillingType::CarUsage);
    }

    const auto accounts = TBase::GetAvailableWallets(permissions, TAdministrativeAction::EAction::Observe);
    R_ENSURE(accounts, TBase::ConfigHttpStatus.UnknownErrorStatus, "cannot fetch available wallets");

    TSet<TString> filtredAccountNames;
    for (const auto& acc : accountNames) {
        if (accounts->contains(acc)) {
            filtredAccountNames.emplace(acc);
        }
    }

    auto tx = TBase::template BuildTx<NSQL::ReadOnly>();

    TMaybe<typename T::TEvents> events;
    if (!sessionIds.empty()) {
        TEventsGuard eg(r, "GetEventsById");
        if constexpr (std::is_same<T, TFiscalRefundsHistoryManager>::value) {
            events = manager.GetRefundsFromDB(sessionIds, tx);
        } else {
            events = manager.GetBillsFromDB(sessionIds, tx);
        }
    } else if (accountNames && !filtredAccountNames) {
        events = TVector<typename T::TEvent>();
    } else {
        TEventsGuard eg(r, "GetEventsSinceId");
        auto billingWalletsFilter = TBase::template GetHandlerSetting<bool>("billing_wallets_filter").GetOrElse(false);
        auto queryOptions = IBaseSequentialTableImpl::TQueryOptions();
        if (billingWalletsFilter && accountNames) {
            queryOptions.SetGenericCondition("billing_wallets", filtredAccountNames);
        }
        events = manager.GetEvents(cursor + 1, { since, until }, tx, queryOptions);
    }
    if (!events) {
        tx.DoExceptionOnFail(TBase::ConfigHttpStatus);
    }

    NJson::TJsonValue sessionsJson = NJson::JSON_ARRAY;
    TVector<TReportItem<typename T::TEvent>> reportItems;
    TVector<TString> filteredSessionIds;
    for (auto&& ev : *events) {
        if (ev.GetHistoryInstant() > until) {
            break;
        }
        if (ev.GetHistoryInstant() < since) {
            continue;
        }
        if (!billingTypes.contains(ev.GetBillingType())) {
            continue;
        }

        for (auto&& i : ev.GetDetails().GetItems()) {
            if (!accounts->contains(i.GetUniqueName())) {
                continue;
            }
            if (accountNames && !filtredAccountNames.contains(i.GetUniqueName())) {
                continue;
            }
            reportItems.emplace_back(ev, i);
            filteredSessionIds.push_back(reportItems.back().GetRealSessionId());
        }
        cursor = std::max(cursor, ev.GetHistoryEventId());
        if (reportItems.size() >= limit) {
            break;
        }
    }

    THistoryRidesContext context(*TBase::Server);
    {
        auto eg = g.BuildEventGuard("InitializeHistoryRidesContext");
        auto ydbTx = TBase::template BuildYdbTx<NSQL::ReadOnly>("compiled_entity_history");
        R_ENSURE(context.InitializeSessions(filteredSessionIds, tx, ydbTx), TBase::ConfigHttpStatus.UnknownErrorStatus, "cannot initialize HistoryRidesContext");
    }

    auto sessions = context.GetSessions(TInstant::Max(), 100 * filteredSessionIds.size());
    THistoryRideObject::FetchFullRiding(TBase::Server, sessions);

    TMap<TString, TAtomicSharedPtr<TFullCompiledRiding>> compiledSessionMap;
    for (auto&& session : sessions) {
        auto compiledRiding = session.GetFullCompiledRiding(tx);
        compiledSessionMap.emplace(session.GetSessionId(), compiledRiding);
    }
    for (auto&& i : reportItems) {
        i.SetCompiledSession(compiledSessionMap[i.GetRealSessionId()]);
    }
    for (auto&& i : reportItems) {
        sessionsJson.AppendValue(i.GetReport(TBase::Server));
    }
    r.AddReportElement("cursor", cursor);
    r.AddReportElement("sessions", std::move(sessionsJson));
    g.SetCode(HTTP_OK);
}

void TCompiledBillsHistory::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Process(g, permissions, DriveApi->GetBillingManager().GetCompiledBills());
}

void TCompiledRefundsHistory::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Process(g, permissions, DriveApi->GetBillingManager().GetCompiledRefunds());
}
