#include "processors.h"

#include <drive/backend/base/server.h>

#include <drive/library/cpp/network/data/data.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/api/search_client/reply.h>
#include <rtline/library/graph/geometry_graph/common/access_type.h>
#include <rtline/library/graph/request/parsers/route_mn.h>

#include <util/string/join.h>


using namespace NRTLine;

namespace {
    const TString StatusOk("OK");
    const TString StatusFail("FAIL");
    const TString StatusTimeout("TIMEOUT");

    const TString CgiAccuracies("accuracies");
    const TString CgiAvoidTolls("avoid_tolls");
    const TString CgiDepartureTime("departure_time");
    const TString CgiDestinations("destinations");
    const TString CgiGta("gta");
    const TString CgiMode("mode");
    const TString CgiOrigins("origins");
    const TString CgiPoints("points");
    const TString CgiPriority("priority");
    const TString CgiPron("pron");
    const TString CgiSuggestPart("part");
    const TString CgiText("text");
    const TString CgiTimeout("timeout");
    const TString CgiTimestamps("timestamps");
    const TString CgiWaypoints("waypoints");
    const TString CgiWeight("weight");

    const double IndicesPrecision = 0.01f;


    const TString& GetCgiParameter(const TCgiParameters& cgi, TStringBuf name, bool required, int defaultCode=HTTP_BAD_REQUEST) {
        const size_t count = cgi.NumOfValues(name);
        if (count == 0) {
            if (required) {
                throw TCodedException(defaultCode) << "parameter '" << name << "' is missing";
            } else {
                return Default<TString>();
            }
        }
        if (count > 1) {
            throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' is specified more than once";
        }
        const TString& value = cgi.Get(name);
        if (value.empty()) {
            throw TCodedException(defaultCode) << "parameter '" << name << "' is empty";
        }
        return value;
    }

    ERouterMode GetRouterMode(const TCgiParameters& cgi) {
        const TString& mode = GetCgiParameter(cgi, CgiMode, false);
        ERouterMode result = ERouterMode::Driving;
        if (!mode.empty() && !TryFromString(mode, result)) {
            throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiMode << "' is incorrect: unknown mode '" << mode << "'";
        }
        return result;
    }

    template <class T>
    T FromExternalString(TStringBuf s, TStringBuf label) {
        T result;
        if (TryFromString(s, result)) {
            return result;
        } else {
            throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << label << "' contains a token '" << s << "' that cannot be parsed as " << TypeName<T>();
        }
    }


    template <class T>
    TVector<T> GetArray(TStringBuf s, TStringBuf label) {
        TVector<T> result;
        if (s.empty()) {
            return result;
        }
        for (auto&& i : StringSplitter(s).Split('|')) {
            result.push_back(FromExternalString<T>(i.Token(), label));
        }
        return result;
    }


    TVector<TGeoCoord> GetCgiCoordinates(const TCgiParameters& cgi, TStringBuf name, bool required) {
        TString value = GetCgiParameter(cgi, name, required);

        TVector<TGeoCoord> result;
        for (auto&& c : StringSplitter(value).Split('|')) {
            auto token = c.Token();
            if (token.empty()) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains an empty coordinate";
            }
            if (token.find(',') == TStringBuf::npos) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains a coordinate '" << token << "' without a separating comma";
            }
            auto latitude = token.Before(',');
            auto longitude = token.After(',');
            if (latitude.empty()) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains a coordinate '" << token << "' with empty latitude";
            }
            if (longitude.empty()) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains a coordinate '" << token << "' with empty longitude";
            }
            auto coordinate = TGeoCoord(
                FromExternalString<double>(longitude, name),
                FromExternalString<double>(latitude, name)
            );
            if (std::abs(coordinate.X) > 180) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains a coordinate '" << token << "' with longitude outside of [-180; 180] range";
            }
            if (std::abs(coordinate.Y) > 90) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << name << "' contains a coordinate '" << token << "' with latitide outside of [-90; 90] range";
            }
            result.emplace_back(std::move(coordinate));
        }
        return result;
    }


    TString GetPermissions(ERouterMode mode, const TString& departureTime, bool internalRequest) {
        TString base;

        switch (mode) {
        case ERouterMode::Driving:
            base = "ptAuto";
            break;
        case ERouterMode::Walking:
            base = "ptPedestrian";
            break;
        case ERouterMode::Transit:
            base = "Transport";
            break;
        }

        TString modificator;
        if (departureTime) {
            if (mode == ERouterMode::Walking) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter 'mode=" << mode << "' and '" << CgiDepartureTime << "' should not be used together";
            }

            ui64 timestamp = FromExternalString<ui64>(departureTime, CgiDepartureTime);
            ui64 now = Seconds();

            constexpr ui64 lowThreshold = 10 * 60; // 10 minutes
            if (!internalRequest) {
                if (timestamp + lowThreshold < now) {
                    throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiDepartureTime << "' is in the past: " << timestamp;
                }
            }

            constexpr ui64 upperThreshold = 30 * 60; // 30 minutes
            if (timestamp > now + upperThreshold || timestamp + lowThreshold < now) {
                modificator = ToString(timestamp);
            }
        }

        if (mode == ERouterMode::Transit && modificator) { // workaround for backend quirks
            base = "ptBus|ptPedestrian|ptSubway";
        }

        if (modificator) {
            return base + '-' + modificator;
        } else {
            return base;
        }
    }


    TString GetMatchingPermissions(ERouterMode mode) {
        switch (mode) {
        case ERouterMode::Driving:
            return "ptAuto";
        case ERouterMode::Walking:
            return "ptPedestrian";
        case ERouterMode::Transit:
            throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiMode << "' is incorrect: mode '" << mode << "' is not supported";
        }
    }


    bool IsRealtimeBehaviour(TStringBuf name) {
        return
            name == "ptAuto" ||
            name == "ptPedestrian" ||
            name == "Transport";
    }


    TString GetTrafficType(const NJson::TJsonValue& behaviourInfo) {
        if (behaviourInfo.Has("time_switch") || !IsRealtimeBehaviour(behaviourInfo["name"].GetString())) {
            return "forecast";
        }
        return "realtime";
    }


    bool IsTrivialAccessType(ui32 accessType) {
        return accessType == 0
            || accessType == NGeoEdge::ptAuto
            || accessType == NGeoEdge::ptPedestrian
        ;
    }


    TString GetAlgorithmDistanceMatrix(ERouterMode mode) {
        if (mode == ERouterMode::Transit) {
            return "layer";
        } else {
            return {};
        }
    }


    TString GetAlgorithmRoute(ERouterMode mode, bool avoidTolls) {
        if (mode == ERouterMode::Transit) {
            return "layer_p2p";
        } else if (avoidTolls) {
            return "layer";
        } else {
            return {};
        }
    }


    ui32 GetPrecision() {
        return 1000;
    }


    void ValidateRequest(const TCgiParameters& cgi) {
        if (cgi.Has("text")) {
            throw TCodedException(HTTP_BAD_REQUEST) << "forbidden parameter: text";
        }
    }


    template<class T>
    TVector<T> GetCgiArray(const TCgiParameters& cgi, TStringBuf name, bool required) {
        const TString& value = GetCgiParameter(cgi, name, required);
        return GetArray<T>(value, name);
    }

    void SumRouteElements(NGraph::TBaseRouter::TRouteElement& dst, TVector<TGeoCoord>& dstTrace, const NGraph::TBaseRouter::TRouteElement& src) {
        for (const auto& coord: src.PolyLine.GetCoords()) {
            if (dstTrace.empty()) {
                dstTrace.push_back(coord);
            } else {
                const auto& last = dstTrace.back();
                if (std::abs(last.X - coord.X) > 0.0001f || std::abs(last.Y - coord.Y) > 0.0001f) {
                    dstTrace.push_back(coord);
                }
            }
        }

        if (dst.Length < 0) {
            dst.Length = 0;
        }
        dst.Length += src.Length;
        if (dst.Time < 0) {
            dst.Time = 0;
        }
        dst.Time += src.Time;
        dst.WaitingTime += src.WaitingTime;
        dst.FC = src.FC;
        dst.AccessType = src.AccessType;
    }
}


void TB2BRequestProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    AssertCorrectConfig(section->GetDirectives().GetValue("RTLineAPIName", RTLineAPIName), "Incorrect 'RTLineApiName' field for matrix calculation processor");
    AssertCorrectConfig(section->GetDirectives().GetValue("RTLineAPINameTransport", RTLineAPINameTransport), "Incorrect 'RTLineApiNameTransport' field for matrix calculation processor");
    RTLineAPINameWalking = section->GetDirectives().Value("RTLineAPINameWalking", RTLineAPIName);
    RTLineAPITimeout = section->GetDirectives().Value("RTLineAPITimeout", RTLineAPITimeout);
    RTLineAPIRealtimeTimeout = section->GetDirectives().Value("RTLineAPIRealtimeTimeout", RTLineAPIRealtimeTimeout);
}


void TB2BRequestProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "RTLineAPIName: " << RTLineAPIName << Endl;
    os << "RTLineAPINameTransport: " << RTLineAPINameTransport << Endl;
    os << "RTLineAPINameWalking: " << RTLineAPINameWalking << Endl;
    os << "RTLineAPITimeout: " << RTLineAPITimeout << Endl;
    os << "RTLineAPIRealtimeTimeout: " << RTLineAPIRealtimeTimeout << Endl;
}


const TString& TB2BRequestProcessorConfig::GetRTLineAPIName(const TCgiParameters& cgi) const {
    ERouterMode mode = GetRouterMode(cgi);
    const TString& source = GetCgiParameter(cgi, "source", false);

    if (source) {
        return source;
    }
    switch (mode) {
    case ERouterMode::Transit:
        return RTLineAPINameTransport;
    case ERouterMode::Walking:
        return RTLineAPINameWalking;
    case ERouterMode::Driving:
        return RTLineAPIName;
    }
}


IRequestProcessor::TPtr TDistanceMatrixProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TDistanceMatrixProcessor(this, context, authModule, server);
}


void TDistanceMatrixProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    MaxMatrixSize = section->GetDirectives().Value("MaxMatrixSize", MaxMatrixSize);
}


void TDistanceMatrixProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "MaxMatrixSize: " << MaxMatrixSize << Endl;
}


IRequestProcessor::TPtr TRouteProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TRouteProcessor(this, context, authModule, server);
}


void TRouteProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    FastEdgeSpeedLimitKmh = section->GetDirectives().Value("FastEdgeSpeedLimitKmh", FastEdgeSpeedLimitKmh);
}


void TRouteProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "FastEdgeSpeedLimitKmh: " << FastEdgeSpeedLimitKmh << Endl;
}


IRequestProcessor::TPtr TMatchProcessorConfig::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    return new TMatchProcessor(this, context, authModule, server);
}


void TMatchProcessorConfig::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
}


void TMatchProcessorConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
}


TString TB2BRequestProcessor::GetStatus(bool isOk, bool timeouted) const {
    if (!isOk) {
        if (timeouted && InternalRequest) {
            return StatusTimeout;
        } else {
            return StatusFail;
        }
    }
    return StatusOk;
}


void TB2BRequestProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    {
        g.MutableReport().SetExternalReport(NJson::JSON_MAP);

        TCgiParameters cgi = Context->GetCgiParameters();
        g.MutableReport().AddEvent("Cgi original: " + cgi.Print());

        RTLineAPIName = Config.GetRTLineAPIName(cgi);
        TryFromString(cgi.Get("internal_request"), InternalRequest);

        TQuery query = CorrectCgi(cgi);
        CorrectPriority(cgi, query);
        TDuration timeout = query.GetTimeout(Config.GetRTLineAPITimeout());
        g.MutableReport().AddEvent("Query to geosaas: " + query.BuildQuery(true) + "&timeout=" + ToString(timeout.MicroSeconds()));

        const auto& searchClient = BaseServer->GetRTLineAPI(RTLineAPIName)->GetSearchClient();
        TSearchReply searchReply = searchClient.SendQuery(query, timeout);
        ComposeReport(searchReply, g);

        if (InternalRequest) {
            const NMetaProtocol::TReport& report = searchReply.GetReport();
            NJson::TJsonValue jsonEvents = NJson::JSON_ARRAY;
            for (auto& event: report.GetEventLog()) {
                jsonEvents.AppendValue(event);
            }
            g.MutableReport().AddEvent(std::move(jsonEvents));
        }
    }
}


void TB2BRequestProcessor::CorrectPriority(const TCgiParameters& cgi, TQuery& query) {
    TString priority = GetCgiParameter(cgi, CgiPriority, false);
    if (priority == "realtime") {
        query.AddExtraParam("pron", "longsearch");

        ui64 maxTimeout = Config.GetRTLineAPIRealtimeTimeout().MicroSeconds();
        ui64 timeout = 0;
        TryFromString(GetCgiParameter(cgi, CgiTimeout, false), timeout);
        if (timeout == 0 || timeout > maxTimeout) {
            timeout = maxTimeout;
        }
        query.SetTimeout(TDuration::MicroSeconds(timeout));
    } else if (priority.empty() || priority == "normal") {
        ui64 timeout = 0;
        TryFromString(GetCgiParameter(cgi, CgiTimeout, false), timeout);
        if (timeout > 0) {
            query.SetTimeout(TDuration::MicroSeconds(timeout));
        }
    } else {
        throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiPriority << "' has to be one of the following: normal, realtime";
    }
}


void TDistanceMatrixProcessor::ComposeReport(const TSearchReply& searchReply, TJsonReport::TGuard& g) {
    if (searchReply.GetCode() == HTTP_OK) {
        TCgiParameters cgi = Context->GetCgiParameters();

        NGraphResponse::TRouteMNParser parser;
        parser.ParseFromReport(searchReply.GetReport());

        g.MutableReport().AddReportElement("traffic_type", GetTrafficType(parser.GetBehaviorInfo()));
        if (InternalRequest) {
            g.MutableReport().AddReportElement("behavior_info", NJson::TJsonValue(parser.GetBehaviorInfo()));
        }

        const NGraphResponse::TMatrix* matrix = parser.GetMatrix();
        NJson::TJsonValue jsonMatrix = NJson::JSON_ARRAY;
        if (matrix) {
            for (ui32 row = 0; row < matrix->GetRowsCount(); ++row) {
                NJson::TJsonValue jsonRow = NJson::JSON_MAP;
                jsonRow.InsertValue("elements", NJson::JSON_ARRAY);
                for (ui32 col = 0; col < matrix->GetColsCount(); ++col) {
                    const NGraphResponse::TMatrixElement* element = matrix->GetElement(row, col);
                    NJson::TJsonValue jsonElement = NJson::JSON_MAP;
                    if (element->Correctness() == NGraphResponse::EElementInitialization::CorrectResult) {
                        NJson::TJsonValue value;
                        value.InsertValue("value", static_cast<i64>(element->GetDistance()));
                        jsonElement.InsertValue("distance", value);
                        value.InsertValue("value", element->GetDuration().Seconds());
                        jsonElement.InsertValue("duration", value);

                        jsonElement.InsertValue("status", StatusOk);
                    } else {
                        if (!InternalRequest && parser.GetTimeouted()) {
                            throw TCodedException(HTTP_GATEWAY_TIME_OUT) << "request /distancematrix timeouted";
                        }
                        jsonElement.InsertValue("status", GetStatus(false, parser.GetTimeouted()));
                    }
                    jsonRow["elements"].AppendValue(jsonElement);
                }
                jsonMatrix.AppendValue(jsonRow);
            }
            g.MutableReport().AddReportElement("rows", std::move(jsonMatrix));
        }

        g.SetCode(HTTP_OK);
    } else {
        g.SetCode(searchReply.GetCode());
    }
}


TQuery TDistanceMatrixProcessor::CorrectCgi(const TCgiParameters& cgi) {
    ValidateRequest(cgi);

    const TVector<TGeoCoord>& from = GetCgiCoordinates(cgi, CgiOrigins, true);
    const TVector<TGeoCoord>& to = GetCgiCoordinates(cgi, CgiDestinations, true);
    const TString& departureTime = GetCgiParameter(cgi, CgiDepartureTime, false);
    const ERouterMode mode = GetRouterMode(cgi);
    const bool internalRequest = IsTrue(GetCgiParameter(cgi, "internal_request", false));

    const auto& permissions = GetPermissions(mode, departureTime, internalRequest);
    const auto algorithm = GetAlgorithmDistanceMatrix(mode);
    const auto precision = GetPrecision();
    const auto avoidTolls = GetCgiParameter(cgi, CgiAvoidTolls, false);

    TString text = Sprintf("type:route_mn;from:%s;to:%s;d:%d;perm:%s;",
        TGeoCoord::SerializeVector(from).data(),
        TGeoCoord::SerializeVector(to).data(),
        precision,
        permissions.data()
    );
    if (algorithm) {
        text += Sprintf("algorithm:%s;", algorithm.data());
    }

    TQuery query;
    query.SetText(text);

    if (avoidTolls) {
        if (FromExternalString<bool>(avoidTolls, CgiAvoidTolls)) {
            query.AddPron("no_tr");
        }
    }

    const auto weight = std::max<ui64>(from.size() * to.size(), 1);
    if (weight > Config.GetMaxMatrixSize()) {
        throw TCodedException(HTTP_BAD_REQUEST) << "the routing matrix is too big: " << weight << " > " << Config.GetMaxMatrixSize();
    }
    query.AddExtraParam(CgiWeight, ToString(weight));

    return query.AddExtraParams(cgi);
}


TString TRouteProcessor::GetMode(ui32 accessType) {
    if (accessType == 0) {
        return DefaultMode.empty() ? GetMode(NGeoEdge::ptAuto) : DefaultMode;
    }
    if (accessType == NGeoEdge::ptAuto) {
        return "driving";
    }
    if (accessType == NGeoEdge::ptPedestrian) {
        return "walking";
    }
    return "transit";
}


TString TRouteProcessor::GetFeatureClass(ui32 accessType, ui32 fc) {
    if (accessType == 0) {
        if (DefaultMode.empty() || DefaultMode == "driving") {
            return GetFeatureClass(NGeoEdge::ptAuto, fc);
        } else {
            return GetFeatureClass(NGeoEdge::ptPedestrian, fc);
        }
    }
    if (accessType == NGeoEdge::ptAuto) {
        return "road." + ToString(fc);
    }
    if (accessType == NGeoEdge::ptPedestrian) {
        return "road." + ToString(fc);
    }
    if (accessType == NGeoEdge::ptSubway) {
        return "transit.subway";
    }
    {
        return "transit.bus." + ToString(TAccessWorker<TAccess>::GetAccessId(accessType));
    }
}


void TRouteProcessor::FillStepElement(NJson::TJsonValue& jsonElement, float correctionCoefficient,
    const NGraph::TBaseRouter::TRouteElement& interval, TVector<TGeoCoord>& trace, TVector<TRouteProcessor::TFastEdgeParams>& fastEdges)
{
    NJson::TJsonValue jsonPoints = NJson::JSON_ARRAY;
    for(const auto& coord: trace) {
        NJson::TJsonValue jsonPoint = NJson::JSON_ARRAY;
        jsonPoint.AppendValue(coord.Y);
        jsonPoint.AppendValue(coord.X);
        jsonPoints.AppendValue(jsonPoint);
    }

    NJson::TJsonValue jsonPolyline = NJson::JSON_MAP;
    jsonPolyline.InsertValue("points", jsonPoints);
    jsonElement.InsertValue("polyline", jsonPolyline);

    jsonElement.InsertValue("length", interval.Length);
    auto duration = interval.Time + correctionCoefficient * interval.Length;
    jsonElement.InsertValue("duration", duration);

    if (interval.Time > 1e-7 && GetSpeedKmh(interval.Length, interval.Time) > Config.GetFastEdgeSpeedLimitKmh()) {
        fastEdges.emplace_back(interval.EdgeExternalId, interval.Length, interval.Time);
    } else if (duration > 1e-7 && GetSpeedKmh(interval.Length, duration) > Config.GetFastEdgeSpeedLimitKmh()) {
        fastEdges.emplace_back(interval.EdgeExternalId, interval.Length, duration, correctionCoefficient);
    }
}


void TRouteProcessor::ReportFastEdges(const TSearchReply& searchReply, const TVector<TRouteProcessor::TFastEdgeParams>& fastLegs, const TVector<TRouteProcessor::TFastEdgeParams>& fastEdges) {
    if (!fastEdges && !fastLegs) {
        return;
    }

    auto formatEdges = [](const TVector<TRouteProcessor::TFastEdgeParams>& edges) -> TString {
        TVector<TString> strEdges;
        for (const auto& edgeParams: edges) {
            TStringStream ss;
            ss << "edgeId:" << edgeParams.EdgeId
                << "," << "l:" << edgeParams.Length
                << "," << "t:" << edgeParams.Time
                << "," << "s:" << GetSpeedKmh(edgeParams.Length, edgeParams.Time)
                << "," << "k:" << edgeParams.CorrectionCoefficient;
            strEdges.push_back(ss.Str());
        }
        return JoinSeq(", ", strEdges);
    };

    DEBUG_LOG << "FastEdges: reqid=" << searchReply.GetReqId() << ", limit " << Config.GetFastEdgeSpeedLimitKmh() << "kmh, legs: " << formatEdges(fastLegs) << Endl;
    DEBUG_LOG << "FastEdges: reqid=" << searchReply.GetReqId() << ", limit " << Config.GetFastEdgeSpeedLimitKmh() << "kmh, edges: " << formatEdges(fastEdges) << Endl;
    FastEdgeSignal.Signal(1);
}


void TRouteProcessor::ComposeReport(const TSearchReply& searchReply, TJsonReport::TGuard& g) {
    if (searchReply.GetCode() == HTTP_OK) {
        TCgiParameters cgi = Context->GetCgiParameters();

        NGraph::TRouter parser(BaseServer->GetRTLineAPI(RTLineAPIName)->GetSearchClient());
        NGraph::TBaseRouter::TRouteObject routeObject = parser.GetRoute(searchReply.GetReport());
        const TVector<NGraph::TBaseRouter::TRoute>& route = routeObject.Route;

        NJson::TJsonValue jsonRoute = NJson::JSON_ARRAY;
        NJson::TJsonValue jsonLegs = NJson::JSON_ARRAY;
        NJson::TJsonValue jsonLeg = NJson::JSON_MAP;
        if (route.ysize() > 0) {
            if (InternalRequest) {
                g.MutableReport().AddReportElement("behavior_info", NJson::TJsonValue(route[0].MetaInfo.BehaviorInfo));
            }

            TVector<TFastEdgeParams> fastEdges;
            TVector<TFastEdgeParams> fastLegs;
            for (const auto& rout: route) {
                if (!InternalRequest && rout.MetaInfo.Timeouted) {
                    throw TCodedException(HTTP_GATEWAY_TIME_OUT) << "request /route timeouted";
                }
                jsonLeg.InsertValue("status", GetStatus(rout.HasPath, rout.MetaInfo.Timeouted));

                float segmentLength = 0;
                float segmentTime = 0;
                for (auto&& edge: rout.Elements) {
                    Y_ENSURE(edge.Length >= 0);
                    segmentLength += edge.Length;
                    segmentTime += edge.Time;
                    if (edge.Time > 1e-7 && GetSpeedKmh(edge.Length, edge.Time) > Config.GetFastEdgeSpeedLimitKmh()) {
                        fastEdges.emplace_back(edge.EdgeExternalId, edge.Length, edge.Time);
                    }
                }
                float k = segmentLength > 0 ? (rout.Time - segmentTime) / segmentLength : 0;

                NJson::TJsonValue jsonSteps = NJson::JSON_ARRAY;
                NJson::TJsonValue jsonElement = NJson::JSON_MAP;
                NGraph::TBaseRouter::TRouteElement interval;
                if (rout.Elements) {
                    interval.EdgeExternalId = rout.Elements.front().EdgeExternalId;
                }
                TVector<TGeoCoord> trace;

                for (const auto& elem: rout.Elements) {
                    bool differentAccessType = interval.AccessType != elem.AccessType;
                    bool differentFC = IsTrivialAccessType(elem.AccessType) && (interval.FC != elem.FC);

                    if (differentAccessType || differentFC) {
                        if (!trace.empty()) {
                            FillStepElement(jsonElement, k, interval, trace, fastLegs);
                            jsonSteps.AppendValue(jsonElement);

                            jsonElement = NJson::JSON_MAP;
                            trace.clear();
                            interval = {};
                            interval.EdgeExternalId = elem.EdgeExternalId;
                        }

                        jsonElement.InsertValue("mode", GetMode(elem.AccessType));
                        jsonElement.InsertValue("feature_class", GetFeatureClass(elem.AccessType, elem.FC));
                        jsonElement.InsertValue("waiting_duration", elem.WaitingTime);
                        if (elem.WaitingTime > 0) {
                            DEBUG_LOG << "WaitingDuration: reqid=" << searchReply.GetReqId() << ", waiting_duration=" << elem.WaitingTime << Endl;
                            WaitingDurationSignal.Signal(1);
                        }
                    }

                    SumRouteElements(interval, trace, elem);
                }

                if (!trace.empty()) {
                    FillStepElement(jsonElement, k, interval, trace, fastLegs);
                    jsonSteps.AppendValue(jsonElement);
                }

                if (!jsonSteps.GetArray().empty()) {
                    jsonLeg.InsertValue("steps", jsonSteps);
                }
                jsonLegs.AppendValue(jsonLeg);
                jsonLeg = NJson::JSON_MAP;
            }
            g.MutableReport().AddReportElement("traffic_type", GetTrafficType(route[0].MetaInfo.BehaviorInfo));

            ReportFastEdges(searchReply, fastLegs, fastEdges);
        } else {
            jsonLeg.InsertValue("status", StatusFail);
            jsonLegs.AppendValue(jsonLeg);
        }
        jsonRoute.InsertValue("legs", jsonLegs);
        g.MutableReport().AddReportElement("route", std::move(jsonRoute));

        g.SetCode(HTTP_OK);
    } else {
        g.SetCode(searchReply.GetCode());
    }
}


TQuery TRouteProcessor::CorrectCgi(const TCgiParameters& cgi) {
    ValidateRequest(cgi);

    const TVector<TGeoCoord>& waypoints = GetCgiCoordinates(cgi, CgiWaypoints, true);
    const TString& departureTime = GetCgiParameter(cgi, CgiDepartureTime, false);
    const ERouterMode mode = GetRouterMode(cgi);
    const bool internalRequest = IsTrue(GetCgiParameter(cgi, "internal_request", false));

    const TString avoidTollsParam = GetCgiParameter(cgi, CgiAvoidTolls, false);
    bool avoidTolls = avoidTollsParam ? FromExternalString<bool>(avoidTollsParam, CgiAvoidTolls) : false;
    const auto& permissions = GetPermissions(mode, departureTime, internalRequest);
    const auto algorithm = GetAlgorithmRoute(mode, avoidTolls);
    const auto precision = GetPrecision();

    if (waypoints.size() < 2) {
        throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiWaypoints << "' contains less than 2 waypoints";
    }
    if (waypoints.size() > 50) {
        throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiWaypoints << "' contains more than 50 waypoints";
    }
    RouterModeSignal.Signal(mode, 1);
    RouterPointsSignal.Signal(mode, waypoints.size());

    const auto& from = waypoints.front();
    const auto& to = waypoints.back();

    TString text = Sprintf("type:route;from:%s;to:%s;d:%d;perm:%s;",
        from.ToString().data(),
        to.ToString().data(),
        precision,
        permissions.data()
    );
    TString middle;
    for (size_t i = 1; i < waypoints.size() - 1; ++i) {
        middle += waypoints[i].ToString() + " ";
    }
    if (middle) {
        text += Sprintf("middle:%s;", middle.data());
    }
    if (algorithm) {
        text += Sprintf("algorithm:%s;", algorithm.data());
    }
    if (mode == ERouterMode::Transit || avoidTolls) {
        text += "layer:0;";
    }

    TQuery query;
    query.SetText(text);

    if (avoidTolls) {
        query.AddPron("no_tr");
    }

    if (!cgi.Has(CgiGta, "FC")) {
        query.AddProperty("FC");
    }
    if (!cgi.Has(CgiGta, "CompressedRoute")) {
        query.AddProperty("CompressedRoute");
    }

    if (!cgi.Has(CgiPron, "report_incomplete_chain")) {
        query.AddPron("report_incomplete_chain");
    }

    return query.AddExtraParams(cgi);
}


void TMatchProcessor::ComposeReport(const TSearchReply& searchReply, TJsonReport::TGuard& g) {
    if (searchReply.GetCode() == HTTP_OK) {
        TCgiParameters cgi = Context->GetCgiParameters();
        NGraph::TRouter parser(BaseServer->GetRTLineAPI(RTLineAPIName)->GetSearchClient());

        const auto& points = GetCgiCoordinates(cgi, CgiPoints, true);
        TVector<ui64> timestamps = GetCgiArray<ui64>(cgi, CgiTimestamps, false);
        NGraph::TBaseRouter::TTimedGeoCoordinates coordinates;
        for (size_t i = 0; i < points.size(); ++i) {
            coordinates.push_back(
                NGraph::TBaseRouter::TTimedGeoCoordinate(
                    points[i],
                    TInstant::Seconds(timestamps.size() ? timestamps[i] : 0)
            ));
        }

        NGraph::TBaseRouter::TMatchingOptions options = {};
        options.FeaturePrecision = 0;
        NGraph::TBaseRouter::TMatch match = parser.GetEdges(searchReply.GetReport(), coordinates, options);

        NJson::TJsonValue jsonPoints = NJson::JSON_ARRAY;
        double timestamp = timestamps.size() ? static_cast<double>(timestamps[0]) : 0.;
        for (yssize_t elemId = 0; elemId < match.Elements.ysize(); ++elemId) {
            const auto& element = match.Elements[elemId];

            for (size_t j = 0; j < element.Projection.size(); ++j) {
                NJson::TJsonValue jsonPoint = NJson::JSON_ARRAY;
                jsonPoint.AppendValue(element.Projection[j].Y);
                jsonPoint.AppendValue(element.Projection[j].X);
                jsonPoint.AppendValue(element.Indices[j]);
                if (timestamps.size()) {
                    ui64 index = static_cast<ui64>(std::floor(element.Indices[j]));
                    jsonPoint.AppendValue(timestamps[index]);
                }

                jsonPoints.AppendValue(jsonPoint);
            }
            timestamp += element.Time;

            if (element.RawIndices.ysize() >= 2 && (element.RawIndices.back() - element.RawIndices.front() < IndicesPrecision)) {
                continue;
            }

            if (element.PolyLine.Size() > 0 && elemId < match.Elements.ysize()-1) {
                NJson::TJsonValue jsonPoint = NJson::JSON_ARRAY;
                jsonPoint.AppendValue(element.PolyLine.LastCoord().Y);
                jsonPoint.AppendValue(element.PolyLine.LastCoord().X);
                jsonPoint.AppendValue(NJson::JSON_NULL);
                if (timestamps.size()) {
                    jsonPoint.AppendValue(timestamp);
                }

                jsonPoints.AppendValue(jsonPoint);
            }
        }

        if (!jsonPoints.GetArray().empty()) {
            g.MutableReport().AddReportElement("points", std::move(jsonPoints));
        }

        g.SetCode(HTTP_OK);
    } else {
        g.SetCode(searchReply.GetCode());
    }
}


TQuery TMatchProcessor::CorrectCgi(const TCgiParameters& cgi) {
    ValidateRequest(cgi);

    const auto& points = GetCgiCoordinates(cgi, CgiPoints, true);
    TVector<double> accuracies = GetCgiArray<double>(cgi, CgiAccuracies, false);
    TVector<ui64> timestamps = GetCgiArray<ui64>(cgi, CgiTimestamps, false);
    const ERouterMode mode = GetRouterMode(cgi);

    const auto& permissions = GetMatchingPermissions(mode);

    if (points.size() < 2) {
        throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiPoints << "' contains less than 2 points";
    }
    if (points.size() > 50) {
        throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiPoints << "' contains more than 50 points";
    }

    TQuery query;
    if (timestamps.size()) {
        if (timestamps.size() != points.size()) {
            throw TCodedException(HTTP_BAD_REQUEST)
                << "parameter '" << CgiTimestamps << "' has timestamps for " << timestamps.size() << " points, "
                << "whereas '" << CgiPoints << "' has " << points.size() << " points";
        }
        query.AddExtraParam("tss", JoinSeq(",", timestamps));
    }

    ui32 d = 15;
    if (accuracies.size()) {
        constexpr double k = 1.0 * 15 / 300;
        double accuracy = 0;
        for (auto&& a : accuracies) {
            if (a <= 0) {
                throw TCodedException(HTTP_BAD_REQUEST) << "parameter '" << CgiAccuracies << "' has incorrect accuracy " << a;
            }
            accuracy += a / accuracies.size();
        }
        if (accuracies.size() != 1 && accuracies.size() != points.size()) {
            throw TCodedException(HTTP_BAD_REQUEST)
                << "parameter '" << CgiAccuracies << "' has accuracies for " << accuracies.size() << " points, "
                << "whereas '" << CgiPoints << "' has " << points.size() << " points";
        }
        d = std::max<ui32>(std::round(k * accuracy), 1);
    }

    TString text = Sprintf("type:match;c_set:%s;d:%d;perm:%s",
        TGeoCoord::SerializeVector(points).data(),
        d,
        permissions.data()
    );
    query.SetText(text);

    return query.AddExtraParams(cgi);
}


TDistanceMatrixProcessorConfig::TFactory::TRegistrator<TDistanceMatrixProcessorConfig> TDistanceMatrixProcessorConfig::Registrator("distancematrix");
TRouteProcessorConfig::TFactory::TRegistrator<TRouteProcessorConfig> TRouteProcessorConfig::Registrator("route");
TMatchProcessorConfig::TFactory::TRegistrator<TMatchProcessorConfig> TMatchProcessorConfig::Registrator("match");
