#pragma once

#include <drive/backend/processors/common_app/processor.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/api/search_client/reply.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/unistat/signals.h>


enum class ERouterMode {
    Driving /* "driving" */,
    Transit /* "transit" */,
    Walking /* "walking" */
};

class TB2BRequestProcessorConfig: public TCommonAppConfig {
private:
    using TBase = TCommonAppConfig;

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;

public:
    using TBase::TBase;

    virtual void ToString(IOutputStream& os) const override;

    const TString& GetRTLineAPIName(const TCgiParameters& cgi) const;

    TDuration GetRTLineAPITimeout() const {
        return RTLineAPITimeout;
    }

    TDuration GetRTLineAPIRealtimeTimeout() const {
        return RTLineAPIRealtimeTimeout;
    }

private:
    TString RTLineAPIName;
    TString RTLineAPINameTransport;
    TString RTLineAPINameWalking;
    TDuration RTLineAPITimeout = TDuration::Seconds(5);
    TDuration RTLineAPIRealtimeTimeout = TDuration::Seconds(1);
};


class TB2BRequestProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    TB2BRequestProcessor(const TB2BRequestProcessorConfig *config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server)
        : TBase(*config, context, authModule, server)
        , InternalRequest(false)
        , Config(*config)
    {
    }

    void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

protected:
    virtual NRTLine::TQuery CorrectCgi(const TCgiParameters& cgi) = 0;
    virtual void ComposeReport(const NRTLine::TSearchReply& searchReply, TJsonReport::TGuard& g) = 0;

    void CorrectPriority(const TCgiParameters& cgi, NRTLine::TQuery& query);
    TString GetStatus(bool isOk, bool timeouted) const;

protected:
    bool InternalRequest;
    TString RTLineAPIName;

private:
    const TB2BRequestProcessorConfig& Config;
};


class TDistanceMatrixProcessorConfig: public TB2BRequestProcessorConfig {
private:
    using TBase = TB2BRequestProcessorConfig;
protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;
public:
    using TBase::TBase;

    virtual void ToString(IOutputStream& os) const override;

    ui32 GetMaxMatrixSize() const {
        return MaxMatrixSize;
    }

private:
    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override;

private:
    static TFactory::TRegistrator<TDistanceMatrixProcessorConfig> Registrator;

    ui32 MaxMatrixSize = 100;
};


class TDistanceMatrixProcessor: public TB2BRequestProcessor {
public:
    TDistanceMatrixProcessor(const TDistanceMatrixProcessorConfig* config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server)
        : TB2BRequestProcessor(config, context, authModule, server)
        , Config(*config)
    {
    }

protected:
    virtual NRTLine::TQuery CorrectCgi(const TCgiParameters& cgi) override;
    virtual void ComposeReport(const NRTLine::TSearchReply& searchReply, TJsonReport::TGuard& g) override;

private:
    const TDistanceMatrixProcessorConfig& Config;
};


class TRouteProcessorConfig: public TB2BRequestProcessorConfig {
private:
    using TBase = TB2BRequestProcessorConfig;

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;
public:
    using TBase::TBase;

    virtual void ToString(IOutputStream& os) const override;

    double GetFastEdgeSpeedLimitKmh() const {
        return FastEdgeSpeedLimitKmh;
    }
private:
    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override;

private:
    static TFactory::TRegistrator<TRouteProcessorConfig> Registrator;

    double FastEdgeSpeedLimitKmh = 1000;
};


namespace {
    TNamedSignalSimple FastEdgeSignal("fast_edge");
    TNamedSignalSimple WaitingDurationSignal("waiting_duration");
}


class TRouteProcessor: public TB2BRequestProcessor {
private:
    struct TFastEdgeParams {
        ui64 EdgeId;
        double Length;
        double Time;
        double CorrectionCoefficient;

        TFastEdgeParams() {}
        TFastEdgeParams(ui64 edgeId, double length, double time, double k=0)
            : EdgeId(edgeId)
            , Length(length)
            , Time(time)
            , CorrectionCoefficient(k)
        {}
    };

    TEnumSignal<ERouterMode, double> RouterModeSignal;
    const NUnistat::TIntervals IntervalsRouterPoints = { 2, 3, 5, 10, 20, 30, 50 };
    TEnumSignal<ERouterMode, double> RouterPointsSignal;

public:
    TRouteProcessor(const TRouteProcessorConfig* config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server)
        : TB2BRequestProcessor(config, context, authModule, server)
        , RouterModeSignal({ "frontend-mode" }, false)
        , RouterPointsSignal({ "frontend-router_points" }, IntervalsRouterPoints)
        , Config(*config)
    {
    }

protected:
    virtual NRTLine::TQuery CorrectCgi(const TCgiParameters& cgi) override;
    virtual void ComposeReport(const NRTLine::TSearchReply& searchReply, TJsonReport::TGuard& g) override;

private:
    TString GetMode(ui32 accessType);
    TString GetFeatureClass(ui32 accessType, ui32 fc);
    void FillStepElement(NJson::TJsonValue& jsonElement, float correctionCoefficient, const NGraph::TBaseRouter::TRouteElement& interval, TVector<TGeoCoord>& trace, TVector<TRouteProcessor::TFastEdgeParams>& fastEdges);
    void ReportFastEdges(const NRTLine::TSearchReply& searchReply, const TVector<TRouteProcessor::TFastEdgeParams>& fastLegs, const TVector<TRouteProcessor::TFastEdgeParams>& fastEdges);
    static double GetSpeedKmh(double length, double time) {
        return 3.6 * length / time;
    }
private:
    const TRouteProcessorConfig& Config;
    TString DefaultMode;
};


class TMatchProcessorConfig: public TB2BRequestProcessorConfig {
private:
    using TBase = TB2BRequestProcessorConfig;
protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;
public:
    using TBase::TBase;

    virtual void ToString(IOutputStream& os) const override;

private:
    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override;

private:
    static TFactory::TRegistrator<TMatchProcessorConfig> Registrator;

};


class TMatchProcessor: public TB2BRequestProcessor {
public:
    TMatchProcessor(const TMatchProcessorConfig* config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server)
        : TB2BRequestProcessor(config, context, authModule, server)
        , Config(*config)
    {
    }

protected:
    virtual NRTLine::TQuery CorrectCgi(const TCgiParameters& cgi) override;
    virtual void ComposeReport(const NRTLine::TSearchReply& searchReply, TJsonReport::TGuard& g) override;

private:
    TString GetFeatureClass(ui32 accessType, ui32 fc);

private:
    const TMatchProcessorConfig& Config;
    TString DefaultMode;
};
