#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/covid_pass/client.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <util/draft/date.h>

void TCheckPassProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto pass = GetString(cgi, "pass");
    const auto until = GetTimestamp(cgi, "until", Now());

    TSet<EPassApiType> types;
    try {
        TString apiSettings = Server->GetSettings().GetValueDef<TString>("covid_pass.api_types", "moscow");
        StringSplitter(apiSettings).SplitBySet(", \n\r").SkipEmpty().ParseInto(&types);
    } catch(...) {
        R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect covid_pass.api_types");
    }

    TMap<EPassApiType, NThreading::TFuture<TCovidPassClient::TPassData>> passes;
    for (auto&& type : types) {
        auto client = Server->GetCovidPassClient(type);
        if (!client) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "NullClient")
                ("type", ToString(type))
            );
            continue;
        }
        passes.emplace(type, client->GetPassData(pass, until));
    }

    bool valid = false;
    bool hasErrors = false;
    for (auto&& [type, asyncPassData] : passes) {
        asyncPassData.Wait(Context->GetRequestDeadline());
        g.AddEvent(NJson::TMapBuilder
            ("event", "CovidPassCheckResult")
            ("type", ToString(type))
            ("pass", pass)
            ("response", NJson::ToJson(asyncPassData))
        );
        if (asyncPassData.HasValue()) {
            valid = valid || asyncPassData.GetValue().GetIsValid();
        } else {
            hasErrors = true;
        }
    }
    if (!hasErrors || valid) {
        g.AddReportElement("valid", valid);
    } else {
        g.AddReportElement("valid", NJson::JSON_NULL);
    }
    g.SetCode(HTTP_OK);
}

namespace {
    struct TPass {
        TString CarNumber;
        TString CarId;
        TString PassCode;
        TInstant CheckTimestamp;
        bool Valid;
    };
    using TPassKey = std::pair<TString, TString>;
}

void TRnisPassCheckProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    Y_UNUSED(permissions);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto dateString = GetString(cgi, "date");
    auto date = TDate(dateString, "%Y-%m-%d");
    auto dateStart = TInstant::Seconds(date.GetStart());
    auto dateFinish = dateStart + TDuration::Days(1);

    const auto tagName = GetHandlerSetting<TString>("pass_tag_name").GetOrElse(TUserIsolationPassTag::TypeName);
    R_ENSURE(tagName, ConfigHttpStatus.NotImplementedState, "incorrect tag name: " << tagName);

    const auto& userTagManager = Server->GetDriveDatabase().GetTagsManager().GetUserTags();
    auto session = BuildTx<NSQL::ReadOnly>();

    TVector<TDBTag> tags;
    R_ENSURE(userTagManager.RestoreTags({}, { tagName }, tags, session), ConfigHttpStatus.UnknownErrorStatus, "cannot RestoreTags", session);

    TVector<TString> userIds;
    g.AddEvent(NJson::TMapBuilder
        ("event", "AcquiredTags")
        ("count", tags.size())
    );
    for (auto&& tag : tags) {
        {
            if (!tag || tag->GetName() != tagName) {
                continue;
            }
            auto passTag = tag.GetTagAs<TUserIsolationPassTag>();
            if (!passTag) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "BadTagCast")
                    ("tag_id", tag.GetTagId())
                );
                continue;
            }

            if (passTag->GetValidationTime() < dateStart) {
                continue;
            }
            userIds.push_back(tag.GetObjectId());
        }
    }
    g.AddEvent(NJson::TMapBuilder
        ("event", "AcquiredCandidates")
        ("user_ids", NJson::ToJson(userIds))
    );

    TMap<TPassKey, TPass> passes;
    for (auto&& userId : userIds) {
        auto optionalEvents = userTagManager.GetEventsByObject(userId, session, 0, dateStart);
        if (!optionalEvents) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "GetEventsFailure")
                ("user_id", userId)
            );
            continue;
        }
        for (auto&& ev : *optionalEvents) {
            if (!ev) {
                continue;
            }
            if (ev->GetName() != tagName) {
                continue;
            }
            if (ev.GetHistoryInstant() < dateStart) {
                continue;
            }
            if (ev.GetHistoryInstant() > dateFinish) {
                break;
            }

            auto passTag = ev.GetTagAs<TUserIsolationPassTag>();
            if (!passTag) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "BadTagEventCast")
                    ("tag_id", ev.GetTagId())
                );
                continue;
            }

            auto carObject = Server->GetDriveDatabase().GetCarManager().GetObject(passTag->GetCarId());
            if (!carObject) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "BadCarId")
                    ("car_id", passTag->GetCarId())
                );
                continue;
            }

            TPass pass;
            pass.CarNumber = carObject->GetNumber();
            pass.CarId = carObject->GetIMEI();
            pass.CheckTimestamp = ev.GetHistoryInstant();
            pass.PassCode = passTag->GetToken();
            pass.Valid = passTag->IsValid();
            auto key = std::make_pair(pass.CarNumber, pass.PassCode);
            passes[key] = std::move(pass);
        }
    }

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (auto&& [key, pass] : passes) {
        NJson::TJsonValue p;
        p["reg_number"] = pass.CarNumber;
        p["attid"] = pass.CarId;
        p["digital_code"] = pass.PassCode;
        p["check_time"] = (pass.CheckTimestamp + TDuration::Hours(3)).FormatGmTime("%Y-%m-%d %H:%M:%S");
        p["check_result"] = pass.Valid ? 1 : 0;
        report.AppendValue(std::move(p));
    }

    g.SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}
