#include "processor.h"

#include <drive/backend/billing/accounts/bonus.h>
#include <drive/backend/billing/trust/charge_logic.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/game/logic.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tag_modification.h>
#include <drive/backend/tags/tags_filter.h>

using TPaymentsMap = TMap<TString, TCachedPayments>;
using TRefundsMap = TMap<TString, TVector<TRefundTask>>;
using TOptionalPaymentsRefunds = TMaybe<std::pair<TPaymentsMap, TRefundsMap>>;

bool GetPaymentsRefunds(const TBillingManager& billingManager,  const TSet<TString>& sessionIds, TPaymentsMap& payments, TRefundsMap& refunds, NDrive::TEntitySession& session) {
    auto sessions = MakeVector(sessionIds);
    return billingManager.GetPaymentsManager().GetPayments(payments, sessions, session, false)
        && billingManager.GetPaymentsManager().GetRefundsDB().GetSessionsRefunds(sessions, refunds, session);
}

void TGetUserPaymentsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Payment);
    const TBillingManager& billingManager = DriveApi->GetBillingManager();

    const TString& sessionId = Context->GetCgiParameters().Get("session_id");
    const TString& userId = Context->GetCgiParameters().Get("user_id");
    bool showAll = IsTrue(Context->GetCgiParameters().Get("details"));

    TInstant until = ModelingNow();
    TInstant since = until - TDuration::Days(3);

    since = GetTimestamp(Context->GetCgiParameters(), "since", since);
    until = GetTimestamp(Context->GetCgiParameters(), "until", until);

    R_ENSURE(since < until, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect interval");

    TMaybe<TVector<TBillingTask>> billingTasks;
    TSet<TString> sessionIds;
    TVector<TBillingHistoryEvent> ticketsHistory;
    {
        TMaybe<TVector<TBillingHistoryEvent>> historyTasks;
        auto session = BuildTx<NSQL::ReadOnly>();
        if (!!userId) {
            billingTasks = billingManager.GetActiveTasksManager().GetUsersTasks(NContainer::Scalar(userId), session);
            R_ENSURE(billingTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing tasks: " + session.GetStringReport());
            historyTasks = billingManager.GetHistoryManager().GetUserHistory(userId, {since, until}, session);
            R_ENSURE(historyTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing history: " + session.GetStringReport());
        } else {
            R_ENSURE(sessionId , ConfigHttpStatus.UserErrorState, "One of 'session_id' or 'user_id' should present");

            billingTasks = billingManager.GetActiveTasksManager().GetSessionsTasks(NContainer::Scalar(sessionId), session);
            R_ENSURE(billingTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing tasks: " + session.GetStringReport());
            if (billingTasks->empty()) {
                historyTasks = billingManager.GetHistoryManager().GetSessionHistory(sessionId, {since, until}, session);
                R_ENSURE(historyTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing history: " + session.GetStringReport());
            } else {
                historyTasks.ConstructInPlace();
            }

            TString taskUserId;
            if (!billingTasks->empty()) {
                taskUserId = billingTasks->front().GetUserId();
            } else if (!historyTasks->empty()) {
                taskUserId = historyTasks->front().GetUserId();
            }
            auto realSessionHistoryTasks = billingManager.GetHistoryManager().GetRealSessionHistory(sessionId, taskUserId, {since, until}, session);
            R_ENSURE(realSessionHistoryTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing history tasks by real session: " + session.GetStringReport());
            TSet<TString> historySessionsIds;
            for (auto&& task : *realSessionHistoryTasks) {
                if (!historySessionsIds.contains(task.GetId()) && task.GetId() != sessionId) {
                    historySessionsIds.insert(task.GetId());
                }
            }

            auto realSessionBillingTasks = billingManager.GetActiveTasksManager().GetSessionsTasks(MakeVector(historySessionsIds), session);
            R_ENSURE(realSessionBillingTasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing tasks by real session: " + session.GetStringReport());

            TSet<TString> foundSessionIds;
            for (auto&& task : *realSessionBillingTasks) {
                foundSessionIds.insert(task.GetId());
                billingTasks->push_back(std::move(task));
            }
            for (auto&& task : *realSessionHistoryTasks) {
                if (!foundSessionIds.contains(task.GetId())) {
                    historyTasks->emplace_back(std::move(task));
                }
            }
        }

        for (auto&& task : *historyTasks) {
            if (sessionIds.contains(task.GetId())) {
                continue;
            }
            sessionIds.emplace(task.GetId());
            ticketsHistory.emplace_back(std::move(task));
        }
    }


    TSet<TString> activeTasks;
    for (auto&& task : *billingTasks) {
        activeTasks.emplace(task.GetId());
    }

    const bool showPayments = showAll && permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Payment);
    NJson::TJsonValue report;
    {
        NJson::TJsonValue& current = report["current"];

        TPaymentsMap payments;
        TRefundsMap refunds;
        {
            auto session = BuildTx<NSQL::ReadOnly>();
            if (showPayments && !GetPaymentsRefunds(billingManager, activeTasks, payments, refunds, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        for (auto&& task : *billingTasks) {
            NJson::TJsonValue taskReport = task.SerializeToJson();
            auto itPayments = payments.find(task.GetId());
            auto itRefunds = refunds.find(task.GetId());
            NJson::TJsonValue paymentsReport = GetPaymentsReport(itPayments != payments.end() ? itPayments->second.GetTimeline() : TVector<TPaymentTask>(), itRefunds != refunds.end() ? itRefunds->second : TVector<TRefundTask>());
            taskReport.InsertValue("payments", std::move(paymentsReport));
            current.AppendValue(std::move(taskReport));
        }
    }

    NJson::TJsonValue& compiledHistory = report["compiled_history"];
    const auto& compiledBills = billingManager.GetCompiledBills();
    TVector<TCompiledBill> events;
    if (userId) {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto bills = compiledBills.GetUserFullBillsFromDB(userId, since, until, session);
        R_ENSURE(bills, ConfigHttpStatus.UnknownErrorStatus, "cannot GetEvents from IndexByUser(" << userId << ")", session);
        events = MakeVector(NContainer::Values(*bills));
    } else {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto bill = compiledBills.GetFullBillFromDB(sessionId, session);
        if (bill) {
            events.emplace_back(std::move(*bill));
        }
    }

    TSet<TString> compiledTickets;
    for (auto&& bill : events) {
        if (sessionIds.contains(bill.GetSessionId())) {
            compiledTickets.emplace(bill.GetSessionId());
        } else {
            sessionIds.emplace(bill.GetSessionId());
        }
    }

    TPaymentsMap payments;
    TRefundsMap refunds;
    {
        auto session = billingManager.BuildSession(true);
        if (showAll && !GetPaymentsRefunds(billingManager, sessionIds, payments, refunds, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    auto ticketsIt =  ticketsHistory.begin();
    for (auto&& bill : events) {
        if (!bill.GetFinal()) {
            continue;
        }
        if (activeTasks.contains(bill.GetSessionId())) {
            continue;
        }
        while (ticketsIt != ticketsHistory.end() && ticketsIt->GetHistoryInstant() < bill.GetFinalTime()) {
            if (activeTasks.contains(ticketsIt->GetId())) {
                ++ticketsIt;
                continue;
            }
            if (!compiledTickets.contains(ticketsIt->GetId())) {
                NJson::TJsonValue compiledReport;
                BuildReport(*ticketsIt, compiledReport, payments, refunds);
                compiledHistory.AppendValue(compiledReport);
            }
            ++ticketsIt;
        }
        NJson::TJsonValue compiledReport;
        BuildReport(bill, compiledReport, payments, refunds);
        compiledHistory.AppendValue(compiledReport);
    }
    while (ticketsIt != ticketsHistory.end()) {
        if (activeTasks.contains(ticketsIt->GetId())) {
            ++ticketsIt;
            continue;
        }
        if (!compiledTickets.contains(ticketsIt->GetId())) {
            NJson::TJsonValue compiledReport;
            BuildReport(*ticketsIt, compiledReport, payments, refunds);
            compiledHistory.AppendValue(compiledReport);
        }
        ++ticketsIt;
    }

    if (userId && GetHandlerSetting<bool>("same_persons").GetOrElse(false)) {
        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TString> personsIds;
        {
            auto allPersons = DriveApi->GetUsersData()->GetSamePersons(userId, session);
            if (!allPersons) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            Transform(allPersons->begin(), allPersons->end(), std::back_inserter(personsIds), [](const auto& person) { return person.GetUserId(); });
        }

        auto tasks = billingManager.GetActiveTasksManager().GetUsersTasks(personsIds, session);
        R_ENSURE(tasks.Defined(), ConfigHttpStatus.UnknownErrorStatus, "error while retrieving billing tasks: " + session.GetStringReport());


        TSet<TString> sameSessionIds;
        for (const auto& task : *tasks) {
            sameSessionIds.emplace(task.GetId());
        }

        TPaymentsMap pays;
        TRefundsMap refs;
        if (showAll && !GetPaymentsRefunds(billingManager, sameSessionIds, pays, refs, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        NJson::TJsonValue& samePersons = report["same_persons"];
        for (auto&& task : *tasks) {
            NJson::TJsonValue taskReport = task.SerializeToJson();
            auto ptrPayments = pays.FindPtr(task.GetId());
            auto ptrRefunds = refs.FindPtr(task.GetId());
            NJson::TJsonValue paymentsReport = GetPaymentsReport(ptrPayments ? ptrPayments->GetTimeline() : TVector<TPaymentTask>(), ptrRefunds ? *ptrRefunds : TVector<TRefundTask>());
            taskReport.InsertValue("payments", std::move(paymentsReport));
            samePersons.AppendValue(std::move(taskReport));
        }
    }

    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TRunPaymentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Payment);

    TString sessionId = GetString(requestData, "session_id", false);
    TString userId = GetString(requestData, "user_id", false);
    TString paymethod = GetString(requestData, "paymethod", false);
    TString debitingUser = GetString(requestData, "debiting_user", false);
    R_ENSURE(sessionId || !!userId, ConfigHttpStatus.UserErrorState, "Need in 'session_id' or 'user_id'");

    TBillingTask::TExecContext context;
    context.SetPaymethod(paymethod);
    context.SetUserId(debitingUser);
    auto session = BuildTx<NSQL::Writable>();
    if (sessionId) {
        R_ENSURE(DriveApi->GetBillingManager().GetActiveTasksManager().BoostSession(sessionId, permissions->GetUserId(), context, session), ConfigHttpStatus.UnknownErrorStatus, session.GetStringReport());
    } else {
        R_ENSURE(DriveApi->GetBillingManager().GetActiveTasksManager().BoostUserSessions(userId, permissions->GetUserId(), context, session), ConfigHttpStatus.UnknownErrorStatus, session.GetStringReport());
    }
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    g.SetCode(HTTP_OK);
}

void TCancelPaymentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Payment);

    TString sessionId = GetString(requestData, "session_id", true);
    TString comment = GetString(requestData, "comment", true);
    auto session = BuildTx<NSQL::Writable>();
    session.SetComment(comment);

    if (!DriveApi->GetBillingManager().GetActiveTasksManager().CancelTask(sessionId, permissions->GetUserId(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!DriveApi->GetBillingManager().GetActiveTasksManager().BoostSession(sessionId, permissions->GetUserId(), TBillingTask::TExecContext(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

auto GetPaymenthodsReportBuilder(NJson::TJsonValue&& userReport, bool cardsOnly) {
    return [userReport = std::move(userReport), cardsOnly](const NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>& r) -> NJson::TJsonValue {
        TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>> cards = r.GetValue();
        if (cardsOnly) {
            cards = TBillingManager::GetUserPaymentCards(cards.Get());
        }
        if (!cards) {
            ythrow yexception() << "cannot fetch cards";
        }
        NJson::TJsonValue cardsJson(NJson::JSON_ARRAY);
        for (auto&& card : *cards) {
            auto cardReport = card.GetReport();
            if (userReport.IsDefined()) {
                cardReport["user"] = userReport;
            }
            cardsJson.AppendValue(std::move(cardReport));
        }
        return cardsJson;
    };
}

void TTrustInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Payment);

    const TString paymentId = GetString(Context->GetCgiParameters(), "payment_id", false);
    const TString userId = GetString(Context->GetCgiParameters(), "user_id", false);
    const bool cardsOnly = GetValue<bool>(Context->GetCgiParameters(), "cards_only", false).GetOrElse(false);
    R_ENSURE(paymentId || !!userId, ConfigHttpStatus.UserErrorState, "Need in 'payment_id' or 'user_id'");
    const TString terminal = GetString(Context->GetCgiParameters(), "terminal", false);

    EBillingType billingType = EBillingType::CarUsage;
    R_ENSURE(!terminal || TryFromString(terminal, billingType), ConfigHttpStatus.SyntaxErrorStatus, "Incorrect 'terminal'");

    NJson::TJsonValue report;
    if (userId) {
        auto userPermissions = DriveApi->GetUserPermissions(userId, TUserPermissionsFeatures());
        R_ENSURE(userPermissions, ConfigHttpStatus.UnknownErrorStatus, "Can't construct permissions for user '" + userId + "'");
        auto userCards = DriveApi->GetUserPaymentMethods(*userPermissions, *Server, false).Apply(GetPaymenthodsReportBuilder(NJson::JSON_NULL, cardsOnly));
        TVector<NThreading::TFuture<NJson::TJsonValue>> paymethodReports;
        if (GetHandlerSetting<bool>("same_persons").GetOrElse(false)) {
            auto session = BuildTx<NSQL::ReadOnly>();
            auto allPersons = DriveApi->GetUsersData()->GetSamePersons(userId, session);
            if (!allPersons) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }

            for (auto&& person : *allPersons) {
                auto userPermissions = DriveApi->GetUserPermissions(person.GetUserId(), TUserPermissionsFeatures());
                R_ENSURE(userPermissions, ConfigHttpStatus.UnknownErrorStatus, "Can't construct permissions for user '" + person.GetUserId() + "'");
                paymethodReports.push_back(DriveApi->GetUserPaymentMethods(*userPermissions, *Server, false).Apply(GetPaymenthodsReportBuilder(person.GetReport(), cardsOnly)));
            }
        }
        auto sameUserCards = NThreading::WaitExceptionOrAll(paymethodReports);
        auto processorReport = g.GetReport();
        g.Release();
        NThreading::WaitExceptionOrAll(sameUserCards, userCards.IgnoreResult()).Subscribe([paymethodReports, userCards, processorReport, report = std::move(report)](const NThreading::TFuture<void>& waiter) mutable {
            TJsonReport::TGuard g(processorReport, HttpCodes::HTTP_OK);
            if (!waiter.HasValue()) {
                g.AddReportElement("error", NThreading::GetExceptionMessage(waiter));
                g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            } else {
                auto& samePersonCardsJson = report.InsertValue("same_persons", NJson::JSON_ARRAY);
                for (auto&& cards : paymethodReports) {
                    for (auto&& card : cards.GetValue().GetArray()) {
                        samePersonCardsJson.AppendValue(std::move(card));
                    }
                }
                auto& cardsJson = report.InsertValue("cards", NJson::JSON_ARRAY);
                for (auto&& card : userCards.GetValue().GetArray()) {
                    cardsJson.AppendValue(std::move(card));
                }
                g.MutableReport().SetExternalReport(std::move(report));
            }
        });
        return;
    } else {
        auto logic = DriveApi->GetBillingManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
        R_ENSURE(logic, ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
        const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
        R_ENSURE(trustLogic, ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
        trustLogic->GetTrustInfo(billingType, paymentId, report);
    }
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TGlobalRefundProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("dry_run", "Только чтение").SetDefault(true);
    scheme.Add<TFSNumeric>("pack_size", "Размер пакета").SetDefault(50);
    scheme.Add<TFSString>("process_name", "Название инцидента").SetRequired(true);
    scheme.Add<TFSNumeric>("since", "Время начала").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSNumeric>("until", "Время окончания").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSNumeric>("since_start", "Учитывать сессии не раньше").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("until_finish", "Учитывать сессии не позже").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSString>("offer_filter", "Фильтр офферов").SetRequired(true);
    scheme.Add<TFSBoolean>("add_refunded", "Добавить уже возвращенные(только для чтения)").SetDefault(true);
    scheme.Add<TFSBoolean>("add_active", "Добавить неоплаченные(только для чтения)").SetDefault(true);

    scheme.Add<TFSNumeric>("max_mileage", "Максимальные пробег");
    scheme.Add<TFSNumeric>("bonus_sum", "Сумма бонусов(если указан бонусные тег)");
    scheme.Add<TFSBoolean>("skip_empty", "Игнорировать нулевые").SetDefault(true);

    auto tagNames = NContainer::Keys(server->GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User));
    scheme.Add<TFSVariants>("compensation_tags", "Теги компенсации").SetVariants(tagNames).SetMultiSelect(true);
    scheme.Add<TFSVariants>("add_tags", "Какие теги навешать").SetVariants(tagNames).SetMultiSelect(true);

    scheme.Add<TFSString>("session_id", "Фиксированная сессия");
    return scheme;
}

void TGlobalRefundProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Payment);
    TString processName = GetString(requestData, "process_name", true);
    TInstant since = GetTimestamp(requestData, "since", true).GetRef();
    TInstant sinceStart = GetTimestamp(requestData, "since_start", false).GetOrElse(TInstant::Zero());
    TInstant until = GetTimestamp(requestData, "until", true).GetRef();
    TInstant untilFinish = GetTimestamp(requestData, "until_finish", false).GetOrElse(TInstant::Zero());
    TString sessionId = GetString(requestData, "session_id", false);
    TString offerFilterStr = GetString(requestData, "offer_filter", true);
    TTagsFilter offerFilter = TTagsFilter::BuildFromString(offerFilterStr);
    TVector<TString> compensationTags = GetStrings(requestData, "compensation_tags", false);
    TVector<TString> addTags = GetStrings(requestData, "add_tags", false);
    TMaybe<ui32> maxMileage = GetValue<ui32>(requestData, "max_mileage", false);
    TMaybe<ui32> bonusSum = GetValue<ui32>(requestData, "bonus_sum", false);
    bool skipEmpty = GetValue<bool>(requestData, "skip_empty", false).GetOrElse(true);
    if (!compensationTags) {
        compensationTags = addTags;
    }
    bool dryRun = GetValue<bool>(requestData, "dry_run", false).GetOrElse(true);
    bool addRefunded = GetValue<bool>(requestData, "add_refunded", false).GetOrElse(true);
    bool addActive = GetValue<bool>(requestData, "add_active", false).GetOrElse(true);
    ui32 packSize = GetValue<ui32>(requestData, "pack_size", false).GetOrElse(50);

    THistoryRidesContext sessionsContext(*Server, since);
    TSet<TString> alreadyCompensation;

    R_ENSURE(DriveApi->HasBillingManager(), ConfigHttpStatus.UnknownErrorStatus, "billing manager undefined");
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("global_refund");
        if (sessionId) {
            R_ENSURE(sessionsContext.InitializeSession(sessionId, tx, ydbTx), {}, "cannot InitializeSession " << sessionId, tx);
        } else {
            R_ENSURE(sessionsContext.Initialize(tx, ydbTx, until), {}, "cannot Initialize until " << until, tx);
        }

        if (compensationTags) {
            auto queryOptions = TTagEventsManager::TQueryOptions();
            queryOptions.SetTags(MakeSet(compensationTags));
            auto optionalEvents = DriveApi->GetTagsManager().GetUserTags().GetEvents({since}, tx, queryOptions);
            R_ENSURE(optionalEvents, {}, "cannot GetEvents since " << since, tx);
            for (const auto& ev : *optionalEvents) {
                alreadyCompensation.insert(ev.GetObjectId());
            }
        }
    }
    bool hasMore = true;
    auto rides = sessionsContext.GetSessions(until, 100000, &hasMore, skipEmpty);
    THistoryRideObject::FetchFullRiding(Server, rides);

    TMap<TString, TSet<TString>> offerTags;
    TMap<TString, THistoryRideObject> filtredRides;
    for (const auto& ride : rides) {
        if (ride.IsActive()) {
            continue;
        }
        if (sinceStart && ride.GetStartTS() < sinceStart) {
            continue;
        }
        if (untilFinish && ride.GetLastTS() > untilFinish) {
            continue;
        }
        if (alreadyCompensation.contains(ride.GetUserId())) {
            continue;
        }
        auto offer = ride.GetOffer();
        if (!offer) {
            continue;
        }
        auto it = offerTags.find(offer->GetBehaviourConstructorId());
        if (it == offerTags.end()) {
            auto action = DriveApi->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
            auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
            if (!offerBuilder) {
                continue;
            }
            it = offerTags.emplace(offer->GetBehaviourConstructorId(), offerBuilder->GetGrouppingTags()).first;
        }
        if (it == offerTags.end()) {
            continue;
        }
        if (!offerFilter.IsMatching(it->second)) {
            continue;
        }
        if (maxMileage) {
            auto dist = ride.GetDistance();
            if (!dist || *dist > *maxMileage) {
                continue;
            }
        }
        auto sessionId = ride.GetSessionId();
        filtredRides.emplace(sessionId, std::move(ride));
    }

    TMap<TString, TVector<TRefundTask>> refunds;
    if (!dryRun || !addRefunded) {
        auto tx = BuildTx<NSQL::ReadOnly>();
        R_ENSURE(
            DriveApi->GetBillingManager().GetPaymentsManager().GetRefundsDB().GetSessionsRefunds(MakeVector(NContainer::Keys(filtredRides)), refunds, tx),
            {},
            "cannot GetSessionsRefunds",
            tx
        );
    }

    TMap<TString, TBillingTask> active;
    if (!dryRun || !addActive) {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto optionalActive = DriveApi->GetBillingManager().GetActiveTasksManager().GetSessionsTasks(MakeVector(NContainer::Keys(filtredRides)), tx);
        R_ENSURE(optionalActive, {}, "cannot GetSessionsTags", tx);
        for (const auto& task : *optionalActive) {
            auto id = task.GetId();
            active.emplace(id, std::move(task));
        }
    }

    ui64 totalSum = 0;
    ui64 promocodeSum = 0;
    ui64 count = 0;

    TVector<ITag::TPtr> commonUserTags;
    for (const auto& tagName : addTags) {
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName, processName);
        if (bonusSum) {
            auto operationTag = dynamic_cast<TOperationTag*>(tag.Get());
            if (operationTag) {
                operationTag->SetAmount(*bonusSum);
                promocodeSum += *bonusSum;
            }
        }
        auto fixedSumTag = dynamic_cast<TFixedSumTag*>(tag.Get());
        if (fixedSumTag) {
            auto description = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
            if (description) {
                auto descriptionImpl = dynamic_cast<const TFixedBillingTagDescription*>(description.Get());
                if (descriptionImpl) {
                    promocodeSum += descriptionImpl->GetAmount();
                }
            }
        }
        commonUserTags.emplace_back(std::move(tag));
    }

    NJson::TJsonValue sessions(NJson::JSON_ARRAY);
    NJson::TJsonValue failedSessions(NJson::JSON_ARRAY);
    NJson::TJsonValue::TArray currentSessions;

    NDrive::TEntitySession tx;
    if (!dryRun) {
        tx = BuildTx<NSQL::Writable>();
    }
    for (const auto& [sessionId, ride] : filtredRides) {
        if (!dryRun && currentSessions.size() >= packSize) {
            if (!tx.Commit()) {
                failedSessions.GetArraySafe().insert(failedSessions.GetArraySafe().end(), currentSessions.begin(), currentSessions.end());
            } else {
                sessions.GetArraySafe().insert(sessions.GetArraySafe().end(), currentSessions.begin(), currentSessions.end());
            }
            currentSessions.clear();
            tx.ClearErrors();
            tx = BuildTx<NSQL::Writable>();
        }
        auto refundIt = refunds.find(sessionId);
        bool wasRefund = false;
        if (refundIt != refunds.end()) {
            for (const auto& refund : refundIt->second) {
                if (refund.GetRealRefund()) {
                    wasRefund = true;
                    break;
                }
            }
        }
        if (wasRefund) {
            continue;
        }

        auto activeIt = active.find(sessionId);
        if (activeIt != active.end()) {
            continue; // TODO: remove debt too
        }
        count++;

        NJson::TJsonValue report;
        report.InsertValue("session_id", ride.GetSessionId());
        report.InsertValue("start_ts", ride.GetStartTS().Seconds());
        report.InsertValue("finish_ts", ride.GetLastTS().Seconds());
        report.InsertValue("user_id", ride.GetUserId());
        report.InsertValue("object_id", ride.GetObjectId());
        report.InsertValue("total_sum", ride.GetSumPrice());

        for (const auto& tag : commonUserTags) {
            auto userProblemTag = dynamic_cast<TUserProblemTag*>(tag.Get());
            if (userProblemTag) {
                userProblemTag->SetSessionId(ride.GetSessionId());
            }
            auto refundTag = dynamic_cast<TRefundTag*>(tag.Get());
            if (refundTag) {
                refundTag->SetAmount(ride.GetSumPrice());
                totalSum += ride.GetSumPrice();
            }
        }

        if (!dryRun && commonUserTags) {
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTags(commonUserTags, permissions->GetUserId(), ride.GetUserId(), Server, tx)) {
                report.InsertValue("error", tx.GetReport());
                failedSessions.AppendValue(std::move(report));
                if (currentSessions) {
                    failedSessions.GetArraySafe().insert(failedSessions.GetArraySafe().end(), currentSessions.begin(), currentSessions.end());
                    currentSessions.clear();
                }
                tx.ClearErrors();
                tx = BuildTx<NSQL::Writable>();
                continue;
            }
            currentSessions.push_back(std::move(report));
        } else {
            currentSessions.push_back(std::move(report));
        }
    }
    if (!tx.Commit()) {
        failedSessions.GetArraySafe().insert(failedSessions.GetArraySafe().end(), currentSessions.begin(), currentSessions.end());
    } else {
        sessions.GetArraySafe().insert(sessions.GetArraySafe().end(), currentSessions.begin(), currentSessions.end());
    }
    currentSessions.clear();
    tx.ClearErrors();

    g.MutableReport().AddReportElement("count", count);
    g.MutableReport().AddReportElement("total_sum", totalSum);
    g.MutableReport().AddReportElement("promocode_sum", promocodeSum * count);
    g.MutableReport().AddReportElement("sessions", std::move(sessions));
    g.MutableReport().AddReportElement("failed_sessions", std::move(failedSessions));
    if (hasMore) {
        g.MutableReport().AddReportElement("has_more", hasMore);
    }
    g.SetCode(HTTP_OK);
}

void TPriceCalculatorProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Payment);
    TInstant since = GetTimestamp(requestData, "since", true).GetRef();
    TInstant until = GetTimestamp(requestData, "until", TInstant::Zero());
    TString sessionId = GetString(requestData, "session_id", true);
    bool useCompiled = IsTrue(Context->GetCgiParameters().Get("use_compiled"));
    bool fromHistory = false;

    auto tx = BuildTx<NSQL::ReadOnly>();
    auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, tx);
    R_ENSURE(optionalSession, {}, "cannot GetSession", tx);
    auto bSession = *optionalSession;
    if (!bSession && useCompiled) {
        //Restore from compiled
        const auto& compiledRidesManager = Server->GetDriveAPI()->GetMinimalCompiledRides();
        auto ydbTX = BuildYdbTx<NSQL::ReadOnly>("price_calculator");
        auto optionalSessions = compiledRidesManager.Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTX);
        R_ENSURE(optionalSessions->size() > 0, ConfigHttpStatus.EmptyRequestStatus, "session " << sessionId << " is not found");
        R_ENSURE(optionalSessions->size() == 1, ConfigHttpStatus.ConflictRequest, "session " << sessionId << " is duplicated");

        TMessagesCollector errors;
        bSession = TBillingSession::BuildFromCompiled(Server, optionalSessions->front(), errors);
        R_ENSURE(bSession, ConfigHttpStatus.SyntaxErrorStatus, errors.GetStringReport());
        fromHistory = true;
    }

    R_ENSURE(bSession, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect session");
    R_ENSURE(since < bSession->GetLastTS(), ConfigHttpStatus.SyntaxErrorStatus, "Incorrect since timestamp");
    if (since < bSession->GetStartTS()) {
        since = bSession->GetStartTS();
    }

    if (until) {
        R_ENSURE(until > bSession->GetStartTS(), ConfigHttpStatus.SyntaxErrorStatus, "Incorrect until timestamp");
        if (until > bSession->GetLastTS()) {
            until = bSession->GetLastTS();
        }
        R_ENSURE(since < until, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect inteval");
    }

    TBillingSession::TBillingCompilation bCompilationStart;
    bCompilationStart.SetUntil(since);
    TBillingSession::TBillingCompilation bCompilationUntil;
    if (until) {
        bCompilationUntil.SetUntil(until);
    }

    if (!bSession->FillCompilation(bCompilationStart) || !bSession->FillCompilation(bCompilationUntil)) {
        R_ENSURE(false, ConfigHttpStatus.UnknownErrorStatus, "Can't build compilation");
    }

    const i32 billingPrice = bCompilationUntil.GetBillingSumPrice() - bCompilationStart.GetBillingSumPrice();
    const i32 reportPrice = bCompilationUntil.GetReportSumPrice() - bCompilationStart.GetReportSumPrice();
    g.MutableReport().AddReportElement("price", billingPrice);
    g.MutableReport().AddReportElement("report_price", reportPrice);
    g.MutableReport().AddReportElement("from_history", fromHistory);
    g.SetCode(HTTP_OK);
}

const IDBEntitiesManager<TVirtualTerminal>* TTerminalsInfoProcessor::GetEntitiesManager(const IServerBase* server) const {
    auto logic = server->GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetBillingManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return nullptr;
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
    if (!trustLogic) {
        return nullptr;
    }
    return &trustLogic->GetTrustClient().GetTerminals();
}

const IDBEntitiesManager<TVirtualTerminal>* TTerminalsUpsertProcessor::GetEntitiesManager(const IServerBase* server) const {
    auto logic = server->GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetBillingManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return nullptr;
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
    if (!trustLogic) {
        return nullptr;
    }
    return &trustLogic->GetTrustClient().GetTerminals();
}

const IDBEntitiesManager<TVirtualTerminal>* TTerminalsRemoveProcessor::GetEntitiesManager(const IServerBase* server) const {
    auto logic = server->GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetBillingManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
    if (!logic) {
        return nullptr;
    }
    const TTrustLogic* trustLogic = logic->GetAsSafe<TTrustLogic>();
    if (!trustLogic) {
        return nullptr;
    }
    return &trustLogic->GetTrustClient().GetTerminals();
}

void TAddCompiledBillProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Payment);
    const TString sessionId = GetString(requestData, "session_id", true);

    auto options = TBillingHistoryManager::TQueryOptions(1, true);
    options.AddGenericCondition("session_id", sessionId);
    options.SetOrderBy({ "history_event_id" });

    auto session = BuildTx<NSQL::Writable>();
    auto lastEvent = DriveApi->GetBillingManager().GetHistoryManager().GetEvents({}, {}, session, options);
    R_ENSURE(lastEvent && lastEvent->size() && lastEvent->front().GetHistoryAction() == EObjectHistoryAction::Remove, ConfigHttpStatus.UserErrorState, "incorrect session state");

    auto bills = DriveApi->GetBillingManager().GetCompiledBills().GetBillsFromDB(sessionId, session);
    R_ENSURE(bills && bills->empty(), ConfigHttpStatus.UserErrorState, "compiled bill already exists");
    if (!DriveApi->GetBillingManager().BuildCompiledBill(lastEvent->front(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

int TCreateTestCard::GetLuhnCheck(const TString& pan) {
    int sum = 0;
    for (size_t i(0); i < pan.size(); ++i) {
        if (i % 2 != 0) {
            sum += pan[i] - '0';
        } else {
            int num(pan[i] - '0');
            num *= 2;
            sum += num > 9 ? num % 10 + 1 : num;
        }
    }
    return sum % 10;
}

void TCreateTestCard::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr, const NJson::TJsonValue&) {
    const TString prefix = Server->GetSettings().GetValueDef<TString>("billing.testing.card_prefix", "510000");
    TString pan = prefix + ToString(100000000 + RandomNumber<ui32>(99999999)) + "0";
    int check = GetLuhnCheck(pan);
    pan.back() = '0' + (10 - check) % 10;
    g.MutableReport().AddReportElement("test_card", pan);
    Y_ASSERT(GetLuhnCheck(pan) == 0);
    g.SetCode(HTTP_OK);
}

bool TSyncUserPayment::ProcessTask(const TPaymentTask& task, TJsonReport::TGuard& /*g*/, const NJson::TJsonValue& /*requestData*/) const {
    auto logic = DriveApi->GetBillingManager().GetLogicByType(NDrive::NBilling::EAccount::Trust);
    R_ENSURE(logic, ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    const ITrustLogic* trustLogic = logic->GetAsSafe<ITrustLogic>();
    R_ENSURE(logic, ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");

    TBillSyncParams params = {/* UpdateBillingTask = */ false, Nothing(), {}};
    return trustLogic->SyncPaymentTask(task, "fake", DriveApi->GetBillingManager().GetPaymentsManager(), ETrustOperatinType::PaymentStart, ETrustOperatinType::PaymentInfo, nullptr, true, params)  == EProduceResult::Ok;
}

bool TCancelUserPayment::ProcessTask(const TPaymentTask& task, TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) const {
    auto session = BuildTx<NSQL::Writable>();
    auto historyTasks = DriveApi->GetBillingManager().GetHistoryManager().GetSessionHistory(task.GetSessionId(), {}, session);
    if (!historyTasks) {
        return false;
    }
    TString userId;
    if (historyTasks->empty()) {
        g.MutableReport().AddReportElement(task.GetPaymentId(), "no history");
        userId = GetString(requestData, "user_id", true);
    } else {
        userId = historyTasks->front().GetUserId();
    }
    if (!DriveApi->GetBillingManager().ProcessFiscalRefundIssue(task, userId, task.GetSum(), false, session)) {
        return false;
    }
    return session.Commit();
}
