#pragma once

#include <drive/backend/processors/common_app/db_entities.h>
#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/billing/manager.h>

template <class THandler, class THandlerConfigImpl>
class TBillingProcessorBase: public TCommonServiceAppProcessorBase {
public:
    using THandlerConfig = TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReportingProcessor<THandler>>;

protected:
    const THandlerConfig& Config;

public:
    TBillingProcessorBase(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TCommonServiceAppProcessorBase(config, context, authModule, server)
        , Config(config)
    {
        Y_UNUSED(Config.Registrator);
    }

protected:
    template <class T>
    void BuildReport(const T& object, NJson::TJsonValue& compiledReport, const TMap<TString, TCachedPayments>& payments, const TMap<TString, TVector<TRefundTask>>& refunds) {
        compiledReport["bill"] = object.GetBill();
        compiledReport["billing_type"] = ::ToString(object.GetBillingType());
        compiledReport["transactions"] = NJson::JSON_ARRAY;
        TString billingTaskId;
        if constexpr (std::is_same<T, TCompiledBill>::value) {
            billingTaskId = object.GetSessionId();
        } else {
            billingTaskId = object.GetId();
        }
        compiledReport["session_id"] = billingTaskId;

        auto it = payments.find(billingTaskId);
        if (it != payments.end()) {
            auto refundsIt = refunds.find(billingTaskId);
            compiledReport["payments"] = GetPaymentsReport(it->second.GetTimeline(), refundsIt == refunds.end() ? TVector<TRefundTask>() : refundsIt->second);
            auto timeline =  it->second.GetTimeline();
            if (!timeline.empty()) {
                compiledReport["timestamp"] = timeline.front().GetCreatedAt().Seconds();
            }
        } else {
            compiledReport.InsertValue("payments", NJson::JSON_ARRAY);
        }
        if (!compiledReport.Has("timestamp")) {
            if constexpr (std::is_same<T, TCompiledBill>::value) {
                compiledReport["timestamp"] = object.GetFinalTime().Seconds();
            } else {
                compiledReport["timestamp"] = object.GetHistoryInstant().Seconds();
            }
        }

        NJson::TJsonValue meta(NJson::JSON_MAP);
        if (object.GetRealSessionId()) {
            meta["real_session_id"] = object.GetRealSessionId();
        }
        if (object.GetComment()) {
            meta["comment"] = object.GetComment();
        }
        compiledReport["meta"] = meta;
    }

    NJson::TJsonValue GetPaymentsReport(const TVector<TPaymentTask>& payments, const TVector<TRefundTask>& sessionRefunds) const {
        TMap<TString, TVector<TRefundTask>> refundsByPayment;

        for (auto&& refund : sessionRefunds) {
            refundsByPayment[refund.GetPaymentId()].push_back(refund);
        }

        NJson::TJsonValue paymentsJson(NJson::JSON_ARRAY);
        for (auto&& payment : payments) {
            NJson::TJsonValue paymentJson;
            payment.ToJson(paymentJson);
            auto it = refundsByPayment.find(payment.GetPaymentId());
            ui32 refunded = 0;
            ui32 waitRefund = 0;
            if (it != refundsByPayment.end()) {
                for (auto&& refund : it->second) {
                    if (refund.GetFinished()) {
                        refunded += refund.GetSum();
                    }
                    waitRefund += refund.GetSum();
                }
            }
            paymentJson["refunded"] = refunded;
            paymentJson["wait_refund"] = waitRefund;
            paymentsJson.AppendValue(paymentJson);
        }
        return paymentsJson;
    }
};

class TGetUserPaymentsProcessor: public TBillingProcessorBase<TGetUserPaymentsProcessor, TEmptyConfig> {
private:
    using TBase = TBillingProcessorBase<TGetUserPaymentsProcessor, TEmptyConfig>;
public:
    TGetUserPaymentsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "user_payments";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TTicketsConfig {
public:
    void InitFeatures(const TYandexConfig::Section* section) {
        MaxTicketRub = section->GetDirectives().Value("MaxTicketRub", MaxTicketRub);
    }

    void ToStringFeatures(IOutputStream& os) const {
        os << "MaxTicketRub: " << MaxTicketRub << Endl;
    }

    ui32 MaxTicketRub = 30000;
};


class TRunPaymentProcessor: public TAppCommonProcessor<TRunPaymentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRunPaymentProcessor, TEmptyConfig>;
public:
    TRunPaymentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "reset_payment";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCancelPaymentProcessor: public TAppCommonProcessor<TCancelPaymentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCancelPaymentProcessor, TEmptyConfig>;
public:
    TCancelPaymentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "cancel_payment";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TTrustInfoProcessor : public TBillingProcessorBase<TTrustInfoProcessor, TEmptyConfig> {
private:
    using TBase = TBillingProcessorBase<TTrustInfoProcessor, TEmptyConfig>;
public:
    TTrustInfoProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "trust_info";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGlobalRefundProcessor: public TAppCommonProcessor<TGlobalRefundProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TGlobalRefundProcessor, TEmptyConfig>;
public:
    TGlobalRefundProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "global_refunds_processor";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
    static NDrive::TScheme GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */);
};

class TPriceCalculatorProcessor: public TAppCommonProcessor<TPriceCalculatorProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TPriceCalculatorProcessor, TEmptyConfig>;
public:
    TPriceCalculatorProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "price_calculator";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TTerminalsInfoProcessor : public TObjectsInfoProcessor<TVirtualTerminal, TTerminalsInfoProcessor> {
private:
    using TBase = TObjectsInfoProcessor<TVirtualTerminal, TTerminalsInfoProcessor>;
protected:
    virtual const IDBEntitiesManager<TVirtualTerminal>* GetEntitiesManager(const IServerBase* server) const override;

    virtual TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Terminal;
    }
public:
    using TBase::TBase;
};

class TTerminalsUpsertProcessor : public TObjectsUpsertProcessor<TVirtualTerminal, TTerminalsUpsertProcessor> {
private:
    using TBase = TObjectsUpsertProcessor<TVirtualTerminal, TTerminalsUpsertProcessor>;
protected:
    virtual const IDBEntitiesManager<TVirtualTerminal>* GetEntitiesManager(const IServerBase* server) const override;

    virtual TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Terminal;
    }
public:
    using TBase::TBase;
};

class TTerminalsRemoveProcessor : public TObjectsRemoveProcessor<TVirtualTerminal, TTerminalsRemoveProcessor> {
private:
    using TBase = TObjectsRemoveProcessor<TVirtualTerminal, TTerminalsRemoveProcessor>;
protected:
    virtual const IDBEntitiesManager<TVirtualTerminal>* GetEntitiesManager(const IServerBase* server) const override;

    virtual TAdministrativeAction::EEntity GetEntityType() const override {
        return TAdministrativeAction::EEntity::Terminal;
    }
public:
    using TBase::TBase;
};

class TAddCompiledBillProcessor : public TAppCommonProcessor<TAddCompiledBillProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TAddCompiledBillProcessor, TEmptyConfig>;
public:
    TAddCompiledBillProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "add_compiled_bill";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCreateTestCard : public TAppCommonProcessor<TCreateTestCard, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCreateTestCard, TEmptyConfig>;
public:
    TCreateTestCard(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "create_test_card";
    }

    static int GetLuhnCheck(const TString& pan);
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

template <typename THandler>
class IUpdatePayments : public TAppCommonProcessor<THandler, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<THandler, TEmptyConfig>;
    using THandlerConfig = TContextAwareConfig<THandler, TEmptyConfig, TSchemeReportingProcessor<THandler>>;

public:
    IUpdatePayments(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        THandler::ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Payment);
        auto paymentIds = THandler::GetStrings(requestData, "payment_id", true);
        auto iterationCount = THandler::template GetValue<ui32>(requestData, "count", false).GetOrElse(5);

        auto session = THandler::template BuildTx<NSQL::Writable>();
        auto payments = THandler::DriveApi->GetBillingManager().GetPaymentsManager().GetPayments(paymentIds, session);
        R_ENSURE(payments, {}, "cannot GetPayments", session);

        TVector<TString> sessions;
        for (auto payment : *payments) {
            R_ENSURE(payment.GetPaymentType() == NDrive::NBilling::EAccount::Trust, THandler::ConfigHttpStatus.UserErrorState, "incorrect payment type for " + payment.GetPaymentId());
            sessions.push_back(payment.GetSessionId());
        }

        auto activeTasks = THandler::DriveApi->GetBillingManager().GetActiveTasksManager().GetSessionsTasks(sessions, session);
        R_ENSURE(activeTasks, {}, "cannot GetActiveTasks", session);
        R_ENSURE(activeTasks->empty(), THandler::ConfigHttpStatus.UserErrorState, "active task already exist for " + activeTasks->front().GetId());

        ui32 iter = 0;
        while (!payments->empty() && iter++ < iterationCount) {
            for (auto it = payments->begin(); it != payments->end();) {
                if (ProcessTask(*it, g, requestData)) {
                    it = payments->erase(it);
                } else {
                    ++it;
                }
            }
        }
        g.MutableReport().AddReportElement("wait_elements", payments->size());
        g.SetCode(HTTP_OK);
    }
    virtual bool ProcessTask(const TPaymentTask& task, TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) const = 0;
};

class TSyncUserPayment : public IUpdatePayments<TSyncUserPayment> {
    using TBase = IUpdatePayments<TSyncUserPayment>;
public:
    using TBase::TBase;
    static TString GetTypeName() {
        return "sync_user_payment";
    }

    virtual bool ProcessTask(const TPaymentTask& task, TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) const override;
};

class TCancelUserPayment : public IUpdatePayments<TCancelUserPayment> {
    using TBase = IUpdatePayments<TCancelUserPayment>;
public:
    using TBase::TBase;
    static TString GetTypeName() {
        return "cancel_user_payment";
    }

    virtual bool ProcessTask(const TPaymentTask& task, TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) const override;
};
