#include "processor.h"

#include <drive/backend/cars/car_model.h>

#include <rtline/util/types/uuid.h>

#include <util/generic/string.h>

void TCarModelsInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    if (requestData.IsNull()) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Models);
        const TVector<TString> ids = GetStrings(Context->GetCgiParameters(), "id", false);

        auto session = BuildTx<NSQL::ReadOnly>();
        TModelsDB::TFetchResult result;
        if (ids.size()) {
            result = Server->GetDriveAPI()->GetModelsData()->FetchInfo(ids, session);
        } else {
            result = Server->GetDriveAPI()->GetModelsData()->FetchInfo(session);
        }

        NJson::TJsonValue jsonResult;
        for (auto&& carModel : result.GetResult()) {
            NJson::TJsonValue modelJson = carModel.second.GetReport(GetLocale(), NDriveModelReport::ReportAll);
            jsonResult.AppendValue(modelJson);
        }

        g.MutableReport().AddReportElement("car_models", std::move(jsonResult));
        g.SetCode(HTTP_OK);
        return;
    }

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Models);

    TString modelCode = GetString(requestData, "code");
    TDriveModelData modelData;
    bool isNewModel = true;
    R_ENSURE(modelCode, ConfigHttpStatus.SyntaxErrorStatus, "empty code");

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    {
        auto result = Server->GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, session).GetResult();
        if (result.size()) {
            modelData = result.begin()->second;
            isNewModel = false;
        }
    }

    if (!isNewModel) {
        R_ENSURE(GetString(requestData, "name"), ConfigHttpStatus.SyntaxErrorStatus, "empty name");
        R_ENSURE(GetString(requestData, "manufacturer"), ConfigHttpStatus.SyntaxErrorStatus, "empty manufacturer");
    }
    R_ENSURE(modelData.Patch(requestData), ConfigHttpStatus.SyntaxErrorStatus, "unable to parse");

    if (!Server->GetDriveAPI()->GetModelsData()->Upsert(modelData, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TCarModelsRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Models);

    const TVector<TString> ids = GetStrings(requestData, "id", true);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    if (!Server->GetDriveAPI()->GetModelsData()->Remove(ids, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

/*
    Two below endpoints are redundant: we can simply pass modified array or specifications now.
    Left for backwards compatibility.
*/

void TCarModelSpecificationAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Models);

    // parse everything and create specification
    TString id = GetString(requestData, "id", false);
    TString icon = GetString(requestData, "icon", false);
    TString name = GetString(requestData, "name");
    TString value = GetString(requestData, "value");
    TString modelCode = GetString(requestData, "model_id");
    auto position = GetValue<ui32>(requestData, "position").GetRef();
    if (!id) {
        id = NUtil::CreateUUID();
    }
    TDriveModelSpecification specification(id, name, value, position, icon);

    // find the model, which will receive this new specification
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    TDriveModelData modelData;
    {
        auto result = Server->GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, session).GetResult();
        R_ENSURE(result.size() > 0, ConfigHttpStatus.SyntaxErrorStatus, "unknown model");
        modelData = result.begin()->second;
    }
    R_ENSURE(
        modelData.MutableSpecifications().Upsert(std::move(specification)),
        HTTP_BAD_REQUEST,
        "cannot UpsertSpecification",
        session
    );

    // save updated model and forcibly update cache
    if (!Server->GetDriveAPI()->GetModelsData()->Upsert(modelData, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    Server->GetDriveAPI()->GetModelsData()->FetchInfo({ modelCode }, Now());

    g.MutableReport().AddReportElement("id", id);
    g.SetCode(HTTP_OK);
}

void TCarModelSpecificationRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Models);

    const TVector<TString> idsVector = GetStrings(requestData, "id", true);
    TSet<TString> idsForRemoval;
    for (auto&& id : idsVector) {
        idsForRemoval.insert(id);
    }

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto modelsMap = Server->GetDriveAPI()->GetModelsData()->FetchInfo(session).GetResult();
    for (auto&& element : modelsMap) {
        TVector<TDriveModelSpecification> newSpecifications;
        auto model = element.second;
        const auto& specifications = model.GetSpecifications().GetSpecifications();
        for (auto&& specification : model.GetSpecifications().GetSpecifications()) {
            if (!idsForRemoval.contains(specification.GetId())) {
                newSpecifications.push_back(specification);
            }
        }
        if (newSpecifications.size() != specifications.size()) {
            model.SetSpecifications(std::move(newSpecifications));
            R_ENSURE(Server->GetDriveAPI()->GetModelsData()->Upsert(model, session), ConfigHttpStatus.UnknownErrorStatus, "unable to upsert");
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    Server->GetDriveAPI()->GetModelsData()->FetchInfo(Now());
    g.SetCode(HTTP_OK);
}
