#include "processor.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/database/attachment_context.h>
#include <drive/backend/processors/leasing/add_cars.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/uuid.h>

void TCarVinValidationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
    TString vin = GetString(Context->GetCgiParameters(), "vin");
    auto session = BuildTx<NSQL::ReadOnly>();
    auto vinVerificationResult = Server->GetDriveAPI()->GetCarsData()->ValidateVIN(vin, session);
    g.MutableReport().SetExternalReport(vinVerificationResult.SerializeToJson());
    g.SetCode(HTTP_OK);
}

void TCarUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Car);

    R_ENSURE(GetString(requestData, "id"), ConfigHttpStatus.SyntaxErrorStatus, "empty id");

    // try to get existing car by id
    TDriveCarInfo currentCarInfo;
    TString carId = GetString(requestData, "id");
    TString isForce = GetString(Context->GetCgiParameters(), "force", false);

    auto session = BuildTx<NSQL::Writable>();
    bool isNewCar = false;
    {
        auto gCarsData = Server->GetDriveAPI()->GetCarsData()->FetchInfo(carId, session);
        auto result = gCarsData.GetResult();
        if (result.empty()) {
            R_ENSURE(isForce || GetString(requestData, "vin"), ConfigHttpStatus.SyntaxErrorStatus, "empty vin");
            R_ENSURE(isForce || GetString(requestData, "model_code"), ConfigHttpStatus.SyntaxErrorStatus, "empty model_code");
            isNewCar = true;
        }
    }

    bool hasErrors = false;
    NJson::TJsonValue errors = NJson::JSON_ARRAY;
    if (!isForce) {
        if (requestData.Has("vin")) {
            TString vin = GetString(requestData, "vin");
            auto vinVerificationResult = Server->GetDriveAPI()->GetCarsData()->ValidateVIN(vin, session);
            if (vinVerificationResult.ValidationStatus != TCarsDB::TVinValidationResult::EVinValidationStatus::VinOK) {
                NJson::TJsonValue errorReport;
                errorReport["field"] = "vin";
                errorReport["details"] = vinVerificationResult.SerializeToJson();
                errors.AppendValue(std::move(errorReport));
                hasErrors = true;
            }
        }
    }
    {
        if (requestData.Has("vin") && GetString(requestData, "vin").size() > 17) {
            R_ENSURE(GetString(requestData, "vin").size() <= 17, ConfigHttpStatus.SyntaxErrorStatus, "vin is longer than 17 chars");
        }

        if (requestData.Has("registration_id")) {
            TString comment;
            ui64 registrationIdInt;
            R_ENSURE(GetString(requestData, "registration_id").size() <= 10, ConfigHttpStatus.SyntaxErrorStatus, "registration_id is longer than 10 chars");
            R_ENSURE(TryFromString(GetString(requestData, "registration_id"), registrationIdInt), ConfigHttpStatus.SyntaxErrorStatus, "registration_id is not int");
        }

        if (requestData.Has("model_code")) {
            auto modelCode = GetString(requestData, "model_code");
            R_ENSURE(
                !!DriveApi->GetModelsData()->FetchInfo(modelCode, session).GetResultPtr(modelCode),
                ConfigHttpStatus.SyntaxErrorStatus,
                "unknown model_code"
            );
        }
    }

    if (hasErrors) {
        g.MutableReport().AddReportElement("errors", std::move(errors));
        g.SetCode(HTTP_BAD_REQUEST);
        return;
    }

    // Create or modify the car
    if (!Server->GetDriveAPI()->GetCarsData()->UpdateCarFromJSON(requestData, carId, permissions->GetUserId(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    // If it's the new car, put the necessary tags
    if (isNewCar) {
        auto modelCode = GetString(requestData, "model_code");
        auto gModelsData = Server->GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, session);
        auto result = gModelsData.GetResult();  // Can't be empty, otherwise the creation above will fail
        auto defaultTags = result.begin()->second.GetDefaultTags();
        for (auto&& defaultTag : defaultTags) {
            ITag::TPtr tagData = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(defaultTag.TagName);
            tagData->SetTagPriority(defaultTag.Priority);
            if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagData, permissions->GetUserId(), carId, Server, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }

    TCarGenericAttachment attachment;
    THolder<TCarRegistryDocument> currentRegistryDocument;
    if (DriveApi->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, attachment, Now())) {
        currentRegistryDocument.Reset(new TCarRegistryDocument(*dynamic_cast<const TCarRegistryDocument*>(attachment.Get())));
    } else {
        currentRegistryDocument.Reset(new TCarRegistryDocument());
    }
    if (currentRegistryDocument->PatchWithCarJSON(requestData, Server->GetDriveAPI()->GetModelsData(), Context->GetRequestStartTime())) {
        TCarGenericAttachment attachment(currentRegistryDocument.Release());
        if (!DriveApi->GetCarAttachmentAssignments().Attach(attachment, carId, permissions->GetUserId(), session, Server)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TCarModificationsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car);

    const auto& cgi = Context->GetCgiParameters();
    auto carId = GetString(cgi, "car_id");
    auto since = GetTimestamp(cgi, "since", false);
    auto until = GetTimestamp(cgi, "until", false);
    auto tx = BuildTx<NSQL::ReadOnly>();
    auto optionalEvents = Server->GetDriveDatabase().GetCarManager().GetHistoryManager().GetEvents(carId, { since, until }, tx);
    R_ENSURE(optionalEvents, {}, "cannot GetEvents for " << carId, tx);

    NJson::TJsonValue changesHistoryReport = NJson::JSON_ARRAY;
    for (auto&& event : *optionalEvents) {
        changesHistoryReport.AppendValue(event.BuildReportItem());
    }
    g.MutableReport().AddReportElement("history", std::move(changesHistoryReport));
    g.SetCode(HTTP_OK);
}

void TCarAttachedHardwareProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    auto carId = GetString(cgi, "car_id");
    auto reportType = GetString(cgi, "report");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car, carId);
    R_ENSURE(DriveApi->GetCarsData()->FetchInfo(carId).size() != 0, ConfigHttpStatus.PermissionDeniedStatus, "no such car");

    NAttachmentReport::TReportTraits traits = 0;
    if (reportType == "service_app") {
        traits = NAttachmentReport::ReportServiceApp;
    } else if (reportType == "all") {
        traits = NAttachmentReport::ReportAll;
    } else if (reportType == "basic_info") {
        traits = EDocumentAttachmentType::CarRegistryDocument;
    } else if (reportType == "major_info") {
        traits = EDocumentAttachmentType::CarRegistryDocument | EDocumentAttachmentType::CarInsurancePolicy;
    }

    auto tx = BuildTx<NSQL::ReadOnly>();
    const auto optionalAttachments = DriveApi->GetCarAttachmentAssignments().GetActiveAttachments(carId, traits, tx);
    R_ENSURE(optionalAttachments, {}, "cannot GetActiveAttachments", tx);
    const auto& attachments = *optionalAttachments;

    NJson::TJsonValue hardwareReport = NJson::JSON_ARRAY;
    TCarAttachmentReportContext context(DriveApi);
    context.SetRegistryDocumentTraits(permissions->GetCarRegistryReportTraits());
    for (auto item : attachments) {
        hardwareReport.AppendValue(item.BuildReport(context));
    }

    g.MutableReport().AddReportElement("attachments", std::move(hardwareReport));
    g.SetCode(HTTP_OK);
}

void TCarAssignAttachmentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Car);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString deviceCode = GetString(requestData, "device_code");
    TString carId = GetString(cgi, "car_id");
    auto type = GetValue<EDocumentAttachmentType>(cgi, "type", false);
    bool dryRun = GetValue<bool>(cgi, "dry_run", false).GetOrElse(false);
    bool isForce = false;
    if (!!GetString(cgi, "force", false)) {
        isForce = true;
    }

    auto locale = GetLocale();
    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(DriveApi->GetCarsData()->FetchInfo(carId, session).size() != 0, ConfigHttpStatus.PermissionDeniedStatus, "no such car", session);

    TCarGenericAttachment attachment;
    if (!DriveApi->GetCarGenericAttachments().UpsertFromServiceApp(deviceCode, attachment, permissions->GetUserId(), session, type)) {
        throw TCodedException(HTTP_BAD_REQUEST)
            .SetLocalizedMessage(NDrive::TLocalization::UnableToUpsertDevice())
            .SetErrorCode("unable_to_upsert_device")
            .AddPublicInfo("may_retry_with_force", false);
    }

    auto optionalAssignment = DriveApi->GetCarAttachmentAssignments().GetActiveAttachmentAssignment(attachment.GetId(), session);
    R_ENSURE(optionalAssignment, {}, "cannot GetActiveAttachmentAssignment for " << attachment.GetId(), session);
    auto assignment = std::move(*optionalAssignment);
    if (assignment) {
        if (assignment.GetCarId() == carId) {
            throw TCodedException(HTTP_BAD_REQUEST)
                .SetLocalizedMessage(NDrive::TLocalization::DeviceAlreadyAttached())
                .SetErrorCode("device_already_attached")
                .AddPublicInfo("may_retry_with_force", false);
        } else if (!isForce) {
            auto carFetchResult = DriveApi->GetCarsData()->FetchInfo(assignment.GetCarId(), session);
            auto carPtr = carFetchResult.GetResultPtr(assignment.GetCarId());
            NJson::TJsonValue carReport;
            TString modelCode;
            if (!carPtr) {
                carReport["id"] = assignment.GetCarId();
            } else {
                carReport = carPtr->GetReport(locale, permissions->GetDeviceReportTraits() | NDeviceReport::EReportTraits::ReportVIN);
                modelCode = carPtr->GetModel();
            }

            NJson::TJsonValue modelReport;
            auto modelFetchResult = DriveApi->GetModelsData()->FetchInfo(modelCode, session);
            auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
            if (!modelPtr) {
                modelReport["code"] = modelCode;
            } else {
                modelReport = modelPtr->GetReport(locale, NDriveModelReport::ReportAll);
            }

            throw TCodedException(HTTP_BAD_REQUEST)
                .SetLocalizedMessage(NDrive::TLocalization::DeviceAttachedToDifferentCar())
                .SetErrorCode("device_attached_to_different_car")
                .AddPublicInfo("may_retry_with_force", true)
                .AddInfo("car", std::move(carReport))
                .AddInfo("model", std::move(modelReport));
        }
    }

    bool wasDetached = false;
    if (assignment) {
        if (!DriveApi->GetCarAttachmentAssignments().Detach(assignment.GetId(), permissions->GetUserId(), session, Server)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        wasDetached = true;
    }

    R_ENSURE(
        DriveApi->GetCarAttachmentAssignments().Attach(attachment, carId, permissions->GetUserId(), session, Server, wasDetached),
        {},
        "cannot Attach",
        session
    );

    TCarAttachmentReportContext context(DriveApi);
    g.AddReportElement("attachment", attachment.BuildReport(context));

    if (dryRun) {
        g.AddReportElement("status", "dry_run");
        g.SetCode(HTTP_OK);
        return;
    }

    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    g.MutableReport().AddReportElement("status", "success");
    g.SetCode(HTTP_OK);
}

void TCarAttachedHardwareHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Car);

    TString carId = GetString(Context->GetCgiParameters(), "car_id");

    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(DriveApi->GetCarsData()->FetchInfo(carId, session).size() != 0, ConfigHttpStatus.PermissionDeniedStatus, "no such car");
    const auto optionalEvents = DriveApi->GetCarAttachmentAssignments().GetHistoryManager().GetEvents(0, session, NSQL::TQueryOptions()
        .AddGenericCondition("car_id", carId)
    );
    R_ENSURE(optionalEvents, {}, "cannot GetEvents", session);

    NJson::TJsonValue changesHistoryReport = NJson::JSON_ARRAY;
    TSet<TString> objectIds;
    const auto& history = *optionalEvents;
    for (auto&& historyItem : history) {
        changesHistoryReport.AppendValue(historyItem.BuildReportItem());
        objectIds.insert(historyItem.GetGenericAttachmentId());
    }

    auto attachmentIds = MakeVector(objectIds);
    auto optionalAttachments = DriveApi->GetCarGenericAttachments().GetAttachments(attachmentIds, session);
    R_ENSURE(optionalAttachments, {}, "cannot GetAttachments", session);

    NJson::TJsonValue attachmentsReport = NJson::JSON_ARRAY;
    TCarAttachmentReportContext context(DriveApi);
    context.SetRegistryDocumentTraits(permissions->GetCarRegistryReportTraits());
    const auto& attachments = *optionalAttachments;
    for (auto&& attachment : attachments) {
        attachmentsReport.AppendValue(attachment.BuildReport(context));
    }

    g.MutableReport().AddReportElement("attachments", std::move(attachmentsReport));
    g.MutableReport().AddReportElement("history", std::move(changesHistoryReport));
    g.SetCode(HTTP_OK);
}

void TCarUnassignAttachmentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString attachmentId = GetString(cgi, "attachment_id", false);
    TString carId = GetUUID(cgi, "car_id", false);
    bool useAdmPermissions = GetHandlerSetting<bool>("use_adm_permissions").GetOrElse(true);

    R_ENSURE(carId || attachmentId, ConfigHttpStatus.SyntaxErrorStatus, "either car_id or attachment_id should be present");
    R_ENSURE(carId.empty() || attachmentId.empty(), ConfigHttpStatus.SyntaxErrorStatus, "only one of car_id or attachment_id should be present");

    if (attachmentId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Car);
    }

    if (carId) {
        TString type = GetString(cgi, "type", false);
        TString cppType = GetString(cgi, "cpp_type", false);
        R_ENSURE(type || cppType, ConfigHttpStatus.SyntaxErrorStatus, "either type or cpp_type should be present");
        R_ENSURE(type.empty() || cppType.empty(), ConfigHttpStatus.SyntaxErrorStatus, "only one of type or cpp_type should be present");

        if (useAdmPermissions) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Car);
        } else {
            auto session = BuildTx<NSQL::ReadOnly>();
            bool found = NDrive::ContainsActionInTag(permissions->GetUserId(), DriveApi, ConfigHttpStatus, carId, session, "detach-device");
            R_ENSURE(found, ConfigHttpStatus.PermissionDeniedStatus, "action detach-device not found amoung performed tags for " << carId);
        }

        auto tx = BuildTx<NSQL::ReadOnly>();
        const auto optionalAttachments = DriveApi->GetCarAttachmentAssignments().GetActiveAttachments(carId, NAttachmentReport::ReportServiceApp, tx);
        R_ENSURE(optionalAttachments, {}, "cannot GetActiveAttachments", tx);
        const auto& attachments = *optionalAttachments;

        for (auto&& item : attachments) {
            if ((cppType && cppType == ToString(item.GetType())) ||
                (type && type == CarAttachmentType(item.GetType()))) {
                attachmentId = item.GetId();
                break;
            }
        }

        R_ENSURE(attachmentId, HTTP_NOT_FOUND, "cannot find assignment for " << (type ? ("type: " + type) : ("cppType: " + cppType)), tx);
    }

    auto session = BuildTx<NSQL::Writable>();
    auto optionalAssignment = DriveApi->GetCarAttachmentAssignments().GetActiveAttachmentAssignment(attachmentId, session);
    R_ENSURE(optionalAssignment, {}, "cannot GetActiveAttachmentAssignment for " << attachmentId, session);
    auto a = std::move(*optionalAssignment);
    R_ENSURE(a, HTTP_NOT_FOUND, "cannot find assignment for attachment " << attachmentId, session);
    if (!DriveApi->GetCarAttachmentAssignments().Detach(a.GetId(), permissions->GetUserId(), session, Server) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TCarRegistryUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Car);

    auto modelCodesMapping = DriveApi->GetModelsData()->GetRegistryModelCodesMapping();

    TInstant docActuality = Now();

    R_ENSURE(requestData.Has("cars") && requestData["cars"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "no \"cars\" array in the root of request body");

    auto session = BuildTx<NSQL::Writable>();

    const auto& cgi = Context->GetCgiParameters();
    const bool hasOrganization = GetValue<bool>(cgi, "has_organization", false).GetOrElse(false);
    TMaybe<TString> organizationCarTag;
    if (hasOrganization) {
        auto organizationAffiliationTagPtr = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, session);
        organizationCarTag = Yensured(organizationAffiliationTagPtr)->GetOwningCarTagName();
    }

    auto allVins = DriveApi->GetCarsData()->GetAllVins(session);
    auto gModelsData = DriveApi->GetModelsData()->FetchInfo(session);
    auto modelsData = gModelsData.GetResult();
    const auto createModel = GetHandlerSettingDef("create_model", false);
    TSet<TString> processedVins;
    for (auto& entryRaw : requestData["cars"].GetArray()) {
        R_ENSURE(entryRaw.IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "no \"cars\" array in the root of request body");

        NJson::TJsonValue entry;
        NJson::TJsonValue::TMapType entryMap;
        if (!entryRaw.GetMap(&entryMap)) {
            continue;
        }
        for (auto&& it : entryMap) {
            try {
                entry[StripString(ToLowerUTF8(it.first))] = it.second;
            } catch (const std::exception& e) {
                ERROR_LOG << "unable to make lowercase from " << it.first << ": " << FormatExc(e) << Endl;
            }
        }

        R_ENSURE(entry.Has("vin") && entry["vin"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "no \"vin\" field in car description or vin is not string");

        TCarGenericAttachment currentRegistryDocument;
        auto vin = entry["vin"].GetString();
        TString carId;

        if (!processedVins.emplace(vin).second) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "SkipProcessedVin")
                ("vin", vin)
            );
            continue;
        }

        if (!allVins.contains(vin)) {
            const bool isLegacyModelKey = entry.Has("модель") && entry["модель"].IsString();
            const bool isLegacyManufacturerKey = entry.Has("марка") && entry["марка"].IsString();
            const bool isCommonModelKey = entry.Has("registry_model") && entry["registry_model"].IsString();
            const bool isCommonManufacturerKey = entry.Has("registry_manufacturer") && entry["registry_manufacturer"].IsString();

            R_ENSURE((isLegacyModelKey || isCommonModelKey), ConfigHttpStatus.SyntaxErrorStatus, "no model specified for a car with vin " + vin);
            R_ENSURE((isLegacyManufacturerKey || isCommonManufacturerKey), ConfigHttpStatus.SyntaxErrorStatus, "no manufacturer specified for a car with vin " + vin);
            R_ENSURE((!isLegacyModelKey || !isCommonModelKey), ConfigHttpStatus.SyntaxErrorStatus, "ambiguous model specified for a car with vin " + vin);
            R_ENSURE((!isLegacyManufacturerKey || !isCommonManufacturerKey), ConfigHttpStatus.SyntaxErrorStatus, "ambiguous manufacturer specified for a car with vin " + vin);

            auto model = StripString(ToLowerUTF8(entry[isLegacyModelKey ? "модель" : "registry_model"].GetString()));
            auto manufacturer = StripString(ToLowerUTF8(entry[isLegacyManufacturerKey ? "марка" : "registry_manufacturer"].GetString()));

            const auto modelKey = std::make_pair(manufacturer, model);
            if (createModel && !modelCodesMapping.contains(modelKey)) {
                NDrivematics::TAddLeasingCarsProcessor::CreateModel(
                    vin,
                    manufacturer,
                    model,
                    modelCodesMapping,
                    *DriveApi,
                    session
                );
            }

            R_ENSURE(
                modelCodesMapping.contains(modelKey),
                ConfigHttpStatus.SyntaxErrorStatus,
                "the following model key pair is not known: (" + manufacturer + ", " + model + ")"
            );

            NJson::TJsonValue minCarJson;
            carId = NUtil::CreateUUID();

            TString modelCode = modelCodesMapping[modelKey];

            minCarJson["id"] = carId;
            minCarJson["vin"] = vin;
            minCarJson["model_code"] = modelCode;
            if (entry.Has("грз") && entry["грз"].IsString()) {
                minCarJson["number"] = ToLowerUTF8(entry["грз"].GetString());
            }
            if (entry.Has("стс") && entry["стс"].IsString()) {
                minCarJson["registration_id"] = entry["стс"].GetString();
            }

            if (!DriveApi->GetCarsData()->UpdateCarFromJSON(minCarJson, carId, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            g.AddEvent(NJson::TMapBuilder
                ("event", "UpdateCarFromJson")
                ("car", minCarJson)
            );

            if (modelsData.contains(modelCode)) {
                auto defaultTags = modelsData[modelCode].GetDefaultTags();
                for (auto&& defaultTag : defaultTags) {
                    ITag::TPtr tagData = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(defaultTag.TagName);
                    if (!tagData) {
                        g.AddEvent(NJson::TMapBuilder
                            ("event", "SkipWhenCreatingTag")
                            ("vin", vin)
                            ("modelCode", modelCode)
                            ("tag", defaultTag.TagName)
                        );
                        continue;
                    }
                    tagData->SetTagPriority(defaultTag.Priority);
                    if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagData, permissions->GetUserId(), carId, Server, session)) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                }
            }

            if (organizationCarTag) {
                ITag::TPtr tagData = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(*organizationCarTag);
                R_ENSURE(tagData, ConfigHttpStatus.UnknownErrorStatus, "organization tag is not valid");
                if (!Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagData, permissions->GetUserId(), carId, Server, session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        } else {
            carId = allVins[vin];
            if (!DriveApi->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, docActuality)) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "TryGetAttachmentOfTypeError")
                    ("car_id", carId)
                    ("vin", vin)
                );
            }
        }

        THolder<TCarRegistryDocument> regDocument;
        if (currentRegistryDocument.GetType() == EDocumentAttachmentType::CarRegistryDocument) {
            auto ptr = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            if (!!ptr) {
                regDocument.Reset(new TCarRegistryDocument(*ptr));
            }
        }
        if (!regDocument) {
            regDocument.Reset(new TCarRegistryDocument());
        }

        if (!regDocument->PatchWithRawRegistryData(entry)) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "PatchWithRawRegistryDataError")
                ("car_id", carId)
                ("entry", entry)
            );
        }

        TCarGenericAttachment attachment(regDocument.Release());
        if (!DriveApi->GetCarAttachmentAssignments().Attach(attachment, carId, permissions->GetUserId(), session, Server)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}


void TCarModelsListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);

    NJson::TJsonValue result = NJson::JSON_ARRAY;
    auto modelsFetchResult = DriveApi->GetModelsData()->GetCached();
    for (auto&& it : modelsFetchResult) {
        result.AppendValue(it.second.GetReport(GetLocale(), NDriveModelReport::UserReport));
    }

    g.MutableReport().AddReportElement("models", std::move(result));
    g.SetCode(HTTP_OK);
}

void TCarInsuranceUploadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Car);

    R_ENSURE(requestData.Has("cars") && requestData["cars"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "no \"cars\" array in the root of request body");
    R_ENSURE(requestData.Has("type") && requestData["type"].GetString(), ConfigHttpStatus.SyntaxErrorStatus, "no type specified");
    R_ENSURE(!requestData.Has("force") || requestData["force"].IsBoolean(), ConfigHttpStatus.SyntaxErrorStatus, "\"force\" should be boolean");

    ui32 baseCost = 35;
    if (requestData.Has("base_cost")) {
        R_ENSURE(requestData["base_cost"].IsUInteger(), ConfigHttpStatus.SyntaxErrorStatus, "base cost is not uint");
        baseCost = requestData["base_cost"].GetUInteger();
    }

    bool force = (!requestData.Has("force") || requestData["force"].GetBoolean());
    auto provider = GetValue<NDrive::EInsuranceProvider>(requestData, "type", true);

    TSet<TString> vinsForFetch;
    TMap<TString, THolder<TCarInsurancePolicy>> projectedUpdate;

    for (auto&& row : requestData["cars"].GetArray()) {
        R_ENSURE(row.IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "element of 'cars' is not a map");

        NJson::TJsonValue entry;
        NJson::TJsonValue::TMapType entryMap;
        if (!row.GetMap(&entryMap)) {
            continue;
        }

        for (auto&& it : entryMap) {
            try {
                entry[StripString(ToLowerUTF8(it.first))] = it.second;
            } catch (const std::exception& e) {
                ERROR_LOG << "unable to make lowercase from " << it.first << ": " << FormatExc(e) << Endl;
            }
        }

        if (!entry.Has("vin") && !entry["vin"].IsString()) {
            ERROR_LOG << "entry has no vin field, skipping: " << row.GetStringRobust() << Endl;
            continue;
        }

        TString vin;
        try {
            vin = ToUpperUTF8(StripString(entry["vin"].GetString()));
        } catch (const yexception& ex) {
            ERROR_LOG << "vin normalization failed: " << entry["vin"].GetString() << Endl;
            continue;
        }

        THolder<TCarInsurancePolicy> policy = MakeHolder<TCarInsurancePolicy>();
        if (!policy->ConstructFromAdmin(entry, *provider, baseCost)) {
            continue;
        }

        vinsForFetch.insert(vin);
        projectedUpdate.emplace(std::move(vin), std::move(policy));
    }

    auto session = BuildTx<NSQL::Writable>();
    auto carsFR = Server->GetDriveAPI()->GetCarsData()->FetchCarsByVIN(vinsForFetch, session);
    TMap<TString, TDriveCarInfo> carByVIN;

    TVector<TString> carIds;
    carIds.reserve(carsFR.size());

    for (auto&& carIt : carsFR) {
        TString vin = carIt.second.GetVin();
        carByVIN.emplace(vin, std::move(carIt.second));
        carIds.push_back(carIt.first);
    }

    auto presentAttachments = Server->GetDriveAPI()->GetCarAttachmentAssignments().GetAssignmentsOfType(carIds, EDocumentAttachmentType::CarInsurancePolicy, Context->GetRequestStartTime());

    for (auto&& policyIt : projectedUpdate) {
        auto carIt = carByVIN.find(policyIt.first);
        if (carIt == carByVIN.end()) {
            continue;
        }

        auto carId = carIt->second.GetId();
        const auto& newPolicy = policyIt.second.Get();
        auto existingInsurances = presentAttachments.equal_range(carId);

        bool isPresent = false;
        TVector<TString> removalCandidates;

        for (auto&& it = existingInsurances.first; it != existingInsurances.second; ++it) {
            auto existingPolicy = dynamic_cast<const TCarInsurancePolicy*>(it->second.Get());
            if (!existingPolicy) {
                continue;
            }

            // Check non-zero intersecion. In this case this is either the same policy (skip then), or current policy needs to be removed
            bool hasIntersection = (Max(existingPolicy->GetValidFrom(), newPolicy->GetValidFrom()) < Min(existingPolicy->GetValidUntil(), newPolicy->GetValidUntil()));
            bool isDuplicate = true;

            if (existingPolicy->GetAgreementPartnerNumber() != newPolicy->GetAgreementPartnerNumber() || existingPolicy->GetAgreementNumber() != newPolicy->GetAgreementNumber()) {
                isDuplicate = false;
            }
            if (existingPolicy->GetBaseCost() != newPolicy->GetBaseCost() || existingPolicy->GetPerMinuteCost() != newPolicy->GetPerMinuteCost()) {
                isDuplicate = false;
            }
            if (existingPolicy->GetValidFrom() != newPolicy->GetValidFrom() || existingPolicy->GetValidUntil() != newPolicy->GetValidUntil()) {
                isDuplicate = false;
            }

            if (isDuplicate) {
                isPresent = true;
            } else if (hasIntersection) {
                removalCandidates.emplace_back(it->second.GetId());
            }
        }

        if (!isPresent) {
            if (!force && !removalCandidates.empty()) {
                continue;
            }

            bool isFailed = false;
            auto recordsToDetach = Server->GetDriveAPI()->GetCarAttachmentAssignments().GetAssignmentIdsForAttachments(removalCandidates, Context->GetRequestStartTime());
            for (auto&& id : recordsToDetach) {
                if (!Server->GetDriveAPI()->GetCarAttachmentAssignments().Detach(id, permissions->GetUserId(), session, Server)) {
                    ERROR_LOG << "unable to detach old insurance policy: " << id << Endl;
                    isFailed = true;
                    break;
                }
            }
            if (isFailed) {
                continue;
            }

            TCarGenericAttachment newInsuranceAttachment(policyIt.second.Release());

            if (!Server->GetDriveAPI()->GetCarAttachmentAssignments().Attach(newInsuranceAttachment, carId, permissions->GetUserId(), session, Server)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
    R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    g.SetCode(HTTP_OK);
}

void TOsagoUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Car);

    R_ENSURE(DriveApi->HasMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "mds client undefined");
    const TS3Client::TBucket* mdsBucket = DriveApi->GetMDSClient().GetBucket(Config.GetMdsBucket());
    R_ENSURE(mdsBucket, ConfigHttpStatus.UnknownErrorStatus, "mdsBucket " + Config.GetMdsBucket() + " undefined");

    auto carId = GetString(requestData, "car_id", true);
    auto gCars = DriveApi->GetCarsData()->FetchInfo(carId, TInstant::Zero());
    auto carPtr = gCars.GetResultPtr(carId);
    R_ENSURE(carPtr, ConfigHttpStatus.PermissionDeniedStatus, "no such car");

    TCarGenericAttachment attachment;
    THolder<TCarRegistryDocument> currentRegistryDocument;
    if (DriveApi->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, attachment, TInstant::Zero())) {
        currentRegistryDocument.Reset(new TCarRegistryDocument(*dynamic_cast<const TCarRegistryDocument*>(attachment.Get())));
    } else {
        currentRegistryDocument.Reset(new TCarRegistryDocument());
        currentRegistryDocument->SetNumber(carPtr->GetNumber()).SetRegistrationId(carPtr->GetRegistrationID());
    }

    if (!currentRegistryDocument->GetVin()) {
        currentRegistryDocument->SetVin(carPtr->GetVin());
    }

    bool isUpdate = false;
    TMessagesCollector errors;
    R_ENSURE(currentRegistryDocument->UpdateOSAGO(isUpdate, *mdsBucket, *Server, errors), ConfigHttpStatus.UnknownErrorStatus, errors.GetStringReport());

    if (isUpdate) {
        TCarGenericAttachment attachmentNew(currentRegistryDocument.Release());
        auto session = BuildTx<NSQL::Writable>();
        if (!DriveApi->GetCarAttachmentAssignments().Attach(attachmentNew, carId, permissions->GetUserId(), session, Server) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    } else {
        g.SetCode(HTTP_NO_CONTENT);
    }
}

void TFuelingCardsUploadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Car);
    R_ENSURE(requestData.Has("cars") && requestData["cars"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "no \"cars\" array in the root of request body");

    TVector<TFuelCard::TCardFields> customCardFields;
    {
        const TString setting = Server->GetSettings().GetValueDef<TString>("service_app.settings.fuel_cards", "[]");
        NJson::TJsonValue json;
        R_ENSURE(NJson::ReadJsonFastTree(setting, &json) && json.IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "invalid fueling cards settings");
        for (const auto& customData : json.GetArray()) {
            TFuelCard::TCardFields fields;
            R_ENSURE(TryFromJson(customData, fields), ConfigHttpStatus.SyntaxErrorStatus, "invalid fueling cards setting: " + customData.GetStringRobust());
            customCardFields.emplace_back(std::move(fields));
        }
    }

    TMap<TString, TVector<THolder<TFuelCard>>> projectedUpdate;
    for (auto&& row : requestData["cars"].GetArray()) {
        R_ENSURE(row.IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "element of 'cars' is not a map");
        NJson::TJsonValue entry;
        for (auto&& it : row.GetMap()) {
            entry[StripString(ToLowerUTF8(it.first))] = it.second;
        }
        TString rowVin;
        R_ENSURE(NJson::ParseField(entry["vin"], rowVin, true), ConfigHttpStatus.SyntaxErrorStatus, "entry has no vin field, skipping: " + row.GetStringRobust());
        TString vin = ToUpperUTF8(StripString(rowVin));
        auto constructed = TFuelCard::ConstructFromAdmin(entry, customCardFields);
        if (constructed.empty()) {
            continue;
        }
        R_ENSURE(projectedUpdate.emplace(vin, std::move(constructed)).second, ConfigHttpStatus.SyntaxErrorStatus, "duplicated vin: " + vin);
    }

    const bool force = GetValue<bool>(Context->GetCgiParameters(), "force", false).GetOrElse(true);
    auto session = BuildTx<NSQL::Writable>();
    auto carByVIN = DriveApi->GetCarVins()->FetchInfo(MakeSet(NContainer::Keys(projectedUpdate)), session);
    TVector<TString> carIds;
    carIds.reserve(carByVIN.size());
    Transform(carByVIN.begin(), carByVIN.end(), std::back_inserter(carIds), [](const std::pair<TString, TDriveCarInfo>& item) { return item.second.GetId(); });
    auto presentAttachments = Server->GetDriveAPI()->GetCarAttachmentAssignments().GetAssignmentsOfType(carIds, EDocumentAttachmentType::CarFuelCard, Context->GetRequestStartTime());
    for (auto&& fuelingCardIt : projectedUpdate) {
        auto carIt = carByVIN.GetResult().find(fuelingCardIt.first);
        if (carIt == carByVIN.end()) {
            continue;
        }

        auto carId = carIt->second.GetId();
        auto existingCards = presentAttachments.equal_range(carId);
        auto& newCards = fuelingCardIt.second;
        TVector<TString> removalCandidates;
        for (auto&& it = existingCards.first; it != existingCards.second; ++it) {
            auto existingCard = dynamic_cast<const TFuelCard*>(it->second.Get());
            if (!existingCard) {
                continue;
            }
            auto cardIt = FindIf(newCards, [&existingCard](const THolder<TFuelCard> &item) {
                return item->GetCode() == existingCard->GetCode();
            });
            if (cardIt != newCards.end()) {
                if (!force || ((*cardIt)->GetPin() == existingCard->GetPin() && (*cardIt)->GetStation() == existingCard->GetStation())) {
                    newCards.erase(cardIt);
                } else {
                    removalCandidates.emplace_back(it->second.GetId());
                }
            }
        }
        if (!removalCandidates.empty()) {
            auto recordsToDetach = Server->GetDriveAPI()->GetCarAttachmentAssignments().GetAssignmentIdsForAttachments(removalCandidates, Context->GetRequestStartTime());
            for (auto&& id : recordsToDetach) {
                R_ENSURE(Server->GetDriveAPI()->GetCarAttachmentAssignments().Detach(id, permissions->GetUserId(), session, Server), ConfigHttpStatus.UnknownErrorStatus, "unable to detach old fueling cards", session);
            }
        }
        for (auto&& card : newCards) {
            TCarGenericAttachment cardAttachment(card.Release());
            if (!Server->GetDriveAPI()->GetCarAttachmentAssignments().Attach(cardAttachment, carId, permissions->GetUserId(), session, Server)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}
