#include "processor.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/radar.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users_controller/ucontroller.h>
#include <drive/library/cpp/maps_router/router.h>
#include <drive/backend/offers/user_context.h>

#include <rtline/protos/proto_helper.h>

class TScannerPropertiesParser {
    R_FIELD(TDuration, WalkingDuration, TDuration::Minutes(15));
    R_FIELD(TDuration, Livetime, TDuration::Minutes(15));
    R_FIELD(TDuration, Delay, TDuration::Zero());
    R_FIELD(TVector<TString>, FilterIdentifiers);
    R_FIELD(TRadarUserTag::EAction, ActionRadar, TRadarUserTag::EAction::Order);
    R_OPTIONAL(TGeoCoord, Position);
    R_OPTIONAL(TVector<TGeoCoord>, PolygonSearchArea);

public:
    TScannerPropertiesParser(const NJson::TJsonValue& requestData, const THttpStatusManagerConfig& configHttpStatus, const IRequestProcessor& processor) {
        if (TMaybe<TString> filterSetting = processor.GetHandlerSetting<TString>("filter")) {
            StringSplitter(*filterSetting).SplitBySet(", ").SkipEmpty().Collect(&FilterIdentifiers);
        } else if (requestData.Has("filter")) {
            StringSplitter(requestData["filter"].GetStringRobust()).SplitBySet(", ").SkipEmpty().Collect(&FilterIdentifiers);
        }
        if (TMaybe<TDuration> walkingTime = processor.GetHandlerSetting<TDuration>("walking_time")) {
            SetWalkingDuration(*walkingTime);
        } else if (requestData.Has("walking_time")) {
            R_ENSURE(requestData["walking_time"].IsInteger(), configHttpStatus.SyntaxErrorStatus, "incorrect walking_time", EDriveSessionResult::IncorrectRequest);
            SetWalkingDuration(TDuration::Seconds(requestData["walking_time"].GetInteger()));
            R_ENSURE(processor.GetHandlerSettingDef<bool>("global_scanner_available", false) || WalkingDuration, configHttpStatus.SyntaxErrorStatus, "incorrect walking time", NDrive::MakeError("scanner.walking_time.incorrect"));
        }
        if (TMaybe<TDuration> livetime = processor.GetHandlerSetting<TDuration>("livetime")) {
            SetLivetime(*livetime);
        } else if (requestData.Has("livetime")) {
            R_ENSURE(requestData["livetime"].IsInteger(), configHttpStatus.SyntaxErrorStatus, "incorrect livetime", EDriveSessionResult::IncorrectRequest);
            SetLivetime(TDuration::Seconds(requestData["livetime"].GetInteger()));
        }
        if (TMaybe<TDuration> delay = processor.GetHandlerSetting<TDuration>("delay")) {
            SetDelay(*delay);
        } else if (requestData.Has("delay")) {
            R_ENSURE(requestData["delay"].IsInteger(), configHttpStatus.SyntaxErrorStatus, "incorrect delay", EDriveSessionResult::IncorrectRequest);
            SetDelay(TDuration::Seconds(requestData["delay"].GetInteger()));
        } else if (requestData.Has("current_user_ts") && requestData.Has("start_user_ts")) {
            R_ENSURE(requestData["current_user_ts"].IsInteger(), configHttpStatus.SyntaxErrorStatus, "incorrect current_user_ts", EDriveSessionResult::IncorrectRequest);
            R_ENSURE(requestData["start_user_ts"].IsInteger(), configHttpStatus.SyntaxErrorStatus, "incorrect start_user_ts", EDriveSessionResult::IncorrectRequest);
            const i64 currentTs = requestData["current_user_ts"].GetInteger();
            const i64 startTs = requestData["start_user_ts"].GetInteger();
            R_ENSURE(currentTs <= startTs, configHttpStatus.SyntaxErrorStatus, "incorrect start_user_ts < current_user_ts", EDriveSessionResult::IncorrectRequest);
            SetDelay(TDuration::Seconds(startTs - currentTs));
        }

        if (WalkingDuration) {
            R_ENSURE(requestData.Has("lat") && requestData["lat"].IsDouble() && requestData.Has("lon") && requestData["lon"].IsDouble(), configHttpStatus.SyntaxErrorStatus, "incorrect lat/lon", EDriveSessionResult::IncorrectRequest);
            Position = TGeoCoord(requestData["lon"].GetDouble(), requestData["lat"].GetDouble());
        } else {
            TGeoCoord c;
            TMaybe<TString> position = processor.GetHandlerSetting<TString>("user_position");
            if (position && c.DeserializeFromString(*position)) {
                Position = c;
            }
        }
        if (TMaybe<TRadarUserTag::EAction> actionRadar = processor.GetHandlerSetting<TRadarUserTag::EAction>("action")) {
            ActionRadar = *actionRadar;
        } else if (requestData.Has("action")) {
            R_ENSURE(requestData["action"].IsString() && TryFromString(requestData["action"].GetString(), ActionRadar), configHttpStatus.SyntaxErrorStatus, "incorrect action", EDriveSessionResult::IncorrectRequest);
        }
        R_ENSURE(NJson::ParseField(requestData, "search_area", PolygonSearchArea, false), configHttpStatus.SyntaxErrorStatus, EDriveSessionResult::IncorrectRequest);
    }
};

namespace {
    NDrive::TLocationTag GetAreaTag(const NDrive::TLocationTags& locationTags, const NDrive::TLocationTags& regionLocationTags) {
        for (auto&& tag : locationTags) {
            if (!regionLocationTags.contains(tag)) {
                continue;
            }
            return tag;
        }
        return {};
    }
}

void TStartScannerProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TSet<TString>* tagNames = Yensured(permissions)->GetTagNamesByActionPtr(TTagAction::ETagAction::Add);
    R_ENSURE(
        tagNames && tagNames->contains(TRadarUserTag::TypeName),
        ConfigHttpStatus.PermissionDeniedStatus,
        "no permissions for add scanner tag",
        EDriveSessionResult::NoUserPermissions
    );

    TScannerPropertiesParser params(requestData, ConfigHttpStatus, *this);
    const auto driveApi = Server->GetDriveAPI();
    const auto localization = Server->GetLocalization();
    const auto locale = GetLocale();
    const auto& settings = Server->GetSettings();
    auto session = BuildTx<NSQL::Writable>();
    if (driveApi && driveApi->HasBillingManager()) {
        const TBillingManager& billingManager = driveApi->GetBillingManager();
        const bool checkDebt = settings.GetValue<bool>("billing.book.check_debt").GetOrElse(true);
        TMaybe<ui32> debt;
        if (checkDebt) {
            debt = billingManager.GetDebt(permissions->GetUserId(), session);
            R_ENSURE(debt, {}, "cannot GetDebt for " << permissions->GetUserId(), session);
        }

        if (debt && *debt > 99) {
            auto landingString = settings.GetValue<TString>("warning_screens.checkers.user_debt_check.landings.main").GetOrElse({});
            SubstGlobal(landingString, "_Debt_", localization->FormatPrice(locale, *debt));
            NJson::TJsonValue landing;
            NJson::ReadJsonFastTree(landingString, &landing);
            localization->ApplyResourcesForJson(landing, locale);
            session.SetLanding(landing);
            R_ENSURE(*debt < 100, ConfigHttpStatus.PaymentRequiredState, "debt_payment_required: " << debt, EDriveSessionResult::PaymentRequired, session);
        }
        CheckUserCards(permissions, nullptr);
    }

    TGeoCoord scannerCoord;
    R_ENSURE(params.HasPosition(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect scanner position info", EDriveSessionResult::IncorrectRequest);
    scannerCoord = params.GetPositionUnsafe();

    TMaybe<TGeoCoord> userCoord = GetUserLocation();
    R_ENSURE(userCoord, ConfigHttpStatus.SyntaxErrorStatus, "incorrect user position info", EDriveSessionResult::IncorrectRequest);

    TInstant startTime = ModelingNow() + params.GetDelay();

    ITag::TPtr tag = Yensured(driveApi)->GetTagsManager().GetTagsMeta().CreateTag(TRadarUserTag::TypeName);
    auto tagRadar = Yensured(std::dynamic_pointer_cast<TRadarUserTag>(tag));
    const TDuration reqTimeout = Server->GetSettings().GetValueDef<TDuration>("radar.search_area.request_timeout", TDuration::Seconds(5));
    if (params.HasPolygonSearchArea()) {
        tagRadar->SetSearchArea(params.GetPolygonSearchAreaUnsafe());
    } else if (params.GetWalkingDuration()) {
        auto polygonFutureCoords = TUserOfferContext::GetSearchAreaFromRouter(*Server, permissions, reqTimeout, params.GetWalkingDuration(), scannerCoord);
        R_ENSURE(polygonFutureCoords.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async polygonFutureCoords");
        tagRadar->SetSearchArea(polygonFutureCoords.ExtractValue(Context->GetRequestDeadline() - Now()));
    }
    tagRadar->SetSLAInstant(startTime + params.GetLivetime());
    tagRadar->SetWalkDuration(params.GetWalkingDuration());
    tagRadar->SetFilterIdentifiers(params.GetFilterIdentifiers());
    tagRadar->SetStartTimestamp(startTime);
    tagRadar->SetAction(params.GetActionRadar());
    tagRadar->SetRadarPosition(scannerCoord);
    R_ENSURE(
        !tagRadar->GetSearchArea().empty() || (!tagRadar->GetFilterIdentifiers().empty() && !tagRadar->GetWalkDuration()),
        ConfigHttpStatus.UnknownErrorStatus, "unable to build search area"
    );
    const TString& userId = permissions->GetUserId();
    if (!DriveApi->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, Server, session, EUniquePolicy::Rewrite)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (DriveApi->HasBillingManager()) {
        session.SetComment("scanner");
        if (!DriveApi->GetBillingManager().GetActiveTasksManager().BoostUserSessions(permissions->GetUserId(), permissions->GetUserId(), TBillingTask::TExecContext(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TStopScanner::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TSet<TString>* tagNames = permissions->GetTagNamesByActionPtr(TTagAction::ETagAction::Remove);
    R_ENSURE(
        tagNames && tagNames->contains(TRadarUserTag::TypeName),
        ConfigHttpStatus.PermissionDeniedStatus,
        "no permissions for remove scanner tag",
        EDriveSessionResult::NoUserPermissions
    );

    auto session = BuildTx<NSQL::Writable>();
    TDBTags tags;
    if (!DriveApi->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), {TRadarUserTag::TypeName}, tags, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (tags.size()) {
        R_ENSURE(tags.size() == 1, ConfigHttpStatus.UserErrorState, "have to been once scanner tag", EDriveSessionResult::IncorrectRequest);

        if (!DriveApi->GetTagsManager().GetUserTags().RemoveTag(tags.front(), permissions->GetUserId(), Server, session, true) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    g.SetCode(HTTP_OK);
}

void TAreaScanner::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& requestData) {
    TEventsGuard eg(g.MutableReport(), "TAreaScanner");
    TMaybe<i32> numAreas = GetValue<i32>(requestData, "num_areas", true);
    R_ENSURE(*numAreas > 0, ConfigHttpStatus.SyntaxErrorStatus, "incorrect num_areas (too small)", EDriveSessionResult::IncorrectRequest);
    TScannerPropertiesParser params(requestData, ConfigHttpStatus, *this);
    R_ENSURE(*numAreas <= 25, ConfigHttpStatus.SyntaxErrorStatus, "incorrect num_areas (too large)", EDriveSessionResult::IncorrectRequest);

    TGeoCoord coord;
    if (params.HasPosition()) {
        coord = params.GetPositionUnsafe();
    } else {
        auto userLocation = GetUserLocation();
        R_ENSURE(userLocation, ConfigHttpStatus.SyntaxErrorStatus, "incorrect user position info", EDriveSessionResult::IncorrectRequest);
        coord = *userLocation;
    }

    R_ENSURE(Server->GetUsersController(), ConfigHttpStatus.UnknownErrorStatus, "incorrect_server_configuration");
    TVector<TWalkingArea> areas;
    {
        TEventsGuard eg(g.MutableReport(), "GetAreas");
        auto useMapsRouter = GetHandlerSetting<bool>("radar.pedestrian_router.isochrone_enabled").GetOrElse(false);
        R_ENSURE(Server->GetUsersController()->GetAreas(params.GetWalkingDuration(), coord, *numAreas, areas, useMapsRouter), ConfigHttpStatus.UnknownErrorStatus, "cannot_restore_area");
    }
    NJson::TJsonValue jsonArea(NJson::JSON_ARRAY);
    for (ui32 i = 0; i < areas.size(); ++i) {
        NJson::TJsonValue areaJson;
        jsonArea.AppendValue(areas[i].SerializeToJson());
    }
    g.MutableReport().AddReportElement("area", std::move(jsonArea));

    NJson::TJsonValue flags(NJson::JSON_MAP);
    {
        NDrive::TLocationTags regionLocationTags = StringSplitter(
            Server->GetSettings().GetValue<TString>("areas.region.tags").GetOrElse("msc_area,spb_area,kazan_area")
        ).Split(',');
        NDrive::TLocationTags locationTags = DriveApi->GetTagsInPoint(coord);
        NDrive::TLocationTag targetArea = GetAreaTag(locationTags, regionLocationTags);
        if (locationTags.contains("radar_disable_autobook")) {
            flags.InsertValue("disable_autobook", true);
        }

        auto userLocation = GetUserLocation();
        NDrive::TLocationTags userLocationTags = userLocation ? DriveApi->GetTagsInPoint(*userLocation) : NDrive::TLocationTags();
        NDrive::TLocationTag userArea = GetAreaTag(userLocationTags, regionLocationTags);
        if (targetArea != userArea) {
            flags.InsertValue("disable_filters", true);
        }
    }
    g.MutableReport().AddReportElement("flags", std::move(flags));
    g.SetCode(HTTP_OK);
}
