#include "processor.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/condition.h>
#include <drive/backend/chat_robots/robot_manager.h>
#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/logging/logging.h>
#include <drive/backend/support_center/categorizer/model.h>

namespace NChatOutputHelpers {
    void GetExpectedActionReport(NJson::TJsonValue& result, const IChatRobot::TPtr chatRobot, const TString& topic, const IChatUserContext::TPtr ctx) {
        TChatRobotScriptItem currentChatItem;
        TChatContext chatContext;
        TInstant actuality = TInstant::Zero();
        result["owner_id"] = ctx->GetUserId();
        if (chatRobot->GetChatContextByChat(ctx->GetUserId(), topic, chatContext, actuality) && chatRobot->GetCurrentScriptItem(ctx->GetUserId(), topic, currentChatItem, actuality) && currentChatItem.GetActionTypeInterface() != NChatRobot::EUserAction::NoAction) {
            NJson::TJsonValue actionJson = currentChatItem.GetActionReport().IsDefined() ? ctx->UnescapeJson(currentChatItem.GetActionReport(), chatContext) : NJson::JSON_MAP;
            actionJson["type"] = ToString(currentChatItem.GetActionTypeInterface());
            actionJson["text"] = chatRobot->UnescapeMacroses(ctx->GetLocale(), currentChatItem.GetActionButtonText(), ctx->GetUserId(), topic, NDrive::NChat::TMessage::OnViewMacroPrefix, NDrive::NChat::TMessage::OnViewMacroSuffix);
            result["expected_action"] = std::move(actionJson);
            if (currentChatItem.GetActionType() == NChatRobot::EUserAction::Tree || currentChatItem.GetActionType() == NChatRobot::EUserAction::ContextButtons) {
                result["schema"] = currentChatItem.GetSchema().SerializeToJson(ctx, chatContext);
            } else if (currentChatItem.GetActionType() == NChatRobot::EUserAction::DeeplinkAction) {
                result["expected_action"]["link"] = chatRobot->UnescapeMacroses(ctx->GetLocale(), currentChatItem.GetLink(), ctx->GetUserId(), topic, NDrive::NChat::TMessage::OnViewMacroPrefix, NDrive::NChat::TMessage::OnViewMacroSuffix);
            }
        } else {
            if (currentChatItem.GetActionTypeInterface() == NChatRobot::EUserAction::NoAction) {
                result["expected_action"] = NJson::JSON_NULL;
            } else {
                result["expected_action"] = ToString(NChatRobot::EUserAction::ChatClosed);
            }
        }
        if (currentChatItem.GetAllowedMessageTypes().size()) {
            NJson::TJsonValue typesList = NJson::JSON_ARRAY;
            for (auto&& type : currentChatItem.GetAllowedMessageTypes()) {
                typesList.AppendValue(ToString(type));
            }
            result["allowed_types"] = std::move(typesList);
        }
        if (chatRobot->GetChatConfig().GetChatScript().HasRatings()) {
            result["ratings"] = NJson::ToJson(chatRobot->GetChatConfig().GetChatScript().GetMessageRatings());
        }
    }

    NJson::TJsonValue GetFaqUrl(const IChatRobot::TPtr chatRobot) {
        NJson::TJsonValue result;
        if (chatRobot->GetFaqUrl()) {
            result = chatRobot->GetFaqUrl();
        } else {
            result = NJson::JSON_NULL;
        }
        return result;
    }

    NJson::TJsonValue GetSupportUrl(const IChatRobot::TPtr chatRobot) {
        NJson::TJsonValue result;
        if (chatRobot->GetSupportUrl()) {
            result = chatRobot->GetSupportUrl();
        } else {
            result = NJson::JSON_NULL;
        }
        return result;
    }

    NJson::TJsonValue BuildParticipantsJsonReport(const TUsersDB* usersData, const TSet<TString>& participantIds) {
        NJson::TJsonValue result = NJson::JSON_ARRAY;
        if (!usersData) {
            return result;
        }
        auto usersFR = usersData->FetchInfo(participantIds);
        for (auto&& userIt : usersFR) {
            result.AppendValue(userIt.second.GetChatReport());
        }
        return result;
    }
}

void TRobotChatHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto locale = GetLocale();
    auto userId = permissions->GetUserId();
    auto chatId = GetString(Context->GetCgiParameters(), "chat_id");
    auto chatIdRaw = chatId;
    auto sinceStr = GetString(Context->GetCgiParameters(), "since", false);
    auto sinceIdStr = GetString(Context->GetCgiParameters(), "since_id", false);

    auto localization = Server->GetLocalization();
    auto messageTraits = permissions->GetMessageVisibilityTraits();

    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);

    bool isAdminRequest = false;
    auto delegatedUserId = GetString(Context->GetCgiParameters(), "user_id", false);
    if (delegatedUserId && delegatedUserId != userId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatRobot, chatId);
        auto session = BuildTx<NSQL::ReadOnly>();
        auto tag = IChatRobotImpl::GetTagByTopicLink(chatIdRaw, delegatedUserId, session, *Server);
        if (!tag.Defined()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        } else if (tag->HasData()) {
            auto tagsForObserve = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
            R_ENSURE(tagsForObserve.contains((*tag)->GetName()), ConfigHttpStatus.PermissionDeniedStatus, "no permissions to observe chats with tag " + (*tag)->GetName());
        }
    }
    if (delegatedUserId) {
        isAdminRequest = true;
        userId = std::move(delegatedUserId);
    }

    ui32 since = 0;
    if (sinceStr != "") {
        R_ENSURE(TryFromString(sinceStr, since), ConfigHttpStatus.SyntaxErrorStatus, "'since' should be either missing or int");
    }
    ui64 sinceId = 0;
    if (sinceIdStr != "") {
        R_ENSURE(TryFromString(sinceIdStr, sinceId), ConfigHttpStatus.SyntaxErrorStatus, "'since_id' should be either missing or int");
    }
    R_ENSURE(!sinceId || !since, ConfigHttpStatus.SyntaxErrorStatus, "only one of {'since_id', 'since'} should be specified");

    auto chatRobot = Server->GetChatRobot(chatId);
    {
        R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
        bool chatExists = chatRobot->HasChat(userId, topic);
        if (!chatExists) {
            auto session = BuildChatSession(false);
            R_ENSURE(chatRobot->EnsureChat(userId, topic, session, true), HTTP_NOT_FOUND, "chat " + chatIdRaw + " not found");
            if (!session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            auto externalUserId = GetExternalUserId();
            auto origin = GetOrigin();
            auto userLocation = GetUserLocation();
            chatRobot->CheckIn(userId, permissions->GetUserId(), topic, origin, externalUserId, userLocation.Get(), Context->GetRequestStartTime());
        }
    }
    TString actualStepId = "intro";
    TChatRobotScriptItem currentScriptItem;
    {
        if (!chatRobot->GetChatConfig().GetIsStatic()) {
            bool hasActualStepId = chatRobot->GetActualStateId(userId, topic, actualStepId);
            if (userId == permissions->GetUserId()) {
                TChatRobotScriptItem cachedScriptItem;
                bool hasScriptItem = chatRobot->GetCurrentScriptItem(userId, topic, cachedScriptItem, TInstant::Zero());
                if (!hasScriptItem) {
                    chatRobot->RefreshStates(Context->GetRequestStartTime());
                } else {
                    if (!hasActualStepId || actualStepId != cachedScriptItem.GetId()) {
                        chatRobot->RefreshStates(Context->GetRequestStartTime());
                    }
                    if (!hasActualStepId) {
                        actualStepId = chatRobot->GetChatConfig().GetInitialStepId();
                    }
                }
            }
        } else {
            sinceId = 0;
            if (topic) {
                actualStepId = topic;
                R_ENSURE(chatRobot->GetChatConfig().GetChatScript().GetScriptItemById(actualStepId, currentScriptItem), ConfigHttpStatus.UnknownErrorStatus, "could not get current script item");
            }
        }
    }

    TChatUserContext::TPtr chatContextPtr;
    if (!isAdminRequest) {
        chatContextPtr = BuildChatContext(permissions);
    } else {
        chatContextPtr.Reset(new TChatUserContext());
        chatContextPtr->SetUserId(userId);
    }
    R_ENSURE(chatContextPtr, HTTP_INTERNAL_SERVER_ERROR, "cannot construct context ");
    chatContextPtr->SetChatId(chatId);
    chatContextPtr->SetChatTopic(topic);

    NJson::TJsonValue result;
    NChatOutputHelpers::GetExpectedActionReport(result, chatRobot, topic, chatContextPtr);
    if (chatRobot->GetChatConfig().GetIsStatic()) {
        if (
            permissions->GetStatus() != NDrive::UserStatusActive &&
            permissions->GetStatus() != NDrive::UserStatusOnboarding
        ) {
            result["expected_action"] = NJson::JSON_NULL;
        }
    }

    result["faq_url"] = NChatOutputHelpers::GetFaqUrl(chatRobot);
    result["support_url"] = NChatOutputHelpers::GetSupportUrl(chatRobot);
    result["user_id"] = userId;
    TInstant nextInstant = TInstant::Zero();
    NDrive::NChat::TOptionalMessageEvents rawMessages;
    if (!chatRobot->GetChatConfig().GetIsStatic()) {
        NJson::TJsonValue messagesReport = NJson::JSON_ARRAY;
        {
            auto session = BuildChatSession(false);
            if (since) {
                rawMessages = chatRobot->GetChatMessagesSinceTimestamp(userId, topic, TInstant::Seconds(since), session, messageTraits);
            } else {
                rawMessages = chatRobot->GetChatMessages(userId, topic, session, messageTraits, sinceId);
            }
            if (!rawMessages) {
                R_ENSURE(rawMessages, HTTP_INTERNAL_SERVER_ERROR, "cannot get messages", session);
            }
            if (!rawMessages->empty() && (!chatRobot->MarkMessagesRead(rawMessages->back(), permissions->GetUserId(), session) || !session.Commit())) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        for (auto&& message : *rawMessages) {
            if (message.GetType() == NDrive::NChat::TMessage::EMessageType::Delay) {
                ui32 shift = 0;
                if (TryFromString(message.GetText(), shift)) {
                    nextInstant = Max(nextInstant, message.GetHistoryInstant() + TDuration::Seconds(shift));
                }
                if (nextInstant > Context->GetRequestStartTime()) {
                    break;
                }
                continue;
            }
            messagesReport.AppendValue(chatRobot->GetMessageReport(locale, message, userId, topic));
        }
        result["messages"] = std::move(messagesReport);
        if (nextInstant > Context->GetRequestStartTime()) {
            result["expected_action"] = NJson::JSON_NULL;
        }
    } else {
        TInstant landingTime;
        TSet<TString> viewedLandings;
        if (topic) {
            auto userLandingsFR = Server->GetDriveAPI()->GetUserLandingData()->FetchInfo(userId);
            if (userLandingsFR.size() == 1) {
                auto userLandings = std::move(userLandingsFR.MutableResult().begin()->second);
                for (auto&& landing : userLandings.GetLandings()) {
                    viewedLandings.insert(landing.GetId());
                }
                TString landingId;
                auto driveLandingsFR = Server->GetDriveAPI()->GetLandingsDB()->GetCachedObjectsVector(viewedLandings);
                for (auto&& landing : driveLandingsFR) {
                    if (landing.GetChatId() == chatId && landing.GetChatMessagesGroup() == topic) {
                        landingId = landing.GetId();
                        break;
                    }
                }
                for (auto&& landing : userLandings.GetLandings()) {
                    if (landing.GetId() == landingId) {
                        landingTime = landing.GetLastAcceptedAt();
                        break;
                    }
                }
            }
        }
        result["messages"] = chatRobot->GetCurrentActionMessagesReport(locale, userId, topic, false, landingTime);
    }

    if (sinceId == 0 && result["messages"].GetArray().size() == 0) {
        result["messages"] = chatRobot->GetCurrentActionMessagesReport(locale, userId, topic, true);
    }

    if (isAdminRequest) {
        if (rawMessages) {
            auto participantIds = chatRobot->GetNonRobotParticipants(*rawMessages);
            result["users"] = NChatOutputHelpers::BuildParticipantsJsonReport(Server->GetDriveAPI()->GetUsersData(), participantIds);
        }
        result["user_last_viewed"] = chatRobot->GetLastViewedMessageId(userId, topic, userId);
    } else if (!chatRobot->GetChatConfig().GetIsStatic()) {
        result["last_viewed_others"] = chatRobot->GetChatConfig().GetMarkAllRead() ? chatRobot->GetLastViewedMessageId(userId, topic, userId) : chatRobot->GetLastViewedMessageIdExcept(userId, topic, userId);
    }

    if (!isAdminRequest) {
        NJson::TJsonValue chatDescriptor;
        const auto& configTitle = chatRobot->GetChatConfig().GetTitle();
        if (configTitle) {
            auto title = localization ? localization->GetLocalString(locale, configTitle) : configTitle;
            chatDescriptor["title"] = title;
            chatDescriptor["name"] = title;
        }
        NJson::InsertField(chatDescriptor, "styles", chatRobot->GetChatConfig().GetTheme().GetStyles());

        if (!chatRobot->GetChatConfig().GetIsStatic()) {
            NDrive::NChat::TChat chatMetaData;
            R_ENSURE(chatRobot->GetChat(userId, topic, chatMetaData), ConfigHttpStatus.UnknownErrorStatus, "could not get chat");
            if (!configTitle) {
                auto chatTitle = chatRobot->GetChatTitle(chatMetaData);
                auto title = localization ? localization->GetLocalString(locale, chatTitle) : chatTitle;
                chatDescriptor["name"] = std::move(title);
            }
            chatDescriptor["flags"] = chatMetaData.GetFlagsReport();
            chatDescriptor["icon"] = chatRobot->GetChatIcon(chatMetaData);
        }

        result["descriptor"] = std::move(chatDescriptor);
    }

    if (userId == permissions->GetUserId()) {
        TVector<TDBTag> tagsToRemove;
        auto taggedUser = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetCachedObject(userId);
        if (taggedUser) {
            for (auto&& tag : taggedUser->GetTags()) {
                if (tag && tag->GetName() != TUserChatShowTag::TypeName) {
                    continue;
                }
                auto tagImpl = tag.GetTagAs<TUserChatShowTag>();
                if (!tagImpl) {
                    continue;
                }
                if (tagImpl->GetRemoveOnView()) {
                    tagsToRemove.emplace_back(std::move(tag));
                }
            }
        }
        if (!tagsToRemove.empty()) {
            auto session = BuildTx<NSQL::Writable>();
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsToRemove, permissions->GetUserId(), Server, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
            }
            if (!session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
            }
        }
    }
    g.MutableReport().SetExternalReport(std::move(result));
    g.SetCode(HTTP_OK);
}

void FinalizeWithJson(TJsonReport::TGuard& g, NJson::TJsonValue&& jsonReport, HttpCodes httpCode, const TString& userId) {
    Y_UNUSED(userId);
    if (httpCode != HTTP_OK) {
        TCodedException exception(httpCode);
        if (jsonReport.IsMap()) {
            for (auto&& [key, value] : jsonReport.GetMap()) {
                exception.AddInfo(key, value);
            }
        } else {
            exception.AddInfo("", jsonReport);
        }
        g.SetCode(std::move(exception));
    } else {
        g.MutableReport().SetExternalReport(std::move(jsonReport));
        g.SetCode(httpCode);
    }
}

void TRobotChatActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto locale = GetLocale();
    auto userId = permissions->GetUserId();
    auto operatorId = userId;
    auto topicLink = GetString(Context->GetCgiParameters(), "chat_id");
    TString chatId, topic;
    IChatRobot::ParseTopicLink(topicLink, chatId, topic);

    auto requestedUserId = GetString(Context->GetCgiParameters(), "user_id", false);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    R_ENSURE(!chatRobot->GetChatConfig().GetIsStatic(), ConfigHttpStatus.SyntaxErrorStatus, "can't send messages to static chat");

    auto message = GetString(requestData, "message");
    TVector<TMessageAttachment> attachments;
    if (requestData.Has("attachments")) {
        R_ENSURE(requestData["attachments"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'attachments' should be an array");
        if (requestData.Has("content_types")) {
            R_ENSURE(requestData["content_types"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'content_types' should be an array");
        }
        auto attachmentsJson = requestData["attachments"].GetArray();
        auto contentTypesJson = requestData.Has("content_types") ? requestData["content_types"].GetArray() : NJson::TJsonValue::TArray();
        for (size_t i = 0; i < attachmentsJson.size(); ++i) {
            R_ENSURE(attachmentsJson[i].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "each element of 'attachments' should be a string");
            TString contentType;
            if (contentTypesJson.size() > i) {
                R_ENSURE(contentTypesJson[i].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "each element of 'content_types' should be a string");
                contentType = contentTypesJson[i].GetString();
            }
            attachments.push_back({attachmentsJson[i].GetString(), contentType});
        }
    }

    NDrive::NChat::TMessage::EMessageType type = NDrive::NChat::TMessage::EMessageType::Plaintext;
    if (requestData.Has("message_type")) {
        R_ENSURE(requestData["message_type"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "'message_type' should be string");
        R_ENSURE(TryFromString(requestData["message_type"].GetString(), type), ConfigHttpStatus.SyntaxErrorStatus, "'message_type' should be enum element");
    }

    ui32 messageTraits = 0;
    if (requestData.Has("traits")) {
        auto allowedMessageTraits = permissions->GetMessageVisibilityTraits();
        R_ENSURE(requestData["traits"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'traits' are specified in body, but they're not an array");
        for (auto&& traitJsonRaw : requestData["traits"].GetArray()) {
            R_ENSURE(traitJsonRaw.IsString(), ConfigHttpStatus.SyntaxErrorStatus, "some of traits is not string");
            NDrive::NChat::TMessage::EMessageTraits trait;
            R_ENSURE(TryFromString(traitJsonRaw.GetString(), trait), ConfigHttpStatus.SyntaxErrorStatus, "unknown trait: " + traitJsonRaw.GetString());
            R_ENSURE((allowedMessageTraits & (ui32)trait), ConfigHttpStatus.PermissionDeniedStatus, "no permissions to operate trait: " + traitJsonRaw.GetString());
            messageTraits |= (ui32)trait;
        }
    }

    if (requestedUserId && requestedUserId != userId) {
        NAccessVerification::TAccessVerificationTraits permissionTraits = NAccessVerification::EAccessVerificationTraits::TagsAccess;
        if (messageTraits) {
            permissionTraits |= NAccessVerification::EAccessVerificationTraits::ChatObserveAccess;
        } else {
            permissionTraits |= NAccessVerification::EAccessVerificationTraits::ChatWriteAccess;
        }
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatRobot, chatId);
        CheckUserGeneralAccessRights(requestedUserId, permissions, permissionTraits, true);
        userId = std::move(requestedUserId);
    }

    auto externalUserId = GetExternalUserId();
    auto origin = GetOrigin();
    TMaybe<TGeoCoord> userLocation = GetUserLocation();
    chatRobot->CheckIn(userId, operatorId, topic, origin, externalUserId, userLocation.Get(), Context->GetRequestStartTime());

    TChatUserContext::TPtr context;
    if (userId == permissions->GetUserId()) {
        context = BuildChatContext(permissions);
    } else {
        auto userPermissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        R_ENSURE(userPermissions, ConfigHttpStatus.SyntaxErrorStatus, "cannot get user permissions");
        context = BuildChatContext(userPermissions);
    }

    NDrive::NChat::TMessage chatMessage(message, messageTraits, type);
    chatMessage.SetLink(GetString(requestData, "link", false));
    chatMessage.SetIcon(GetString(requestData, "icon", false));
    TMaybe<TExternalMessageData> externalData;
    if (userId && messageTraits == 0) {
        TSet<TString> tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TSupportChatTag::TypeName});
        if (tagNames.size()) {
            auto taggedUser = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetCachedObject(userId);
            if (taggedUser) {
                for (auto&& tag : taggedUser->GetTags()) {
                    if (tag && !tagNames.contains(tag->GetName())) {
                        continue;
                    }
                    auto tagImpl = tag.GetTagAs<TSupportChatTag>();
                    if (tagImpl && tagImpl->GetExternalChatInfo().GetId() && tagImpl->GetTopicLink() == topicLink) {
                        INFO_LOG << "ChatActionProcessor: accepting action, chatId: "  << chatId << ", userId: " << userId << ", requestedUserId: " << requestedUserId << Endl;
                        R_ENSURE(Server->GetTaxiChatClient(), ConfigHttpStatus.SyntaxErrorStatus, "Taxi client not configured");
                        externalData = TExternalMessageData(Server->GetTaxiChatClient(), message, tagImpl->GetExternalChatInfo().GetId());
                        R_ENSURE(type == NDrive::NChat::TMessage::EMessageType::Plaintext, ConfigHttpStatus.SyntaxErrorStatus, "Can't send external message of non-text type");
                        R_ENSURE(attachments.empty(), ConfigHttpStatus.SyntaxErrorStatus, "Can't send external message with attachments");
                        chatMessage.SetExternalStatus(NDrive::NChat::TMessage::EExternalStatus::Undefined);
                        break;
                    }
                }
            }
        }
    }

    context->SetChatId(chatId);
    context->SetChatTopic(topic);
    context->SetMessage(message);
    if (userLocation) {
        context->SpecifyGeo(userLocation.GetRef());
    }

    auto eventLogState = NDrive::TEventLog::CaptureState();
    chatRobot->AcceptUserResponse(context, chatMessage, attachments, operatorId).Apply([
        eventLogState = std::move(eventLogState),
        report = g.GetReport(),
        chatRobot,
        externalData,
        locale,
        operatorId,
        topic,
        userId
    ] (const auto& future) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        TJsonReport::TGuard g(report, HTTP_INTERNAL_SERVER_ERROR);
        if (future.HasException() || !future.HasValue()) {
            FinalizeWithJson(g, NThreading::GetExceptionInfo(future), HTTP_INTERNAL_SERVER_ERROR, userId);
            return;
        }

        NJson::TJsonValue jsonReport, errorReport;
        TMaybe<NDrive::NChat::TMessageEvent> lastMessage;
        NThreading::TFuture<bool> replyFuture;
        if (externalData && externalData->GetTaxiChatClient()) {
            replyFuture = externalData->GetTaxiChatClient()->AddMessage(externalData->GetChatId(), externalData->GetMessage());
            replyFuture.Wait();
        } else if (externalData && !externalData->GetTaxiChatClient()) {
            jsonReport["status"] = "error";
            jsonReport["error"] = "TaxiChatClient is null";
            FinalizeWithJson(g, std::move(jsonReport), HTTP_INTERNAL_SERVER_ERROR, userId);
            return;
        }

        HttpCodes resultCode = HTTP_OK;
        auto session = chatRobot->BuildChatEngineSession(false);
        if (!chatRobot->GetLastMessage(userId, topic, session, 0, lastMessage, operatorId)) {
            lastMessage = Nothing();
        }
        if (replyFuture.Initialized()) {
            if (replyFuture.HasException() || !replyFuture.HasValue() || !replyFuture.GetValue()) {
                errorReport.InsertValue("reply_error", NThreading::GetExceptionInfo(replyFuture));
                if (lastMessage && (!chatRobot->DeleteMessage(operatorId, lastMessage->GetHistoryEventId(), session)) || !session.Commit()) {
                    errorReport.InsertValue("delete_message_error", session.GetReport());
                }
                resultCode = HTTP_INTERNAL_SERVER_ERROR;
            } else if (lastMessage && !chatRobot->EditMessage(operatorId, lastMessage->GetHistoryEventId(), NDrive::NChat::TMessageEdit({}, {}, NDrive::NChat::TMessage::EExternalStatus::Sent), session)) {
                errorReport.InsertValue("edit_message_error", session.GetReport());
            }
        }
        if (resultCode == HTTP_OK && lastMessage) {
            if (!chatRobot->UpdateLastViewedMessageId(userId, topic, operatorId, lastMessage->GetHistoryEventId(), session) || !session.Commit()) {
                errorReport.InsertValue("update_viewed_error", session.GetReport());
            }
        }

        if (!lastMessage) {
            jsonReport["sent_message_id"] = NJson::JSON_NULL;
        } else {
            jsonReport = chatRobot->GetMessageReport(locale, *lastMessage, userId, topic);
            jsonReport["sent_message_id"] = lastMessage->GetHistoryEventId();
        }
        jsonReport.InsertValue("debug", std::move(errorReport));
        FinalizeWithJson(g, std::move(jsonReport), resultCode, userId);
    });
    g.Release();
}

struct TUserChatDescription {
    const TString ChatRobotId;
    const IChatRobot::TPtr ChatRobotPtr;
    const NDrive::NChat::TChat ChatDescription;
    const TString OriginatorId;
    TInstant TimestampOverride;

    TUserChatDescription() = default;

    TUserChatDescription(const TString& chatRobotId, const IChatRobot::TPtr chatRobotPtr, const NDrive::NChat::TChat& chatDescription, const TString& originatorId = "")
        : ChatRobotId(chatRobotId)
        , ChatRobotPtr(chatRobotPtr)
        , ChatDescription(chatDescription)
        , OriginatorId(originatorId)
    {
        CHECK_WITH_LOG(chatRobotPtr);
    }

    NJson::TJsonValue GetReport(const bool isAdmin = false) const {
        NJson::TJsonValue chatDescription;

        TString id = ChatRobotId;
        if (ChatDescription.GetTopic()) {
            id += "." + ChatDescription.GetTopic();
        }
        chatDescription["id"] = id;

        if (!isAdmin) {
            TChatRobotScriptItem currentChatItem;
            if (ChatRobotPtr->GetCurrentScriptItem(OriginatorId, ChatDescription.GetTopic(), currentChatItem, TInstant::Zero())) {
                chatDescription["expected_action"] = ToString(currentChatItem.GetActionTypeInterface());
            } else {
                chatDescription["expected_action"] = ToString(NChatRobot::EUserAction::ChatClosed);
            }
        }

        chatDescription["is_menu_landing"] = ChatDescription.GetIsMenuLanding();
        if (ChatDescription.GetPreview()) {
            chatDescription["preview"] = ChatDescription.GetPreview();
        }
        if (!!ChatRobotPtr->GetChatConfig().GetTitle()) {
            chatDescription["title"] = ChatRobotPtr->GetChatConfig().GetTitle();
            chatDescription["name"] = ChatRobotPtr->GetChatConfig().GetTitle();
        } else {
            chatDescription["name"] = ChatRobotPtr->GetChatTitle(ChatDescription);
            if (ChatRobotPtr->GetChatConfig().GetIsStatic()) {
                chatDescription["title"] = ChatRobotPtr->GetChatTitle(ChatDescription);
            }
        }

        if (ChatDescription.GetCreatedAt() != TInstant::Zero()) {
            chatDescription["created_at"] = ChatDescription.GetCreatedAt().Seconds();
        }

        if (!ChatDescription.GetIcon()) {
            chatDescription["icon"] = ChatRobotPtr->GetChatIcon(ChatDescription);
        } else {
            chatDescription["icon"] = ChatDescription.GetIcon();
        }
        NJson::InsertField(chatDescription, "styles", ChatRobotPtr->GetChatConfig().GetTheme().GetStyles());
        chatDescription["flags"] = ChatDescription.GetFlagsReport();

        if (OriginatorId) {
            chatDescription["originator"] = OriginatorId;
            //participantIds.insert(it.OriginatorId);
        }
        return chatDescription;
    }
};

void TRobotChatUnreadsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto locale = GetLocale();
    auto userId = permissions->GetUserId();
    auto chatRobots = Server->GetChatRobots();
    R_ENSURE(chatRobots, ConfigHttpStatus.SyntaxErrorStatus, "no chat robots");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString chatId(GetString(cgi, "chat_id", false));
    NJson::TJsonValue report = NJson::JSON_ARRAY;

    auto messageTraits = permissions->GetMessageVisibilityTraits();
    TChatUserContext::TPtr ctx = BuildChatContext(permissions);
    auto session = BuildChatSession(true);
    for (auto&& chatRobotIt : chatRobots) {
        for (auto&& chat : chatRobotIt.second->GetTopics(ctx, true)) {
            if (chat.GetAtLeastOneFlag(NDrive::NChat::TChat::HideMessagesFlags)) {
                continue;
            }

            auto chatRobot = Server->GetChatRobot(chatRobotIt.first);
            if (!chatRobot || (chatId && chatRobotIt.first + (chat.GetTopic() ? "." + chat.GetTopic() : "") != chatId)) {
                continue;
            }

            auto chatUnreads = chatRobot->GetUnreadMessages(chat.GetId(), permissions->GetUserId(), messageTraits, session);
            if (!chatUnreads) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (chatUnreads->empty()) {
                continue;
            }
            NJson::TJsonValue messagesList = NJson::JSON_ARRAY;
            for (auto&& message : *chatUnreads) {
                messagesList.AppendValue(chatRobot->GetMessageReport(locale, message, userId, chat.GetTopic()));
            }
            NJson::TJsonValue chatReport;
            chatReport["messages"] = std::move(messagesList);
            chatReport["chat_id"] = chatRobotIt.first + (chat.GetTopic() ? "." + chat.GetTopic() : "");
            chatReport["is_hidden"] = chatRobotIt.second->GetChatConfig().GetIsHidden();
            report.AppendValue(std::move(chatReport));
        }
    }
    g.MutableReport().AddReportElement("chats", std::move(report));
    g.SetCode(HTTP_OK);
}

class TChatRankData {
    R_READONLY(bool, IsActionRequired, false);
    R_READONLY(TInstant, LastActivityInstant, TInstant::Zero());
    R_READONLY(bool, IsLanding, false);
    R_READONLY(TInstant, LandingOverride, TInstant::Zero());
    R_READONLY(TInstant, LastAcceptedOverride, TInstant::Zero());

public:
    TChatRankData(const bool actionRequired, const TInstant lastActivityInstant, bool isLanding, TInstant landingOverride = TInstant::Zero(), TInstant lastAcceptedOverride = TInstant::Zero())
        : IsActionRequired(actionRequired)
        , LastActivityInstant(lastActivityInstant)
        , IsLanding(isLanding)
        , LandingOverride(landingOverride)
        , LastAcceptedOverride(lastAcceptedOverride)
    {
    }

    TInstant GetLandingInstant() const {
        TInstant landingTime = LandingOverride ? LandingOverride : LastAcceptedOverride;
        return landingTime ? landingTime : LastActivityInstant;
    }

    bool operator < (const TChatRankData& other) const {
        if (IsActionRequired ^ other.GetIsActionRequired()) {
            return IsActionRequired;
        }
        if (IsLanding && other.GetIsLanding()) {
            return GetLandingInstant() > other.GetLandingInstant();
        }
        return LastActivityInstant > other.GetLastActivityInstant();
    }
};

void TRobotChatListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto locale = GetLocale();
    auto userId = GetString(Context->GetCgiParameters(), "user_id", false);
    if (userId && userId != permissions->GetUserId()) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatsList);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
    } else {
        userId = permissions->GetUserId();
    }

    auto messageTraits = permissions->GetMessageVisibilityTraits();

    bool isAdminRequest = userId != permissions->GetUserId();
    IChatUserContext::TPtr ctx;
    if (isAdminRequest) {
        ctx = new TChatUserContext();
        ctx->SetUserId(userId);
    } else {
        ctx = BuildChatContext(permissions);
    }

    auto allowedStatuses = MakeSet(SplitString(GetHandlerSettingDef<TString>("allowed_statuses", ""), ","));

    TVector<TUserChatDescription> userChats;
    bool enableStories = permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Stories);
    if (requestData != NJson::JSON_NULL) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
        isAdminRequest = true;
        R_ENSURE(requestData.IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "post payload should be map");
        R_ENSURE(requestData.Has("chats") && requestData["chats"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "post payload should have an array 'chats'");
        for (auto&& chat : requestData["chats"].GetArray()) {
            R_ENSURE(chat.Has("user_id") && chat["user_id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "no 'user_id' in chat request");
            R_ENSURE(chat.Has("chat_id") && chat["chat_id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "no 'user_id' in chat request");
            auto userId = chat["user_id"].GetString();
            auto chatId = chat["chat_id"].GetString();
            TString topic = "";
            IChatRobot::ParseTopicLink(chatId, chatId, topic);
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatRobot, chatId);

            auto chatRobot = Server->GetChatRobot(chatId);
            if (!chatRobot) {
                continue;
            }

            NDrive::NChat::TChat chatModel;
            if (!chatRobot->GetChat(userId, topic, chatModel, false)) {
                continue;
            }

            userChats.push_back(TUserChatDescription(chatId, chatRobot, chatModel, userId));
        }
    } else {
        if (allowedStatuses.contains(permissions->GetStatus())
        || permissions->GetStatus() == NDrive::UserStatusActive
        || permissions->GetStatus() == NDrive::UserStatusOnboarding)
        {
            if (enableStories) {
                TMap<TString, TInstant> landingTimestamp;
                auto userLandingsFR = Server->GetDriveAPI()->GetUserLandingData()->FetchInfo(userId);
                TVector<TLanding> landingDescriptions;
                if (userLandingsFR.size() == 1) {
                    TSet<TString> landingIds;
                    auto userLandings = std::move(userLandingsFR.MutableResult().begin()->second);
                    for (auto&& landing : userLandings.GetLandings()) {
                        landingIds.insert(landing.GetId());
                        landingTimestamp[landing.GetId()] = landing.GetLastAcceptedAt();
                    }
                    landingDescriptions = Server->GetDriveAPI()->GetLandingsDB()->GetCachedObjectsVector(landingIds);
                }
                TChatContext emptyContext;
                for (auto&& ld : landingDescriptions) {
                    TString chatId = ld.GetChatId();
                    if (chatId == "announcements") {
                        chatId = "static_announcements";
                    } else if (!chatId) {
                        continue;
                    }
                    if (!ld.GetChatEnabled() || ld.GetChatDeadline() < Context->GetRequestStartTime()) {
                        continue;
                    }
                    if (ld.GetCheckAuditoryConditionInList() && ld.GetAuditoryCondition() && !ld.GetAuditoryCondition()->IsMatching(ctx, emptyContext)) {
                        continue;
                    }
                    auto chatRobot = Server->GetChatRobot(chatId);
                    if (!chatRobot) {
                        continue;
                    }
                    NDrive::NChat::TChat description;
                    {
                        if (chatId != "static_announcements" && !chatRobot->GetChat(userId, ld.GetChatMessagesGroup(), description)) {
                            continue;
                        } else {
                            description.SetId(0);
                            description.SetHandlerChatId("static_announcements");
                        }
                    }
                    description.SetTopic(ld.GetChatMessagesGroup());
                    description.SetIsMenuLanding(true);
                    description.SetCreatedAt(ld.GetTimestampOverride());

                    description.SetPreview(ld.GetPreview());
                    if (ld.GetChatIcon()) {
                        description.SetIcon(ld.GetChatIcon());
                    }
                    if (ld.GetChatTitle()) {
                        description.SetTitle(ld.GetChatTitle());
                    } else {
                        TChatRobotScriptItem item;
                        chatRobot->GetChatConfig().GetChatScript().GetScriptItemById(ld.GetChatMessagesGroup(), item);
                        if (!item.GetPreActionMessages().empty()) {
                            const auto& firstPreActionMessage = item.GetPreActionMessages().front();
                            if (firstPreActionMessage.GetType() == NDrive::NChat::IMessage::EMessageType::Plaintext) {
                                static const TString trimFront = "[color=#FFFFFF][b=32]";
                                static const TString trimBack = "[/b][/color]";
                                TString title = firstPreActionMessage.GetText();
                                if (title.StartsWith(trimFront)) {
                                    title = title.substr(trimFront.size());
                                }
                                if (title.EndsWith(trimBack)) {
                                    title = title.substr(0, title.size() - trimBack.size());
                                }
                                description.SetTitle(title);
                            }
                        }
                    }
                    TUserChatDescription d(chatId, chatRobot, description, userId);
                    d.TimestampOverride = landingTimestamp[ld.GetId()];
                    userChats.emplace_back(std::move(d));
                }
            }

            auto chatRobots = Server->GetChatRobots();
            R_ENSURE(chatRobots, ConfigHttpStatus.SyntaxErrorStatus, "no chat robots");
            for (auto&& chatRobotIt : chatRobots) {
                if (!isAdminRequest) {
                    if (chatRobotIt.second->GetChatConfig().GetIsHidden()) {
                        continue;
                    }
                }
                if (enableStories && chatRobotIt.first == "announcements") {
                    continue;
                }
                {
                    for (auto&& chat : chatRobotIt.second->GetTopics(ctx, true)) {
                        if (chat.GetFlag(NDrive::NChat::TChat::EChatFlags::Hidden)) {
                            continue;
                        }
                        userChats.push_back(TUserChatDescription(chatRobotIt.first, chatRobotIt.second, chat, userId));
                    }
                }
            }
        }
    }

    TMap<std::pair<TString, TString>, TDBTag> tagsForChats;
    TMap<TString, TDBTag> containerTags;
    if (isAdminRequest) {
        TSet<TString> userIds;
        for (auto&& chatRequest : userChats) {
            userIds.insert(chatRequest.OriginatorId);
        }
        TSet<TString> tagNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TSupportChatTag::TypeName, TUserContainerTag::TypeName, TSupportOutgoingCommunicationTag::TypeName});
        if (userIds.size() && tagNames.size()) {
            const auto& userTagManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
            auto session = BuildTx<NSQL::ReadOnly>();
            auto optionalTags = userTagManager.RestoreTags(userIds, MakeVector(tagNames), session);
            if (optionalTags) {
                for (auto&& tag : *optionalTags) {
                    auto tagImpl = tag.GetTagAs<ITopicLinkOwner>();
                    if (tagImpl) {
                        tagsForChats[std::make_pair(tag.GetObjectId(), tagImpl->GetTopicLink())] = tag;
                        continue;
                    }
                    auto containerTag = tag.GetTagAs<TUserContainerTag>();
                    if (containerTag) {
                        ITag::TPtr containerImpl = containerTag->RestoreTag();
                        auto topicLinkTag = containerImpl ? dynamic_cast<ITopicLinkOwner*>(containerImpl.Get()) : nullptr;
                        if (topicLinkTag) {
                            TDBTag tagCopy = tag;
                            tagCopy.SetData(containerImpl);
                            tagsForChats[std::make_pair(tag.GetObjectId(), topicLinkTag->GetTopicLink())] = tagCopy;
                            containerTags[tag.GetTagId()] = tag;
                        }
                        continue;
                    }
                }
            }
        }
    }

    ui32 totalUnread = 0;
    TSet<TString> participantIds;
    TVector<std::pair<TChatRankData, NJson::TJsonValue>> feed;
    auto tagsForObserve = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
    auto session = BuildChatSession(true);
    for (auto&& it : userChats) {
        NJson::TJsonValue chatDescription = it.GetReport(isAdminRequest);

        if (isAdminRequest) {
            TString id = it.ChatRobotId;
            if (it.ChatDescription.GetTopic()) {
                id += "." + it.ChatDescription.GetTopic();
            }
            auto tagIt = tagsForChats.find(std::make_pair(it.OriginatorId, id));
            if (tagIt != tagsForChats.end()) {
                if (!tagsForObserve.contains(tagIt->second->GetName())) {
                    INFO_LOG << "TRobotChatListProcessor: skipping chat because user " << permissions->GetUserId() << " has no rights to observe chat tag " << tagIt->second->GetName() << Endl;
                    continue;
                }
                chatDescription["tag_id"] = tagIt->second.GetTagId();
                if (tagIt->second->GetPerformer()) {
                    chatDescription["tag_performer_id"] = tagIt->second->GetPerformer();
                    participantIds.insert(tagIt->second->GetPerformer());
                } else {
                    chatDescription["tag_performer_id"] = NJson::JSON_NULL;
                }
                chatDescription["tag_data"] = tagIt->second->SerializeToJson();
                auto containerIt = containerTags.find(tagIt->second.GetTagId());
                if (containerIt != containerTags.end()) {
                    chatDescription["container_tag_data"] = containerIt->second->SerializeToJson();
                }
            } else {
                chatDescription["tag_id"] = NJson::JSON_NULL;
                chatDescription["tag_data"] = NJson::JSON_NULL;
            }
            TChatUserContext::TPtr ctx = new TChatUserContext();
            ctx->SetLocale(locale);
            ctx->SetUserId(it.OriginatorId);
            NChatOutputHelpers::GetExpectedActionReport(chatDescription, it.ChatRobotPtr, it.ChatDescription.GetTopic(), ctx);
        }

        NJson::TJsonValue messagesReport;
        if (auto totalCount = it.ChatRobotPtr->GetMessagesCount(it.ChatDescription.GetId(), messageTraits, session)) {
            messagesReport["total"] = *totalCount;
        } else {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        if (auto realUnreadCount = it.ChatRobotPtr->GetUnreadMessagesCount(it.ChatDescription.GetId(), permissions->GetUserId(), messageTraits, session))
        {
            auto statUnreadCount = it.ChatDescription.GetAtLeastOneFlag(NDrive::NChat::TChat::HideMessagesFlags) ? 0 : *realUnreadCount;
            messagesReport["unread"] = *realUnreadCount;
            totalUnread += statUnreadCount;
        } else {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        chatDescription["stats"] = std::move(messagesReport);

        if (it.OriginatorId) {
            participantIds.insert(it.OriginatorId);
        }

        TMaybe<NDrive::NChat::TMessageEvent> lastMessage;
        if (it.ChatRobotPtr->GetLastMessage(it.ChatDescription.GetId(), session, messageTraits, lastMessage) && lastMessage) {
            chatDescription["last_message"] = it.ChatRobotPtr->GetMessageReport(locale, *lastMessage, it.ChatDescription.GetObjectId(), it.ChatDescription.GetTopic());
            TMaybe<NDrive::NChat::TMessageEvent> lastMessageForRanking;
            if (it.ChatRobotPtr->GetLastMessage(it.ChatDescription.GetId(), session, 0, lastMessageForRanking) && lastMessageForRanking) {
                bool isActionNeeded = lastMessageForRanking->GetHistoryUserId() != permissions->GetUserId();
                if (!isAdminRequest) {
                    isActionNeeded = it.ChatRobotPtr->GetChatConfig().IsPinned();
                }
                feed.emplace_back(
                    std::make_pair(
                        TChatRankData(isActionNeeded, lastMessageForRanking->GetHistoryInstant(), it.ChatDescription.GetIsMenuLanding(), it.ChatDescription.GetCreatedAt(), it.TimestampOverride),
                        chatDescription
                    )
                );
                participantIds.insert(lastMessage->GetHistoryUserId());
            }
        }
    }

    Sort(
        feed.begin(),
        feed.end(),
        [](const std::pair<TChatRankData, NJson::TJsonValue>& lhs, const std::pair<TChatRankData, NJson::TJsonValue>& rhs) {
            return lhs.first < rhs.first;
        }
    );

    NJson::TJsonValue chatsList = NJson::JSON_ARRAY;
    for (auto&& it = feed.begin(); it != feed.end(); ++it) {
        chatsList.AppendValue(std::move(it->second));
    }

    g.MutableReport().AddReportElement("chats", std::move(chatsList));
    g.MutableReport().AddReportElement("users", NChatOutputHelpers::BuildParticipantsJsonReport(Server->GetDriveAPI()->GetUsersData(), participantIds));
    g.MutableReport().AddReportElement("total_unread", totalUnread);
    g.MutableReport().AddReportElement("user_id", userId);
    g.SetCode(HTTP_OK);
}

class TFeedChat {
    R_READONLY(TInstant, Timestamp);
    R_READONLY(NJson::TJsonValue, Report);

private:
    const TFilteredSupportChat* FilteredChat;

public:
    TFeedChat() = default;

    TFeedChat(const TInstant timestamp, const NJson::TJsonValue& report, const TFilteredSupportChat* filteredChat)
        : Timestamp(timestamp)
        , Report(report)
        , FilteredChat(filteredChat)
    {
    }

    const TFilteredSupportChat& GetFilteredChat() const {
        return *FilteredChat;
    }
};

template <typename T>
TSet<TString> GetFileteredTagIds(const TVector<T>& filteredRequests) {
    TSet<TString> tagIds;
    for (auto&& req : filteredRequests) {
        tagIds.emplace(req.GetTag().GetTagId());
    }
    return tagIds;
}

NJson::TJsonValue TChatAdminFeedProcessor::GetFilteredRequestsReport(const TVector<TBasicFilteredSupportRequest>& requests, TSet<TString>& participantIds, const TSupportRequestCategorizationDB* categorizer) {
    TMap<TString, TInstant> tagCreationTs;
    {
        auto session = BuildTx<NSQL::ReadOnly>();
        tagCreationTs = GetTagsEventTimestamps(EObjectHistoryAction::Add, GetFileteredTagIds(requests), session);
    }
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& request : requests) {
        NJson::TJsonValue report;
        report["tag_data"] = request.GetTag().BuildJsonReport();
        auto creationTimeIt = tagCreationTs.find(request.GetTag().GetTagId());
        if (creationTimeIt != tagCreationTs.end()) {
            report["tag_data"]["tag_created"] = creationTimeIt->second.Seconds();
        }
        if (categorizer) {
            auto fullCategorization = categorizer->GetActualCategorization(request.GetTag().GetTagId());
            report["tag_data"]["categorizations"] = fullCategorization.BuildReport();
        }
        result.AppendValue(std::move(report));
        participantIds.insert(request.GetTag().GetObjectId());
        if (request.GetTag()->GetPerformer()) {
            participantIds.insert(request.GetTag()->GetPerformer());
        }
    }
    return result;
}

TMap<TString, TInstant> TChatAdminFeedProcessor::GetTagsEventTimestamps(const EObjectHistoryAction action, const TSet<TString>& tagIds, NDrive::TEntitySession& session) const {
    TMap<TString, TInstant> tagCreationTs;
    TTagEventsManager::TQueryOptions queryOptions;
    queryOptions.AddGenericCondition("history_action", ToString(action));
    queryOptions.SetTagIds(tagIds);
    auto optionalEvents = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetEvents(TRange<TInstant>(), session, queryOptions);
    if (!optionalEvents) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    for (auto&& tagEvent : *optionalEvents) {
        tagCreationTs[tagEvent.GetTagId()] = tagEvent.GetHistoryTimestamp();
    }
    return tagCreationTs;
}

void TChatAdminFeedProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    const auto locale = GetLocale();
    size_t totalUnread = 0;

    TSet<TString> participantIds;
    TSet<TString> usedTagNames;

    TSet<TString> params = MakeSet(GetStrings(cgi, "request_types", false));
    if (params.empty()) {
        params.emplace("chat");
    }

    bool isReversedOrder = GetValue<bool>(cgi, "rev", false).GetOrElse(false);

    bool reportCategorizations = GetValue<bool>(cgi, "report_categorizations", false).GetOrElse(false);
    TSupportRequestCategorizationDB* categorizer = nullptr;
    if (reportCategorizations) {
        categorizer = Server->GetSupportCenterManager()->GetSupportRequestCategorizer();
        R_ENSURE(categorizer, ConfigHttpStatus.ServiceUnavailable, "support request categorizer not configured");
    }

    if (params.contains("chat")) {
        const bool disableAnswered = GetHandlerSettingDef<bool>("disable_answered", false);
        const bool countUnreadMessages = GetHandlerSettingDef<bool>("count_unread_messages", true);

        TChatSupportRequestsFilter filter;
        filter.InitFromCgi(Server, cgi, permissions);
        filter.SetChatRobotId("");
        filter.MutableMetricFilter().SetType(
            disableAnswered
            ? TChatMetricFilter::EMetricType::SinceLastUserGroupStart
            : TChatMetricFilter::EMetricType::AbsoluteWait
        );
        filter.MutableMetricFilter().SetMinValue(0);
        filter.MutableMetricFilter().SetMaxValue(TDuration::Days(3660).Seconds());
        {
            auto chatId = GetString(cgi, "chat_id", false);
            if (chatId) {
                TString chatRobotId;
                TString chatTopicId;
                IChatRobot::ParseTopicLink(chatId, chatRobotId, chatTopicId);
                filter.SetChatRobotId(chatRobotId);
                filter.SetChatTopic(chatTopicId);
            }
        }
        auto limit = GetValue<size_t>(cgi, "limit", false).GetOrElse(Max<size_t>());

        TVector<TFilteredSupportChat> filteredChats_;
        {
            TMessagesCollector errors;
            R_ENSURE(filter.GetSupportRequests(Server, filteredChats_, TInstant::Zero(), permissions->GetUserId(), errors), ConfigHttpStatus.SyntaxErrorStatus, "could not acquire chats according to filter " + errors.GetStringReport());
        }
        const TVector<TFilteredSupportChat> filteredChats = std::move(filteredChats_);

        TVector<TFeedChat> urgentChats;
        TVector<TFeedChat> answeredChats;
        auto chatRobots = Server->GetChatRobots();
        TMap<TString, TInstant> tagCreationTs;
        {
            auto session = BuildTx<NSQL::ReadOnly>();
            tagCreationTs = GetTagsEventTimestamps(EObjectHistoryAction::Add, GetFileteredTagIds(filteredChats), session);
        }
        for (auto&& chatEntry : filteredChats) {
            const TString& userId = chatEntry.GetTag().GetObjectId();

            auto tagImpl = chatEntry.GetTag().GetTagAs<TSupportChatTag>();
            if (!tagImpl) {
                continue;
            }
            TString chatRobotId;
            TString chatRobotTopic;
            IChatRobot::ParseTopicLink(tagImpl->GetTopicLink(), chatRobotId, chatRobotTopic);

            auto chatRobot = Server->GetChatRobot(chatRobotId);
            if (!chatRobot) {
                continue;
            }

            NJson::TJsonValue chatReport;
            chatReport["id"] = tagImpl->GetTopicLink();
            chatReport["originator"] = userId;
            chatReport["tag_data"] = chatEntry.GetTag().BuildJsonReport();
            auto creationTimeIt = tagCreationTs.find(chatEntry.GetTag().GetTagId());
            if (creationTimeIt != tagCreationTs.end()) {
                chatReport["tag_data"]["tag_created"] = creationTimeIt->second.Seconds();
            }
            if (reportCategorizations && categorizer) {
                auto fullCategorization = categorizer->GetActualCategorization(chatEntry.GetTag().GetTagId());
                chatReport["tag_data"]["categorizations"] = fullCategorization.BuildReport();
            }

            // header data
            NDrive::NChat::TChat chat;
            if (chatRobot->GetChat(userId, chatRobotTopic, chat)) {
                chatReport["name"] = chatRobot->GetChatTitle(chat);
            } else {
                continue;
            }

            // message stats
            {
                NJson::TJsonValue messagesReport;
                messagesReport["total"] = chatEntry.GetMessagesCount().Total;
                messagesReport["unread"] = chatEntry.GetMessagesCount().Unread;
                chatReport["stats"] = std::move(messagesReport);
                totalUnread +=
                    countUnreadMessages
                    ? chatEntry.GetMessagesCount().Unread
                    : chatEntry.GetMessagesCount().Unread ? 1 : 0;
            }

            chatReport["last_message"] = chatRobot->GetMessageReport(locale, chatEntry.GetDisplayMessage(), userId, chatRobotTopic);

            if (disableAnswered
                || (chatEntry.GetDisplayMessage().GetHistoryUserId() == userId || chatEntry.GetDisplayMessage().GetHistoryUserId().StartsWith("robot-"))
                && !tagImpl->IsMuted()) {
                urgentChats.emplace_back(TFeedChat(Context->GetRequestStartTime() - TDuration::Seconds(chatEntry.GetMetricValue()), std::move(chatReport), &chatEntry));
            } else {
                answeredChats.emplace_back(TFeedChat(chatEntry.GetDisplayMessage().GetHistoryInstant(), std::move(chatReport), &chatEntry));
            }
        }

        const auto comparator = [isReversedOrder](const TFeedChat& lhs, const TFeedChat& rhs) {
            return (lhs.GetTimestamp() < rhs.GetTimestamp()) ^ isReversedOrder;
        };
        Sort(urgentChats.begin(), urgentChats.end(), comparator);
        Sort(answeredChats.begin(), answeredChats.end(), comparator);
        urgentChats.resize(std::min(urgentChats.size(), limit));
        answeredChats.resize(std::min(answeredChats.size(), limit));

        NJson::TJsonValue chatsReport;
        {
            NJson::TJsonValue urgentChatsJson = NJson::JSON_ARRAY;
            for (auto&& chat : urgentChats) {
                urgentChatsJson.AppendValue(std::move(chat.GetReport()));
                participantIds.insert(chat.GetFilteredChat().GetTag().GetObjectId());
                participantIds.insert(chat.GetFilteredChat().GetDisplayMessage().GetHistoryUserId());
            }
            chatsReport.InsertValue("urgent", std::move(urgentChatsJson));
        }
        {
            NJson::TJsonValue answeredChatsJson = NJson::JSON_ARRAY;
            for (auto&& chat : answeredChats) {
                answeredChatsJson.AppendValue(std::move(chat.GetReport()));
                participantIds.insert(chat.GetFilteredChat().GetTag().GetObjectId());
                participantIds.insert(chat.GetFilteredChat().GetDisplayMessage().GetHistoryUserId());
            }
            chatsReport.InsertValue("answered", std::move(answeredChatsJson));
        }

        for (auto&& tag : filter.GetQueryTagNames(Server)) {
            usedTagNames.insert(tag);
        }

        g.MutableReport().AddReportElement("chats", std::move(chatsReport));
        g.MutableReport().AddReportElement("total_unread", totalUnread);
    }

    if (params.contains("call")) {
        TCallSupportRequestsFilter filter;
        filter.InitFromCgi(Server, cgi, permissions);
        TVector<TBasicFilteredSupportRequest> filteredCalls;
        {
            TMessagesCollector errors;
            R_ENSURE(filter.GetSupportRequests(Server, filteredCalls, errors), ConfigHttpStatus.SyntaxErrorStatus, "could not acquire calls according to filter " + errors.GetStringReport());
        }
        auto callsList = GetFilteredRequestsReport(filteredCalls, participantIds, categorizer);

        for (auto&& tag : filter.GetQueryTagNames(Server)) {
            usedTagNames.insert(tag);
        }

        g.MutableReport().AddReportElement("calls", std::move(callsList));
    }

    if (params.contains("outgoing")) {
        NJson::TJsonArray outgoingList;
        for (auto& filter : {
                TAtomicSharedPtr<TSimpleSupportRequestsFilter>(new TOutgoingSupportRequestsFilter()),
                TAtomicSharedPtr<TSimpleSupportRequestsFilter>(new TOutgoingFleetSupportRequestsFilter())
            })
        {
            filter->InitFromCgi(Server, cgi, permissions);
            TVector<TBasicFilteredSupportRequest> filteredOutgoing;
            {
                TMessagesCollector errors;
                R_ENSURE(filter->GetSupportRequests(Server, filteredOutgoing, errors), ConfigHttpStatus.SyntaxErrorStatus, "could not acquire outgoing requests according to filter " + errors.GetStringReport());
            }

            auto filteredReport = GetFilteredRequestsReport(filteredOutgoing, participantIds, categorizer);
            for (const auto& r : filteredReport.GetArray()) {
                outgoingList.AppendValue(std::move(r));
            }

            for (auto&& tag : filter->GetQueryTagNames(Server)) {
                usedTagNames.insert(tag);
            }
        }

        g.MutableReport().AddReportElement("outgoing", std::move(outgoingList));
    }

    if (params.contains("status_tags")) {
        TSet<TString> statusTags = MakeSet(SplitString(GetHandlerSettingDef<TString>("feed.status_tags", ""), ","));
        NJson::TJsonValue tagsReport(NJson::JSON_ARRAY);
        auto taggedUser = DriveApi->GetTagsManager().GetUserTags().GetCachedObject(permissions->GetUserId());
        if (taggedUser) {
            for (const auto& tag : taggedUser->GetTags()) {
                if (!tag) {
                    continue;
                }
                if (!statusTags.contains(tag->GetName())) {
                    continue;
                }
                tagsReport.AppendValue(tag.BuildJsonReport());
            }
        }

        for (auto&& tag : statusTags) {
            usedTagNames.insert(tag);
        }

        g.MutableReport().AddReportElement("status_tags", std::move(tagsReport));
    }

    if (params.contains("container")) {
        TUserContainerSupportRequestsFilter filter;
        filter.InitFromCgi(Server, cgi, permissions);
        TVector<TBasicFilteredSupportRequest> filteredContainers;
        {
            TMessagesCollector errors;
            R_ENSURE(filter.GetSupportRequests(Server, filteredContainers, errors), ConfigHttpStatus.SyntaxErrorStatus, "could not acquire container requests according to filter " + errors.GetStringReport());
        }
        TMaybe<TSet<TString>> containedNames;
        if (cgi.Has("contained_tag_names")) {
            containedNames = MakeSet(GetStrings(cgi, "contained_tag_names"));
        }
        if (cgi.Has("contained_tag_types")) {
            TSet<TString> containedTypes = MakeSet(GetStrings(cgi, "contained_tag_types"));
            if (containedNames.Defined()) {
                auto reqTypeNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames(containedTypes);
                containedNames->insert(reqTypeNames.begin(), reqTypeNames.end());
            } else {
                containedNames = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames(containedTypes);
            }
        }
        bool checkIfChat = GetValue<bool>(cgi, "is_chat", false).GetOrElse(false);

        TMap<TString, TInstant> tagCreationTs;
        {
            auto session = BuildTx<NSQL::ReadOnly>();
            tagCreationTs = GetTagsEventTimestamps(EObjectHistoryAction::Add, GetFileteredTagIds(filteredContainers), session);
        }
        NJson::TJsonValue result = NJson::JSON_ARRAY;
        for (auto&& request : filteredContainers) {
            auto tag = request.GetTag();
            auto containerTag = tag.GetTagAs<TUserContainerTag>();
            if (!containerTag) {
                continue;
            }
            auto restoredTag = containerTag->RestoreTag();
            if (!restoredTag) {
                continue;
            }
            if (checkIfChat) {
                auto topicLinkOwner = dynamic_cast<ITopicLinkOwner*>(restoredTag.Get());
                if (!topicLinkOwner) {
                    continue;
                }
            }
            if (containedNames.Defined() && !containedNames->contains(restoredTag->GetName())) {
                continue;
            }

            NJson::TJsonValue tagReport = restoredTag->SerializeToJson();
            tagReport.InsertValue("object_id", tag.GetObjectId());
            tagReport.InsertValue("tag_id", tag.GetTagId());
            auto creationTimeIt = tagCreationTs.find(request.GetTag().GetTagId());
            if (creationTimeIt != tagCreationTs.end()) {
                tagReport.InsertValue("tag_created", creationTimeIt->second.Seconds());
            }
            if (reportCategorizations && categorizer) {
                auto fullCategorization = categorizer->GetActualCategorization(request.GetTag().GetTagId());
                tagReport["categorizations"] = fullCategorization.BuildReport();
            }
            result.AppendValue(std::move(tagReport));

            participantIds.insert(tag.GetObjectId());
        }

        for (auto&& tag : filter.GetQueryTagNames(Server)) {
            usedTagNames.insert(tag);
        }
        if (containedNames.Defined()) {
            for (auto&& tag : *containedNames) {
                usedTagNames.insert(tag);
            }
        }
        g.MutableReport().AddReportElement("container", std::move(result));
    }

    NJson::TJsonValue tagsReport = NJson::JSON_ARRAY;
    for (auto&& tagName : usedTagNames) {
        auto td = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
        if (!td) {
            continue;
        }
        tagsReport.AppendValue(td->BuildJsonReport(locale));
    }
    g.MutableReport().AddReportElement("tags", std::move(tagsReport));
    g.MutableReport().AddReportElement("users", NChatOutputHelpers::BuildParticipantsJsonReport(Server->GetDriveAPI()->GetUsersData(), participantIds));
    g.SetCode(HTTP_OK);
}

void TChatFilterProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
    TChatSupportRequestsFilter filter;
    R_ENSURE(filter.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "could not deserialize filter from json");
    filter.SetPermissions(permissions);
    TVector<TFilteredSupportChat> filteredChats;
    TMessagesCollector errors;
    R_ENSURE(filter.GetSupportRequests(Server, filteredChats, TInstant::Zero(), permissions->GetUserId(), errors), ConfigHttpStatus.SyntaxErrorStatus, "could not acquire chats according to filter " + errors.GetStringReport());
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& filteredChat : filteredChats) {
        result.AppendValue(filteredChat.BuildReport());
    }
    g.MutableReport().AddReportElement("chats", std::move(result));
    g.SetCode(HTTP_OK);
}

void TRobotChatResourceProcessor::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    // Deduce chat
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString chatId(GetString(cgi, "chat_id", false));
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);

    auto userId = permissions->GetUserId();
    {
        auto operatedUserId = GetString(cgi, "user_id", false);
        if (operatedUserId && operatedUserId != userId) {
            R_ENSURE(
                permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatResource),
                HTTP_FORBIDDEN,
                "no permissions to Observe::ChatResource"
            );
            userId = std::move(operatedUserId);
        }
    }

    TBlob postData = Context->GetBuf();
    TMemoryInput inp(postData.Data(), postData.Size());
    auto postDataStr = inp.ReadAll();
    TString resourceId = GetString(cgi, "resource_id", false);

    bool shared = GetValue<bool>(cgi, "shared", false).GetOrElse(false);
    if (shared && postDataStr && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatResource)) {
        R_ENSURE(
            permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatResource),
            HTTP_FORBIDDEN,
            "no permissions to Modify::ChatResource"
        );
    }

    if (postDataStr) {
        TString contentType(Context->GetRequestData().HeaderInOrEmpty("Content-Type"));
        if (shared) {
            ProcessUploadShared(g, userId, resourceId, postDataStr, contentType);
        } else {
            ProcessUpload(g, userId, chatId, topic, resourceId, postDataStr, contentType);
        }
    } else {
        if (!resourceId) {
            resourceId = GetString(cgi, "photo_id", false);
        }
        if (!chatId) {
            auto photoFetchResult = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(resourceId);
            auto photo = photoFetchResult.GetResultPtr(resourceId);
            if (photo) {
                chatId = photo->GetOriginChat();
                IChatRobot::ParseTopicLink(chatId, chatId, topic);
            }
        }
        ProcessAcquire(g, userId, chatId, topic, resourceId);
    }
}

void TRobotChatResourceProcessor::ProcessUploadShared(TJsonReport::TGuard& g, const TString& userId, const TString& resourceId, const TString& content, const TString& contentType) {
    R_ENSURE(Server->GetChatRobotsManager(), ConfigHttpStatus.ServiceUnavailable, "chat robots manager storage not configured");
    auto mediaStorage = Server->GetChatRobotsManager()->GetChatRobotsMediaStorage();
    R_ENSURE(mediaStorage != nullptr, ConfigHttpStatus.ServiceUnavailable, "chat media storage not configured");
    {
        auto timeout = Context->GetRequestDeadline() - Now();
        auto session = Yensured(Server->GetChatEngine())->BuildSession(false, false, TDuration::Zero(), timeout);
        session.SetOriginatorId(userId);
        R_ENSURE(mediaStorage->RegisterResource(userId, resourceId, contentType, true, session), {}, "cannot RegisterResource", session);
        R_ENSURE(session.Commit(), {}, "cannot Commit", session);
    }
    auto callback = MakeAtomicShared<TChatMediaSimpleCallback>(g.GetReport(), userId);
    g.Release();
    mediaStorage->UploadResource(userId, resourceId, contentType, content, callback);
}

void TRobotChatResourceProcessor::ProcessUpload(TJsonReport::TGuard& g, const TString& userId, const TString& chatId, const TString& topic, const TString& resourceId, const TString& content, const TString& contentType) {
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    auto session = chatRobot->BuildChatEngineSession(false);
    {
        R_ENSURE(chatRobot->EnsureChat(userId, topic, session, true), HTTP_NOT_FOUND, "chat not found");
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    auto callback = MakeAtomicShared<TChatMediaSimpleCallback>(g.GetReport(), userId);
    auto context = MakeAtomicShared<TChatUserContext>();
    context->SetUserId(userId);
    context->SetChatId(chatId);
    context->SetChatTopic(topic);
    g.Release();
    chatRobot->AcceptMediaResource(context, resourceId, contentType, content, callback);
}

void MakeResourceReplyReport(const NThreading::TFuture<TChatResourceAcquisitionResult>& result, IServerReportBuilder::TPtr report, const NDrive::TEventLog::TState& eventLogState) {
    Y_ENSURE(report);
    NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
    if (result.HasValue()) {
        if (result.GetValue().GetLocation().empty()) {
            TBuffer reportBuffer;
            reportBuffer.Assign(result.GetValue().GetContent().AsCharPtr(), result.GetValue().GetContent().Size());
            report->Finish(HTTP_OK, result.GetValue().GetDescription().GetContentType(), reportBuffer);
        } else {
            auto context = report->GetContext();
            auto processorName = report->GetHandlerName();
            Y_ENSURE(context);
            context->AddReplyInfo("Location", result.GetValue().GetLocation());
            context->MakeSimpleReply(TBuffer(), HTTP_FOUND);
            NDrive::TUnistatSignals::OnReply(processorName, HTTP_FOUND, *context);
            NDrive::TEventLog::Log(NDrive::TEventLog::ResponseInfo, *context, HTTP_FOUND, NJson::TMapBuilder
                ("Location", result.GetValue().GetLocation())
            );
        }
    } else {
        HttpCodes code = HTTP_INTERNAL_SERVER_ERROR;
        auto currentException = NThreading::GetException(result);
        NJson::TJsonValue jsonReport = NThreading::GetExceptionInfo(result);
        try {
            result.GetValue();
        } catch(const NDrive::NChat::TBadRequestException&) {
            code = HTTP_BAD_REQUEST;
        } catch(const NDrive::NChat::TAccessForbiddenException&) {
            code = HTTP_FORBIDDEN;
        }

        TJsonReport::TGuard g(report);
        g.SetExternalReport(std::move(jsonReport));
        g.SetCode(code);
    }
}

void TRobotChatResourceProcessor::ProcessAcquire(TJsonReport::TGuard& g, const TString& userId, const TString& chatId, const TString& topic, const TString& resourceId) {
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    {
        auto session = chatRobot->BuildChatEngineSession(false);
        R_ENSURE(chatRobot->EnsureChat(userId, topic, session, true), HTTP_NOT_FOUND, "chat not found");
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    auto useCache = GetString(Context->GetCgiParameters(), "allow_redirects", false);
    auto context = MakeAtomicShared<TChatUserContext>();
    context->SetUserId(userId);
    context->SetChatId(chatId);
    context->SetChatTopic(topic);
    auto eventLogState = NDrive::TEventLog::CaptureState();
    chatRobot->GetMediaResource(context, resourceId, !useCache).Subscribe([
        eventLogState,
        report = g.GetReport()
    ](const auto& result) {
        MakeResourceReplyReport(result, report, eventLogState);
    });
    g.Release();
}

void TRobotChatResetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = permissions->GetUserId();
    auto chatId = GetString(Context->GetCgiParameters(), "chat_id");
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);

    auto chatRobot = Server->GetChatRobot(chatId);

    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    R_ENSURE(chatRobot->ResetChat(userId, topic), ConfigHttpStatus.UnknownErrorStatus, "can't reset state");

    if (!!dynamic_cast<const TRegistrationChatBot*>(chatRobot.Get())) {
        auto session = BuildTx<NSQL::Writable>();
        auto userId = permissions->GetUserId();
        auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
        R_ENSURE(userFetchResult, {}, "cannot FetchInfo for " << userId, session);
        R_ENSURE(!userFetchResult.empty(), ConfigHttpStatus.UnknownErrorStatus, "no such user");
        auto p = userFetchResult.MutableResult().begin();
        auto userData = std::move(p->second);
        userData.SetStatus(NDrive::UserStatusOnboarding);
        userData.SetFirstName("");
        userData.SetLastName("");
        userData.SetPName("");
        if (!DriveApi->GetUsersData()->UpdateUser(userData, userId, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.SetCode(HTTP_OK);
}

void TRobotChatRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto userId = GetString(requestData, "user_id");
    auto chatId = GetString(requestData, "chat_id");
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);

    auto dropState = GetString(requestData, "drop_state", false);

    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");

    if (dropState) {
        R_ENSURE(chatRobot->RemoveChat(userId, topic, permissions->GetUserId()), ConfigHttpStatus.UnknownErrorStatus, "can't remove chat");
    } else {
        R_ENSURE(chatRobot->ArchiveChat(userId, topic), ConfigHttpStatus.UnknownErrorStatus, "can't archive chat");
    }

    g.SetCode(HTTP_OK);
}

void TRobotChatResourcePreviewProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = permissions->GetUserId();

    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString chatId(GetString(cgi, "chat_id"));
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    TString resourceId(GetString(cgi, "resource_id"));

    {
        TString requestedUserId(GetString(cgi, "user_id", false));
        if (requestedUserId && requestedUserId != permissions->GetUserId()) {
            CheckUserGeneralAccessRights(requestedUserId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess | NAccessVerification::EAccessVerificationTraits::ChatObserveAccess, true);
            userId = std::move(requestedUserId);
        }
    }

    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    {
        auto session = BuildChatSession(false);
        R_ENSURE(chatRobot->EnsureChat(userId, topic, session, true), HTTP_NOT_FOUND, "chat not found");
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    auto useCache = GetString(Context->GetCgiParameters(), "allow_redirects", false);
    auto context = MakeAtomicShared<TChatUserContext>();
    context->SetUserId(userId);
    context->SetChatId(chatId);
    context->SetChatTopic(topic);
    TMap<TString, TString> resourcePreviewOverrides;
    auto previewOverridesJson = Server->GetSettings().GetJsonValue("resource_preview.overrides");
    for (auto [type, previewPath] : previewOverridesJson.GetMap()) {
        resourcePreviewOverrides[type] = previewPath.GetStringRobust();
    }
    auto eventLogState = NDrive::TEventLog::CaptureState();
    chatRobot->GetMediaResourcePreview(context, resourceId, !useCache, resourcePreviewOverrides).Subscribe([
        report = g.GetReport(),
        eventLogState,
        chatRobot,
        userId
    ] (const NThreading::TFuture<TChatResourceAcquisitionResult>& result) {
        if (result.HasValue() && result.GetValue().IsGenerated()) {
            auto description = result.GetValue().GetDescription();
            chatRobot->UploadResourcePreview(TString(result.GetValue().GetContent().AsCharPtr(), result.GetValue().GetContent().Size()), description, result.GetValue().GetPreviewType()).Apply([description, chatRobot, userId](const auto& uploadResult) mutable {
                if (uploadResult.HasException() || !uploadResult.HasValue()) {
                    ERROR_LOG << "TRobotChatResourcePreviewProcessor::ProcessServiceRequest: failed to upload preview for " << description.GetId() << ", error: " << NThreading::GetExceptionMessage(uploadResult) << Endl;
                } else {
                    description.SetHasPreview(true);
                    auto session = chatRobot->BuildChatEngineSession(false);
                    if (!chatRobot->UpdateMediaResourceDescription(description, userId, session) || !session.Commit()) {
                        ERROR_LOG << "TRobotChatResourcePreviewProcessor::ProcessServiceRequest: failed to update resource preview description for " << description.GetId() << Endl;
                    }
                }
            });
        }
        MakeResourceReplyReport(result, report, eventLogState);
    });
    g.Release();
}

void TRobotChatEditMessageProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto userId = GetString(requestData, "user_id");
    auto chatId = GetString(requestData, "chat_id");
    bool force = false;
    if (requestData.Has("force")) {
        R_ENSURE(requestData["force"].IsBoolean(), ConfigHttpStatus.SyntaxErrorStatus, "'force' parameter is not boolean");
        force = requestData["force"].GetBoolean();
    }
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);

    auto operatorId = permissions->GetUserId();

    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    R_ENSURE(!chatRobot->GetChatConfig().GetIsStatic(), ConfigHttpStatus.SyntaxErrorStatus, "can't send messages to static chat");

    R_ENSURE(requestData.Has("message_id") && requestData["message_id"].IsInteger(), ConfigHttpStatus.SyntaxErrorStatus, "no 'message_id' or it is not integer");
    ui64 messageId = requestData["message_id"].GetInteger();

    auto newText = GetString(requestData, "text", false);
    auto isDelete = GetString(requestData, "delete", false);
    R_ENSURE(!newText.empty() ^ !isDelete.empty(), ConfigHttpStatus.SyntaxErrorStatus, "only one of {'text', 'is_delete'} should be specified");
    R_ENSURE(force || chatRobot->GetLastViewedMessageId(userId, topic, userId) < messageId, ConfigHttpStatus.SyntaxErrorStatus, "user had already read this message");

    if (newText) {
        auto newType = GetValue<NDrive::NChat::TMessage::EMessageType>(requestData, "type", false);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatMessage);
        auto session = BuildChatSession(false);
        if (!chatRobot->EditMessage(permissions->GetUserId(), messageId, NDrive::NChat::TMessageEdit(newText, newType), session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::ChatMessage);
        auto session = BuildChatSession(false);
        if (!chatRobot->DeleteMessage(permissions->GetUserId(), messageId, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.SetCode(HTTP_OK);
}

void TChatRobotRateMessage::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto requestUserId = permissions->GetUserId();
    auto userId = GetString(requestData, "user_id", false);
    if (!userId) {
        userId = requestUserId;
    }
    if (userId != requestUserId) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatMessage);
    }
    auto ratingText = GetString(requestData, "rating", true);
    ui64 messageId = GetValue<ui64>(requestData, "message_id", true).GetOrElse(0);
    auto chatId = GetString(requestData, "chat_id");
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    R_ENSURE(!chatRobot->GetChatConfig().GetIsStatic(), ConfigHttpStatus.SyntaxErrorStatus, "cannot edit messages in static chat");
    {
        auto session = BuildChatSession(false);
        if (!chatRobot->EditMessage(requestUserId, messageId, NDrive::NChat::TMessageEdit({}, {}, {}, ratingText, false), session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    g.SetCode(HTTP_OK);
}

void TChatRobotMarkReadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto userId = permissions->GetUserId();
    auto chatId = GetString(cgi, "chat_id");
    auto messageIdRaw = GetString(cgi, "message_id");
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    ui64 messageId;
    R_ENSURE(TryFromString(messageIdRaw, messageId), ConfigHttpStatus.SyntaxErrorStatus, "'message_id' should be int");

    auto messageTraits = permissions->GetMessageVisibilityTraits();

    ui64 currentLastViewed = chatRobot->GetLastViewedMessageId(userId, topic, userId);
    NDrive::NChat::TOptionalMessageEvents viewedMessages;
    {
        auto session = BuildChatSession(false);
        viewedMessages = chatRobot->GetChatMessagesRange(userId, topic, messageTraits, currentLastViewed + 1, messageId + 1, session);
        if (!viewedMessages || !chatRobot->UpdateLastViewedMessageId(userId, topic, userId, messageId, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    TSet<TString> processedLandingIds;
    TVector<TLandingAcceptance> landingAcceptances;
    for (auto&& message : *viewedMessages) {
        if (message.GetType() != NDrive::NChat::TMessage::EMessageType::Introscreen || processedLandingIds.contains(message.GetText())) {
            continue;
        }
        landingAcceptances.emplace_back(TLandingAcceptance().SetId(message.GetText()).SetLastAcceptedAt(Context->GetRequestStartTime()));
        processedLandingIds.insert(message.GetText());
    }
    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->AcceptLandings(*permissions, landingAcceptances, Context->GetRequestStartTime(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot commit tags session");
    g.SetCode(HTTP_OK);
}

void TChatRobotStickerListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetChatEngine(), ConfigHttpStatus.ServiceUnavailable, "chat engine not configured");
    auto stickerManager = Server->GetChatEngine()->GetStickerManager();
    R_ENSURE(stickerManager, ConfigHttpStatus.ServiceUnavailable, "sticker manager not configured");
    g.MutableReport().AddReportElement(
        "stickers",
        stickerManager->GetStickersReport(
            Config.GetIsAdminProcessor() & permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatSticker)
        )
    );
    g.SetCode(HTTP_OK);
}

void TChatRobotStickerEditProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetChatEngine(), ConfigHttpStatus.ServiceUnavailable, "chat engine not configured");
    auto stickerManager = Server->GetChatEngine()->GetStickerManager();
    R_ENSURE(stickerManager, ConfigHttpStatus.ServiceUnavailable, "sticker manager not configured");
    NDrive::NChat::TSticker stickerData;
    R_ENSURE(stickerData.ParseFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "can't deserialize sticker from json");
    auto command = GetString(requestData, "operation");
    auto session = BuildChatSession();
    if (command == "add") {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::ChatSticker);
        R_ENSURE(stickerData.GetUrl(), ConfigHttpStatus.SyntaxErrorStatus, "url is not specified");
        if (!stickerManager->AddSticker(stickerData, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else if (command == "update") {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatSticker);
        R_ENSURE(stickerData.GetId(), ConfigHttpStatus.SyntaxErrorStatus, "id is not specified");
        if (!stickerManager->EditSticker(stickerData, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else if (command == "remove") {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::ChatSticker);
        R_ENSURE(stickerData.GetId(), ConfigHttpStatus.SyntaxErrorStatus, "id is not specified");
        if (!stickerManager->RemoveSticker(stickerData.GetId(), permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        R_ENSURE(false, ConfigHttpStatus.SyntaxErrorStatus, "unknown operation: '" + command + "', only {'add', 'update', 'remove'} are supported");
    }
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "can't commit session");
    g.SetCode(HTTP_OK);
}

void TChatRobotStickerHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatSticker);
    R_ENSURE(Server->GetChatEngine(), ConfigHttpStatus.ServiceUnavailable, "chat engine not configured");
    auto stickerManager = Server->GetChatEngine()->GetStickerManager();
    R_ENSURE(stickerManager, ConfigHttpStatus.ServiceUnavailable, "sticker manager not configured");

    TVector<TAtomicSharedPtr<TObjectEvent<NDrive::NChat::TSticker>>> history;
    R_ENSURE(stickerManager->GetHistoryManager().GetEventsAll(TInstant::Zero(), history, TInstant::Zero()), ConfigHttpStatus.UnknownErrorStatus, "can't get history");
    NJson::TJsonValue historyJson = NJson::JSON_ARRAY;
    for (auto&& entry : history) {
        historyJson.AppendValue(entry->BuildReportItem());
    }

    g.MutableReport().AddReportElement("history", std::move(historyJson));
    g.SetCode(HTTP_OK);
}

void TEditChatFlagsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto userId = permissions->GetUserId();

    auto chatId = GetString(cgi, "chat_id");
    auto flagRaw = GetString(cgi, "flag");
    bool enabled = GetValue<bool>(cgi, "enabled").GetRef();

    NDrive::NChat::TChat::EChatFlags flag;
    R_ENSURE(TryFromString(flagRaw, flag), ConfigHttpStatus.SyntaxErrorStatus, "unknown flag");

    TString topic;
    NDrive::NChat::TChat chat;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "unknown chat logic");
    R_ENSURE(chatRobot->GetChat(userId, topic, chat), ConfigHttpStatus.SyntaxErrorStatus, "no such chat");

    if (chat.GetFlag(flag) != enabled) {
        if (enabled) {
            chat.SetFlag(flag);
        } else {
            chat.DropFlag(flag);
        }
        auto session = BuildChatSession(false);
        R_ENSURE(Server->GetChatEngine()->GetChats().Upsert(chat, permissions->GetUserId(), session) && session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "failed db write");
    }

    g.SetCode(HTTP_OK);
}

TVector<TContextMapInfo> BuildChatContextMapFromRequest(const NJson::TJsonValue::TArray& context) {
    TVector<TContextMapInfo> contextValues;
    for (auto&& field : context) {
        TString key, value;
        R_ENSURE(NJson::ParseField(field, "key", key, true) && NJson::ParseField(field, "value", value, true), HTTP_BAD_REQUEST, "incorrect context data");
        TContextMapInfo mapField(key, value);
        EContextDataType dataType;
        if (key == "longterm_session_id") {
            mapField.Type = EContextDataType::SessionId;
        } else if (TryFromString(key, dataType)) {
            mapField.Type = dataType;
        }
        contextValues.emplace_back(std::move(mapField));
    }
    return contextValues;
}

bool UpsertChatRobotContext(const IChatRobot& chatRobot, const NJson::TJsonValue::TArray& context, const TString& userId, const TString& topic, NDrive::TEntitySession& session) {
    TVector<TContextMapInfo> contextValues = BuildChatContextMapFromRequest(context);
    if (!contextValues.empty()) {
        return chatRobot.AddToContext(contextValues, userId, topic, session);
    }
    return true;
}

void TChatNodeJumpProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Send, TAdministrativeAction::EEntity::ChatMessage);
    auto userId = GetString(requestData, "user_id");
    auto chatId = GetString(requestData, "chat_id");
    auto nodeName = GetString(requestData, "node_name");
    bool sendMessages = true;
    if (requestData.Has("send_messages")) {
        R_ENSURE(requestData["send_messages"].IsBoolean(), ConfigHttpStatus.SyntaxErrorStatus, "send_messages should be boolean");
        sendMessages = requestData["send_messages"].GetBoolean();
    }
    TString topic;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "there is no chat robot");
    auto session = BuildChatSession(/* readOnly = */ false);
    if (requestData["context"].IsArray()) {
        R_ENSURE(UpsertChatRobotContext(*chatRobot, requestData["context"].GetArray(), userId, topic, session), ConfigHttpStatus.UnknownErrorStatus, "could not upsert context", session);
    }
    R_ENSURE(chatRobot->MoveToStep(userId, topic, nodeName, &session, sendMessages) && session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "could not move to step", session);
    g.SetCode(HTTP_OK);
}

void TChatDialogueClassifierProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage);
    R_ENSURE(Server->GetTaxiSupportClassifier(), ConfigHttpStatus.ServiceUnavailable, "taxi support classifier not configured");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto userId = GetString(cgi, "user_id");
    auto chatId = GetString(cgi, "chat_id");
    TString topic;
    TString mlRequestId = chatId + "+" + userId;
    IChatRobot::ParseTopicLink(chatId, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "there is no chat robot");
    NDrive::NChat::TOptionalMessageEvents messages;
    {
        auto session = BuildChatSession(true);
        messages = chatRobot->GetChatMessages(userId, topic, session);
        if (!messages) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    TVector<NJson::TJsonValue> dialogue;
    TVector<NJson::TJsonValue> userMonologue;
    bool isMonologue = true;
    for (auto&& message : *messages) {
        NJson::TJsonValue dialogueTurn;
        dialogueTurn["author_id"] = message.GetHistoryUserId();
        dialogueTurn["message"] = message.GetText();
        isMonologue = isMonologue && (!dialogue || (dialogueTurn["author_id"] == dialogue.back()["author_id"]));
        dialogue.push_back(dialogueTurn);
        if (isMonologue) {
            userMonologue.push_back(dialogueTurn);
        }
    }
    auto asyncDialoguePreds = Server->GetTaxiSupportClassifier()->Classify(dialogue, userId, mlRequestId);
    auto asyncUserMonologuePreds = Server->GetTaxiSupportClassifier()->Classify(userMonologue, userId, mlRequestId);

    asyncUserMonologuePreds.Wait(Context->GetRequestDeadline());
    R_ENSURE(!asyncUserMonologuePreds.HasException(), ConfigHttpStatus.UnknownErrorStatus, NThreading::GetExceptionMessage(asyncUserMonologuePreds));
    R_ENSURE(asyncUserMonologuePreds.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "taxi support classifier error");
    NJson::TJsonValue userMonologueJson =  PostprocessClassifyResult(asyncUserMonologuePreds.ExtractValue());
    NDrive::TEventLog::Log("ClassifyUserMonologue", NJson::TMapBuilder
        ("user_id", userId)
        ("features", userMonologueJson)
    );

    asyncDialoguePreds.Wait(Context->GetRequestDeadline());
    R_ENSURE(!asyncDialoguePreds.HasException(), ConfigHttpStatus.UnknownErrorStatus, NThreading::GetExceptionMessage(asyncDialoguePreds));
    R_ENSURE(asyncDialoguePreds.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "taxi support classifier error");
    NJson::TJsonValue dialogueJson =  PostprocessClassifyResult(asyncDialoguePreds.ExtractValue());
    g.MutableReport().AddReportElement("features", std::move(dialogueJson));
    g.SetCode(HTTP_OK);
}

NJson::TJsonValue TChatDialogueClassifierProcessor::PostprocessClassifyResult(const NDrive::TSupportPrediction& classifyResult) {
    auto predictions = classifyResult.Elements;
    Sort(predictions.begin(), predictions.end(), [](const NDrive::TSupportPrediction::TElement& lhs, const NDrive::TSupportPrediction::TElement& rhs){return lhs.Probability < rhs.Probability;});

    NJson::TJsonValue featuresJson = NJson::JSON_ARRAY;
    for (auto&& pred : predictions) {
        NJson::TJsonValue featureReport;
        featureReport["id"] = pred.Topic;
        featureReport["confidence"] = pred.Probability;

        auto classId = pred.Topic;
        if (Server->GetSupportCenterManager() && Server->GetSupportCenterManager()->GetSupportRequestCategorizer()) {
            auto maybeCategory = Server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->GetNode(classId);
            if (maybeCategory) {
                featureReport["class"] = maybeCategory->GetLabel();
            } else {
                featureReport["class"] = pred.Topic;
            }
        }

        featuresJson.AppendValue(std::move(featureReport));
    }
    return featuresJson;
}

void TChatBulkEvolveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::ChatMessage);

    auto chatId = GetString(requestData, "chat_id");
    auto userIds = GetStrings(requestData, "user_ids", false);
    auto stepNames = GetStrings(requestData, "step_names");
    auto topic = GetString(requestData, "topic");
    auto targetStepName = GetString(requestData, "target_step_name");
    bool sendMessages = true;
    if (requestData.Has("send_messages")) {
        R_ENSURE(requestData["send_messages"].IsBoolean(), ConfigHttpStatus.SyntaxErrorStatus, "'send_messages' is not bool");
        sendMessages = requestData["send_messages"].GetBoolean();
    }

    auto chatRobot = Server->GetChatRobot(chatId);
    size_t remaining = 0;
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "no such chat robot");
    auto users = chatRobot->GetUsersInNodes(topic, MakeSet(stepNames));
    if (users.size() > 2000 && userIds.empty()) {
        remaining = users.size() - 2000;
        users.resize(2000);
    }

    NJson::TJsonValue processedEntries = NJson::JSON_ARRAY;
    auto userIdsSet = MakeSet(userIds);
    auto session = BuildChatSession();
    for (auto&& [userId, chatSearchId] : users) {
        if (userIdsSet.size() && !userIdsSet.contains(userId)) {
            continue;
        }
        TString topic;
        if (chatSearchId.size() > userId.size() + 1 + chatId.size()) {
            topic = chatSearchId.substr(userId.size() + 1 + chatId.size() + 1);
        }
        R_ENSURE(chatRobot->MoveToStep(userId, topic, targetStepName, &session, sendMessages), ConfigHttpStatus.UnknownErrorStatus, "could not move to step");
        NJson::TJsonValue entry;
        entry["user_id"] = userId;
        entry["chat_id"] = chatId + (topic ? ("." + topic) : "");
        processedEntries.AppendValue(std::move(entry));
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().AddReportElement("remaining", remaining);
    g.MutableReport().AddReportElement("processed_chats", std::move(processedEntries));
    g.SetCode(HTTP_OK);
}

void TChatGetOrCreateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString chatId, topic;
    auto userId = permissions->GetUserId();
    auto topicLink = requestData.Has("chat_id") ? GetString(requestData, "chat_id", false) : GetString(cgi, "chat_id", false);
    if (topicLink.empty()) {
        chatId = requestData.Has("chat_type") ? GetString(requestData, "chat_type") : GetString(cgi, "chat_type");
        topic = requestData.Has("topic") ? GetString(requestData, "topic") :  GetString(cgi, "topic", false);
    } else {
        IChatRobot::ParseTopicLink(topicLink, chatId, topic);
    }
    R_ENSURE(!chatId.empty(), ConfigHttpStatus.SyntaxErrorStatus, "Neither chat_type nor chat_id not provided");

    auto locale = GetLocale();
    auto messageTraits = permissions->GetMessageVisibilityTraits();
    bool getIfOpen = requestData.Has("get_if_open") ? GetValue<bool>(requestData, "get_if_open").GetOrElse(false) : GetValue<bool>(cgi, "get_if_open", false).GetOrElse(false);
    bool skipHidden = requestData.Has("skip_hidden") ? GetValue<bool>(requestData, "skip_hidden").GetOrElse(true) : GetValue<bool>(cgi, "skip_hidden", false).GetOrElse(true);
    auto moveToNode = requestData.Has("node") ? GetString(requestData, "node", false) : GetString(cgi, "node", false);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_type'");
    R_ENSURE(!chatRobot->GetChatConfig().GetIsStatic(), ConfigHttpStatus.SyntaxErrorStatus, "cannot create static chats");
    bool isNewChat = false;

    if (getIfOpen && topicLink.empty()) {
        if (topic.empty()) {
            TChatUserContext::TPtr ctx = BuildChatContext(permissions);
            for (auto&& chat : chatRobot->GetTopics(ctx, true)) {
                if (!chatRobot->GetIsChatInClosedNode(userId, chat.GetTopic(), Context->GetRequestStartTime())) {
                    if (skipHidden && chat.GetFlag(NDrive::NChat::TChat::EChatFlags::Hidden)) {
                        continue;
                    }
                    topic = chat.GetTopic();
                    break;
                }
            }
        } else {
            if (chatRobot->GetIsChatInClosedNode(userId, topic, Context->GetRequestStartTime())) {
                topic = "";
            } else if (skipHidden) {
                NDrive::NChat::TChat chat;
                if (chatRobot->GetChat(userId, topic, chat) && chat.GetFlag(NDrive::NChat::TChat::EChatFlags::Hidden)) {
                    topic = "";
                }
            }
        }
    }
    if (topicLink.empty() && topic.empty()) {
        isNewChat = true;
        topic = ToString(TInstant::Now().Seconds());
    }

    auto session = BuildChatSession(false);

    TVector<TContextMapInfo> contextMap;
    if (requestData["context"].IsArray()) {
        contextMap = BuildChatContextMapFromRequest(requestData["context"].GetArray());
    }
    if (moveToNode.empty()) {
        R_ENSURE(chatRobot->EnsureChat(userId, topic, session, false, contextMap), HTTP_NOT_FOUND, "could not ensure chat");
    } else {
        R_ENSURE(chatRobot->MoveToStep(userId, topic, moveToNode, &session, true, "", chatRobot->GetChatConfig().GetRobotUserId(), {}, contextMap), ConfigHttpStatus.SyntaxErrorStatus, "could not move to step");
    }

    NDrive::NChat::TChat chat;
    if (!chatRobot->GetChat(userId, topic, chat, true, session.GetTransaction())) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TUserChatDescription chatDescription(chatId, chatRobot, chat, userId);
    auto descriptionReport = chatDescription.GetReport();

    NJson::TJsonValue messagesReport;
    if (auto totalCount = chatRobot->GetMessagesCount(chatDescription.ChatDescription.GetId(), messageTraits, session)) {
        messagesReport["total"] = *totalCount;
    } else {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (auto realUnreadCount = chatRobot->GetUnreadMessagesCount(chatDescription.ChatDescription.GetId(), userId, messageTraits, session)) {
        messagesReport["unread"] = *realUnreadCount;
    } else {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    descriptionReport["stats"] = std::move(messagesReport);

    TMaybe<NDrive::NChat::TMessageEvent> lastMessage;
    if (chatRobot->GetLastMessage(chatDescription.ChatDescription.GetId(), session, messageTraits, lastMessage) && lastMessage) {
        descriptionReport["last_message"] = chatRobot->GetMessageReport(
            locale,
            *lastMessage,
            chatDescription.ChatDescription.GetObjectId(),
            chatDescription.ChatDescription.GetTopic()
        );
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.MutableReport().SetExternalReport(std::move(descriptionReport));
    g.SetCode(HTTP_OK);
}

void TChatGetSuggest::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto requestUserId = permissions->GetUserId();
    auto topicLink = GetString(Context->GetCgiParameters(), "chat_id");
    auto messageText = GetString(requestData, "message", false);
    TString chatId, topic;
    IChatRobot::ParseTopicLink(topicLink, chatId, topic);
    auto chatRobot = Server->GetChatRobot(chatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "there is no chat robot");
    auto userId = GetString(Context->GetCgiParameters(), "user_id", false);
    if (userId.empty()) {
        userId = requestUserId;
    }

    NDrive::NChat::TMessageEvents messages;
    {
        auto session = BuildChatSession(true);
        auto optionalMessages = chatRobot->GetChatMessages(userId, topic, session, NDrive::NChat::TMessage::DefaultUserTraits, 0);
        R_ENSURE(optionalMessages, ConfigHttpStatus.SyntaxErrorStatus, "cannot get chat messages", session);
        messages = std::move(*optionalMessages);
        if (messageText) {
            NDrive::NChat::IMessage::EMessageType messageType = NDrive::NChat::IMessage::EMessageType::Plaintext;
            R_ENSURE(NJson::ParseField(requestData, "type", NJson::Stringify(messageType), false), ConfigHttpStatus.SyntaxErrorStatus, "cannot parse type");
            NDrive::NChat::TMessage message(messageText, 0, messageType);
            messages.emplace_back(TObjectEvent<NDrive::NChat::TMessage>(message, EObjectHistoryAction::Add, Now(), userId, userId, ""));
        }
    }

    TChatUserContext::TPtr context;
    if (userId && requestUserId != userId) {
        auto userPermissions = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        R_ENSURE(userPermissions, ConfigHttpStatus.SyntaxErrorStatus, "cannot get user permissions");
        context = BuildChatContext(userPermissions);
    } else {
        context = BuildChatContext(permissions);
    }
    context->SetChatId(chatId);
    context->SetChatTopic(topic);
    context->SetMessage(messageText);

    TChatRobotScriptItem currentScriptItem;
    R_ENSURE(chatRobot->GetCurrentScriptItem(userId, topic, currentScriptItem, Context->GetRequestStartTime()), ConfigHttpStatus.SyntaxErrorStatus, "cannot get current script item");
    auto suggestFuture = currentScriptItem.GetNodeResolver()->GetSuggest(context, messages, true);
    auto report = g.GetReport();
    suggestFuture.Subscribe([
        chatRobot,
        context,
        currentScriptItem,
        report,
        topic
    ](const NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse>& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();

        if (result.HasValue()) {
            auto suggestedOptions = currentScriptItem.GetNodeResolver()->GetSuggestedChatOptions(result.GetValue());
            NJson::TJsonValue resultJson = NJson::JSON_MAP;
            NChatOutputHelpers::GetExpectedActionReport(resultJson, chatRobot, topic, context);
            NJson::TJsonValue schema = NJson::JSON_MAP;
            schema["options"] = suggestedOptions;
            resultJson["schema"] = schema;
            r.SetExternalReport(std::move(resultJson));
        } else {
            r.AddReportElement("error", NThreading::GetExceptionInfo(result));
            g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        }
    });
    g.Release();
}
