#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/auth/common/auth.h>

#include <rtline/util/types/accessor.h>

#include <util/string/type.h>

// base

class TCommonAppConfig: public IAuthRequestProcessorConfig {
private:
    TString HandlerName = "default";

private:
    using TBase = IAuthRequestProcessorConfig;

private:
    TMap<TString, ui32> MinimalBuildsInfo;
    bool MinimalBuildsInfoOverriden = false;

private:
    bool ReadMinimalBuildsInfo(const TYandexConfig::Section* section);

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;

public:
    virtual const TString& GetHandlerName() const override {
        return HandlerName;
    }

    const TMap<TString, ui32>& GetMinimalBuildsInfo() const {
        return MinimalBuildsInfo;
    }

    TCommonAppConfig(const TString& name)
        : IAuthRequestProcessorConfig(name)
    {
    }

    virtual void ReadDefaults(const TYandexConfig::Section* section) override;
    virtual void ToString(IOutputStream& os) const override;
};

// registrators

template <class THandler, class THandlerConfig>
class TDriveUserAppConfig: public TCommonAppConfig, public THandlerConfig {
private:
    using TBase = TCommonAppConfig;
    using TNativeBase = THandlerConfig;
    using TSelf = TDriveUserAppConfig<THandler, THandlerConfig>;

public:
    using TBase::TBase;
    using TFactory = typename TBase::TFactory;

    static const TFactory::TRegistrator<TSelf> Registrator;

    static TString GetTypeName() {
        return THandler::GetTypeName();
    }

    virtual void ToString(IOutputStream& os) const override {
        TBase::ToString(os);
        TNativeBase::ToStringFeatures(os);
    }

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override {
        TBase::DoInit(section);
        TNativeBase::InitFeatures(section);
    }

    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override {
        return new THandler(*this, context, authModule, &server->GetAsSafe<NDrive::IServer>());
    }
};

template <class THandler, class THandlerConfig>
const TDriveUserAppConfig<THandler, THandlerConfig>::TFactory::template TRegistrator<TDriveUserAppConfig<THandler, THandlerConfig>> TDriveUserAppConfig<THandler, THandlerConfig>::Registrator(THandler::GetTypeName());

template <typename THandler, typename THandlerConfigImpl, typename TSchemeReporter>
class TContextAwareConfig: public TCommonAppConfig, public THandlerConfigImpl {
private:
    using TBase = TCommonAppConfig;
    using TNativeBase = THandlerConfigImpl;

    using TSelf = TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>;

    R_READONLY(TString, SchemeParamName, "scheme");

public:
    using TBase::TBase;
    using TFactory = typename TBase::TFactory;

    static const TFactory::TRegistrator<TSelf> Registrator;

    static TString GetTypeName() {
        return THandler::GetTypeName();
    }

    virtual void ToString(IOutputStream& os) const override;

protected:
    virtual void DoInit(const TYandexConfig::Section* section) override;

    virtual IRequestProcessor::TPtr DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const override;
};

template <typename THandler, typename THandlerConfigImpl, typename TSchemeReporter>
void TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>::DoInit(const TYandexConfig::Section* section) {
    TBase::DoInit(section);
    TNativeBase::InitFeatures(section);
    SchemeParamName = section->GetDirectives().Value("SchemeParamName", SchemeParamName);
}

template <typename THandler, typename THandlerConfigImpl, typename TSchemeReporter>
void TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    TNativeBase::ToStringFeatures(os);
    os << "SchemeParamName: " << SchemeParamName << Endl;
}

template <typename THandler, typename THandlerConfigImpl, typename TSchemeReporter>
IRequestProcessor::TPtr TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>::DoConstructAuthProcessor(IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server) const {
    const auto& cgi = context->GetCgiParameters();
    if (cgi.Has(SchemeParamName) && IsTrue(cgi.Get(SchemeParamName))) {
        return MakeIntrusive<TSchemeReporter>(*this, context, authModule, server->GetAsPtrSafe<NDrive::IServer>());
    }
    return MakeIntrusive<THandler>(*this, context, authModule, server->GetAsPtrSafe<NDrive::IServer>());
}

template <class THandler, class THandlerConfigImpl, typename TSchemeReporter>
const TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>::TFactory::template TRegistrator<TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>> TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReporter>::Registrator(THandler::GetTypeName());

// implementations

class TEmptyConfig {
public:
    void InitFeatures(const TYandexConfig::Section* /*section*/) {
    }

    void ToStringFeatures(IOutputStream& /*os*/) const {
    }
};

class TPackProcessorConfig {
    R_READONLY(ui32, MaxItems, 1000);
public:
    void InitFeatures(const TYandexConfig::Section* section) {
        MaxItems = section->GetDirectives().Value("MaxItems", MaxItems);
    }

    void ToStringFeatures(IOutputStream& os) const {
        os << "MaxItems: " << MaxItems << Endl;
    }
};
