#pragma once

#include "processor.h"

#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/common/serializable.h>
#include <library/cpp/case_insensitive_string/case_insensitive_string.h>

template <class TBaseImpl, class TConfig>
class TDeviceChangeCheckProcessor: public TAppCommonProcessor<TBaseImpl, TConfig> {
    using TBase = TAppCommonProcessor<TBaseImpl, TConfig>;

public:
    using typename TBase::THandlerConfig;
    using TBase::TBase;
    using TBase::Server;
    using TBase::Context;

    TDeviceChangeCheckProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    bool AddTagOnDeviceChange(const TString& userId, const TUserDevice& newDevice, const TMaybe<TObjectEvent<TUserDevice>>& oldDevice, NDrive::TEntitySession& session) const {
        bool antifraudEnabled = TBase::template GetHandlerSettingDef<bool>("antifraud.enabled", false);
        if (!antifraudEnabled) {
            return true;
        }

        auto selfieTagName = TBase::template GetHandlerSetting<TString>("antifraud.selfie_tag");
        bool shouldSkip = TBase::template GetHandlerSettingDef<bool>("antifraud.skip_if_selfie_tag", false);
        if (selfieTagName && shouldSkip) {
            TMaybe<TVector<TDBTag>> tags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, { *selfieTagName }, session);
            if (!tags) {
                return false;
            }
            if (shouldSkip && !tags->empty()) {
                return true;
            }
        }

        TString tagData = TBase::template GetHandlerSettingDef<TString>("antifraud.tag_data", "");
        if (!tagData) {
            session.SetErrorInfo("DeviceChangeCheckProcessor", "antifraud.tag_data not defined");
            return false;
        }

        if (!oldDevice.Defined()) {
            bool addTagIfNoOldDevice = TBase::template GetHandlerSettingDef<bool>("antifraud.add_tag_no_old_device", false);
            if (addTagIfNoOldDevice) {
                return AddTag(tagData, userId, session);
            } else {
                return true;
            }
        }

        bool checkDeviceIdChanged = TBase::template GetHandlerSettingDef<bool>("antifraud.check_device_id", false);
        bool checkUserAgentChanged = TBase::template GetHandlerSettingDef<bool>("antifraud.check_user_agent", false);
        if (!checkDeviceIdChanged && !checkUserAgentChanged) {
            return true;
        }
        if (!checkDeviceIdChanged || newDevice.GetDeviceId() != oldDevice->GetDeviceId()) {
            if (checkUserAgentChanged) {
                auto newAgent = GetDeviceName(userId, newDevice.GetDescription());
                auto oldAgent = GetDeviceName(userId, oldDevice->GetDescription());
                if (newAgent && oldAgent && *newAgent == *oldAgent) {
                    return true;
                }
                if ((!newAgent || !oldAgent) && !TBase::template GetHandlerSettingDef<bool>("antifraud.add_tag_on_failed_name_parsing", false)) {
                    return true;
                }
            }
            return AddTag(tagData, userId, session);
        }
        return true;
    }

private:
    bool AddTag(const TString& tagData, const TString& userId, NDrive::TEntitySession& session) const {
        TMessagesCollector errors;
        ITag::TPtr tag = IJsonSerializableTag::BuildFromString(Server->GetDriveAPI()->GetTagsManager(), tagData, &errors);
        if (!tag) {
            session.SetErrorInfo("DeviceChangeCheckProcessoor", "cannot construct tag from " + tagData + " errors: " + errors.GetStringReport());
            return false;
        }
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, Server, session)) {
            return false;
        }
        return true;
    }

    TMaybe<TString> GetDeviceName(const TString& userId, const TString& description) const {
        NJson::TJsonValue descriptionJson;
        if (!NJson::ReadJsonFastTree(description, &descriptionJson)) {
            NDrive::TEventLog::Log("DeviceChangeCheckProcessor", NJson::TMapBuilder("user_id", userId)("error", "cannot parse device description: " + description));
            return Nothing();
        }
        TString userAgent;
        for (auto&& [key, value] : descriptionJson["headers"].GetMap()) {
            TCaseInsensitiveString caseInsensitiveKey(key);
            if (caseInsensitiveKey == "user-agent") {
                if (!NJson::ParseField(value, userAgent, true)) {
                    NDrive::TEventLog::Log("DeviceChangeCheckProcessor", NJson::TMapBuilder("user_id", userId)("error", "cannot parse user agent from device description: " + description));
                    return Nothing();
                }
                break;
            }
        }
        auto posBracket = userAgent.find('(');
        if (posBracket == TString::npos) {
            NDrive::TEventLog::Log("DeviceChangeCheckProcessor", NJson::TMapBuilder("user_id", userId)("error", "cannot parse device name from user agent: " + userAgent));
            return Nothing();
        }
        userAgent = userAgent.substr(posBracket + 1, userAgent.size() - posBracket - 1);
        auto posEnd = userAgent.find(';');
        if (posEnd == TString::npos) {
            posEnd = userAgent.find(')');
        }
        if (posEnd == TString::npos) {
            NDrive::TEventLog::Log("DeviceChangeCheckProcessor", NJson::TMapBuilder("user_id", userId)("error", "cannot parse device name from user agent: " + userAgent));
            return Nothing();
        }
        TString deviceName = userAgent.substr(0, posEnd);
        return deviceName;
    }
};
