#pragma once

#include "processor.h"

#include <drive/backend/common/localization.h>

namespace NDrive {
    class IServer;
}

template <class TObjectContainer, class TBaseImpl>
class TObjectsInfoProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::Server;

protected:
    virtual const IDBEntitiesManager<TObjectContainer>* GetEntitiesManager(const IServerBase* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsInfoProcessor() = default;

    TObjectsInfoProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_info";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, GetEntityType());

        const TSet<typename TObjectContainer::TId> ids = MakeSet(TBase::template GetValues<typename TObjectContainer::TId>(Context->GetCgiParameters(), "ids", false));

        TMap<typename TObjectContainer::TId, TObjectContainer> objects;
        R_ENSURE(GetEntitiesManager(Server)->GetObjects(objects), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_infos");
        NJson::TJsonValue reportJson(NJson::JSON_ARRAY);
        for (auto&& i : objects) {
            if (ids.size() && !ids.contains(i.first)) {
                continue;
            }
            reportJson.AppendValue(i.second.GetReport());
        }
        g.MutableReport().AddReportElement("objects", std::move(reportJson));

        g.SetCode(HTTP_OK);
    }
};

template <class TObjectContainer, class TBaseImpl>
class TObjectsUpsertProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

protected:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::GetStrings;
    using TBase::Server;

protected:
    virtual const IDBEntitiesManager<TObjectContainer>* GetEntitiesManager(const IServerBase* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsUpsertProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_upsert";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, GetEntityType());

        R_ENSURE(requestData["objects"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'objects' is not array");

        const bool force = IsTrue(Context->GetCgiParameters().Get("force"));
        TVector<TObjectContainer> objectContainers;
        for (auto&& i : requestData["objects"].GetArraySafe()) {
            TObjectContainer container;
            TMessagesCollector errors;
            R_ENSURE(TBaseDecoder::DeserializeFromJsonVerbose(container, i, errors), ConfigHttpStatus.SyntaxErrorStatus, "incorrect object json: " + errors.GetStringReport());
            R_ENSURE(PreprocessContainer(container, permissions, errors), ConfigHttpStatus.SyntaxErrorStatus, "PreprocessContainer error: " + errors.GetStringReport());
            objectContainers.emplace_back(std::move(container));
        }

        auto session = BuildTx<NSQL::Writable>();
        UpsertObjects(force, objectContainers, permissions, session);
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }

protected:
    virtual void UpsertObjects(const bool force, const TVector<TObjectContainer>& containers, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
        for (auto&& container : containers) {
            if (force) {
                if (!GetEntitiesManager(Server)->ForceUpsertObject(container, permissions->GetUserId(), session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            } else {
                if (!GetEntitiesManager(Server)->UpsertObject(container, permissions->GetUserId(), session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        }
    }

    virtual bool PreprocessContainer(const TObjectContainer& /*container*/, TUserPermissions::TPtr /*permissions*/, TMessagesCollector& /*errors*/) const {
        return true;
    }
};

template <class TObjectContainer, class TBaseImpl>
class TObjectsRemoveProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::GetStrings;
    using TBase::Server;

protected:
    virtual const IDBEntitiesManager<TObjectContainer>* GetEntitiesManager(const IServerBase* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsRemoveProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, GetEntityType());
        R_ENSURE(requestData["ids"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'ids' is not array");
        TSet<TString> ids;
        TJsonProcessor::ReadContainer(requestData, "ids", ids);
        auto session = BuildTx<NSQL::Writable>();

        R_ENSURE(GetEntitiesManager(Server)->RemoveObject(ids, permissions->GetUserId(), session), ConfigHttpStatus.UnknownErrorStatus, "cannot remove object");
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        g.SetCode(HTTP_OK);
    }
};

template <class TObjectContainer, class TBaseImpl>
class TObjectsProposeProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::GetStrings;
    using TBase::Server;

protected:
    virtual const IDBEntitiesWithPropositionsManager<TObjectContainer>* GetEntitiesManager(const NDrive::IServer* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsProposeProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_propose";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Propose, GetEntityType());
        const TString& comment = Context->GetCgiParameters().Get("comment");
        R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment is missing", EDriveSessionResult::IncorrectRequest);

        auto session = BuildTx<NSQL::Writable>();

        TObjectContainer container;
        R_ENSURE(TBaseDecoder::DeserializeFromJson(container, requestData), ConfigHttpStatus.SyntaxErrorStatus, "incorrect object json");

        TObjectProposition<TObjectContainer> proposition(container, 1);
        proposition.SetDescription(comment);
        if (GetEntitiesManager(Server)->GetPropositions()->Propose(proposition, permissions->GetUserId(), session) == EPropositionAcceptance::Problems) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        g.SetCode(HTTP_OK);
    }
};

template <class TObjectContainer, class TBaseImpl>
class TObjectsConfirmProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::GetStrings;
    using TBase::Server;

protected:
    virtual const IDBEntitiesWithPropositionsManager<TObjectContainer>* GetEntitiesManager(const NDrive::IServer* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsConfirmProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_confirm";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        const TAdministrativeAction::EEntity entity = GetEntityType();
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, entity);
        const TString& comment = Context->GetCgiParameters().Get("comment");
        R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment is missing", EDriveSessionResult::IncorrectRequest);

        auto session = BuildTx<NSQL::Writable>();

        const TSet<TString> propositionIds = MakeSet(GetStrings(requestData, "proposition_ids"));
        auto optionalSessions = GetEntitiesManager(Server)->GetPropositions()->Get(propositionIds, session);
        R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
        auto sessions = *optionalSessions;

        for (auto&& i : propositionIds) {
            auto it = sessions.find(i);
            R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
            //            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, entity, it->second->GetType());
        }

        session.SetComment(comment);
        for (auto&& i : propositionIds) {
            auto it = sessions.find(i);
            const EPropositionAcceptance confirmResult = GetEntitiesManager(Server)->GetPropositions()->Confirm(it->first, permissions->GetUserId(), session);
            if (confirmResult == EPropositionAcceptance::ConfirmWaiting) {
            } else if (confirmResult == EPropositionAcceptance::ReadyForCommit) {
                if (!GetEntitiesManager(Server)->UpsertObject(it->second, permissions->GetUserId(), session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            } else {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }
};

template <class TObjectContainer, class TBaseImpl>
class TObjectsRejectProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Context;
    using TBase::GetStrings;
    using TBase::Server;

protected:
    virtual const IDBEntitiesWithPropositionsManager<TObjectContainer>* GetEntitiesManager(const NDrive::IServer* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TObjectsRejectProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_reject";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        const TAdministrativeAction::EEntity entity = GetEntityType();
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Reject, entity);
        const TString& comment = Context->GetCgiParameters().Get("comment");
        R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must being", EDriveSessionResult::IncorrectRequest);

        auto session = BuildTx<NSQL::Writable>();

        const TSet<TString> propositionIds = MakeSet(GetStrings(requestData, "proposition_ids"));
        auto optionalSessions = GetEntitiesManager(Server)->GetPropositions()->Get(propositionIds, session);
        R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
        auto sessions = *optionalSessions;

        for (auto&& i : propositionIds) {
            auto it = sessions.find(i);
            R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        }

        session.SetComment(comment);
        for (auto&& i : propositionIds) {
            auto it = sessions.find(i);
            const EPropositionAcceptance confirmResult = GetEntitiesManager(Server)->GetPropositions()->Reject(it->first, permissions->GetUserId(), session);
            if (confirmResult != EPropositionAcceptance::Rejected) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }
};

template <class TObjectContainer, class TBaseImpl>
class TPropositionsListProcessor: public TAppCommonProcessor<TBaseImpl, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TBaseImpl, TEmptyConfig>;

public:
    using typename TBase::THandlerConfig;

private:
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        return TBase::template BuildTx<traits>();
    }
    using TBase::ConfigHttpStatus;
    using TBase::Server;

protected:
    virtual const IDBEntitiesWithPropositionsManager<TObjectContainer>* GetEntitiesManager(const NDrive::IServer* server) const = 0;
    virtual TAdministrativeAction::EEntity GetEntityType() const = 0;

public:
    TPropositionsListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return TObjectContainer::GetTableName() + "_propositions";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) override {
        R_ENSURE(GetEntitiesManager(Server), ConfigHttpStatus.NotImplementedState, "not configured");
        const TAdministrativeAction::EEntity entity = GetEntityType();
        this->ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, entity);

        auto session = BuildTx<NSQL::ReadOnly>();

        auto optionalPropositions = Yensured(GetEntitiesManager(Server)->GetPropositions())->Get(session);
        R_ENSURE(optionalPropositions, {}, "cannot get propositions", session);
        auto propositions = *optionalPropositions;

        TSet<TString> propositionUserIds;
        NJson::TJsonValue report(NJson::JSON_ARRAY);
        for (auto&& i : propositions) {
            i.second.FillUsers(propositionUserIds);
            report.AppendValue(i.second.BuildJsonReport());
        }

        g.MutableReport().AddReportElement("propositions", std::move(report));

        auto gPropositionUsers = Server->GetDriveAPI()->GetUsersData()->FetchInfo(propositionUserIds, session);
        NJson::TJsonValue usersJson = NJson::JSON_ARRAY;
        for (auto&& user : gPropositionUsers) {
            usersJson.AppendValue(user.second.GetReport(permissions->GetUserReportTraits()));
        }
        g.MutableReport().AddReportElement("propositions_users", std::move(usersJson));
        g.SetCode(HTTP_OK);
    }
};
