#include "fetcher.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/coloring.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/images/database.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/report/json.h>
#include <drive/backend/sessions/matcher/session_matcher.h>
#include <drive/backend/signalq/signals/tag.h>
#include <drive/backend/surge/surge_snapshot.h>

#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/protocol/sensor.h>
#include <drive/telematics/protocol/settings.h>

#include <library/cpp/json/json_writer.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/iterator.h>

#include <saas/util/types/cast.h>

DECLARE_FIELDS_JSON_SERIALIZER(TCarsFetcher::TModelSpecificationFilter);

TSet<NDrive::TSensorId> TCarsFetcher::DefaultSensorIds = {
    CAN_FUEL_LEVEL_P,
    CAN_FUEL_DISTANCE_KM,
    NDrive::NVega::SvrRawState,
};

NJson::TJsonValue TCarsFetcher::BuildTagJson(const TDBTag& tag, const TString& userId) const {
    TString displayName = tag->GetName();
    auto tagDescription = DriveApi.GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());

    NJson::TJsonValue tagJson(tag.SerializeToJson());
    tagJson.InsertValue("display_name", tagDescription ? tagDescription->GetDisplayName() : tag->GetName());

    if (tag->GetPerformer() == userId) {
        tagJson.InsertValue("selected", true);
    } else if (tag->GetPerformer() != "") {
        auto* userData = TagPerformers.GetResultPtr(tag->GetPerformer());
        NJson::TJsonValue& pInfo = tagJson.InsertValue("performer_info", NJson::JSON_MAP);
        pInfo.InsertValue("id", tag->GetPerformer());
        if (userData) {
            pInfo.InsertValue("name", userData->GetFullName());
            pInfo.InsertValue("phone", userData->GetPhone());
        }
    }
    if (auto it = TagsCreationTimes.find(tag.GetTagId()); it != TagsCreationTimes.end()) {
        tagJson.InsertValue("creation_time", it->second.Seconds());
    }
    return tagJson;
}

template <class T>
TMaybe<T> TCarsFetcher::GetSetting(const TString& key) const {
    if (Permissions) {
        return Permissions->GetSetting<T>(Server.GetSettings(), key);
    } else {
        return Server.GetSettings().GetValue<T>(key);
    }
}

bool TCarsFetcher::FetchData(TUserPermissions::TPtr permissions, const TSet<TString>& fetchedCarIds) {
    return FetchData(permissions, &fetchedCarIds);
}

bool TCarsFetcher::FetchData(TUserPermissions::TPtr permissions, const TSet<TString>* fetchedCarIdsExt) {
    if (DataFetched)
        return true;
    if (!permissions) {
        return false;
    }
    auto actuality = UseCache ? TInstant::Zero() : Now();

    Permissions = permissions;
    auto eg = NDrive::BuildEventGuard("fetch.full");
    auto tx = DriveApi.BuildTx<NSQL::ReadOnly | NSQL::RepeatableRead | NSQL::Deferred>();

    UserId = permissions->GetUserId();
    UserObservableTags = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe, NEntityTagsManager::EEntityType::Car);
    TMaybe<TSet<TString>> fetchedCarIdsLocal;
    const TSet<TString>* fetchedCarIds = fetchedCarIdsExt;
    if (IsRealtime) {
        auto eg = NDrive::BuildEventGuard("fetch.sensors.rt");
        SnapshotsFull = Server.GetSnapshotsManager().GetSnapshots();
        if (!fetchedCarIds) {
            if (CarTagsFilter && CarTagsFilter->IsInclusive()) {
                auto fetchedCarIdsLocalSet = Server.GetDriveAPI()->GetTagsManager().GetDeviceTags().PrefilterObjects(*CarTagsFilter, nullptr, actuality);
                if (!fetchedCarIdsLocalSet) {
                    return false;
                }
                fetchedCarIdsLocal = std::move(*fetchedCarIdsLocalSet);
            }

            if (HasUserLocation() && PreLimit < SnapshotsFull.ById().size() && (!fetchedCarIdsLocal || PreLimit < fetchedCarIdsLocal->size())) {
                fetchedCarIdsLocal = SnapshotsFull.GetNearestIds(GetUserLocationRef(), PreLimit, ReportTraits & NDeviceReport::EReportTraits::ReportIncorrectLocation, WithClusters, fetchedCarIdsLocal.Get());
            }
            fetchedCarIds = fetchedCarIdsLocal.Get();
        }
    }

    if (IsRealtime) {
        auto eg = NDrive::BuildEventGuard("fetch.sensors");
        if ((ReportTraits & (NDeviceReport::ReportSurgeInfo | NDeviceReport::ReportFutures | NDeviceReport::ReportHiddenDiscounts)) && !!Server.GetSurgeConstructor()) {
            RTFactors = Server.GetSurgeConstructor()->GetInfoActual();
        }
    }

    TStateFiltersDB::TObjectStates objectStates;
    if (StatusFilter) {
        objectStates = Server.GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
    }

    auto itRTFactorsInfo = RTFactors.begin();
    auto itRTSnapshotInfo = SnapshotsFull.begin();

    const bool dontUseFutures = !permissions->GetSetting<bool>("offers.futures.enabled", true);
    const ui32 secondsLimit = permissions->GetSetting<TDuration>("offers.futures.min_duration", TDuration::Max()).Seconds();

    TMap<TString, std::pair<TString, size_t>> tagNameToStrGroup;
    {
        TString groupsTagsStr = permissions->GetSetting<TString>("tags.groups.city", "");
        NJson::TJsonValue groupsJson;
        if (!groupsTagsStr.empty() && NJson::ReadJsonTree(groupsTagsStr, &groupsJson)) {
            for (auto&& m : groupsJson.GetArray()) {
                if (m["city"].IsString() && m["tags"].IsArray()) {
                    auto tagsArray = m["tags"].GetArray();
                    auto cityName = m["city"].GetString();
                    for (size_t i = 0; i < tagsArray.size(); ++i) {
                        tagNameToStrGroup[tagsArray[i].GetString()] = {cityName, i};
                    }
                }
            }
        }
    }

    TMap<TString, TDriveModelData> modelInfos;
    auto action = [&](const TTaggedObject& car) {
        if (!!CarTagsFilter && !CarTagsFilter->IsMatching(car)) {
            return false;
        }
        if (!!AreaTagsFilter && Advance(itRTSnapshotInfo, SnapshotsFull.end(), car.GetId()) && !AreaTagsFilter->IsMatching(itRTSnapshotInfo->second.GetLocationTagsGuarantee())) {
            return false;
        }
        if (LocationTypeFilter && Advance(itRTSnapshotInfo, SnapshotsFull.end(), car.GetId())) {
            auto location = itRTSnapshotInfo->second.GetLocation();
            auto locationType = location ? location->Type : NDrive::TLocation::Unknown;
            if (!LocationTypeFilter->contains(locationType)) {
                return false;
            }
        }
        if (StatusFilter) {
            auto status = objectStates.find(car.GetId());
            if (status == objectStates.end()) {
                return false;
            }
            if (!StatusFilter->contains(status->second)) {
                return false;
            }
        }

        const bool hasFactors = Advance(itRTFactorsInfo, RTFactors.end(), car.GetId());
        EFuturesStage futuresStage = (IsRealtime && hasFactors && itRTFactorsInfo->second.HasFinishDuration()) ? EFuturesStage::Futurable : EFuturesStage::NoFutures;
        if ((ReportTraits & NDeviceReport::ReportFutures) == 0) {
            futuresStage = EFuturesStage::NoFutures;
        }
        bool isAvailable = false;
        if ((ReportTraits & NDeviceReport::EReportTraits::ReportOrganizationOnly) && FleetsFilter) {
            bool checkOrganization = false;
            for (auto&& tag : car.GetTags()) {
                if (auto organizationTag = tag.GetTagAs<IOrganizationTag>(); organizationTag && FleetsFilter->contains(organizationTag->GetParentId())) {
                    checkOrganization = true;
                    break;
                }
            }
            if (!checkOrganization) {
                return false;
            }
        }
        if (CheckVisibility && IsRealtime) {
            TUserPermissions::TUnvisibilityInfoSet unvisibilityInfo;
            switch (permissions->GetVisibility(car, NEntityTagsManager::EEntityType::Car, &unvisibilityInfo)) {
                case TUserPermissions::EVisibility::SelectedStrict:
                    SelectedOnly = !NoSelected;
                case TUserPermissions::EVisibility::Selected:
                    if (NoSelected) {
                        return false;
                    }
                    SelectedByUser.emplace(car.GetId());
                case TUserPermissions::EVisibility::Visible:
                    isAvailable = true;
                    break;
                default:
                    if ((futuresStage == EFuturesStage::NoFutures) || unvisibilityInfo != (TUserPermissions::TUnvisibilityInfoSet)TUserPermissions::EUnvisibilityInfo::Busy || dontUseFutures) {
                        return false;
                    }
            }
        } else {
            isAvailable = true;
        }

        if (hasFactors && !isAvailable) {
            const auto& factors = itRTFactorsInfo->second;
            auto finishDuration = factors.OptionalFinishDuration();
            if (finishDuration && *finishDuration > secondsLimit) {
                return false;
            }
        }

        if (ModelSpecificationFilters) {
            auto carInfo = DriveApi.GetCarManager().GetObject(car.GetId());
            R_ENSURE(carInfo, HTTP_INTERNAL_SERVER_ERROR, "cannot GetObject " << car.GetId(), tx);
            auto modelInfo = MapFindPtr(modelInfos, carInfo->GetModel());
            if (!modelInfo) {
                auto modelFetchResults = DriveApi.GetModelsDB().GetCachedOrFetch(carInfo->GetModel(), tx);
                auto modelFetchResult = modelFetchResults.ExtractResult(carInfo->GetModel());
                R_ENSURE(modelFetchResult, HTTP_INTERNAL_SERVER_ERROR, "cannot Fetch model " << carInfo->GetModel(), tx);
                auto [p, inserted] = modelInfos.emplace(carInfo->GetModel(), std::move(*modelFetchResult));
                Y_ASSERT(inserted);
                modelInfo = &p->second;
            }
            for (auto&& msf : *ModelSpecificationFilters) {
                auto specification = NDrive::GetSpecification(msf.Name, carInfo.Get(), modelInfo);
                if (!specification) {
                    return false;
                }
                if (!msf.Values.contains(specification->GetValue())) {
                    return false;
                }
            }
        }

        TVector<const TDBTag*> filteredTags;
        bool isRiding = false;
        {
            for (auto&& tag : car.GetTags()) {
                if (permissions->GetActionsByTagIdx(tag->GetDescriptionIndex()) & (TTagAction::TTagActions)TTagAction::ETagAction::Observe) {
                    filteredTags.emplace_back(&tag);
                }
                if (tag.GetTagAs<TTagReservationFutures>()) {
                    if (tag->GetPerformer() != permissions->GetUserId()) {
                        futuresStage = EFuturesStage::NoFutures;
                    } else {
                        futuresStage = EFuturesStage::Futures;
                    }
                } else if (IsCorrectFuturesState(tag->GetName())) {
                    isRiding = true;
                }
            }
        }
        if (filteredTags.size()) {
            ObservableTagsData.emplace(car.GetId(), std::move(filteredTags));
        }
        if ((futuresStage != EFuturesStage::NoFutures) && isRiding) {
            Y_ASSERT(Futures.empty() || (Futures.back().first < car.GetId()));
            Futures.emplace_back(std::make_pair(car.GetId(), futuresStage));
        }

        if (auto tagIt = ObservableTagsData.find(car.GetId()); tagIt != ObservableTagsData.end()) {
            TString groupStr;
            size_t orderIndex = std::numeric_limits<size_t>::max();
            for (auto&& tag : tagIt->second) {
                auto tagImpl = tag->GetData();
                if (!tagImpl) {
                    continue;
                }

                if (auto it = tagNameToStrGroup.find(tagImpl->GetName()); it != tagNameToStrGroup.end()
                    && it->second.second < orderIndex) {
                    orderIndex = it->second.second;
                    groupStr = it->second.first;
                }
            }

            if (groupStr) {
                CityGroups.emplace(car.GetId(), groupStr);
            }
        }

        if (isAvailable) {
            AvailableCars.emplace(car.GetId());
        }

        return true;
    };

    {
        auto eg = NDrive::BuildEventGuard("fetch.ApplyFilter");
        if (!Server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCurrentSnapshot(DevicesSnapshot, fetchedCarIds, actuality)) {
            return false;
        }
        DevicesSnapshot.ApplyFilter(action, fetchedCarIds);
    }

    TSet<TString> fullCarsList = AvailableCars;
    for (auto&& i : Futures) {
        fullCarsList.emplace(i.first);
    }

    if (IsRealtime) {
        auto eg = NDrive::BuildEventGuard("fetch.sensors.rt");
        SensorIds.emplace(CAN_FUEL_LEVEL_P);
        Snapshots = Server.GetSnapshotsManager().GetSnapshotsVector(fullCarsList);
    }

    {
        auto eg = NDrive::BuildEventGuard("fetch.cars.data");
        if (ReportTraits & (NDeviceReport::ReportCurrentOffer | NDeviceReport::ReportFutures)) {
            auto billingInfo = DriveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", actuality);
            if (!!billingInfo) {
                CarCurrentSessions = billingInfo->GetSessionsActualByObjects(&fullCarsList);
            }
            if (ReportTraits & NDeviceReport::ReportCurrentOfferDetails) {
                auto language = enum_cast<ELanguage>(Locale);
                auto geocoder = DriveApi.HasGeocoderClient() ? &DriveApi.GetGeocoderClient() : nullptr;
                TSet<TString> users;
                for (auto&& [car, session] : CarCurrentSessions) {
                    auto startEvent = session->GetFirstEvent();
                    users.emplace(session->GetUserId());
                    if (startEvent) {
                        auto snapshot = (*startEvent)->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
                        if (snapshot) {
                            if (auto location = snapshot->GetHistoryLocation()) {
                                TPointInfo point;
                                point.Point = *location;
                                if (geocoder) {
                                    point.Address = geocoder->Decode(location->GetCoord(), language);
                                }
                                Starts[session->GetSessionId()] = std::move(point);
                            }
                        }
                    }
                }
                SessionUsers = DriveApi.GetUsersData()->FetchInfo(users);
            }
        }
        CarInfosOriginal = DriveApi.GetCarsData()->FetchInfo(fullCarsList, TInstant::Zero());
        if (ReportTraits & NDeviceReport::ReportStatus) {
            auto eg = NDrive::BuildEventGuard("fetch.statuses");
            CarStatuses = DriveApi.GetStateFiltersDB()->GetObjectStates();
            if (!DriveApi.GetStateFiltersDB()->GetAllObjectsFromCache(FullCarStatuses)) {
                return false;
            }
            if (RecalculateStatuses) {
                auto eg = NDrive::BuildEventGuard("fetch.recalculate_statuses");
                for (auto&& [id, taggedObject] : DevicesSnapshot) {
                    if (!taggedObject) {
                        continue;
                    }
                    const auto snapshot = SnapshotsFull.GetSnapshot(id);
                    const auto& sensors = snapshot ? snapshot->GetSensors() : Default<NDrive::TSensors>();
                    RecalculatedStatuses[id] = DriveApi.GetStateFiltersDB()->CalcStatus(taggedObject->GetTags(), sensors);
                }
            }
        }

        if (ReportTraits & EReportTraits::ReportTagDetails) {
            auto eg = NDrive::BuildEventGuard("fetch.tags");
            TSet<TString> userIds;
            for (auto&& i : DevicesSnapshot) {
                for (auto&& tag : i.second->GetTags()) {
                    if (!!tag->GetPerformer()) {
                        userIds.emplace(tag->GetPerformer());
                    }
                }
            }
            TagPerformers = DriveApi.GetUsersData()->FetchInfo(userIds);
        }
    }

    if (ReportTraits & NDeviceReport::ReportClusters) {
        auto eg = NDrive::BuildEventGuard("fetch.clusters");
        Clusters.Initialize(DriveApi, Server.GetSettings());
    }
    if (ReportTraits & NDeviceReport::ReportFilters) {
        auto eg = NDrive::BuildEventGuard("fetch.filters");
        Filters = permissions->GetFilterActions();
    }
    if (ReportTraits & NDeviceReport::ReportFormerNumbers) {
        auto eg = NDrive::BuildEventGuard("fetch.former_numbers");
        for (auto&& [id, info] : CarInfosOriginal) {
            auto optionalFormerNumbers = DriveApi.GetCarManager().FetchFormerNumbers(id, tx);
            if (!optionalFormerNumbers) {
                return false;
            }
            FormerNumbers[id] = std::move(*optionalFormerNumbers);
        }
    }
    if (ReportTraits & NDeviceReport::ReportHeadId) {
        auto eg = NDrive::BuildEventGuard("fetch.head_ids");
        auto objectIds = MakeVector(NContainer::Keys(CarInfosOriginal));
        if (!DriveApi.GetHeadAccountManager().GetHeadIds(objectIds, HeadIds)) {
            return false;
        }
        if (!DriveApi.GetHeadAccountManager().GetHeadDeviceIds(objectIds, HeadDeviceIds)) {
            return false;
        }
    }
    if (ReportTraits & NDeviceReport::ReportDocuments) {
        auto eg = NDrive::BuildEventGuard("fetch.documents");
        auto objectIds = MakeVector(NContainer::Keys(CarInfosOriginal));
        if (!objectIds.empty()) {
            CarDocumentAttachments = DriveApi.GetCarAttachmentAssignments().GetAssignmentsOfType(objectIds, EDocumentAttachmentType::CarRegistryDocument);
        }
    }
    if (ReportTraits & NDeviceReport::ReportSignalq) {
        auto eg = NDrive::BuildEventGuard("fetch.car_signal_devices");
        auto objectIds = MakeVector(NContainer::Keys(CarInfosOriginal));
        if (!objectIds.empty()) {
            CarSignalDevices = DriveApi.GetCarAttachmentAssignments().GetAssignmentsOfType(objectIds, EDocumentAttachmentType::CarSignalDevice);
        }
    }
    if (ReportTraits & NDeviceReport::ReportValidatedPhotos) {
        auto eg = NDrive::BuildEventGuard("fetch.host_by_source_mapping");
        if (DriveApi.HasMDSClient()) {
            ImageHostBySourceMapping = TCommonImageData::GetHostByImageSourceMapping(DriveApi.GetMDSClient());
        }
    }

    TVector<TOrderedCarInfo> cars;
    OverLimitedCars.clear();
    {
        auto eg = NDrive::BuildEventGuard("fetch.merge");
        TVector<TString> clusters;
        auto itTagsObservable = ObservableTagsData.begin();
        auto itCityGroups = CityGroups.begin();
        auto itTagsFullDevices = DevicesSnapshot.begin();
        auto itSnapshots = Snapshots.GetIterator();
        auto itRTFactorsInfo = RTFactors.begin();
        auto itRecalculatedStatus = RecalculatedStatuses.begin();
        auto itState = CarStatuses.begin();
        auto itSession = CarCurrentSessions.begin();
        auto itAvailable = AvailableCars.begin();
        auto itFutures = Futures.begin();
        auto itHeadId = HeadIds.begin();
        auto itHeadDeviceId = HeadDeviceIds.begin();
        auto itDocumentAttachment = CarDocumentAttachments.begin();
        auto itCarSignalDevice = CarSignalDevices.begin();
        cars.reserve(CarInfosOriginal.GetResult().size());
        const bool showModelsAll = (ReportTraits & NDeviceReport::ReportShowAllModels) || ((ReportTraits & (NDeviceReport::ReportModels)) && ((ReportTraits & NDeviceReport::ReportCars) == 0));
        for (auto it = CarInfosOriginal.GetResult().begin(); it != CarInfosOriginal.GetResult().end(); ++it) {
            const TDriveCarInfo& info = it->second;
            const TRTDeviceSnapshot* snapshot = itSnapshots.SkipTo(it->first);

            if (SignalqStatusFilter && !SignalqStatusFilter->contains(MakeSignalqStatus(snapshot))) {
                continue;
            }

            i32 modelIdx = -1;
            if (showModelsAll) {
                modelIdx = ModelIndexes.emplace(info.GetModel(), ModelIndexes.size()).first->second;
            }
            if (SelectedOnly && !SelectedByUser.contains(info.GetId())) {
                continue;
            }

            const bool isAvailable = AdvanceSimple(itAvailable, AvailableCars.end(), it->first);
            EFuturesStage futuresStage = EFuturesStage::NoFutures;
            if (Advance(itFutures, Futures.end(), it->first)) {
                futuresStage = itFutures->second;
            }

            TCluster* cluster = nullptr;
            const TBillingSession* billingSession = nullptr;
            if (Advance(itSession, CarCurrentSessions.end(), it->first) && !!itSession->second) {
                billingSession = dynamic_cast<const TBillingSession*>(itSession->second.Get());
            }
            const TPointInfo* startAddress = nullptr;
            if (billingSession) {
                startAddress = Starts.FindPtr(billingSession->GetSessionId());
            }
            const TDriveUserData* sessionUser = nullptr;
            if (billingSession) {
                sessionUser = SessionUsers.GetResultPtr(billingSession->GetUserId());
            }
            if (isAvailable) {
                if (snapshot && !Clusters.Register(snapshot->GetClusterIds(), cluster)) {
                    continue;
                }
            } else if (futuresStage != EFuturesStage::NoFutures) {
                if (Clusters.HasSelectedClusterId()) {
                    continue;
                }
                if (billingSession) {
                    const TFixPointOffer* fpOffer = dynamic_cast<const TFixPointOffer*>(billingSession->GetCurrentOffer().Get());
                    if (fpOffer && (Clusters.Check(fpOffer->GetFinishAreaIds()))) {
                        continue;
                    }
                }
            }

            TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> documentAttachment;
            if (Advance(itDocumentAttachment, CarDocumentAttachments.end(), it->first)) {
                documentAttachment = itDocumentAttachment->second.GetImpl();
            }

            TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> carSignalDevice;
            if (!CarSignalDevices.empty() && Advance(itCarSignalDevice, CarSignalDevices.end(), it->first)) {
                carSignalDevice = itCarSignalDevice->second.GetImpl();
            }

            NDrive::TLocation location;
            bool hasLocation = false;
            if (IsRealtime) {
                hasLocation = snapshot && snapshot->GetLocation(location);
                if (!hasLocation) {
                    if ((ReportTraits & NDeviceReport::EReportTraits::ReportIncorrectLocation) == 0) {
                        continue;
                    }
                } else if (HasRect() && !GetRectUnsafe().Contain(location.GetCoord())) {
                    continue;
                }
            }
            NThreading::TFuture<NDrive::TGeocoder::TResponse> geocodedLocationFuture;
            if (IsRealtime && ReportTraits & EReportTraits::ReportGeocodedLocation && ReportTraits & EReportTraits::ReportLeasingStats && hasLocation) {
                const auto language = enum_cast<ELanguage>(Locale);
                const auto geocoder = DriveApi.HasGeocoderClient() ? &DriveApi.GetGeocoderClient() : nullptr;
                if (geocoder) {
                    geocodedLocationFuture = geocoder->Decode(location.GetCoord(), language);
                }
            }

            modelIdx = ModelIndexes.emplace(info.GetModel(), ModelIndexes.size()).first->second;

            if (cars.size() >= Limit && SortMethod == ESortMethod::None) {
                if (!!info.GetNumber()) {
                    OverLimitedCars.emplace_back(info.GetNumber());
                }
            } else {
                const TTaggedObject* taggedObject = Advance(itTagsFullDevices, DevicesSnapshot.end(), it->first) ? itTagsFullDevices->second.Get() : nullptr;
                const auto recalculatedStatus = Advance(itRecalculatedStatus, RecalculatedStatuses.end(), it->first);
                const auto cachedStatus = Advance(itState, CarStatuses.end(), it->first);
                const TString* stateInfo = recalculatedStatus ? recalculatedStatus : cachedStatus;
                TVector<const TDBTag*>* observableTags = Advance(itTagsObservable, ObservableTagsData.end(), it->first) ? &itTagsObservable->second : nullptr;
                const TString* cityGroup = Advance(itCityGroups, CityGroups.end(), it->first) ? &itCityGroups->second : nullptr;
                const TCarRTFactors* surgeInfo = Advance(itRTFactorsInfo, RTFactors.end(), it->first) ? &itRTFactorsInfo->second : nullptr;
                if ((futuresStage != EFuturesStage::NoFutures) || isAvailable) {
                    cars.emplace_back(info, observableTags, taggedObject, snapshot, surgeInfo, stateInfo,
                                      billingSession, cluster, isAvailable, futuresStage, documentAttachment, carSignalDevice,
                                      startAddress, sessionUser, std::move(geocodedLocationFuture));
                    cars.back().CityGroup = cityGroup;
                    cars.back().HeadId = Advance(itHeadId, HeadIds.end(), it->first) ? &itHeadId->second : nullptr;
                    cars.back().ModelIdx = modelIdx;
                    cars.back().HeadDeviceId = Advance(itHeadDeviceId, HeadDeviceIds.end(), it->first) ? &itHeadDeviceId->second : nullptr;
                }
            }
        }
    }
    {
        auto eg = NDrive::BuildEventGuard("fetch.models");
        for (auto&& model : SelectedModels) {
            ModelIndexes.emplace(model, ModelIndexes.size());
        }
        ModelsInfo = DriveApi.GetModelsData()->GetCachedOrFetch(MakeSet(NContainer::Keys(ModelIndexes)), tx);
        ModelsInfoVector.resize(ModelIndexes.size(), nullptr);
        for (auto&& i : ModelIndexes) {
            ModelsInfoVector[i.second] = ModelsInfo.GetResultPtr(i.first);
        }
    }
    if (ReportTraits & NDeviceReport::ReportCars) {
        auto eg = NDrive::BuildEventGuard("fetch.view");
        auto view = CreateView(cars, SortMethod);
        for (ui32 overlimitIdx = /*Limit*/ 0; overlimitIdx < view.size(); ++overlimitIdx) {
            TStringBuf sbNumber = cars[view[overlimitIdx]].Info.GetNumber();
            if (!!sbNumber) {
                OverLimitedCars.emplace_back(std::move(sbNumber));
            }
        }
        bool needUpdateModelIds = (view.size() > Limit) || PageSize;
        if (view.size() > Limit) {
            view.resize(Limit);
        } else if (PageSize) {
            ui32 offset = (PageNumber - 1) * PageSize;
            if (offset >= view.size()) {
                view.clear();
            } else {
                if (view.size() - offset > PageSize) {
                    CanGetMorePages = true;
                }
                view = TVector<size_t>(view.begin() + offset, view.begin() + offset + Min(static_cast<size_t>(PageSize), view.size() - offset));
            }
        }
        if (needUpdateModelIds) {
            TSet<TString> modelIds;
            for (auto&& i : view) {
                modelIds.emplace(cars[i].Info.GetModel());
            }
            for (auto it = ModelsInfo.MutableResult().begin(); it != ModelsInfo.MutableResult().end();) {
                if (!modelIds.contains(it->first)) {
                    it = ModelsInfo.MutableResult().erase(it);
                } else {
                    ++it;
                }
            }
        }
        CarsView = std::move(view);
    }
    Cars = std::move(cars);
    LocationOptions = Server.GetSnapshotsManager().GetLocationOptions();
    if (ReportCreationTime && ReportTraits & NDeviceReport::ReportCars && ReportTraits & NDeviceReport::ReportTagDetails) {
        TSet<TString> tagIds;
        for (const auto& [_, dbTags] : ObservableTagsData) {
            for (const auto* dbTag : dbTags) {
                if (dbTag) {
                    tagIds.insert(dbTag->GetTagId());
                }
            }
        }
        auto optionalEvents = DriveApi.GetTagsManager().GetDeviceTags().GetEvents(TRange<TInstant>{}, tx, TTagEventsManager::TQueryOptions()
            .SetTagIds(std::move(tagIds))
            .SetActions({EObjectHistoryAction::Add})
        );
        if (!optionalEvents) {
            return false;
        }
        {
            for (const auto& ev : *optionalEvents) {
                if (ev) {
                    TagsCreationTimes[ev.GetTagId()] = ev.GetHistoryTimestamp();
                }
            }
        }
    }
    if (ModelTraits & NDriveModelReport::ReportModelSpecifications) {
        auto eg = NDrive::BuildEventGuard("fetch.model_specifications");
        auto inserted = TSet<TString>();
        for (auto&& [id, info] : CarInfosOriginal) {
            inserted.clear();
            for (auto&& specification : info.GetSpecifications().GetSpecifications()) {
                const auto& name = specification.GetName();
                const auto& value = specification.GetValue();
                ModelSpecificationStats[name][value] += 1;
                inserted.insert(name);
            }
            auto model = GetModelInfo(info.GetModel());
            R_ENSURE(model, HTTP_INTERNAL_SERVER_ERROR, "cannot GetModelInfo " << info.GetModel());
            for (auto&& specification : model->GetSpecifications().GetSpecifications()) {
                const auto& name = specification.GetName();
                if (inserted.contains(name)) {
                    continue;
                }

                const auto& value = specification.GetValue();
                ModelSpecificationStats[name][value] += 1;
            }
        }
    }

    DataFetched = true;
    FetchFinishInstant = Now();
    return true;
}

namespace {
    class TSensorLocationSignal: public TUnistatSignal<double> {
    public:
        TSensorLocationSignal()
            : TUnistatSignal<double>({ "frontend-sensors-location" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class THasSensorLocationSignal: public TUnistatSignal<double> {
    public:
        THasSensorLocationSignal()
            : TUnistatSignal<double>({ "frontend-sensors-location-accept" }, EAggregationType::Sum, "dmmm")
        {
        }
    };

    class TNoLocationSignal: public TUnistatSignal<double> {
    public:
        TNoLocationSignal()
            : TUnistatSignal<double>({ "frontend-no-location" }, EAggregationType::Sum, "dmmm")
        {
        }
    };

    class TLocationOldSignal: public TUnistatSignal<double> {
    public:
        TLocationOldSignal()
            : TUnistatSignal<double>({ "frontend-old-location" }, EAggregationType::Sum, "dmmm")
        {
        }
    };

    TSensorLocationSignal LocationSensorsSignal;
    TNoLocationSignal LocationNoSignal;
    TLocationOldSignal LocationOldSignal;
    THasSensorLocationSignal HasSensorLocationSignal;
}

void TCarsFetcher::SerializeLocationInfo(NJson::TJsonWriter& locationReport, const NDrive::TLocation& location, const TRTDeviceSnapshot* snapshot) const {
    locationReport.Write("lat", location.Latitude);
    locationReport.Write("lon", location.Longitude);
    if (ReportTraits & EReportTraits::ReportLocationCourse) {
        locationReport.Write("course", location.Course);
    }
    if (ReportTraits & EReportTraits::ReportLocationDetails) {
        locationReport.Write("name", location.Name);
        locationReport.Write("type", ToString(location.Type));
        locationReport.Write("precision", location.Precision);
        locationReport.Write("timestamp", location.Timestamp.Seconds());

        if (location.Base) {
            locationReport.Write("base_latitude", location.Base->Latitude);
            locationReport.Write("base_longitude", location.Base->Longitude);
            locationReport.Write("base_timestamp", location.Base->Timestamp.Seconds());
        }

        locationReport.OpenArray("tags");
        for (auto&& tag : snapshot->GetTagsInPoint()) {
            locationReport.Write(tag);
        }
        locationReport.CloseArray();

        const TMap<TString, TArea> areas = DriveApi.GetAreasDB()->GetCachedObjectsMap(snapshot->GetAreaIds());

        locationReport.OpenArray("areas");
        for (auto&&[areaId, area] : areas) {
            locationReport.OpenMap();
            locationReport.Write("id", areaId);
            locationReport.Write("title", area.GetTitle());
            if (area.GetTags().size()) {
                locationReport.OpenArray("tags");
                for (auto&& tag : area.GetTags()) {
                    locationReport.Write(tag);
                }
                locationReport.CloseArray();
            }
            locationReport.CloseMap();
        }
        locationReport.CloseArray();
    }
}

void TCarsFetcher::SerializeLagInfo(NJson::TJsonWriter& lagReport, const NDrive::TLocation& location, const TRTDeviceSnapshot* snapshot) const {
    NDrive::THeartbeat heartbeat;
    if (snapshot->GetHeartbeat(heartbeat)) {
        lagReport.Write("heartbeat", heartbeat.Timestamp.Seconds());
        lagReport.Write("host", heartbeat.Host);
    } else {
        lagReport.WriteNull("heartbeat");
        lagReport.WriteNull("host");
    }
    if (heartbeat.Created) {
        lagReport.Write("created", heartbeat.Created.Seconds());
    } else {
        lagReport.WriteNull("created");
    }
    if (!location.IsZero()) {
        lagReport.Write("location", location.Timestamp.Seconds());
    } else {
        lagReport.WriteNull("location");
    }
    if (auto configuratorHeartbeat = snapshot->GetConfiguratorHeartbeat()) {
        lagReport.Write("configurator", configuratorHeartbeat->Timestamp.Seconds());
    } else {
        lagReport.WriteNull("configurator");
    }
    if (auto speed = NDrive::ISensorApi::FindSensor(snapshot->GetSensors(), VEGA_SPEED)) {
        lagReport.Write("blackbox", speed->Timestamp.Seconds());
    } else {
        lagReport.WriteNull("blackbox");
    }
}

namespace {
    bool IsValidBleMac(const NDrive::TSensor& sensor) {
        if (!(sensor == NDrive::NVega::BleMac)) {
            return false;
        }
        if (!std::holds_alternative<TBuffer>(sensor.Value)) {
            return false;
        }
        const auto& value = std::get<TBuffer>(sensor.Value);
        if (value.size() != 6) {
            return false;
        }
        for (auto i = value.Begin(); i != value.End(); ++i) {
            if (*i != 0) {
                return true;
            }
        }
        return false;
    }

    TMaybe<TString> GetGeocodedFromPreciseLocation(const NDrive::TLocation& location) {
        if (location.Name == NDrive::GeocodedLocationName) {
            return location.Content;
        } else if (location.Name == NDrive::BeaconsLocationName) {
            NJson::TJsonValue value;
            if (ReadJsonFastTree(location.Content, &value)) {
                if (value["geocoded_location"].IsString()) {
                    return value["geocoded_location"].GetString();
                }
            }
        }
        return {};
    }

    TMaybe<TString> GetParkingPlaceFromPreciseLocation(const NDrive::TLocation& location) {
        if (location.Name == NDrive::BeaconsLocationName) {
            NJson::TJsonValue value;
            if (ReadJsonFastTree(location.Content, &value)) {
                if (value["parking_place"].IsString()) {
                    return value["parking_place"].GetString();
                }
            }
        }
        return {};
    }

    double GetLitersPer100Km(const TDriveModelData* /*modelInfo*/) {
        return 10;
    }

    TMaybe<NDrive::TSensor> GetFuelLevelPercent(const NDrive::TMultiSensor& sensors, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo) {
        auto percentSensor = NDrive::ISensorApi::GetSensorFamily(sensors, CAN_FUEL_LEVEL_P);
        auto litersSensor = NDrive::ISensorApi::GetSensorFamily(sensors, CAN_CUSTOM_FUEL_VOLUME);

        return NDrive::GetFuelLevel(percentSensor, litersSensor, carInfo, modelInfo);
    }

    TMaybe<i32> GetDistanceFromFuelLevel(const NDrive::TMultiSensor& sensors, const TDriveCarInfo* carInfo, const TDriveModelData* modelInfo) {
        auto fuelLevelSensor = GetFuelLevelPercent(sensors, carInfo, modelInfo);
        if (!!fuelLevelSensor) {
            const double tankVolume = NDrive::GetFuelTankVolume(carInfo, modelInfo);
            const double litersPer100Km = GetLitersPer100Km(modelInfo);
            double fuelingValue = 0.;
            if (fuelLevelSensor->TryConvertTo<double>(fuelingValue)) {
                return i32(tankVolume * fuelingValue / litersPer100Km);
            }
        }

        return {};
    }
}

void TCarsFetcher::SerializeTelematic(NJson::TJsonWriter& telematicsReport, const TOrderedCarInfo& carInfo, const TRTDeviceSnapshot* snapshot) const {
    const TString& carId = carInfo.Info.GetId();
    NDrive::TMultiSensor sensors = snapshot->GetSensors();
    sensors = TTelematicsSensorTag::Transform(carId, std::move(sensors), Server);

    for (auto&& i : SensorIds) {
        TString fieldName = i.GetName();
        auto s = NDrive::ISensorApi::GetSensorFamily(sensors, i.Id);
        switch (i.Id) {
            case CAN_ENGINE_IS_ON:
                fieldName = "is_engine_on";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_FUEL_LEVEL_P:
                fieldName = "fuel_level";
                s = GetFuelLevelPercent(sensors, &carInfo.Info, carInfo.ModelInfo);
                telematicsReport.Write(fieldName, s ? s->GetJsonValue() : NJson::JSON_NULL);
                break;
            case CAN_CUSTOM_FUEL_VOLUME:
                fieldName = "fuel_liters_level";
                telematicsReport.Write(fieldName, s ? s->GetJsonValue() : NJson::JSON_NULL);
            case CAN_ODOMETER_KM:
                fieldName = "mileage";
                telematicsReport.Write(fieldName, s ? s->GetJsonValue() : NJson::JSON_NULL);
                break;
            case CAN_DRIVER_DOOR:
                fieldName = "is_front_left_door_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_PASS_DOOR:
                fieldName = "is_front_right_door_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_L_REAR_DOOR:
                fieldName = "is_rear_left_door_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_R_REAR_DOOR:
                fieldName = "is_rear_right_door_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_HOOD:
                fieldName = "is_hood_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_TRUNK:
                fieldName = "is_trunk_open";
                telematicsReport.Write(fieldName, NDrive::TSensor::TryConvertTo<bool>(s).GetOrElse(false));
                break;
            case CAN_FUEL_DISTANCE_KM:
            {
                auto modelInfo = carInfo.ModelInfo;
                bool fuelDistanceEnabled = modelInfo ? modelInfo->GetFuelDistanceEnabled() : true;
                double fuelingValue = 0.;
                if (!fuelDistanceEnabled || !s || s->IsZero()) {
                    auto distance = GetDistanceFromFuelLevel(sensors, &carInfo.Info, modelInfo);
                    if (distance) {
                        telematicsReport.Write(fieldName, distance.GetRef());
                    }
                } else if (s && s->TryConvertTo<double>(fuelingValue)) {
                    const auto& mileageDecreasingCoefficient = GetSetting<ui32>("telematics.mileage_decreasing_coefficient").GetOrElse(0);
                    if (mileageDecreasingCoefficient != 0) {
                        fuelingValue = fuelingValue * fuelingValue / mileageDecreasingCoefficient;
                    }
                    telematicsReport.Write(fieldName, fuelingValue);
                    const auto& averageSpeed = GetSetting<ui32>("telematics.average_speed").GetOrElse(0);
                    if (averageSpeed != 0) {
                        // remainingTime [sec] = (fuelDistance [km] / averageSpeed [km/h]) * 3600 [sec/h]
                        const auto& remainingTime = static_cast<ui32>((fuelingValue / averageSpeed) * 3600);
                        // round to 10 — ex: 157 -> 150; 34 -> 30
                        telematicsReport.Write("remaining_time", remainingTime - remainingTime % (10 * 60));
                    }
                }
                break;
            }
            case NDrive::NVega::BleSessionKey.Id:
            {
                auto hideBleInfo = GetSetting<bool>("telematics.ble_info.hide").GetOrElse(false);
                if (ReportTraits & EReportTraits::ReportBluetoothInfo && !hideBleInfo) {
                    auto decodeBleInfo = GetSetting<bool>("telematics.ble_info.decode").GetOrElse(false);
                    NJson::TJsonValue sensor;
                    if (decodeBleInfo) {
                        sensor = s ? NJson::TJsonValue(s->ConvertTo<TString>()) : NJson::JSON_NULL;
                    } else {
                        sensor = s ? s->GetJsonValue() : NJson::JSON_NULL;
                    }
                    telematicsReport.Write(fieldName, sensor);
                } else {
                    telematicsReport.Write(fieldName, TString(64, '1'));
                }
                break;
            }
            case NDrive::NVega::BlePasskey.Id:
                if (s && !(*s == NDrive::NVega::BlePasskey)) {
                    s = NDrive::ISensorApi::FindSensor(sensors, NDrive::NVega::BlePasskey);
                }
                fieldName = "ble_passkey";
                if (ReportTraits & EReportTraits::ReportBluetoothInfo) {
                    auto sensor = s ? s->GetJsonValue() : NJson::JSON_NULL;
                    telematicsReport.Write(fieldName, sensor);
                } else {
                    telematicsReport.Write(fieldName, 111111);
                }
                break;
            case NDrive::NVega::BleMac.Id:
            {
                auto hideBleInfo = GetSetting<bool>("telematics.ble_info.hide").GetOrElse(false);
                if (ReportTraits & (EReportTraits::ReportBluetoothInfo | EReportTraits::ReportBluetoothMAC) && !hideBleInfo) {
                    auto decodeBleInfo = GetSetting<bool>("telematics.ble_info.decode").GetOrElse(false);
                    NJson::TJsonValue sensor;
                    if (decodeBleInfo) {
                        sensor = s ? NJson::TJsonValue(s->ConvertTo<TString>()) : NJson::JSON_NULL;
                    } else {
                        sensor = (s && IsValidBleMac(*s)) ? s->GetJsonValue() : NJson::JSON_NULL;
                    }
                    telematicsReport.Write(fieldName, sensor);
                } else {
                    telematicsReport.Write(fieldName, "111111111111");
                }
                break;
            }
            case NDrive::NVega::SvrRawState.Id:
                if (s) try {
                    NDrive::NVega::TSvrRawState state;
                    state.Parse(s->ConvertTo<TBuffer>());

                    auto treatZeroAsNull = GetSetting<bool>(
                        TStringBuilder() << "telematics.sensor." << NDrive::NVega::GetSensorName(state.TankLevelAverageId) << ".treat_zero_as_null"
                    ).GetOrElse(true);

                    fieldName = "second_fuel_level";
                    if (!state.TankLevelAverage && !state.SerialNumber && treatZeroAsNull) {
                        telematicsReport.WriteNull(fieldName);
                    } else if (auto calibration = GetSetting<NDrive::NVega::TSensorCalibration>(
                        TStringBuilder() << "telematics.sensor." << NDrive::NVega::GetSensorName(state.TankLevelAverageId) << ".calibration"
                    )) {
                        telematicsReport.Write(fieldName, calibration->Get(state.TankLevelAverage));
                    } else {
                        telematicsReport.Write(fieldName, state.GetTankLevelAverage());
                    }

                    fieldName = "second_fuel_distance";
                    if (!state.TankLevelAverage && !state.SerialNumber && treatZeroAsNull) {
                        telematicsReport.WriteNull(fieldName);
                    } else if (auto calibration = GetSetting<NDrive::NVega::TSensorCalibration>(
                        TStringBuilder() << "telematics.sensor." << NDrive::NVega::GetSensorName(state.DistanceLeftId) << ".calibration"
                    )) {
                        telematicsReport.Write(fieldName, calibration->Get(state.TankLevelAverage));
                    } else {
                        telematicsReport.Write(fieldName, state.DistanceLeft);
                    }
                    telematicsReport.Write("second_fuel_type", "gas");
                } catch (...) {
                    auto evlog = NDrive::GetThreadEventLogger();
                    if (evlog) {
                        evlog->AddEvent(NJson::TMapBuilder
                            ("event", "InvalidSvrRawState")
                            ("error", CurrentExceptionInfo())
                        );
                    }
                }
                break;
            default:
                if (const auto fieldNameOverride = GetSetting<TString>("telematics." + fieldName + ".field_name")) {
                    fieldName = *fieldNameOverride;
                }
                telematicsReport.Write(fieldName, s ? s->GetJsonValue() : NJson::JSON_NULL);
        }
        if (s && (ReportTraits & EReportTraits::ReportSensorTimestamp)) {
            telematicsReport.Write(fieldName + "_updated_at", s->Since.Seconds());
        }
    }
}

void TCarsFetcher::SerializeEnclosingAreas(NJson::TJsonWriter& areasReport, const NDrive::TLocation& preciseLocation) const {
    auto actor = [&areasReport](const TFullAreaInfo& areaInfo) {
        areasReport.Write(areaInfo.GetArea().GetIdentifier());
        return true;
    };
    Yensured(DriveApi.GetAreasDB())->ProcessAreasInPoint(preciseLocation.GetCoord(), actor);
}

void TCarsFetcher::SerializeLeasingStats(NJson::TJsonWriter& leasingStatsReport, const TOrderedCarInfo& carInfo, ui32 daysCount) const {
    if (carInfo.TaggedObject) {
        for (const auto& dbTag : carInfo.TaggedObject->GetTags()) {
            const auto* leasingStatsTag = dbTag.GetTagAs<NDrivematics::TLeasingStatsTag>();
            if (leasingStatsTag) {
                auto maybeAggregatedStats = leasingStatsTag->AggregateStats(daysCount);
                if (!maybeAggregatedStats) {
                    continue;
                }
                const auto& aggregatedStats = *maybeAggregatedStats;
                ForEach(aggregatedStats.GetFields(), [&leasingStatsReport] (auto&& field) {
                    if (!(field.GetTraits() & NDrivematics::ReportInCarList)) {
                        return;
                    }
                    leasingStatsReport.OpenMap(field.GetName());
                    leasingStatsReport.Write("type", ToString(field.GetStatsType()));
                    leasingStatsReport.Write("value", field.Value);
                    leasingStatsReport.CloseMap();
                });
            } else if (dbTag->GetName() == NDrivematics::FreshIssueDateTagName) {
                leasingStatsReport.Write("fresh_issue_date", true);
            }
        }
    }
}

void TCarsFetcher::SerializeTaxiCompany(NJson::TJsonWriter& taxiCompanyReport, const TString& carId) const {
    const auto& tagsManager = DriveApi.GetTagsManager();
    auto taxiCompanyTagsNames = tagsManager.GetTagsMeta().GetRegisteredTagNames({NDrivematics::TTaxiCompanyTag::TypeName});
    TVector<TTaggedDevice> taggedObjects;
    R_ENSURE(tagsManager.GetDeviceTags().GetObjectsFromCache({carId}, taxiCompanyTagsNames, taggedObjects, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can not restore tags");
    if (!taggedObjects.empty()) {
        TVector<TDBTag> taxiCompanyTags;
        taggedObjects.front().FilterTags(taxiCompanyTagsNames, &taxiCompanyTags);
        Y_ASSERT(taxiCompanyTags.size() == 1);
        const auto& tag = taxiCompanyTags.front();
        auto description = std::dynamic_pointer_cast<const NDrivematics::TTaxiCompanyTag::TDescription>(tagsManager.GetTagsMeta().GetDescriptionByName(tag->GetName()));
        Y_ENSURE(description);
        taxiCompanyReport.Write("name", description->GetTaxiCompanyName());
        taxiCompanyReport.Write("id", tag->GetName());
    }
}

class TSortedObservableTag {
private:
    const TDBTag* Tag = nullptr;
    ui32 Abilities = 0;
    i64 Priority = 0;
    R_READONLY(TString, Color);
public:
    TSortedObservableTag(const TDBTag* tag, TUserPermissions::TPtr permissions, const NDrive::IServer& server)
        : Tag(tag)
    {
        if (!!Tag && !!Tag->GetData() && !!permissions) {
            auto actions = permissions->GetActionsByTagIdx(Tag->GetData()->GetDescriptionIndex());
            if ((*Tag)->GetPerformer() == permissions->GetUserId()) {
                Abilities = 5;
            } else if (actions & (ui32)TTagAction::ETagAction::Perform) {
                Abilities = 4;
            } else if (actions & (ui32)TTagAction::ETagAction::Observe) {
                Abilities = 3;
            } else {
                Abilities = 2;
            }
            Priority = (*Tag)->GetTagPriority(0);
            auto description = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Tag->GetData()->GetName());
            if (description && description->GetColor()) {
                Color = description->GetColor();
            }
        }
    }

    bool operator <(const TSortedObservableTag& item) const {
        return std::tie(Abilities, Priority, Color) < std::tie(item.Abilities, item.Priority, item.Color);
    }
};

TCarsFetcher::ESignalqStatus TCarsFetcher::MakeSignalqStatus(const TRTDeviceSnapshot* snapshot) const {
    // order of 'if'-s is defined by Signalq managers and repeats the behaviour of the similar endpoint in Signalq uservices
    const auto lastCarEventPtr = snapshot ? snapshot->GetLastSignalqEventPtr() : nullptr;
    if (!snapshot || !snapshot->HasSignalqStatus() || lastCarEventPtr && lastCarEventPtr->GetType() == "shutdown") {
        return ESignalqStatus::TurnedOff;
    }

    const auto& status = snapshot->GetSignalqStatusRef();
    const auto actualStatusGap =
        Server.GetSettings().GetValue<TDuration>("signalq.signalq_status.actual_status_gap");
    if (Now() - status.GetStatusAt() > actualStatusGap) {
        return ESignalqStatus::Offline;
    }

    const auto statesPtr = status.GetStatesPtr();
    if (statesPtr) {
        const auto cameraPose = (*statesPtr)["dms.analytics.CameraPose"];
        if (cameraPose.IsBoolean() && cameraPose.GetBoolean()) {
            return ESignalqStatus::FacedAway;
        }
        const auto trashFrames = (*statesPtr)["dms.analytics.TrashFrames"];
        if (trashFrames.IsBoolean() && trashFrames.GetBoolean()) {
            return ESignalqStatus::CameraClosed;
        }
    }

    return ESignalqStatus::TurnedOn;
}

const TString OsagoDefaultTitle = "ОСАГО";
const TString RegistrationDefaultTitle = "СТС";

bool TCarsFetcher::BuildOneCarInfoReport(const TOrderedCarInfo& carInfo, const TSet<TFilterAction>& filters, NJson::TJsonWriter& carReport, const TSet<TString>& supportAllowedPhotos) const {
    const TString& carId = carInfo.Info.GetId();
    auto locale = GetLocale();
    auto localization = Server.GetLocalization();

    carInfo.Info.FillReport(Locale, ReportTraits, carReport);
    if ((ReportTraits & EReportTraits::ReportAvailability) && carInfo.TaggedObject && Permissions) {
        carReport.Write(
            "available",
            Permissions->GetVisibility(*carInfo.TaggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible
        );
    }
    if ((ReportTraits & EReportTraits::ReportClusters) && carInfo.Cluster && carInfo.Cluster->IsActive()) {
        carReport.Write("cluster", carInfo.Cluster->Id);
    }
    if ((ReportTraits & EReportTraits::ReportHeadId)) {
        if (carInfo.HeadId) {
            carReport.Write("head_id", *carInfo.HeadId);
        }
        if (carInfo.HeadDeviceId) {
            carReport.Write("head_device_id", *carInfo.HeadDeviceId);
        }
    }
    if (ReportTraits & EReportTraits::ReportFormerNumbers) {
        auto formerNumbers = FormerNumbers.FindPtr(carId);
        carReport.Write("former_numbers", NJson::ToJson(formerNumbers));
    }
    if (ReportTraits & EReportTraits::ReportStatus) {
        carReport.Write("status", carInfo.GetStatus());
    }
    if (carInfo.CityGroup) {
        carReport.Write("city_group", *carInfo.CityGroup);
    }
    if ((ReportTraits & EReportTraits::ReportSelected) && !SelectedByUser.empty()) {
        carReport.Write("selected", SelectedByUser.contains(carId));
    }
    if ((ReportTraits & (EReportTraits::ReportShortProps | EReportTraits::ReportDocuments)) && carInfo.CarRegistryDocument) {
        auto registryDocumentImpl = dynamic_cast<const TCarRegistryDocument*>(carInfo.CarRegistryDocument.Get());
        if (registryDocumentImpl) {
            TCarAttachmentReportContext ctx(Server.GetDriveAPI());
            NJson::TJsonValue documentsReport = NJson::JSON_ARRAY;
            if (registryDocumentImpl->GetOsagoMDSKey()) {
                carReport.Write("osago_mds_key", registryDocumentImpl->GetOsagoMDSKey());
                NJson::TJsonValue reportItem;
                reportItem["title"] = localization ? localization->GetLocalString(locale, "document.osago.title", OsagoDefaultTitle) : OsagoDefaultTitle;
                reportItem["link"] = ctx.GetMdsLink("carsharing-car-documents", registryDocumentImpl->GetOsagoMDSKey());
                documentsReport.AppendValue(std::move(reportItem));
            }
            if (registryDocumentImpl->GetRegistrationMDSKey()) {
                carReport.Write("registration_mds_key", registryDocumentImpl->GetRegistrationMDSKey());
                NJson::TJsonValue reportItem;
                reportItem["title"] = localization ? localization->GetLocalString(locale, "document.registration.title", RegistrationDefaultTitle) : RegistrationDefaultTitle;
                reportItem["link"] = ctx.GetMdsLink("carsharing-car-documents", registryDocumentImpl->GetRegistrationMDSKey());
                documentsReport.AppendValue(std::move(reportItem));
            }
            if (registryDocumentImpl->GetMajorAppMDSKey()) {
                carReport.Write("major_app_mds_key", registryDocumentImpl->GetMajorAppMDSKey());
            }
            if (registryDocumentImpl->GetLessor()) {
                carReport.Write("lessor", registryDocumentImpl->GetLessor());
            }
            carReport.Write("documents", std::move(documentsReport));
        }
    }

    if (ReportTraits & EReportTraits::ReportSignalq) {
        auto carSignalDeviceImpl = dynamic_cast<const TCarSignalDevice*>(carInfo.CarSignalDevice.Get());
        if (carSignalDeviceImpl) {
            carReport.OpenMap("signalq");
            carReport.Write("serial_number", carSignalDeviceImpl->GetSerialNumber());
            if (!carSignalDeviceImpl->GetImei().empty()) {
                carReport.Write("imei", carSignalDeviceImpl->GetImei());
            }
            carReport.CloseMap();
        }
        carReport.Write("signalq_status", ::ToString(MakeSignalqStatus(carInfo.Snapshot)));
    }

    if (!!carInfo.SpecialType) {
        carReport.Write("st", carInfo.SpecialType);
    }

    if (((ReportTraits & EReportTraits::ReportTags) || (ReportTraits & EReportTraits::ReportColors)) && carInfo.TaggedObject) {
        TString usageType = "free";
        for (auto&& tag : carInfo.TaggedObject->GetTags()) {
            if (!tag->GetPerformer()) {
                continue;
            }
            if (tag->GetPerformer() == UserId) {
                usageType = "my";
                break;
            } else {
                usageType = "busy";
                break;
            }
        }
        TSet<TString> colors;
        if (carInfo.ObservableTags && !!Permissions) {
            TSortedObservableTag mainTag(nullptr, Permissions, Server);
            for (auto&& i : *carInfo.ObservableTags) {
                TSortedObservableTag currentSortedTag(i, Permissions, Server);
                if (mainTag < currentSortedTag) {
                    mainTag = currentSortedTag;
                }

            }
            if (mainTag.GetColor()) {
                colors.emplace(mainTag.GetColor());
            } else {
                colors.emplace(GetSetting<TString>("cars_fetcher.default_tag_color").GetOrElse("#80ECECEC"));
            }
        }
        TJsonProcessor::WriteContainerArray(carReport, "colors", colors, false);
        carReport.Write("usage", usageType);
    }

    if ((ReportTraits & EReportTraits::ReportTagDetails) && carInfo.ObservableTags) {
        carReport.OpenArray("tags");
        for (auto&& tag : *carInfo.ObservableTags) {
            carReport.Write(BuildTagJson(*tag, UserId));
        }
        carReport.CloseArray();
    }

    if (ReportTraits & EReportTraits::ReportInsuranceAgreementNumber) {
        TCarInsurancePolicy policy;
        if (DriveApi.GetCarAttachmentAssignments().GetEffectiveInsurancePolicy(carId, Now(), policy, TInstant::Zero())) {
            carReport.Write("insurance_agreement_number", policy.GetAgreementNumber());
            carReport.Write("insurance_provider", ::ToString(policy.GetProvider()));
        } else {
            carReport.WriteNull("insurance_agreement_number");
        }
    }

    if (GetIsRealtime()) {
        if (carInfo.RTFactors) {
            if (ReportTraits & NDeviceReport::ReportSurgeInfo) {
                carReport.Write("surge", carInfo.RTFactors->SerializeToJson());
            }
        }
        bool locationInitialized = false;
        if (carInfo.FuturesStage != EFuturesStage::NoFutures) {
            carInfo.FillLocationFutures(carReport, ReportTraits, *Server.GetLocalization());
        }
        const TRTDeviceSnapshot* snapshot = carInfo.Snapshot;
        if (snapshot && (carInfo.FuturesStage == EFuturesStage::NoFutures || (ReportTraits & NDeviceReport::ReportRTFuturesPosition) || (carInfo.InUsageBy(UserId)))) {
            NDrive::TLocation& location = carInfo.Location;
            auto preciseLocation = snapshot->GetPreciseLocation(TDuration::Max(), LocationOptions);
            if (carInfo.IsAvailable) {
                if (snapshot->GetLocation(location, TDuration::Max(), LocationOptions)) {
                    carReport.OpenMap("location");
                    SerializeLocationInfo(carReport, location, snapshot);
                    carReport.CloseMap();
                    locationInitialized = true;
                    if (FetchFinishInstant >= location.Timestamp) {
                        LocationSensorsSignal.Signal((FetchFinishInstant - location.Timestamp).MilliSeconds());
                        HasSensorLocationSignal.Signal(1);
                    }
                    if ((FetchFinishInstant - location.Timestamp).Seconds() > 300) {
                        LocationOldSignal.Signal(1);
                        DEBUG_LOG << "Too old location signal: " << carInfo.Info.GetId() << Endl;
                    }
                } else {
                    DEBUG_LOG << "No location signal: " << carInfo.Info.GetId() << Endl;
                    LocationNoSignal.Signal(1);
                }
            }

            if (ReportTraits & EReportTraits::ReportLocationDetails) {
                carReport.OpenArray("locations");
                if (carInfo.IsAvailable) {
                    for (auto&& location : snapshot->GetLocations()) {
                        carReport.Write(location.ToJson());
                    }
                }
                carReport.CloseArray();
            }

            if (ReportTraits & EReportTraits::ReportLag) {
                carReport.OpenMap("lag");
                SerializeLagInfo(carReport, location, snapshot);
                carReport.CloseMap();
            }

            if (ReportTraits & EReportTraits::ReportGeocodedLocation) {
                if (ReportTraits & EReportTraits::ReportLeasingStats) {
                    if (locationInitialized) {
                        if (auto asyncGeocoded = carInfo.GeocodedLocationFuture; asyncGeocoded.Initialized()) {
                            auto eg = NDrive::BuildEventGuard("wait_geocoded_location");
                            if (asyncGeocoded.Wait(Deadline) && asyncGeocoded.HasValue()) {
                                carReport.Write("geocoded_location", asyncGeocoded.GetValue().Title);
                            }
                        }
                    } else {
                        carReport.WriteNull("geocoded_location");
                    }
                } else {
                    auto geocoded = (preciseLocation && preciseLocation->Content) ? GetGeocodedFromPreciseLocation(*preciseLocation) : Nothing();
                    if (geocoded) {
                        carReport.Write("geocoded_location", localization ? localization->ApplyResources(*geocoded, locale) : *geocoded);
                    }
                }
            }

            if (ReportTraits & EReportTraits::ReportBeaconParkingPlace) {
                auto parkingPlace = (preciseLocation && preciseLocation->Content) ? GetParkingPlaceFromPreciseLocation(*preciseLocation) : Nothing();
                if (parkingPlace) {
                    carReport.Write("parking_place", localization ? localization->ApplyResources(*parkingPlace, locale) : *parkingPlace);
                }
            }

            if (ReportTraits & EReportTraits::ReportEnclosingAreas) {
                carReport.OpenArray("areas");
                if (preciseLocation) {
                    SerializeEnclosingAreas(carReport, *preciseLocation);
                }
                carReport.CloseArray();
            }
            if (snapshot->GetSensors().size() && SensorIds.size()) {
                carReport.OpenMap("telematics");
                SerializeTelematic(carReport, carInfo, snapshot);
                carReport.CloseMap();
            }
            if (ReportTraits & EReportTraits::ReportCommonSpeed) {
                auto speed = snapshot->GetSpeed();
                if (speed) {
                    carReport.Write("speed", *speed);
                }
            }
        }
        if (!locationInitialized) {
            carReport.WriteNull("location");
        }
    }

    if ((ReportTraits & EReportTraits::ReportPatches)) {
        carInfo.FillPatches(carReport, Patches);
    }

    if ((ReportTraits & EReportTraits::ReportViews)) {
        carInfo.FillView(carReport, Views);
    }

    if (ReportTraits & EReportTraits::ReportShortProps) {
        carInfo.FillShortProps(carReport, Properties, ReportTraits, GetDiscountIntervalsCount());
    }

    if ((ReportTraits & EReportTraits::ReportFilters) && carInfo.TaggedObject && filters.size()) {
        const TVector<ui32> fltrs = CalcFilters(carInfo.TaggedObject->GetTags(), &filters);
        if (carInfo.Cluster) {
            carInfo.Cluster->AddFiltersInfo(fltrs);
        }
        carReport.OpenArray("filters");
        for (auto&& i : fltrs) {
            carReport.Write(i);
        }
        carReport.CloseArray();
    }

    if (ReportTraits & EReportTraits::ReportAggressiveScore) {
        auto scoringTag = carInfo.TaggedObject ? carInfo.TaggedObject->GetFirstTagByClass<TScoringCarTag>() : TDBTag();
        auto scoringTagImpl = scoringTag.GetTagAs<TScoringCarTag>();
        if (scoringTagImpl) {
            carReport.Write("aggressive_rank", NJson::ToJson(scoringTagImpl->OptionalRank()));
            carReport.Write("aggressive_score", scoringTagImpl->GetValue());
            carReport.Write("aggressive_score_tag_id", scoringTag.GetTagId());
        } else {
            carReport.WriteNull("aggressive_rank");
            carReport.WriteNull("aggressive_score");
            carReport.WriteNull("aggressive_score_tag_id");
        }
    }

    if ((ReportTraits & EReportTraits::ReportCurrentOffer) && !!carInfo.Session && !!carInfo.Session->GetCurrentOffer()) {
        carReport.Write("offer_name", carInfo.Session->GetCurrentOffer()->GetName());
        if (!!carInfo.Session->GetCurrentOffer()->GetGroupName()) {
            carReport.Write("offer_group_name", carInfo.Session->GetCurrentOffer()->GetGroupName());
        }

        if (ReportTraits & EReportTraits::ReportCurrentOfferDetails) {
            if (auto standardOffer = std::dynamic_pointer_cast<TStandartOffer>(carInfo.Session->GetCurrentOffer());
                !ChargeAccountsFilter
                || (ReportTraits & EReportTraits::ReportServiceSessions)
                || (!!standardOffer && ChargeAccountsFilter->contains(standardOffer->GetSelectedCharge()))) {
                carReport.OpenMap("current_session");

                if (!!standardOffer) {
                    carReport.Write("insurance_type", standardOffer->GetInsuranceType());
                }
                carReport.Write("session_id", carInfo.Session->GetSessionId());
                carReport.Write("duration", (carInfo.Session->GetLastTS() - carInfo.Session->GetStartTS()).Seconds());
                if (carInfo.StartSessionPoint) {
                    carReport.Write("start_point", carInfo.StartSessionPoint->Point.GetCoord().ToString());
                    if (auto asyncGeocoded = carInfo.StartSessionPoint->Address; asyncGeocoded.Initialized()) {
                        if (asyncGeocoded.Wait(Deadline) && asyncGeocoded.HasValue()) {
                            carReport.Write("start_point_address", asyncGeocoded.GetValue().Title);
                        }
                    }
                }
                if (carInfo.SessionUser) {
                    carReport.Write("user_details", carInfo.SessionUser->GetPublicReport());
                }
                carReport.CloseMap();
            }
        }
    }

    if (ReportTraits & EReportTraits::ReportLeasingStats) {
        carReport.OpenMap("leasing_stats");
        SerializeLeasingStats(carReport, carInfo, GetSetting<ui32>("leasing_cabinet.aggregation.days_count").GetOrElse(60));
        carReport.CloseMap();
    }

    if (ReportTraits & EReportTraits::ReportTaxiCompany) {
        carReport.OpenMap("taxi_company");
        SerializeTaxiCompany(carReport, carId);
        carReport.CloseMap();
    }

    if (ReportTraits & EReportTraits::ReportValidatedPhotos) {
        auto photos = DriveApi.GetImagesDB().GetCachedValidatedImages(carId);
        if (ReportTraits & EReportTraits::ReportSupportFilterPhotos) {
            photos->MutableReportContext().SetSupportAllowedVerdicts(supportAllowedPhotos);
        }
        if (ReportTraits & EReportTraits::ReportValidatedPhotosDups) {
            photos->MutableReportContext().SetSkipIdenticalDups(false);
        }
        photos->MutableReportContext().SetSkipElementDups(!(ReportTraits & EReportTraits::ReportValidatedPhotosElementDups));
        photos->MutableReportContext().SetReportMarkups(ReportTraits & EReportTraits::ReportValidatedPhotosMarkup);
        carReport.Write("photos", photos->GetReport(Locale, Server, ImageHostBySourceMapping));
    }
    return true;
}

bool TCarsFetcher::BuildOneCarInfoReport(const TOrderedCarInfo& carInfo, const TSet<TFilterAction>& filters, NJson::TJsonValue& result) const {
    TString jsonStr;
    {
        TStringOutput so(jsonStr);
        NJson::TJsonWriter jWriter(&so, false);
        jWriter.OpenMap();
        if (!BuildOneCarInfoReport(carInfo, filters, jWriter)) {
            return false;
        }
        jWriter.CloseMap();
    }
    return NJson::ReadJsonFastTree(jsonStr, &result);
}

bool TCarsFetcher::BuildOneCarInfoReport(const TString& carId, const TSet<TFilterAction>& filters, NJson::TJsonValue& result) const {
    TString jsonStr;
    {
        TStringOutput so(jsonStr);
        NJson::TJsonWriter jWriter(&so, false);
        jWriter.OpenMap();
        if (!BuildOneCarInfoReport(carId, filters, jWriter)) {
            return false;
        }
        jWriter.CloseMap();
    }
    return NJson::ReadJsonFastTree(jsonStr, &result);
}

bool TCarsFetcher::GetNeedFueling(const TString& objectId) const {
    const TDriveCarInfo* info = CarInfosOriginal.GetResultPtr(objectId);
    if (!info) {
        return false;
    }
    const TDriveModelData* modelInfo = ModelsInfo.GetResultPtr(info->GetModel());
    if (!modelInfo) {
        return false;
    }
    auto pred = [&objectId](const std::pair<TString, TRTDeviceSnapshot>& snapshot) -> bool {
        return snapshot.first == objectId;
    };
    auto itSnapshot = FindIf(Snapshots.GetSelection().begin(), Snapshots.GetSelection().end(), pred);
    if (itSnapshot == Snapshots.GetSelection().end()) {
        return false;
    }
    auto fuelLevelSensor = NDrive::ISensorApi::GetSensorFamily(itSnapshot->second.GetSensors(), CAN_FUEL_LEVEL_P);
    if (!!fuelLevelSensor) {
        ui64 percLimit = TUserFuelingTag::GetFuelingPercentLimit(info->GetModel(), Permissions, Server);
        double litersLimit = TUserFuelingTag::GetFuelingVolumeLimit(info->GetModel(), Permissions, Server);

        double flSensor;
        if (!fuelLevelSensor->TryConvertTo<double>(flSensor)) {
            return false;
        }

        auto needFuelingLiters = NDrive::GetNeedFuelingLiters(flSensor, info, modelInfo);
        return
            (needFuelingLiters >= litersLimit) &&
            (flSensor <= percLimit);
    } else {
        return false;
    }
}

bool TCarsFetcher::GetNeedTankerFueling(const TString& objectId) const {
    const TDriveCarInfo* info = CarInfosOriginal.GetResultPtr(objectId);
    if (!info) {
        return false;
    }
    auto pred = [&objectId](const std::pair<TString, TRTDeviceSnapshot>& snapshot) -> bool {
        return snapshot.first == objectId;
    };
    auto itSnapshot = FindIf(Snapshots.GetSelection().begin(), Snapshots.GetSelection().end(), pred);
    if (itSnapshot == Snapshots.GetSelection().end()) {
        return false;
    }

    auto configurationTag = DriveApi.GetTagsManager().GetDeviceTags().GetTagFromCache(objectId, TTelematicsConfigurationTag::Type());
    auto configuration = configurationTag ? configurationTag->GetTagAs<TTelematicsConfigurationTag>() : nullptr;
    auto calibratorRequired = GetSetting<bool>("fueling.tanker.calibrator_required").GetOrElse(true);

    TSet<ui16> sensorIds;
    StringSplitter(GetSetting<TString>("fueling.tanker.sensors").GetOrElse("")).Split(',').SkipEmpty().ParseInto(&sensorIds);
    sensorIds.insert(NDrive::NVega::AuxFuelLevel<1>());
    for (auto sensorId : sensorIds) {
        auto sensor = NDrive::ISensorApi::GetSensorFamily(itSnapshot->second.GetSensors(), sensorId);
        if (!sensor) {
            continue;
        }
        auto calibrator = configuration ? configuration->GetCalibrators().FindPtr(sensor->Id) : nullptr;
        bool calibratorValid = calibrator && !calibrator->Empty();
        if (!calibratorValid) {
            if (calibratorRequired) {
                return false;
            }
        }
        auto currentLiters = calibratorValid ? calibrator->Get(*sensor) : sensor->ConvertTo<double>();
        auto maxLiters = std::max<double>(calibratorValid ? calibrator->GetMaxCalibrated() : 2 * currentLiters, 1);
        const double liters = maxLiters - currentLiters;
        const ui64 perc = currentLiters * 100 / maxLiters;
        const ui64 percLimit = TUserFuelingTag::GetTankerFuelingPercentLimit(info->GetModel(), Permissions, Server);
        const double litersLimit = TUserFuelingTag::GetTankerFuelingVolumeLimit(info->GetModel(), Permissions, Server);
        if (liters >= litersLimit && perc <= percLimit && liters > 1e-5) {
            return true;
        }
    }
    return false;
}

i32 TCarsFetcher::GetDiscountIntervalsCount() const {
    if (!DiscountIntervalsCount) {
        DiscountIntervalsCount = Max<i32>(1, GetSetting<i32>("fetcher.discounts.styles.intervals_count").GetOrElse(10));
    }
    return *DiscountIntervalsCount;
}

TMap<TString, NJson::TJsonValue> TCarsFetcher::GetMetaReports() const {
    TMap<TString, NJson::TJsonValue> result;
    if (ReportTraits & EReportTraits::ReportPatches) {
        auto eg = NDrive::BuildEventGuard("report_property_patches");
        result.emplace("property_patches", GetPatchesReportSafe());
    }
    if (ReportTraits & EReportTraits::ReportVisibility) {
        auto eg = NDrive::BuildEventGuard("report_visibility");
        NJson::TJsonValue visibleCarsJson = NJson::JSON_ARRAY;
        for (auto&& i : OverLimitedCars) {
            visibleCarsJson.AppendValue(i);
        }
        result.emplace("visibility", std::move(visibleCarsJson));
    }
    if (ReportTraits & EReportTraits::ReportCurrentOffer) {
        auto eg = NDrive::BuildEventGuard("report_offers");
        result.emplace("offers", GetOffersReportSafe(true));
        result.emplace("offers_array", GetOffersReportSafe(false));
    }
    if (ReportTraits & EReportTraits::ReportStatus) {
        auto eg = NDrive::BuildEventGuard("report_status");
        result.emplace("statuses", GetStatusesReportSafe());
    }
    if (ReportTraits & EReportTraits::ReportSurgeInfo) {
        auto eg = NDrive::BuildEventGuard("report_surge");
        result.emplace("surges", GetSurgesReportSafe());
    }
    if (ReportTraits & EReportTraits::ReportClusters) {
        auto eg = NDrive::BuildEventGuard("report_clusters");
        result.emplace("clusters", GetClustersReportSafe());
    }
    if (ReportTraits & EReportTraits::ReportModels) {
        auto eg = NDrive::BuildEventGuard("report_models");
        result.emplace("models", GetModelsReportSafe(true));
    }
    if (ReportTraits & EReportTraits::ReportViews) {
        auto eg = NDrive::BuildEventGuard("report_views");
        result.emplace("views", GetViewsReportSafe());
    }
    if (ReportTraits & EReportTraits::ReportTags) {
        auto eg = NDrive::BuildEventGuard("report_tags");
        result.emplace("tags", GetTagsReportSafe(true, TagsSortMethod));
    }
    if ((ReportTraits & EReportTraits::ReportTags) || (ReportTraits & EReportTraits::ReportTagsArray)) {
        auto eg = NDrive::BuildEventGuard("report_tags_array");
        result.emplace("tags_array", GetTagsReportSafe(false, TagsSortMethod));
    }
    if (ReportTraits & EReportTraits::ReportShortProps) {
        auto eg = NDrive::BuildEventGuard("report_sf");
        result.emplace("sf", GetShortCarPropertiesReportSafe());
    }
    if (ModelTraits & NDriveModelReport::ReportModelSpecifications) {
        auto eg = NDrive::BuildEventGuard("report_model_specifications");
        result.emplace("model_specifications", GetModelSpecificationReport());
    }
    return result;
}

void TCarsFetcher::BuildReportMeta(TJsonReport& report) const {
    TMap<TString, NJson::TJsonValue> reports = GetMetaReports();
    for (auto&& i : reports) {
        report.AddReportElement(i.first, std::move(i.second));
    }
}

NJson::TJsonValue TCarsFetcher::BuildReportMeta() const {
    TMap<TString, NJson::TJsonValue> reports = GetMetaReports();
    NJson::TJsonValue result = NJson::JSON_MAP;
    for (auto&& i : reports) {
        result.InsertValue(i.first, std::move(i.second));
    }
    return result;
}

TString TCarsFetcher::GetAvailableCarsReportSafe(const TSet<TFilterAction>& filters) {
    TString result;
    TStringOutput so(result);
    NJson::TJsonWriter jWriter(&so, false);
    jWriter.OpenArray();
    for (auto&& i : CarsView) {
        R_ENSURE(i < Cars.size(), HTTP_INTERNAL_SERVER_ERROR, i << " < " << Cars.size());
        jWriter.OpenMap();
        R_ENSURE(
            BuildOneCarInfoReport(Cars[i], filters, jWriter),
            HTTP_INTERNAL_SERVER_ERROR,
            "cannot build report for " << i << ' ' << Cars[i].Info.GetId()
        );
        jWriter.CloseMap();
    }
    jWriter.CloseArray();
    return result;
}

NJson::TJsonValue TCarsFetcher::GetCarIdsByAreaTags(const NDrive::TLocationTags& areaTags) const {
    if (areaTags.empty()) {
        return NJson::TJsonMap();
    }
    NJson::TJsonValue result;
    for (const auto& car : Cars) {
        const auto& snapshot = car.Snapshot;
        if (!snapshot) {
            continue;
        }
        for (const auto& tag : snapshot->GetLocationTagsGuarantee()) {
            if (areaTags.contains(tag)) {
                result[tag].AppendValue(car.Info.GetId());
            }
        }
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetAvailableCarsDebugReport(size_t limit) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (size_t i = 0; i < std::min(CarsView.size(), limit); ++i) {
        auto index = CarsView[i];
        if (index >= Cars.size()) {
            continue;
        }
        const auto& car = Cars[index];
        const auto coordinate = car.Location.GetCoord();
        result.AppendValue(NJson::ToJson(std::tie(
            car.Info.GetNumber(),
            coordinate
        )));
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetClustersReportSafe() const {
    return Clusters.GetReport();
}

NJson::TJsonValue TCarsFetcher::GetFiltersReportSafe(const TSet<TFilterAction::TSequentialId>* explicitFilters) const {
    const TSet<TFilterAction>& filters = Permissions->GetFilterActions();
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    TSet<TString> filterIds;
    for (auto&& filter : filters) {
        if (filter.GetExplicit() && !explicitFilters) {
            continue;
        }
        if (explicitFilters && !explicitFilters->contains(filter.GetSequentialId())) {
            continue;
        }
        filterIds.insert(filter.GetName());
    }
    for (auto&& filter : filters) {
        if (!filterIds.contains(filter.GetName())) {
            continue;
        }
        result.AppendValue(filter.GetReport(&filterIds));
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::TClusters::GetReport() const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (const auto& cluster : ClustersSortedByAreaId) {
        if (!cluster.IsActive()) {
            continue;
        }

        NJson::TJsonValue element;
        element["id"] = cluster.Id;
        element["center"] = cluster.Center.ToString();
        element["count"] = cluster.Count;
        NJson::TJsonValue& location = element.InsertValue("location", NJson::JSON_MAP);
        {
            location["lat"] = cluster.Center.Y;
            location["lon"] = cluster.Center.X;
        }
        for (auto&&[filter, count] : cluster.FilterCount) {
            NJson::TJsonValue f;
            f["id"] = filter;
            f["count"] = count;
            element["filters"].AppendValue(std::move(f));
        }
        if (cluster.Icon) {
            element["icon"] = cluster.Icon;
        }
        if (cluster.Title) {
            element["title"] = cluster.Title;
        }
        if (cluster.Message) {
            element["message"] = cluster.Message;
        }
        result.AppendValue(std::move(element));
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetStatusesReportSafe() const {
    if (!DataFetched) {
        return NJson::JSON_NULL;
    }
    NJson::TJsonValue result = NJson::JSON_ARRAY;

    TMap<TString, ui32> carsCount;
    for (auto&& i : FullCarStatuses) {
        carsCount.emplace(i.GetStateId(), 0);
    }

    for (auto&& i : Cars) {
        ++carsCount[i.GetStatus()];
    }

    for (auto&& i : carsCount) {
        NJson::TJsonValue& statusInfo = result.AppendValue(NJson::JSON_MAP);
        statusInfo.InsertValue("id", i.first);
        statusInfo.InsertValue("cars_count", i.second);
        statusInfo.InsertValue("filter_id", i.first);
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetSurgesReportSafe() const {
    if (!DataFetched) {
        return NJson::JSON_NULL;
    }
    NJson::TJsonValue result = NJson::JSON_MAP;

    const TInstant now = ModelingNow();

    {
        const TVector<TString> idleIntervalsStr = StringSplitter(
            GetSetting<TString>("client.admin.idle_segmentation").GetOrElse("1h, 2h, 3h, 6h, 9h, 1d, 2d, 3d")
        ).SplitBySet(", ").SkipEmpty().ToList<TString>();
        TVector<TDuration> idleIntervals;
        for (auto&& i : idleIntervalsStr) {
            TDuration d;
            if (!TryFromString(i, d)) {
                continue;
            }
            idleIntervals.emplace_back(d);
        }

        if (idleIntervals.size()) {
            TVector<ui32> carsCount;
            carsCount.resize(idleIntervals.size() + 1, 0);
            for (auto&& i : RTFactors.GetCarInfos()) {
                if (!i.second.HasIdleStart()) {
                    continue;
                }
                const TInstant idleStart = i.second.GetIdleStartUnsafe();
                bool found = false;
                for (ui32 d = 0; d < idleIntervals.size(); ++d) {
                    if (now - idleStart < idleIntervals[d]) {
                        ++carsCount[d];
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    ++carsCount.back();
                }
            }

            NJson::TJsonValue& idlesStatReport = result.InsertValue("idles", NJson::JSON_ARRAY);

            for (ui32 d = 0; d < idleIntervals.size(); ++d) {
                NJson::TJsonValue& statInfo = idlesStatReport.AppendValue(NJson::JSON_MAP);
                statInfo.InsertValue("name", "<" + Server.GetLocalization()->FormatDuration(Locale, idleIntervals[d]));
                statInfo.InsertValue("filter_id", idleIntervals[d].Seconds());
                statInfo.InsertValue("cars_count", carsCount[d]);
            }
            NJson::TJsonValue& statInfo = idlesStatReport.AppendValue(NJson::JSON_MAP);
            statInfo.InsertValue("name", ">=" + Server.GetLocalization()->FormatDuration(Locale, idleIntervals.back()));
            statInfo.InsertValue("filter_id", TDuration::Max().Seconds());
            statInfo.InsertValue("cars_count", carsCount.back());
        }

    }

    return result;
}

NJson::TJsonValue TCarsFetcher::GetViewsReportSafe() const {
    return Views.GetViewsReport(Locale, ModelTraits);
}

NJson::TJsonValue TCarsFetcher::GetModelsReportSafe(const bool asMap /*= false*/) const {
    if (!DataFetched) {
        return NJson::JSON_NULL;
    }
    NJson::TJsonValue result;

    TMap<TString, ui32> carsCountByModel;

    for (auto&& i : ModelsInfo.GetResult()) {
        carsCountByModel.emplace(i.first, 0);
    }

    for (auto&& i : Cars) {
        carsCountByModel[i.Info.GetModel()]++;
    }

    result = asMap ? NJson::TJsonValue(NJson::JSON_MAP) : NJson::TJsonValue(NJson::JSON_ARRAY);
    for (auto&& i : ModelsInfo.GetResult()) {
        auto report = i.second.GetReport(Locale, ModelTraits);
        report.InsertValue("cars_count", carsCountByModel[i.first]);
        report.InsertValue("specifications", i.second.GetSpecifications().GetLegacyReport(Locale));
        if (asMap) {
            result.InsertValue(i.first, std::move(report));
        } else {
            result.AppendValue(std::move(report));
        }

    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetModelSpecificationReport() const {
    NJson::TJsonValue result = NJson::JSON_NULL;
    for (auto&& [name, values] : ModelSpecificationStats) {
        TDriveModelSpecification specification;

        NJson::TJsonValue specificationValues = NJson::JSON_ARRAY;
        for (auto&& [value, count] : values) {
            specification.SetNameAndValue(name, value);
            NJson::TJsonValue specificationValue;
            specificationValue.InsertValue("value", value);
            specificationValue.InsertValue("value_localized", specification.GetLocalizedValue(Locale));
            specificationValue.InsertValue("count", count);
            specificationValues.AppendValue(std::move(specificationValue));
        }

        NJson::TJsonValue specificationReport;
        specificationReport.InsertValue("name", name);
        specificationReport.InsertValue("name_localized", specification.GetLocalizedName(Locale));
        specificationReport.InsertValue("values", specificationValues);
        result.AppendValue(std::move(specificationReport));
    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetOffersReportSafe(const bool asMap /*= false*/) const {
    if (!DataFetched) {
        return NJson::JSON_NULL;
    }
    NJson::TJsonValue result;

    TMap<TString, ui32> carsCountBySession;
    TMap<TString, IOffer::TPtr> offerByName;
    for (auto&& i : Cars) {
        if (!!i.Session && i.Session->GetCurrentOffer()) {
            carsCountBySession[i.Session->GetCurrentOffer()->GetFullName()]++;
            offerByName.emplace(i.Session->GetCurrentOffer()->GetFullName(), i.Session->GetCurrentOffer());
        }
    }

    result = asMap ? NJson::TJsonValue(NJson::JSON_MAP) : NJson::TJsonValue(NJson::JSON_ARRAY);
    for (auto&& i : carsCountBySession) {
        NJson::TJsonValue report;
        auto it = offerByName.find(i.first);
        if (it != offerByName.end() && !!it->second->GetGroupName()) {
            report.InsertValue("group_name", it->second->GetGroupName());
        }
        report.InsertValue("name", it->second->GetName());
        report.InsertValue("id", it->second->GetPriceConstructorId());
        report.InsertValue("cars_count", i.second);

        if (asMap) {
            result.InsertValue(i.first, std::move(report));
        } else {
            result.AppendValue(std::move(report));
        }

    }
    return result;
}

NJson::TJsonValue TCarsFetcher::GetPatchesReportSafe() const {
    return Patches.GetReport(Locale);
}

NJson::TJsonValue TCarsFetcher::GetShortCarPropertiesReportSafe() const {
    return Properties.GetReport(Locale);
}

class TTagSortData {
private:
    R_FIELD(ui32, CarsCount, 0);
    R_FIELD(ui32, CarsCountPrior, 0);
    R_FIELD(TTagDescription::TConstPtr, Description);
    R_FIELD(TString, TagName);
public:
    TTagSortData(const TString& tagName)
        : TagName(tagName)
    {
    }

    const TString& GetDisplayName() const {
        return Description ? Description->GetDisplayName() : TagName;
    }

    NJson::TJsonValue GetReport(ELocalization locale) const {
        NJson::TJsonValue result;
        result.InsertValue("name", GetTagName());
        if (!!GetDescription()) {
            result.InsertValue("description", GetDescription()->BuildJsonReport(locale));
        }
        result.InsertValue("cars_count_prior", GetCarsCountPrior());
        result.InsertValue("cars_count_zero", GetCarsCount() - GetCarsCountPrior());
        result.InsertValue("cars_count", GetCarsCount());
        return result;
    }
};

NJson::TJsonValue TCarsFetcher::GetTagsReportSafe(const bool asMap, const ETagsStatSortMethod sortMethod) const {
    if (!DataFetched) {
        return NJson::JSON_NULL;
    }
    NJson::TJsonValue result = asMap ? NJson::JSON_MAP : NJson::JSON_ARRAY;

    TMap<TString, TTagSortData> tagSortInfo;

    for (auto&& i : Cars) {
        if (i.ObservableTags) {
            for (auto&& tag : *i.ObservableTags) {
                auto it = tagSortInfo.find((*tag)->GetName());
                if (it == tagSortInfo.end()) {
                    it = tagSortInfo.emplace((*tag)->GetName(), TTagSortData((*tag)->GetName())).first;
                }
                auto& tagInfo = it->second;
                ++tagInfo.MutableCarsCount();
                if ((*tag)->GetTagPriority(0)) {
                    ++tagInfo.MutableCarsCountPrior();
                }
            }
        }
    }
    if (ReportUserObservableTags) {
        for (auto&& i : UserObservableTags) {
            tagSortInfo.emplace(i, TTagSortData(i));
        }
    }
    TVector<TTagSortData> tagSortInfoVector;
    for (auto&& i : tagSortInfo) {
        i.second.SetDescription(DriveApi.GetTagsManager().GetTagsMeta().GetDescriptionByName(i.first));
        tagSortInfoVector.emplace_back(std::move(i.second));
    }

    const auto pred = [sortMethod](const TTagSortData& left, const TTagSortData& right)-> bool {
        if (sortMethod == ETagsStatSortMethod::Lexical) {
            return left.GetDisplayName() < right.GetDisplayName();
        } else if (sortMethod == ETagsStatSortMethod::CarsCount) {
            if ((left.GetCarsCount() > 0) == (right.GetCarsCount() > 0)) {
                return left.GetDisplayName() < right.GetDisplayName();
            } else if (left.GetCarsCount()) {
                return true;
            } else {
                return false;
            }
        }
        return true;
    };
    std::sort(tagSortInfoVector.begin(), tagSortInfoVector.end(), pred);

    for (auto&& i : tagSortInfoVector) {
        if (asMap) {
            result.InsertValue(i.GetTagName(), i.GetReport(Locale));
        } else {
            result.AppendValue(i.GetReport(Locale));
        }
    }
    return result;
}

TSet<TString> TCarsFetcher::GetCarsIds() const {
    TSet<TString>  result;
    for (auto&& i : CarsView) {
        R_ENSURE(i < Cars.size(), HTTP_INTERNAL_SERVER_ERROR, i << " < " << Cars.size());
        auto carId = TDriveCarInfo::GetObjectId(Cars[i].Info);
        result.insert(carId);
    }
    return result;
}

TMaybe<TCarsFetcher::TOrderedCarInfo> TCarsFetcher::GetOrderedCarInfo(const TString& carId) const {
    auto* carInfo = CarInfosOriginal.GetResultPtr(carId);
    if (!carInfo) {
        return {};
    }
    auto itObserve = ObservableTagsData.find(carId);

    auto itFull = DevicesSnapshot.find(carId);

    auto pred = [&carId](const std::pair<TString, TRTDeviceSnapshot>& snapshot) -> bool {
        return snapshot.first == carId;
    };
    auto itSnapshot = FindIf(Snapshots.GetSelection().begin(), Snapshots.GetSelection().end(), pred);

    auto itRTFactorsInfo = RTFactors.find(carId);
    auto itRecalculatedStatus = RecalculatedStatuses.find(carId);
    auto itStatus = CarStatuses.find(carId);
    auto sessions = CarCurrentSessions.equal_range(carId);
    TVector<const TDBTag*>* tagsObserve = (itObserve == ObservableTagsData.end()) ? nullptr : &itObserve->second;
    const TTaggedObject* taggedObject = (itFull == DevicesSnapshot.end()) ? nullptr : itFull->second.Get();
    const TRTDeviceSnapshot* snapshot = (itSnapshot == Snapshots.GetSelection().end()) ? nullptr : &itSnapshot->second;
    const TCarRTFactors* surgeInfo = (itRTFactorsInfo == RTFactors.end()) ? nullptr : &itRTFactorsInfo->second;
    const TString* recalculatedStatus = (itRecalculatedStatus == RecalculatedStatuses.end()) ? nullptr : &itRecalculatedStatus->second;
    const TString* cachedStatus = (itStatus == CarStatuses.end()) ? nullptr : &itStatus->second;
    const TString* statusInfo = recalculatedStatus ? recalculatedStatus : cachedStatus;
    const bool isAvailable = AvailableCars.contains(carId);
    auto itModelIdx = ModelIndexes.find(carInfo->GetModel());
    EFuturesStage futuresStage = EFuturesStage::NoFutures;
    auto it = Futures.begin();
    if (Advance(it, Futures.end(), carId)) {
        futuresStage = it->second;
    }

    const TBillingSession* billingSession = nullptr;
    for (auto&& [objectId, session] : MakeIteratorRange(sessions)) {
        billingSession = dynamic_cast<const TBillingSession*>(session.Get());
    }

    const TPointInfo* startAddress = nullptr;
    if (billingSession) {
        startAddress = Starts.FindPtr(billingSession->GetSessionId());
    }

    const TDriveUserData* sessionUser = nullptr;
    if (billingSession) {
        sessionUser = SessionUsers.GetResultPtr(billingSession->GetUserId());
    }

    TCarGenericAttachment carDocumentAttachment;
    if (ReportTraits & NDeviceReport::ReportShortProps) {
        DriveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(carInfo->GetId(), EDocumentAttachmentType::CarRegistryDocument, carDocumentAttachment);
    }

    TCarGenericAttachment carSignalDevice;
    if (ReportTraits & NDeviceReport::ReportSignalq) {
        DriveApi.GetCarAttachmentAssignments().TryGetAttachmentOfType(carInfo->GetId(), EDocumentAttachmentType::CarSignalDevice, carSignalDevice);
    }

    NThreading::TFuture<NDrive::TGeocoder::TResponse> geocodedLocationFuture;
    NDrive::TLocation location;
    if (IsRealtime && ReportTraits & EReportTraits::ReportGeocodedLocation && ReportTraits & EReportTraits::ReportLeasingStats && snapshot && snapshot->GetLocation(location)) {
        const auto language = enum_cast<ELanguage>(Locale);
        const auto geocoder = DriveApi.HasGeocoderClient() ? &DriveApi.GetGeocoderClient() : nullptr;
        if (geocoder) {
            geocodedLocationFuture = geocoder->Decode(location.GetCoord(), language);
        }
    }

    TOrderedCarInfo carView(*carInfo, tagsObserve, taggedObject, snapshot, surgeInfo, statusInfo,
                            billingSession, nullptr, isAvailable, futuresStage, carDocumentAttachment.GetImpl(),
                            carSignalDevice.GetImpl(), startAddress, sessionUser, std::move(geocodedLocationFuture));
    carView.ModelInfo = itModelIdx == ModelIndexes.end() ? nullptr : ModelsInfoVector[itModelIdx->second];
    return carView;
}

bool TCarsFetcher::BuildOneCarInfoReport(const TString& carId, const TSet<TFilterAction>& filters, NJson::TJsonWriter& result, const TSet<TString>& supportAllowedPhotos) const {
    auto carInfo = GetOrderedCarInfo(carId);
    if (!carInfo) {
        return false;
    }
    return BuildOneCarInfoReport(*carInfo, filters, result, supportAllowedPhotos);
}

TVector<ui32> TCarsFetcher::CalcFilters(const TVector<TDBTag>& tags, const TSet<TFilterAction>* filters) const {
    TVector<ui32> result;
    if (!filters) {
        filters = &Filters;
    }
    result.reserve(filters->size());

    auto prepared = TTagsFilter::PrepareMatching(tags);
    for (auto&& filter : *filters) {
        if (filter.GetTagsFilter().IsMatching(prepared)) {
            result.push_back(filter.GetSequentialId());
        }
    }
    return result;
}

TVector<size_t> TCarsFetcher::CreateView(const TVector<TOrderedCarInfo>& cars, ESortMethod sortMethod) const {
    TVector<size_t> view;
    view.resize(cars.size(), 0);
    ui32 idx = 0;
    for (auto&& i : cars) {
        view[idx] = idx;
        ++idx;
        i.ModelInfo = ModelsInfoVector[i.ModelIdx];
        if (sortMethod == ESortMethod::Proximity && HasUserLocation()) {
            NDrive::TLocation location;
            if (i.Snapshot && i.Snapshot->GetLocation(location)) {
                i.Distance = GetUserLocationRef().GetLengthTo(location.GetCoord());
            }
        }
    }
    switch (sortMethod) {
    case ESortMethod::None:
        break;
    case ESortMethod::Proximity:
    {
        if (!HasUserLocation()) {
            break;
        }
        std::sort(view.begin(), view.end(), [&cars](size_t l, size_t r) {
            const TOrderedCarInfo& left = cars[l];
            const TOrderedCarInfo& right = cars[r];
            return std::tie(left.Distance, left.Info.GetId()) < std::tie(right.Distance, right.Info.GetId());
        });
        if (EnableRelevance) {
            TVector<ui64> modelCounts(ModelsInfoVector.size(), 0);
            for (auto&& car : cars) {
                modelCounts[car.ModelIdx] += 1;
            }
            TVector<double> modelWeights;
            modelWeights.reserve(modelCounts.size());
            for (auto&& count : modelCounts) {
                modelWeights.push_back(1.0 / count);
            }
            ui64 count = 1;
            for (auto&& car : cars) {
                auto cappedDistance = std::min(car.Distance, 42 * 1000.0);
                auto cappedDistanceSquared = cappedDistance * cappedDistance;
                auto countSquared = count * count;
                car.Relevance = car.Distance - RelevanceModelWeight * modelWeights[car.ModelIdx] - RelevanceModelWeight * cappedDistanceSquared / countSquared;
                count += 1;
            }
        }
        auto begin = std::lower_bound(view.begin(), view.end(), RelevanceDistanceThreshold, [&cars](size_t i, double distance) {
            const TOrderedCarInfo& info = cars[i];
            return info.Distance < distance;
        });
        std::sort(begin, view.end(), [&cars](size_t l, size_t r) {
            const TOrderedCarInfo& left = cars[l];
            const TOrderedCarInfo& right = cars[r];
            return std::tie(left.Relevance, left.Distance, left.Info.GetId()) < std::tie(right.Relevance, right.Distance, right.Info.GetId());
        });
        break;
    }
    case ESortMethod::Rank:
        std::sort(view.begin(), view.end(), [&cars](size_t l, size_t r) {
            const TOrderedCarInfo& left = cars[l];
            const TOrderedCarInfo& right = cars[r];
            const i64 leftRank = left.ModelInfo ? -1 * left.ModelInfo->GetRank() : 0;
            const i64 rightRank = right.ModelInfo ? -1 * right.ModelInfo->GetRank() : 0;
            return std::tie(leftRank, left.Info.GetNumber()) < std::tie(rightRank, right.Info.GetNumber());
        });
        break;
    case ESortMethod::LeasingStats:
    {
        bool recognizedField = false;

        TVector<double> aggrValue;
        aggrValue.resize(view.size());

        for (const auto& vi : view) {
            const TOrderedCarInfo& info = cars[vi];
            if (info.TaggedObject) {
                for (const auto& dbTag : info.TaggedObject->GetTags()) {
                    const auto* leasingStatsTag = dbTag.GetTagAs<NDrivematics::TLeasingStatsTag>();
                    if (leasingStatsTag) {
                        auto maybeAggregatedStats = leasingStatsTag->AggregateStats(GetSetting<ui32>("leasing_cabinet.aggregation.days_count").GetOrElse(60));
                        if (!maybeAggregatedStats) {
                            continue;
                        }
                        const auto& aggregatedStats = *maybeAggregatedStats;
                        ForEach(aggregatedStats.GetFields(), [&aggrValue, &vi, &recognizedField, &fieldName = GetLSSortParams().GetLSOrderField()] (auto&& field) {
                            if ((field.GetTraits() & NDrivematics::ReportInCarList) && field.GetName() == fieldName) {
                                aggrValue[vi] = static_cast<double>(field.Value);
                                recognizedField = true;
                                return;
                            }
                        });
                        R_ENSURE(recognizedField, HTTP_INTERNAL_SERVER_ERROR, "unrecognized sort_field: " << GetLSSortParams().GetLSOrderField());
                    }
                }
            }
        }


        std::sort(view.begin(), view.end(), [&cars, &aggrValue, orderDesc = GetLSSortParams().GetLSOrderDesc()](size_t l, size_t r) {
            const TOrderedCarInfo& left = cars[l];
            const TOrderedCarInfo& right = cars[r];
            if (orderDesc) {
                return std::tie(aggrValue[l], left.Info.GetNumber()) > std::tie(aggrValue[r], right.Info.GetNumber());
            } else {
                return std::tie(aggrValue[l], left.Info.GetNumber()) < std::tie(aggrValue[r], right.Info.GetNumber());
            }
        });
        break;
    }
    }
    return view;
}

namespace {
    const TString DefaultStatus = "unknown";
}

NJson::TJsonValue TCarsFetcher::TProperties::GetReport(ELocalization locale) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& i : Descriptions) {
        result.AppendValue(i->BuildJsonReport(locale));
    }
    return result;
}

void TCarsFetcher::TOrderedCarInfo::FillShortProps(NJson::TJsonWriter& carReport, TCarsFetcher::TProperties& props, const TReportTraits reportTraits, const i32 intervalsCount) const {
    if (!ObservableTags) {
        return;
    }
    TSet<ui32> features;
    for (auto&& i : *ObservableTags) {
        const TDeviceAdditionalFeature* dafTag = dynamic_cast<const TDeviceAdditionalFeature*>(i->GetData().Get());
        if (dafTag) {
            ui32 idx;
            const auto checker = [](TTagDescription::TConstPtr description) -> bool {
                const TDeviceAdditionalFeature::TDescription* d = dynamic_cast<const TDeviceAdditionalFeature::TDescription*>(description.Get());
                return !!d && d->GetVisibility();
            };
            if (props.Register((*i)->GetName(), idx, checker)) {
                features.emplace(idx);
            }
        }
    }
    if (RTFactors && RTFactors->HasHiddenDiscount() && (reportTraits & NDeviceReport::ReportHiddenDiscounts) && RTFactors->GetHiddenDiscountUnsafe() > 0) {
        const double hidden = Min<double>(1, Max<double>(0, RTFactors->GetHiddenDiscountUnsafe()));
        TJsonProcessor::WriteDef<double>(carReport, "hd_normal", hidden, 0);
        ui32 idx;
        const auto checker = [](TTagDescription::TConstPtr description) -> bool {
            const TDeviceAdditionalFeature::TDescription* d = dynamic_cast<const TDeviceAdditionalFeature::TDescription*>(description.Get());
            return !!d && d->GetVisibility();
        };
        if (props.Register("tag_hidden_discount_" + ToString(GetDiscountInterval(hidden, intervalsCount)), idx, checker)) {
            carReport.Write("hd_style", idx);
            features.emplace(idx);
        } else if (props.Register("tag_hidden_discount_default", idx, checker)) {
            carReport.Write("hd_style", idx);
            features.emplace(idx);
        }
    }
    if (features.size()) {
        carReport.Write("sf", NJson::ToJson(features));
    }
}

void TCarsFetcher::TOrderedCarInfo::FillPatches(NJson::TJsonWriter& carReport, TCarsFetcher::TProperties& patches) const {
    if (!ObservableTags) {
        return;
    }
    bool arrPatches = false;
    for (auto&& i : *ObservableTags) {
        const TPropertiesPatchTag* cTag = dynamic_cast<const TPropertiesPatchTag*>(i->GetData().Get());
        if (cTag) {
            ui32 idx;
            if (patches.Register((*i)->GetName(), idx)) {
                if (!arrPatches) {
                    arrPatches = true;
                    carReport.OpenArray("patches");
                }
                carReport.Write(idx);
            }
        }
    }
    if (arrPatches) {
        carReport.CloseArray();
    }
}

void TCarsFetcher::TOrderedCarInfo::FillView(NJson::TJsonWriter& carReport, TCarsFetcher::TViews& views) const {
    TSet<ui32> indexes;
    if (ObservableTags) {
        for (auto&& i : *ObservableTags) {
            auto propertiesPatchTag = i->GetTagAs<TPropertiesPatchTag>();
            if (propertiesPatchTag) {
                indexes.emplace((*i)->GetDescriptionIndex());
            }
        }
    }
    ui64 idx;
    if (views.Register(ModelInfo, Info, indexes, idx)) {
        carReport.Write("view", idx);
    }
}

void TCarsFetcher::TOrderedCarInfo::FillLocationFutures(NJson::TJsonWriter& locationFutures, const TReportTraits reportTraits, const ILocalization& localization) const {
    if (FuturesStage == EFuturesStage::NoFutures) {
        locationFutures.WriteNull("futures_location");
        return;
    }
    const TFixPointOffer* fixPoint = Session ? dynamic_cast<const TFixPointOffer*>(Session->GetCurrentOffer().Get()) : nullptr;
    locationFutures.OpenMap("futures_location");
    // Write location.
    if (fixPoint) {
        locationFutures.Write("lat", fixPoint->GetFinish().Y);
        locationFutures.Write("lon", fixPoint->GetFinish().X);
        if (reportTraits & NDeviceReport::ReportFuturesArea) {
            const auto& area = fixPoint->GetFinishAreaPublic().size() ? fixPoint->GetFinishAreaPublic() : fixPoint->GetFinishArea();
            locationFutures.Write("area", TGeoCoord::SerializeVector(area));
        }
    } else if (RTFactors && RTFactors->HasFinishCoord()) {
        locationFutures.Write("lat", RTFactors->GetFinishCoordUnsafe().Y);
        locationFutures.Write("lon", RTFactors->GetFinishCoordUnsafe().X);
    } else if (reportTraits & NDeviceReport::ReportRTFuturesPosition) {
        NDrive::TLocation location;
        if (Snapshot && Snapshot->GetLocation(location)) {
            locationFutures.Write("lat", location.Latitude);
            locationFutures.Write("lon", location.Longitude);
        }
    }
    // Write actual location.
    if (reportTraits & NDeviceReport::ReportRTFuturesPosition) {
        NDrive::TLocation location;
        if (Snapshot && Snapshot->GetLocation(location)) {
            locationFutures.Write("lat_actual", location.Latitude);
            locationFutures.Write("lon_actual", location.Longitude);
        }
    }
    // Write duration.
    {
        auto duration = GetFinishDurationDef(0);
        locationFutures.Write("duration", (int)duration);
        locationFutures.Write("duration_hr", localization.FormatDuration(Locale, TDuration::Seconds(duration)));
    }
    locationFutures.CloseMap();
}

const TString& TCarsFetcher::TOrderedCarInfo::GetStatus() const {
    if (Status) {
        return *Status;
    } else {
        return DefaultStatus;
    }
}

bool TCarsFetcher::TClusters::Initialize(const TDriveAPI& driveApi, const ISettings& settings) {
    ClustersSortedByAreaId.clear();
    const bool castClusterIdToUi16 = settings.GetValue<bool>("cars_fetcher.cast_cluster_id_to_ui16").GetOrElse(true);
    const auto& areasDb = driveApi.GetAreasDB();
    Y_ENSURE(areasDb->ForObjectsMap([this, castClusterIdToUi16](const auto& id, const auto& area) {
        if (!area.IsCluster()) {
            return;
        }
        auto& cluster = ClustersSortedByAreaId.emplace_back();
        cluster.AreaId = id;
        if (castClusterIdToUi16) {
            cluster.Id = static_cast<ui16>(FnvHash<ui32>(id));
        } else {
            cluster.Id = FnvHash<i32>(id);
        }
        cluster.Center = TGeoCoord::CalcCenter(area.GetCoords());
        if (auto tooltip = area.GetTooltip()) {
            cluster.Icon = tooltip->GetIcon();
            cluster.Title = tooltip->GetTitle();
            cluster.Message = tooltip->GetMessage();
        }
    }));
    const auto& tagsManager = driveApi.GetAreasDB()->GetTagsManager();
    for (auto& cluster : ClustersSortedByAreaId) {
        const auto& expectedTag = tagsManager.GetTagFromCache(cluster.AreaId, TClusterTag::TypeName, TInstant::Zero());
        auto clusterTag = expectedTag ? expectedTag->GetTagAs<TClusterTag>() : nullptr;
        if (clusterTag) {
            cluster.Threshold = clusterTag->GetThreshold();
        }
    }
    std::sort(ClustersSortedByAreaId.begin(), ClustersSortedByAreaId.end(), [](const auto& a, const auto&b) {
        return a.AreaId < b.AreaId;
    });
    return true;
}

TVector<TCarsFetcher::TCluster>::iterator TCarsFetcher::TClusters::FindAny(const TSet<TString>& areaIds) {
    auto begin = ClustersSortedByAreaId.begin();
    const auto& end = ClustersSortedByAreaId.end();
    if ((begin == end) || areaIds.empty()) {
        return end;
    }
    if (*areaIds.crbegin() < begin->AreaId) {
        // all areaIds < all clusters
        return end;
    }
    for (const auto& areaId : areaIds) {
        const auto& it = std::lower_bound(begin, end, areaId, [](const auto& cluster, const auto& areaId){
            return cluster.AreaId < areaId;
        });
        if (it == end) {
            break;
        }
        if (it->AreaId == areaId) {
            return it;
        }
        begin = it;
    }
    return end;
}

bool TCarsFetcher::TClusters::Register(const TSet<TString>& areaIds, TCarsFetcher::TCluster*& cluster) {
    cluster = nullptr;
    if (ClustersSortedByAreaId.empty()) {
        return true;
    }
    const auto& it = FindAny(areaIds);
    if (it != ClustersSortedByAreaId.end()) {
        cluster = &*it;
        ++cluster->Count;
    }
    if (SelectedClusterId) {
        if (!cluster) {
            return false;
        }
        if (*SelectedClusterId != cluster->Id) {
            return false;
        }
    }
    return true;
}

bool TCarsFetcher::TClusters::Check(const TSet<TString>& areaIds) {
    const auto& it = FindAny(areaIds);
    return it != ClustersSortedByAreaId.end();
}

NJson::TJsonValue TCarsFetcher::TModelView::GetViewReport(ELocalization locale, TModelTraits modelTraits) const {
    NJson::TJsonValue result = ModelData.GetReport(locale, modelTraits);
    for (auto&& i : Patches) {
        const TPropertiesPatchTag::TDescription* cTagDescr = dynamic_cast<const TPropertiesPatchTag::TDescription*>(i.Get());
        if (cTagDescr) {
            cTagDescr->ApplyForJson(result);
        }
    }
    return result;
}
