#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/attachment_context.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/users/user.h>
#include <drive/library/cpp/geocoder/api/client.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/rect.h>
#include <rtline/util/types/accessor.h>

class TJsonReport;

class TCarsFetcher {
public:
    enum class ETagsStatSortMethod {
        Lexical /* "lexical" */,
        CarsCount /* "cars_count" */
    };
    enum class ESortMethod {
        None         /* "none" */,
        Rank         /* "rank" */,
        Proximity    /* "proximity" */,
        LeasingStats /* "leasing_stats" */,
    };
    enum class EFuturesStage {
        NoFutures,
        Futurable,
        Futures
    };
    enum ESignalqStatus {
        TurnedOn     /* "turned_on" */,
        TurnedOff    /* "turned_off" */,
        FacedAway    /* "faced_away" */,
        CameraClosed /* "camera_closed" */,
        Offline      /* "offline" */
    };

    // LS - LeasingStats
    struct TLSSortParams {
    public:
        R_FIELD(TString, LSOrderField);
        R_FIELD(bool, LSOrderDesc, true);
    };

    struct TModelSpecificationFilter {
        TString Name;
        TSet<TString> Values;

        DECLARE_FIELDS(
            Field(Name, "name"),
            Field(Values, "values")
        );
    };
    using TModelSpecificationFilters = TVector<TModelSpecificationFilter>;

private:
    static TSet<NDrive::TSensorId> DefaultSensorIds;

private:
    using TClusterId = i32;
    using TCityGroupsData = TMap<TString, TString>;
    using TTagsData = TMap<TString, TVector<const TDBTag*>>;
    using TTagsCreationTimes = TMap<TString, TInstant>;
    using TModelSpecificationStats = TMap<TString, TMap<TString, size_t>>;
    using TModelTraits = NDriveModelReport::TReportTraits;

private:
    TSet<TString> UserObservableTags;

private:
    R_READONLY(TTagsData, ObservableTagsData, {}, mutable);
    R_READONLY(TCityGroupsData, CityGroups, {}, mutable);
    R_READONLY(TSet<TString>, SelectedByUser);
    R_OPTIONAL(TGeoCoord, UserLocation);
    R_OPTIONAL(TRect<TGeoCoord>, Rect);
    R_FIELD(ESortMethod, SortMethod, ESortMethod::None);
    R_FIELD(TLSSortParams, LSSortParams);
    R_FIELD(ETagsStatSortMethod, TagsSortMethod, ETagsStatSortMethod::CarsCount);
    R_FIELD(TClusterId, ClusterId, 0);
    R_FIELD(TSet<NDrive::TSensorId>, SensorIds, DefaultSensorIds);
    R_FIELD(TSet<TString>, SelectedModels, {});
    R_FIELD(TString, UserId);
    R_FIELD(double, RelevanceDistanceThreshold, 2500);
    R_FIELD(double, RelevanceCountWeight, 1.0 / 100);
    R_FIELD(double, RelevanceModelWeight, 100000);
    R_FIELD(ui32, Limit, Max<ui32>());
    R_FIELD(ui32, PreLimit, Max<ui32>());
    R_FIELD(bool, ReportCreationTime, false);
    R_FIELD(bool, CheckVisibility, true);
    R_FIELD(bool, EnableRelevance, false);
    R_FIELD(bool, IsParkingCorrectFuturesState, false);
    R_FIELD(bool, IsRealtime, false);
    R_FIELD(bool, NoSelected, false);
    R_FIELD(bool, RecalculateStatuses, false);
    R_FIELD(bool, ReportUserObservableTags, true);
    R_FIELD(bool, SelectedOnly, false);
    R_FIELD(bool, WithClusters, true);
    R_FIELD(bool, UseCache, true);
    R_FIELD(ui32, PageNumber, 0);
    R_FIELD(ui32, PageSize, 0);
    R_READONLY(bool, CanGetMorePages, false);
    R_READONLY(TTagsCreationTimes, TagsCreationTimes);

    R_OPTIONAL(TSet<ESignalqStatus>, SignalqStatusFilter);
    R_OPTIONAL(TTagsFilter, AreaTagsFilter);
    R_OPTIONAL(TTagsFilter, CarTagsFilter);
    R_OPTIONAL(TSet<NDrive::TLocation::EType>, LocationTypeFilter);
    R_OPTIONAL(TSet<ui64>, FleetsFilter);
    R_OPTIONAL(TSet<TString>, ChargeAccountsFilter);
    R_OPTIONAL(TSet<TString>, StatusFilter);
    R_OPTIONAL(TModelSpecificationFilters, ModelSpecificationFilters);
    R_FIELD(ELocalization, Locale, DefaultLocale);

private:
    bool IsCorrectFuturesState(const TString& currentState) const {
        return (currentState == "old_state_riding") || (currentState == "old_state_parking" && IsParkingCorrectFuturesState);
    }

private:
    struct TCluster {
        TString AreaId;
        TString Icon;
        TString Title;
        TString Message;
        TGeoCoord Center;
        TClusterId Id = 0;
        mutable TMap<ui32, ui64> FilterCount;
        ui64 Count = 0;
        ui64 Threshold = 0;

        bool IsActive() const {
            return Count >= Threshold;
        }
        void AddFiltersInfo(const TVector<ui32>& filters) const {
            for (auto&& filter : filters) {
                ++FilterCount[filter];
            }
        }
    };

    class TModelView {
    private:
        TDriveModelData ModelData;
        TVector<TTagDescription::TConstPtr> Patches;

    public:
        NJson::TJsonValue GetViewReport(ELocalization locale, TModelTraits modelTraits) const;

        TModelView(TDriveModelData&& modelData, TVector<TTagDescription::TConstPtr>&& patches)
            : ModelData(std::move(modelData))
            , Patches(std::move(patches))
        {
        }
    };

    class TViewIndex {
    private:
        const TString ModelCode;
        const ui64 Idx;
        const size_t SpecificationHash;

    public:
        TViewIndex(const TString& modelCode, ui64 idx, size_t specificationHash)
            : ModelCode(modelCode)
            , Idx(idx)
            , SpecificationHash(specificationHash)
        {
        }

        auto Tuple() const {
            return std::tie(
                Idx,
                SpecificationHash,
                ModelCode
            );
        }

        bool operator<(const TViewIndex& item) const {
            return Tuple() < item.Tuple();
        }
    };

    class TViews {
    private:
        const NDrive::IServer& Server;
        TMap<TViewIndex, ui32> ViewIndexes;
        TVector<TModelView> Views;

    public:
        TViews(const NDrive::IServer& server)
            : Server(server)
        {
        }

        NJson::TJsonValue GetViewsReport(ELocalization locale, TModelTraits modelTraits) const {
            NJson::TJsonValue result = NJson::JSON_ARRAY;
            for (auto&& i : Views) {
                result.AppendValue(i.GetViewReport(locale, modelTraits));
            }
            return result;
        }

        bool Register(const TDriveModelData* modelData, const TDriveCarInfo& carInfo, const TSet<ui32>& indexes, ui64& result) {
            if (!modelData) {
                return false;
            }

            result = 0;
            ui32 l = 1;
            for (auto&& i : indexes) {
                result += (i << ++l);
            }

            auto specificationHash = CombineHashes(
                modelData->GetSpecifications().Hash(),
                carInfo.GetSpecifications().Hash()
            );

            auto viewIndex = TViewIndex{modelData->GetCode(), result, specificationHash};
            auto it = ViewIndexes.find(viewIndex);
            if (it == ViewIndexes.end()) {
                TVector<TTagDescription::TConstPtr> tds;
                for (auto&& i : indexes) {
                    auto tagData = Server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByIndex(i);
                    if (tagData) {
                        tds.emplace_back(tagData);
                    }
                }

                auto model = *modelData;
                if (!model.MutableSpecifications().Merge(carInfo.GetSpecifications())) {
                    return false;
                }

                Views.emplace_back(std::move(model), std::move(tds));
                it = ViewIndexes.emplace(viewIndex, Views.size() - 1).first;
            }
            result = it->second;
            return true;
        }
    };

    class TProperties {
    private:
        const NDrive::IServer& Server;
        TMap<TString, ui32> Indexes;
        TVector<TTagDescription::TConstPtr> Descriptions;

    public:
        TProperties(const NDrive::IServer& server)
            : Server(server)
        {
        }

        bool Register(const TString& name, ui32& idx) {
            const auto checker = [](TTagDescription::TConstPtr /*description*/) {
                return true;
            };
            return Register(name, idx, checker);
        }

        template <class TChecker>
        bool Register(const TString& name, ui32& idx, TChecker& checker) {
            auto it = Indexes.find(name);
            if (it != Indexes.end()) {
                idx = it->second;
                return true;
            }
            auto description = Server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(name);
            if (description && checker(description)) {
                Indexes.emplace(name, description->GetIndex());
                Descriptions.emplace_back(description);
                idx = description->GetIndex();
                return true;
            }
            return false;
        }

        NJson::TJsonValue GetReport(ELocalization locale) const;
    };
    using TPatches = TProperties;

    struct TPointInfo {
        NDrive::TLocation Point;
        NThreading::TFuture<NDrive::TGeocoder::TResponse> Address;
    };

public:
    struct TOrderedCarInfo {
    public:
        const TDriveCarInfo& Info;
        const TVector<const TDBTag*>* ObservableTags = nullptr;
        const TString* CityGroup = nullptr;
        const TTaggedObject* TaggedObject = nullptr;
        const TRTDeviceSnapshot* Snapshot = nullptr;
        const TCarRTFactors* RTFactors = nullptr;
        const TString* Status = nullptr;
        const TBillingSession* Session = nullptr;
        const TPointInfo* StartSessionPoint;
        const TDriveUserData* SessionUser;
        const TString* HeadId = nullptr;
        const TString* HeadDeviceId = nullptr;
        const TCluster* Cluster = nullptr;
        TStringBuf SpecialType;
        const bool IsAvailable = true;
        const EFuturesStage FuturesStage = EFuturesStage::NoFutures;
        const TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> CarRegistryDocument;
        const TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> CarSignalDevice;
        const NThreading::TFuture<NDrive::TGeocoder::TResponse> GeocodedLocationFuture;

        mutable double Distance = Max<ui32>();
        mutable ui32 ModelIdx = Max<ui32>();
        mutable i32 Relevance = 0;
        mutable NDrive::TLocation Location;
        mutable const TDriveModelData* ModelInfo = nullptr;

        ELocalization Locale = DefaultLocale;

    public:
        i32 GetDiscountInterval(const double hiddenDiscount, const i32 intervalsCount) const {
            return Max<i32>(0, Min<i32>(hiddenDiscount * intervalsCount, intervalsCount - 1));
        }

    public:
        TOrderedCarInfo(
                const TDriveCarInfo& info,
                TVector<const TDBTag*>* observableTags,
                const TTaggedObject* taggedObject,
                const TRTDeviceSnapshot* snapshot,
                const TCarRTFactors* rtFactors,
                const TString* status,
                const TBillingSession* session,
                const TCluster* cluster,
                const bool isAvailable,
                const EFuturesStage futuresStage,
                const TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> carRegistryDocument,
                const TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> carSignalDevice,
                const TPointInfo* startSessionPoint,
                const TDriveUserData* sessionUser,
                NThreading::TFuture<NDrive::TGeocoder::TResponse>&& geocodedLocationFuture)
            : Info(info)
            , ObservableTags(observableTags)
            , TaggedObject(taggedObject)
            , Snapshot(snapshot)
            , RTFactors(rtFactors)
            , Status(status)
            , Session(session)
            , StartSessionPoint(startSessionPoint)
            , SessionUser(sessionUser)
            , Cluster(cluster)
            , IsAvailable(isAvailable)
            , FuturesStage(futuresStage)
            , CarRegistryDocument(carRegistryDocument)
            , CarSignalDevice(carSignalDevice)
            , GeocodedLocationFuture(std::move(geocodedLocationFuture))
        {
        }

        void FillShortProps(NJson::TJsonWriter& carReport, TCarsFetcher::TProperties& props, const TReportTraits reportTraits, const i32 intervalsCount) const;
        void FillPatches(NJson::TJsonWriter& carReport, TCarsFetcher::TProperties& patches) const;
        void FillView(NJson::TJsonWriter& carReport, TCarsFetcher::TViews& views) const;
        void FillLocationFutures(NJson::TJsonWriter& locationFutures, const TReportTraits reportTraits, const ILocalization& localization) const;

        int GetFinishDurationDef(const int defValue) const {
            return RTFactors ? RTFactors->GetFinishDurationDef(defValue) : defValue;
        }

        bool InUsageBy(const TString& userId) const {
            return Session && Session->GetUserId() == userId;
        }

        const TString& GetStatus() const;
    };

private:
    class TClusters {
    private:
        TVector<TCluster> ClustersSortedByAreaId;
        R_OPTIONAL(i32, SelectedClusterId);

        TVector<TCluster>::iterator FindAny(const TSet<TString>& areaIds);
    public:
        TClusters() = default;

        NJson::TJsonValue GetReport() const;
        bool Initialize(const TDriveAPI& driveApi, const ISettings& settings);

        bool Register(const TSet<TString>& areaIds, TCarsFetcher::TCluster*& cluster);
        bool Check(const TSet<TString>& areaIds);
    };

private:
    const NDrive::IServer& Server;
    const TDriveAPI& DriveApi;
    TUserPermissions::TPtr Permissions;
    TDeviceTagsManager::TCurrentSnapshot DevicesSnapshot;
    TConstDevicesSnapshot SnapshotsFull;

    TCarFactorsSnapshot RTFactors;
    TSet<TString> AvailableCars;
    TVector<std::pair<TString, EFuturesStage>> Futures;
    TCarsDB::TFetchResult CarInfosOriginal;
    TUsersDB::TFetchResult TagPerformers;
    TVector<TOrderedCarInfo> Cars;
    TVector<size_t> CarsView;
    TVector<TStringBuf> OverLimitedCars;
    TMap<TString, TCarsDB::TFormerNumbers> FormerNumbers;
    TMap<TString, ui32> ModelIndexes;
    TMap<TString, TString> HeadIds;
    TMap<TString, TString> HeadDeviceIds;
    TMap<TString, TString> ImageHostBySourceMapping;
    TModelsDB::TFetchResult ModelsInfo;
    TVector<const TDriveModelData*> ModelsInfoVector;
    TDevicesSnapshotManager::TDevicesSnapshotSelection Snapshots;
    TInstant FetchFinishInstant;
    bool DataFetched = false;
    TReportTraits ReportTraits = NDeviceReport::ReportAll;
    TRTDeviceSnapshot::TLocationOptions LocationOptions;
    TVector<TStateFilter> FullCarStatuses;
    TStateFiltersDB::TObjectStates CarStatuses;
    TMap<TString, TString> RecalculatedStatuses;
    std::multimap<TString, IEventsSession<TCarTagHistoryEvent>::TPtr> CarCurrentSessions;
    TMap<TString, TPointInfo> Starts;
    TClusters Clusters;
    mutable TViews Views;
    mutable TPatches Patches;
    mutable TProperties Properties;
    TSet<TFilterAction> Filters;
    mutable TMaybe<i32> DiscountIntervalsCount;
    TInstant Deadline = TInstant::Max();
    TUsersDB::TFetchResult SessionUsers;
    std::multimap<TString, TCarGenericAttachment> CarDocumentAttachments;
    std::multimap<TString, TCarGenericAttachment> CarSignalDevices;
    TModelSpecificationStats ModelSpecificationStats;
    TModelTraits ModelTraits = NDriveModelReport::UserReport;

private:
    TVector<size_t> CreateView(const TVector<TOrderedCarInfo>& cars, ESortMethod sortMethod) const;
    void SerializeTelematic(NJson::TJsonWriter& telematicsReport, const TOrderedCarInfo& carInfo, const TRTDeviceSnapshot* snapshot) const;
    void SerializeLagInfo(NJson::TJsonWriter& lagReport, const NDrive::TLocation& location, const TRTDeviceSnapshot* snapshot) const;
    void SerializeLocationInfo(NJson::TJsonWriter& locationReport, const NDrive::TLocation& location, const TRTDeviceSnapshot* snapshot) const;
    void SerializeEnclosingAreas(NJson::TJsonWriter& areasReport, const NDrive::TLocation& preciseLocation) const;
    void SerializeLeasingStats(NJson::TJsonWriter& leasingStatsReport, const TOrderedCarInfo& carInfo, ui32 daysCount) const;
    void SerializeTaxiCompany(NJson::TJsonWriter& taxiCompanyReport, const TString& carId) const;

    i32 GetDiscountIntervalsCount() const;
    template <class T>
    TMaybe<T> GetSetting(const TString& key) const;

public:
    bool GetNeedFueling(const TString& objectId) const;
    bool GetNeedTankerFueling(const TString& objectId) const;
    bool HasFutures() const {
        return Futures.size();
    }
    bool Empty() const {
        return Cars.empty() && ModelsInfo.empty();
    }
    TClusters& GetClusters() {
        return Clusters;
    }
    const TDeviceLocationOptions& GetLocationOptions() const {
        return LocationOptions;
    }
    const TDevicesSnapshotManager::TDevicesSnapshotSelection& GetSnapshots() const {
        return Snapshots;
    }

    TMap<TString, NJson::TJsonValue> GetMetaReports() const;
    void BuildReportMeta(TJsonReport& report) const;
    NJson::TJsonValue BuildReportMeta() const;

    NJson::TJsonValue BuildTagJson(const TDBTag& tag, const TString& userId) const;

    TCarsFetcher(const NDrive::IServer& server, const TReportTraits traits, TInstant deadline = TInstant::Max(), const TModelTraits modelTraits = NDriveModelReport::UserReport)
        : Server(server)
        , DriveApi(*Server.GetDriveAPI())
        , ReportTraits(traits)
        , Views(Server)
        , Patches(Server)
        , Properties(Server)
        , Deadline(deadline)
        , ModelTraits(modelTraits)
    {
    }

    const TDriveCarInfo* GetCarInfo(const TString& objectId) const {
        return CarInfosOriginal.GetResultPtr(objectId);
    }
    const TDriveModelData* GetModelInfo(const TString& modelId) const {
        return ModelsInfo.GetResultPtr(modelId);
    }

    TMaybe<TOrderedCarInfo> GetOrderedCarInfo(const TString& carId) const;

    [[nodiscard]] ESignalqStatus MakeSignalqStatus(const TRTDeviceSnapshot* snapshot) const;

    [[nodiscard]] bool FetchData(TUserPermissions::TPtr permissions, const TSet<TString>& fetchedCarIds);
    [[nodiscard]] bool FetchData(TUserPermissions::TPtr permissions, const TSet<TString>* fetchedCarIds);

    TString GetAvailableCarsReportSafe(const TSet<TFilterAction>& filters);
    NJson::TJsonValue GetCarIdsByAreaTags(const NDrive::TLocationTags& areaTags) const;
    NJson::TJsonValue GetAvailableCarsDebugReport(size_t limit = std::numeric_limits<size_t>::max()) const;
    NJson::TJsonValue GetClustersReportSafe() const;
    NJson::TJsonValue GetFiltersReportSafe(const TSet<TFilterAction::TSequentialId>* explicitFilters = nullptr) const;
    NJson::TJsonValue GetModelsReportSafe(const bool asMap = false) const;
    NJson::TJsonValue GetModelSpecificationReport() const;
    NJson::TJsonValue GetViewsReportSafe() const;
    NJson::TJsonValue GetStatusesReportSafe() const;
    NJson::TJsonValue GetSurgesReportSafe() const;
    NJson::TJsonValue GetOffersReportSafe(const bool asMap = false) const;
    NJson::TJsonValue GetPatchesReportSafe() const;
    NJson::TJsonValue GetShortCarPropertiesReportSafe() const;
    NJson::TJsonValue GetTagsReportSafe(const bool asMap, const ETagsStatSortMethod sortMethod) const;
    TSet<TString> GetCarsIds() const;

    [[nodiscard]] bool BuildOneCarInfoReport(const TString& carId, const TSet<TFilterAction>& filters, NJson::TJsonValue& result) const;
    [[nodiscard]] bool BuildOneCarInfoReport(const TOrderedCarInfo& carInfo, const TSet<TFilterAction>& filters, NJson::TJsonValue& result) const;
    [[nodiscard]] bool BuildOneCarInfoReport(const TString& carId, const TSet<TFilterAction>& filters, NJson::TJsonWriter& carReport, const TSet<TString>& supportAllowedPhotos = {}) const;
    [[nodiscard]] bool BuildOneCarInfoReport(const TOrderedCarInfo& carInfo, const TSet<TFilterAction>& filters, NJson::TJsonWriter& carReport, const TSet<TString>& supportAllowedPhotos = {}) const;

    TVector<ui32> CalcFilters(const TVector<TDBTag>& tags, const TSet<TFilterAction>* filters = nullptr) const;
};
