#include "processor.h"

#include <drive/backend/auth/blackbox2/auth.h>
#include <drive/backend/auth/fake/fake.h>
#include <drive/backend/auth/jwt/auth.h>
#include <drive/backend/auth/public_key/public_key.h>
#include <drive/backend/auth/tvm/tvm.h>
#include <drive/backend/auth/yang/yang.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/mobile_payment.h>
#include <drive/backend/billing/accounts/yandex_account.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/offers/offers/base.h>

#include <drive/library/cpp/network/data/data.h>
#include <drive/library/cpp/raw_text/phone_number.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/threading/future_cast.h>
#include <drive/library/cpp/user_events_api/client.h>

#include <rtline/util/algorithm/container.h>

#include <util/string/split.h>

namespace {
const TString YaTaxiUserIdHeader = "X-YaTaxi-UserId";

TUsersDB::TOptionalUser GetUserByPhone(const TUsersDB& usersData,
                                       IServerReportBuilder::TEventsGuard& eg,
                                       const TString& normalized,
                                       const TString& phone,
                                       const TString& uid,
                                       NDrive::TEntitySession& tx) {
    auto optionalUser = usersData.GetUserByPhone(normalized, tx);
    eg.AddEvent(NJson::TMapBuilder
        ("event", "GetUserByPhoneResult")
        ("normalized", normalized)
        ("phone", phone)
        ("user_passport_uid", optionalUser ? optionalUser->GetUid() : "")
        ("blackbox_passport_uid", uid)
        ("result", optionalUser ? optionalUser->GetPublicReport() : NJson::JSON_NULL)
    );
    return optionalUser;
}
}

NDrive::TEntitySession TBaseProcessor::BuildChatSession(bool readOnly, bool repeatableRead) const {
    auto timeout = Context->GetRequestDeadline() - Now();
    auto lockTimeout = TDuration::Zero();
    auto statementTimeout = timeout;
    auto result = Yensured(Server->GetChatEngine())->BuildSession(readOnly, repeatableRead, lockTimeout, statementTimeout);
    FillInfoEntitySession(result, Context->GetCgiParameters());
    return result;
}

void TBaseProcessor::FillInfoEntitySession(NDrive::TInfoEntitySession& result, const TCgiParameters& cgi) const {
    result.MutableRequestContext().SetUserChoice(cgi.Get("user_choice"));
    if (PayloadPatch.Defined() && GetPayloadPatchUnsafe().Has("user_choice") && GetPayloadPatchUnsafe()["user_choice"].IsString()) {
        result.MutableRequestContext().SetUserChoice(GetPayloadPatchUnsafe()["user_choice"].GetString());
    }
    if (Comment.Defined()) {
        result.SetComment(*Comment);
    }
    result.SetAlertOnDirty(true);
    result.SetLocale(GetLocale());
    result.SetOriginatorId(OriginatorId);
}

TAtomicSharedPtr<const ISession> TBaseProcessor::GetCurrentUserSession(TUserPermissions::TPtr permissions, const TInstant reqActuality) const {
    const auto& cgi = Context->GetCgiParameters();
    const auto& sessionId = GetString(cgi, "session_id", false);
    TAtomicSharedPtr<const ISession> result;
    ReqCheckCondition(DriveApi->GetUserSession(permissions->GetUserId(), result, sessionId, reqActuality), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    if (result && result->GetUserId() != permissions->GetUserId()) {
        CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
    }
    return result;
}

bool TBaseProcessor::CheckAdmActions(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TAdministrativeAction::EEntity entity, const TString& instance, const TMaybe<TSet<TString>>& instanceTags) const {
    if (!permissions) {
        return false;
    }
    return permissions->CheckAdministrativeActions(action, entity, TypeName, instance, instanceTags);
}

void TBaseProcessor::ReqCheckAdmActions(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TAdministrativeAction::EEntity entity, const TString& instance, const TMaybe<TSet<TString>>& instanceTags) const {
    if (!CheckAdmActions(permissions, action, entity, instance, instanceTags)) {
        TCodedException exception(ConfigHttpStatus.PermissionDeniedStatus);
        const TString errorId = "no_permissions." + ::ToString(entity) + "::" + ::ToString(action);
        exception.SetLocalizedMessage(GetHandlerLocalization(errorId, errorId, GetLocale()));
        throw exception << errorId;
    }
}

void TBaseProcessor::ReqCheckCondition(bool checkValue, ui32 code, const TString& errorId, const TString& additionalMessage) const {
    if (!checkValue) {
        auto locale = GetLocale();
        TString localizedMessage = GetHandlerLocalization(errorId, errorId, locale);
        TString localizedTitle = GetHandlerLocalization(errorId + ".title", TString{}, locale);
        if (additionalMessage) {
            localizedMessage += ". " + additionalMessage;
            localizedTitle += ". " + additionalMessage;
        }

        TCodedException exception(code);
        exception.SetErrorCode(errorId);
        exception.SetLocalizedMessage(std::move(localizedMessage));
        exception.SetLocalizedTitle(std::move(localizedTitle));
        ythrow exception << errorId;
    }
}

void TBaseProcessor::ReqCheckCondition(bool checkValue, ui32 code, EDriveLocalizationCodes errorId) const {
    ReqCheckCondition(checkValue, code, ToString(errorId));
}

bool TBaseProcessor::CheckUserGeneralAccessRights(const TString& userId, TUserPermissions::TPtr permissions, const NAccessVerification::TAccessVerificationTraits& traits, bool doThrow, bool forceShowObjectsWithoutTags) const {
    auto ExitOnError = [this, doThrow](const TString& errorCode, const TString& explanation = {}) -> bool {
        if (!doThrow) {
            return false;
        }
        TCodedException exception(ConfigHttpStatus.PermissionDeniedStatus);
        if (explanation) {
            exception.AddPublicInfo("explanation", explanation);
        }
        exception.SetLocalizedMessage(GetHandlerLocalization(errorCode, errorCode, GetLocale()));
        throw exception << errorCode;
    };

    TTaggedObject taggedUser;
    if (traits & (NAccessVerification::EAccessVerificationTraits::ChatWriteAccess | NAccessVerification::EAccessVerificationTraits::TagsAccess)) {
        taggedUser = DriveApi->GetTagsManager().GetUserTags().GetCachedObject(userId).ExtractValue();
    }

    // Check observe access
    if ((traits & NAccessVerification::EAccessVerificationTraits::ObserveAccess) && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId)) {
        return ExitOnError("no_permissions.user.observe");
    }

    // Check modify access
    if ((traits & NAccessVerification::EAccessVerificationTraits::ModifyAccess) && !permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::User, userId)) {
        return ExitOnError("no_permissions.user.modify");
    }

    // Check chat observe rights
    if ((traits & NAccessVerification::EAccessVerificationTraits::ChatObserveAccess)) {
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatMessage, userId)) {
            return ExitOnError("no_permissions.chat_message.observe");
        }
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatResource, userId)) {
            return ExitOnError("no_permissions.chat_resource.observe");
        }
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::ChatsList, userId)) {
            return ExitOnError("no_permissions.chats_list.observe");
        }
    }

    // Check chat message send
    if (traits & NAccessVerification::EAccessVerificationTraits::ChatWriteAccess) {
        bool canSend = permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Send, TAdministrativeAction::EEntity::ChatMessage, userId);
        bool canForceSend = permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::SendForce, TAdministrativeAction::EEntity::ChatMessage, userId);
        if (!canSend && !canForceSend) {
            return ExitOnError("no_permissions.chat_message.send");
        }
        if (!canForceSend) {
            auto chatId = GetString(Context->GetCgiParameters(), "chat_id", false);
            bool isOK = false;
            for (auto&& tag : taggedUser.GetTags()) {
                auto chatTagPtr = tag.GetTagAs<TSupportChatTag>();
                if (!chatTagPtr) {
                    continue;
                }
                if (chatTagPtr->GetPerformer() == permissions->GetUserId() && chatTagPtr->GetTopicLink() == chatId) {
                    isOK = true;
                    break;
                }
            }
            if (!isOK) {
                return ExitOnError("no_permissions.chat_message.send");
            }
        }
    }

    // Check if access is banned by tags
    if (traits & NAccessVerification::EAccessVerificationTraits::TagsAccess) {
        TUserPermissions::TUnvisibilityInfoSet info = 0;
        bool isVisible = permissions->GetVisibility(taggedUser, NEntityTagsManager::EEntityType::User, &info, forceShowObjectsWithoutTags) != TUserPermissions::EVisibility::NoVisible;
        if (!isVisible) {
            return ExitOnError("no_permissions.user.by_tags_forbidden", TUserPermissions::ExplainInvisibility(info));
        }
    }

    return true;
}

void TBaseProcessor::CheckOrganizationsAccess(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TSet<ui64> parentIds) const {
    auto eventLogger = NDrive::GetThreadEventLogger();
    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "CheckParentId")
            ("parent_id", NJson::ToJson(parentIds))
            ("action", ToString(action))
        );
    }
    auto instances = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::Wallet, TypeName);
    auto organizations = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::B2BOrganization, TypeName);
    if (!instances.Defined() || !organizations.Defined()) {
        return;
    }

    TSet<ui32> organizationIds;
    Transform(organizations->begin(), organizations->end(), std::inserter(organizationIds, organizationIds.begin()), [](auto& organization) {
        ui32 id = 0;
        TryFromString(organization, id);
        return id;
    });

    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "GetAccountsByInstances")
        );
    }

    for (auto&& instance : instances.GetRef()) {
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsByName(instance);
        R_ENSURE(accounts, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch accounts");
        TSet<ui64> accountIds;
        Transform(accounts->begin(), accounts->end(), std::inserter(accountIds, accountIds.begin()), [](auto& account) { return account->GetId(); });
        TSet<ui64> difference;
        SetDifference(parentIds.begin(), parentIds.end(), accountIds.begin(), accountIds.end(), std::inserter(difference, difference.begin()));
        if (difference.empty()) {
            return;
        }
        parentIds = std::move(difference);
    }

    TSet<ui64> difference;
    SetDifference(parentIds.begin(), parentIds.end(), organizationIds.begin(), organizationIds.end(), std::inserter(difference, difference.begin()));
    R_ENSURE(
        difference.empty(),
        ConfigHttpStatus.PermissionDeniedStatus,
        "no permissions to " << action << " account " << JoinSeq(",", difference),
        EDriveSessionResult::IncorrectRequest
    );
}

TExpected<bool, TSet<TString>> TBaseProcessor::CheckWalletsAccess(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TSet<TString> accountNames) const {
    auto eventLogger = NDrive::GetThreadEventLogger();
    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "CheckWalletsAccess")
            ("wallets", NJson::ToJson(accountNames))
            ("action", ToString(action))
        );
    }
    auto instances = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::Wallet, TypeName);
    auto organizations = permissions->GetAdministrativeInstances(action, TAdministrativeAction::EEntity::B2BOrganization, TypeName);
    if (!instances.Defined() || !organizations.Defined()) {
        return true;
    }

    {
        TSet<TString> difference;
        SetDifference(accountNames.begin(), accountNames.end(), instances->begin(), instances->end(), std::inserter(difference, difference.begin()));
        if (difference.empty()) {
            return true;
        }
        accountNames = std::move(difference);
    }

    if (!DriveApi->HasBillingManager()) {
        return MakeUnexpected<TSet<TString>>(std::move(accountNames));
    }

    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "GetAccountsChildren")
        );
    }

    NDrive::NBilling::TAccountParentFilter filter;
    for (const auto& organization : *organizations) {
        ui32 id = 0;
        if (TryFromString(organization, id)) {
            filter.ParentIds.emplace(std::move(id));
        } else {
            return {};
        }
    }
    filter.WithParent = (action == TAdministrativeAction::EAction::Observe || action == TAdministrativeAction::EAction::ObserveStructure || action == TAdministrativeAction::EAction::Add);
    auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetAccountsChildren(filter);
    if (!accounts) {
        return {};
    }
    auto availableNames = MakeSet(NContainer::Keys(*accounts));

    TSet<TString> difference;
    SetDifference(accountNames.begin(), accountNames.end(), availableNames.begin(), availableNames.end(), std::inserter(difference, difference.begin()));
    if (difference.empty()) {
        return true;
    }
    return MakeUnexpected<TSet<TString>>(std::move(difference));
}

TString TBaseProcessor::GetApplicationId() const {
    const auto& rd = Context->GetRequestData();
    const auto& headers = rd.HeadersIn();
    auto userAgentHeader = headers.find("User-Agent");
    if (userAgentHeader == headers.end()) {
        return {};
    }
    auto userAgent = TStringBuf(userAgentHeader->second);
    auto applicationId = userAgent.Before('/');
    return TString{applicationId};
}

TString TBaseProcessor::GetExternalUserId() const {
    const auto& rd = Context->GetRequestData();
    const auto& headers = rd.HeadersIn();
    const auto externalUserIdHeader = GetHandlerSetting<TString>("external_user_id_header").GetOrElse(YaTaxiUserIdHeader);
    auto p = headers.find(externalUserIdHeader);
    if (p != headers.end()) {
        return p->second;
    } else {
        return {};
    }
}

TString TBaseProcessor::GetFallbackUserId() const {
    return GetHandlerSettingDef<TString>("new_user.fallback_id", "");
}

TString TBaseProcessor::GetNewUserPolicy() const {
    return GetHandlerSettingDef<TString>("new_user.policy", "create");
}

TString TBaseProcessor::GetOrigin() const {
    const auto& cgi = Context->GetCgiParameters();
    const auto& rd = Context->GetRequestData();
    const auto& headers = rd.HeadersIn();
    TString result;
    if (!result) {
        result = GetString(cgi, "origin", false);
    }
    if (!result) {
        result = GetHandlerSetting<TString>("default_origin").GetOrElse(result);
    }
    if (!result && headers.contains(YaTaxiUserIdHeader)) {
        result = "taxi";
    }
    return result;
}

bool TBaseProcessor::CheckAdvertisingHeaders() const {
    return GetHandlerSettingDef<bool>("user.check_advertising_headers", false);
}

bool TBaseProcessor::CreateYandexAccount(TUserPermissions::TPtr permissions) const {
    return GetHandlerSettingDef<bool>("user.create_yandex_account", false) && permissions->GetUserFeatures().GetIsPlusUser();
}

bool TBaseProcessor::CreateMobilePaymentAccount(TUserPermissions::TPtr permissions) const {
    return GetHandlerSettingDef<bool>("user.create_mobile_payment_account", false) && permissions->GetUserFeatures().HasMobilePay();
}

bool TBaseProcessor::AddCustomActionsToPermissions() const {
    return GetHandlerSettingDef<bool>("user.add_custom_permissions", true);
}

bool TBaseProcessor::CheckAccessCount(TVector<TDBTag>& dbTags, const TString& eventTagName, const TString& eventName, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
    {
        auto tags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), { eventTagName }, session);
        if (!tags) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        dbTags = *tags;
    }
    if (!dbTags.empty()) {
        auto allowedCount = GetHandlerSetting<ui32>("requests_per_time");
        auto timeForCount = GetHandlerSettingDef<ui32>("requests_since", 86400);
        auto allowedDuration = GetHandlerSetting<ui32>("seconds_between_requests");
        for (auto&& tag : dbTags) {
            auto eventTag = tag.GetTagAs<IEventTag>();
            if (eventTag) {
                auto lastEvent = eventTag->GetLastEvent(eventName);
                if (lastEvent && allowedDuration && lastEvent->GetTimestamp() > (Context->GetRequestStartTime() - TDuration::Seconds(*allowedDuration))) {
                    return false;
                }
                if (allowedCount) {
                    auto eventCount = eventTag->GetEventCount({ eventName }, Context->GetRequestStartTime() - TDuration::Seconds(timeForCount));
                    if (eventCount >= *allowedCount) {
                        return false;
                    }
                }
            } else {
                session.SetErrorInfo("CheckAccessCount", "tag is not event tag " + eventTagName);
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
    return true;
}

bool TBaseProcessor::UpdateAccessCount(TVector<TDBTag>& dbTags, const TString& eventTagName, const TString& eventName, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const {
    if (dbTags.empty()) {
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(eventTagName);
        if (!tag) {
            session.SetErrorInfo("UpdateAccessCount", "cannot create tag " + eventTagName);
            return false;
        }
        auto eventTag = std::dynamic_pointer_cast<IEventTag>(tag);
        if (!eventTag) {
            session.SetErrorInfo("UpdateAccessCount", "tag is not event tag " + eventTagName);
            return false;
        }
        eventTag->AddEvent(eventName, Context->GetRequestStartTime());
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, permissions->GetUserId(), permissions->GetUserId(), Server, session)) {
            return false;
        }
    } else {
        for (auto&& tag : dbTags) {
            auto eventTag = tag.MutableTagAs<IEventTag>();
            if (!eventTag) {
                session.SetErrorInfo("UpdateAccessCount", "tag is not event tag " + eventTagName);
                return false;
            }
            eventTag->AddEvent(eventName, Context->GetRequestStartTime());
        }
    }
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagsData(dbTags, permissions->GetUserId(), session)) {
        return false;
    }
    return true;
}

namespace {
    TSet<TString> BlobContentTypes = {
        "audio",
        "video",
        "image"
    };
    TSet<TString> IncorrectContentTypes = {
        "multipart"
    };
}

NJson::TJsonValue TBaseProcessor::GetRequestData() const {
    auto contentType = TString{Context->GetRequestData().HeaderInOrEmpty("Content-Type")};
    contentType = Strip(ToLowerUTF8(contentType.substr(0, contentType.find("/"))));

    R_ENSURE(!IncorrectContentTypes.contains(contentType), ConfigHttpStatus.SyntaxErrorStatus, "incorrect content type: " << contentType);
    if (BlobContentTypes.contains(contentType)) {
        return NJson::JSON_NULL;
    }

    const auto& post = Context->GetBuf();
    if (post.Empty()) {
        return NJson::JSON_NULL;
    }

    TMemoryInput input(post.data(), post.size());
    try {
        return NJson::ReadJsonTree(&input, /*throwOnError=*/true);
    } catch (const std::exception& e) {
        throw TCodedException(HTTP_BAD_REQUEST) << "cannot parse Json: " << e.what();
    }
}

TString TBaseProcessor::GetUserByPhone(
    TJsonReport::TGuard& g,
    const TUsersDB& usersData,
    const TString& phone,
    const TString& uid,
    EPhoneMatchPolicy matchByPhone,
    NDrive::TEntitySession& tx
) {
    TString userId;
    auto eg = g.BuildEventGuard("GetUserByPhone");

    TPhoneNormalizer phoneNormalizer;
    auto normalized = phoneNormalizer.TryNormalize(phone);
    switch (matchByPhone) {
        case EPhoneMatchPolicy::Force:
            {
                auto optionalUser = ::GetUserByPhone(usersData, eg, normalized, phone, uid, tx);
                R_ENSURE(optionalUser, {}, "cannot GetUserByPhone " << normalized, tx);
                userId = optionalUser->GetUserId();
            }
            break;
        case EPhoneMatchPolicy::Try:
            {
                auto optionalUser = ::GetUserByPhone(usersData, eg, normalized, phone, uid, tx);
                if (optionalUser && (!optionalUser->GetUid() || optionalUser->GetUid() == uid)) {
                    userId = optionalUser->GetUserId();
                }
            }
            break;
        case EPhoneMatchPolicy::Multiuser:
            {
                auto xUserId = Context->GetRequestData().HeaderInOrEmpty("X-UserId");
                if (!xUserId) {
                    xUserId = Context->GetCgiParameters().Get("x-userid");
                }
                R_ENSURE(xUserId, ConfigHttpStatus.UnknownErrorStatus, "X-UserId not found", NDrive::MakeError("auth.find_by_phone.x_user_id_not_found"), tx);

                auto optionalUsers = usersData.GetUsersByPhone(normalized, tx);
                R_ENSURE(optionalUsers, ConfigHttpStatus.UnknownErrorStatus, "cannot get users", NDrive::MakeError("internal_error"), tx);

                for (const auto& user: *optionalUsers) {
                    if (NDrive::UserStatusDeleted != user.GetStatus()) {
                        if (user.GetUserId() == xUserId) {
                            userId = xUserId;
                            break;
                        }
                    }
                }
            }
            break;
        case EPhoneMatchPolicy::Disabled:
            break;
    }
    return userId;
}

void TBaseProcessor::DoAuthProcess(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    const TBaseServerRequestData& rd = Context->GetRequestData();
    TString delegationUserId(rd.HeaderInOrEmpty("UserIdDelegation"));
    TString deviceId(rd.HeaderInOrEmpty("DeviceID"));

    DeviceId = deviceId;
    ClientIp = NUtil::GetClientIp(rd);

    const auto& settings = Server->GetSettings();
    const auto& usersData = DriveApi->GetUserManager();

    TMaybe<TGeoCoord> userLocation = GetUserLocation();
    TUserPermissionsFeatures upFeatures;
    upFeatures.SetClientIP(ClientIp);
    upFeatures.SetUserLocation(userLocation);

    {
        const auto& headers = rd.HeadersIn();
        auto mobilePaymentSupportIt = headers.find("MobilePaymentSupport");
        if (mobilePaymentSupportIt != headers.end()) {
            upFeatures.MutableMobilePay() = mobilePaymentSupportIt->second;
        }
    }
    bool ignoreDeviceId = false;

    auto getUserStatus = [&](const TString& userId) -> TString {
        auto optionalUserId = usersData.FetchInfo(userId);
        for (auto&& userIt : optionalUserId) {
            return userIt.second.GetStatus();
        }
        return "";
    };

    auto registerNewUser = [&](const auto authInfo) {
        auto createUserEventsGuard = g.BuildEventGuard("CreateUser");
        const auto& uid = authInfo->GetUid();
        R_ENSURE(uid, HTTP_UNAUTHORIZED, "cannot register new user without uid");
        auto lockUid = settings.GetValue<bool>("user_registration.lock_uid").GetOrElse(true);
        auto initialStatus = GetHandlerSettingDef<TString>(
            "user_registration.initial_status",
            settings.GetValue<TString>("user_registration.initial_status").GetOrElse(NDrive::UserStatusOnboarding)
        );
        auto forcePhoneVerified = settings.GetValue<bool>("user_registration.force_phone_verified").GetOrElse(false);
        auto onLinkStatus = GetHandlerSetting<TString>("user_registration.on_link_status");

        TPhoneNormalizer phoneNormalizer;
        const auto verifiedPhone = rd.HeaderIn("X-Ya-Phone-Verified");
        const auto verifiedPhoneDefined = verifiedPhone && !verifiedPhone->empty();
        const auto phone = verifiedPhoneDefined ? phoneNormalizer.TryNormalize(*verifiedPhone) : authInfo->GetPhone();
        const auto phoneVerified = verifiedPhoneDefined || forcePhoneVerified;

        auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        auto locked = lockUid ? session.TryLock(uid) : true;
        R_ENSURE(locked, {}, "cannot TryLock " << uid, session);
        R_ENSURE(*locked, ConfigHttpStatus.ConflictRequest, "cannot take lock to register new user " << uid, session);
        auto optionalUserId = usersData.GetUserIdByUidDirect(uid, session);
        R_ENSURE(optionalUserId, {}, "cannot GetUserIdByUidDirect", session);
        auto userId = std::move(*optionalUserId);

        if (userId) {
            auto optionalUser = usersData.FetchInfo(userId);
            for (auto&& userIt : optionalUser) {
                auto& user = userIt.second;
                if (user.GetStatus() == initialStatus) {
                    return userId;
                }

                user.SetStatus(initialStatus);
                R_ENSURE(
                    usersData.UpdateUser(user, user.GetUserId(), session) &&
                    session.Commit(),
                    ConfigHttpStatus.UnknownErrorStatus,
                    "cannot update status of the user " << user.GetUserId(),
                    session
                );

                return user.GetUserId();
            }
        }

        if (!onLinkStatus) {
            auto eg = g.BuildEventGuard("GetOnLinkStatus");
            auto optionalExternalUser = usersData.TryLinkToExistingUser(phone, authInfo->GetEmail(), session);
            R_ENSURE(optionalExternalUser, {}, "cannot TryLinkToExistingUser", session);
            auto externalUser = std::move(*optionalExternalUser);
            auto externalUserPermissions = externalUser
                ? DriveApi->GetUserPermissions(externalUser.GetUserId())
                : nullptr;
            if (externalUserPermissions) {
                onLinkStatus = externalUserPermissions->template GetSetting<TString>("user_registration.on_link_status");
            }
        }
        auto user = usersData.RegisterNewUser(
            "blackbox",
            uid,
            authInfo->GetUsername(),
            session,
            phone,
            authInfo->GetEmail(),
            phoneVerified,
            initialStatus,
            onLinkStatus
        );
        R_ENSURE(user, ConfigHttpStatus.UnknownErrorStatus, "cannot register new user: " << session.GetStringReport());
        userId = user->GetUserId();
        R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot commit transaction: " << session.GetStringReport());
        return userId;
    };

    auto tx = BuildTx<NSQL::ReadOnly | NSQL::Deferred>();
    auto userId = authInfo->GetUserId();
    const auto& uid = authInfo->GetUid();

    const auto matchByPhone = GetHandlerSetting<EPhoneMatchPolicy>("user.authentication.match_by_phone").GetOrElse(EPhoneMatchPolicy::Disabled);
    const auto& phone = authInfo->GetPhone();

    if ((matchByPhone != EPhoneMatchPolicy::Disabled) && phone) {
        userId = GetUserByPhone(g, usersData, phone, uid, matchByPhone, tx);
    }

    if (auto keyAuth = dynamic_cast<const TPublicKeyAuthInfo*>(authInfo.Get())) {
        if (keyAuth->GetRemapUser()) {
            userId = DriveApi->GetHeadAccountManager().GetDriveUserId(keyAuth->GetUserId(), tx);
            R_ENSURE(userId, HTTP_NO_CONTENT, "no session", tx);
        } else {
            userId = keyAuth->GetUserId();
        }
        ignoreDeviceId = true;
    } else if (auto blackboxAuth = dynamic_cast<const TBlackboxAuthInfo*>(authInfo.Get())) {
        ignoreDeviceId = blackboxAuth->GetIgnoreDeviceId();

        if (!userId) {
            userId = usersData.GetUserIdByUid(uid);
        }
        if (userId) {
            auto patchUser = [](TDriveUserData& user, const TBlackboxAuthInfo* blackboxAuth) {
                bool isChanged = false;
                if (!user.GetUid() && blackboxAuth->GetPassportUid()) {
                    user.SetUid(blackboxAuth->GetPassportUid());
                    if (user.GetStatus() == NDrive::UserStatusExternal) {
                        user.SetStatus(NDrive::UserStatusOnboarding);
                    }
                    isChanged = true;
                }
                if (user.GetPhone().empty() && blackboxAuth->GetDefaultPhone()) {
                    user.SetPhone(blackboxAuth->GetDefaultPhone());
                    isChanged = true;
                }
                if (!blackboxAuth->GetLogin().empty() && blackboxAuth->GetLogin() != user.GetLogin()) {
                    user.SetLogin(blackboxAuth->GetLogin());
                    isChanged = true;
                }
                if (user.GetEmail().empty() && blackboxAuth->GetDefaultEmail()) {
                    user.SetEmail(blackboxAuth->GetDefaultEmail());
                    isChanged = true;
                }
                if (user.GetEnvironment() != blackboxAuth->GetEnvironment()) {
                    user.SetEnvironment(blackboxAuth->GetEnvironment());
                    isChanged = true;
                }
                return isChanged;
            };
            auto optionalUser = usersData.GetCachedObject(userId);
            if (optionalUser) {
                auto user = *optionalUser;
                bool isChanged = patchUser(user, blackboxAuth);
                if (isChanged) {
                    auto eg = g.BuildEventGuard("UpdateUser");
                    auto session = BuildTx<NSQL::Writable>();
                    auto userFetchResult = usersData.FetchInfo(userId, session);
                    R_ENSURE(userFetchResult, {}, "cannot FetchInfo for " << userId, session);

                    auto userPtr = userFetchResult.GetResultPtr(userId);
                    R_ENSURE(userPtr, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch user " << userId, session);
                    user = *userPtr;
                    isChanged = patchUser(user, blackboxAuth);
                    if (isChanged) {
                        R_ENSURE(usersData.UpdateUser(user, userId, session), ConfigHttpStatus.UnknownErrorStatus, "cannot update user");
                        R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot commit transaction: " << session.GetStringReport());
                    }
                }
            }
        }
        upFeatures.SetDefaultEmails(MakeSet(blackboxAuth->GetValidatedMails()));
        upFeatures.SetIsPlusUser(blackboxAuth->GetIsPlusUser());
        upFeatures.SetIsYandexUser(blackboxAuth->GetIsYandexoid());
        upFeatures.SetTVMTicket(blackboxAuth->GetTVMTicket());
        upFeatures.SetUid(uid);
        upFeatures.SetPassportName(blackboxAuth->GetName());
        upFeatures.SetPassportDefaultAvatar(blackboxAuth->GetDefaultAvatar());
    } else if (auto tvmAuth = dynamic_cast<const TTvmAuthInfo*>(authInfo.Get())) {
        if (uid) {
            const auto& blackboxInfo = tvmAuth->GetBlackboxInfo();
            upFeatures.SetIsPlusUser(blackboxInfo.IsPlusUser);
            upFeatures.SetIsYandexUser(blackboxInfo.IsYandexoid);
            upFeatures.SetUid(uid);
            upFeatures.SetPassportName(blackboxInfo.Name);
            upFeatures.SetPassportDefaultAvatar(blackboxInfo.DefaultAvatar);
        } else {
            upFeatures.SetIsFallbackUser(true);
        }
    } else if (auto fakeAuth = dynamic_cast<const TFakeAuthInfo*>(authInfo.Get())) {
        upFeatures.SetDefaultEmails(MakeSet(SplitString(GetHandlerSettingDef<TString>("fake_auth.default_emails", ""), ",")));
    } else if (auto yangAuth = dynamic_cast<const TYangAuthInfo*>(authInfo.Get())) {
        upFeatures.SetYangWorkerId(yangAuth->GetWorkerId());
    }
    if (!userId && uid) {
        userId = usersData.GetUserIdByUid(uid);
    }

    auto newUserPolicy = GetNewUserPolicy();
    TString status;
    if (userId) {
        status = getUserStatus(userId);
    }
    if ((!userId || NDrive::IsPreOnboarding(status)) && newUserPolicy == "create") {
        userId = registerNewUser(authInfo);
    }
    if (!userId && newUserPolicy == "fallback") {
        userId = GetFallbackUserId();
        upFeatures.SetIsFallbackUser(true);
    }

    R_ENSURE(
        userId,
        authInfo->GetCode(HTTP_UNAUTHORIZED),
        "user is not registered: " << authInfo->GetMessage(),
        NDrive::MakeError("user_is_not_registered"),
        tx
    );

    NDrive::TEventLog::TUserIdGuard::Set(userId);
    NDrive::TEventLog::Log(NDrive::TEventLog::Authorization, *Context, NJson::TMapBuilder("auth", authInfo->GetInfo()));

    for (auto&& i : CommonAppConfig.GetMinimalBuildsInfo()) {
        const auto* buildInfo = rd.HeaderIn(i.first.data());
        if (!buildInfo) {
            continue;
        }

        ui32 buildVersion;
        R_ENSURE(TryFromString<ui32>(*buildInfo, buildVersion), ConfigHttpStatus.UserErrorState, "Incorrect " << i.first << " header for " << userId);
        R_ENSURE(i.second <= buildVersion, ConfigHttpStatus.UserErrorState, "Too old client for " << userId, EDriveSessionResult::ApplicationOutdated, tx);
    }

    TUserPermissions::TPtr permissionsOriginal;
    TUserPermissions::TPtr permissions;
    {
        TEventsGuard tg(g.MutableReport(), "restore permissions original");
        TInstant actuality = TInstant::Zero();
        bool useCache = IsTrue(rd.HeaderInOrEmpty("UserPermissionsCache"));
        bool forceFetchPermissions = IsTrue(rd.HeaderInOrEmpty("ForceFetchPermissions"));
        permissionsOriginal = DriveApi->GetUserPermissions(userId, upFeatures, actuality, Context, AddCustomActionsToPermissions(), useCache, forceFetchPermissions);
        R_ENSURE(permissionsOriginal, ConfigHttpStatus.PermissionDeniedStatus, "no permissions for user", EDriveSessionResult::NoUserPermissions);
        SetSettings(permissionsOriginal->SettingGetter());

        const TString newDevicePolicy = GetHandlerSettingDef<TString>("new_device.policy", "ignore");

        if (!ignoreDeviceId && newDevicePolicy != "ignore") {
            TEventsGuard eventsGuard(g.MutableReport(), "CheckDeviceId");
            R_ENSURE(deviceId, ConfigHttpStatus.UserErrorState, "no device id", NDrive::MakeError("no_device_id"), tx);
            if (Server->GetUserDevicesManager()) {
                SetNewDeviceStatus(Server->GetUserDevicesManager()->CheckDeviceId(permissionsOriginal->GetUserId(), deviceId));
                switch (GetNewDeviceStatus()) {
                    case ENewDeviceStatus::NoDevices:
                        R_ENSURE(
                            Server->GetUserDevicesManager()->AddFirstDevice(permissionsOriginal->GetUserId(), deviceId, Context),
                            ConfigHttpStatus.UnknownErrorStatus,
                            "cannot add first device",
                            NDrive::MakeError("cannot_add_first_device"),
                            tx
                        );
                        SetNewDeviceStatus(ENewDeviceStatus::Verified);
                        break;
                    case ENewDeviceStatus::Disabled:
                        R_ENSURE(false, ConfigHttpStatus.UserErrorState, "disabled device id", NDrive::MakeError("disabled_device_id"), tx);
                    case ENewDeviceStatus::New:
                    case ENewDeviceStatus::Verification:
                        R_ENSURE(newDevicePolicy != "fail", ConfigHttpStatus.UserErrorState, "new device id", NDrive::MakeError("unverified_device_id"), tx);
                        break;
                    case ENewDeviceStatus::Verified:
                    case ENewDeviceStatus::CannotDeterm:
                        break;
                }
            }
        }

        permissions = permissionsOriginal;
        if (!!delegationUserId) {
            TEventsGuard tg(g.MutableReport(), "restore permissions delegation");
            OriginatorId = userId;
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::User);
            TUserPermissionsFeatures upf;
            upf.SetUserLocation(userLocation);
            permissions = DriveApi->GetUserPermissions(delegationUserId, upf, actuality, Context, AddCustomActionsToPermissions(), useCache);
            R_ENSURE(permissions, ConfigHttpStatus.PermissionDeniedStatus, "no permissions for delegated user: " + delegationUserId);
        }

        // for test
        if (permissions->GetSetting<bool>(settings, "user.disable_plus", false)) {
            TUserPermissionsFeatures upf = permissions->GetUserFeatures();
            upf.SetIsPlusUser(false);
            permissions = DriveApi->GetUserPermissions(permissions->GetUserId(), upf, actuality, Context, AddCustomActionsToPermissions(), useCache);
            R_ENSURE(permissions, ConfigHttpStatus.PermissionDeniedStatus, "no permissions for non plus user: " + permissions->GetUserId());
        } else if (permissions->GetSetting<bool>(settings, "user.enable_plus", false)) {
            TUserPermissionsFeatures upf = permissions->GetUserFeatures();
            upf.SetIsPlusUser(true);
            permissions = DriveApi->GetUserPermissions(permissions->GetUserId(), upf, actuality, Context, AddCustomActionsToPermissions(), useCache);
            R_ENSURE(permissions, ConfigHttpStatus.PermissionDeniedStatus, "no permissions for plus user: " + permissions->GetUserId());
        }
        SetSettings(permissions->SettingGetter());
        UserId = permissions->GetUserId();
    }

    if (Context->GetCgiParameters().Has("comment")) {
        SetComment(Context->GetCgiParameters().Get("comment"));
    }

    {
        TEventsGuard tg(g.MutableReport(), "Process");
        try {
            Parse(Context->GetCgiParameters());
            auto eventTagName = GetHandlerSetting<TString>("event_tag_name");
            auto eventName = GetHandlerSetting<TString>("event_name");
            const bool countOnFail = GetHandlerSetting<bool>("count_on_fail").GetOrElse(true);
            TVector<TDBTag> dbTags;
            if (eventTagName && eventName) {
                auto session = countOnFail ? BuildTx<NSQL::Writable>() : BuildTx<NSQL::ReadOnly>();
                if (!CheckAccessCount(dbTags, *eventTagName, *eventName, permissions, session)) {
                    auto eventErrorCode = GetHandlerSettingDef<TString>("event_error_code", "unknown_code");
                    session.SetLocalizedMessageKey(eventErrorCode);
                    session.SetError(NDrive::MakeError(eventErrorCode));
                    R_ENSURE(false, HTTP_TOO_MANY_REQUESTS, eventErrorCode, session);
                }
                if (countOnFail && (!UpdateAccessCount(dbTags, *eventTagName, *eventName, permissions, session) || !session.Commit())) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
            Process(g, permissions);
            if (!countOnFail && eventTagName && eventName) {
                auto session = BuildTx<NSQL::Writable>();
                if (!UpdateAccessCount(dbTags, *eventTagName, *eventName, permissions, session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        } catch (TCodedException& e) {
            e.AddInfo("user_id", permissions->GetUserId());
            throw e;
        }
    }

    if (CheckAdvertisingHeaders() && Server->GetUserDevicesManager()) {
        Server->GetUserDevicesManager()->RegisterAdvertisingDevice(userId, deviceId, Context, TInstant::Zero());
    }

    if (GetHandlerSettingDef("validate_email_by_passport", false)) {
        TDBTag settingsTag = DriveApi->GetUserSettings(permissions->GetUserId(), tx);
        R_ENSURE(settingsTag.HasData(), ConfigHttpStatus.UnknownErrorStatus, "incorrect settings configuration");
        TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
        R_ENSURE(settingsData, ConfigHttpStatus.UnknownErrorStatus, "incorrect settings configuration");

        auto emailStatusOld = settingsData->GetField("verified_mail");
        auto emailStatusNew = permissions->GetUserFeatures().GetDefaultEmails().contains(permissions->GetEmail());

        if (!emailStatusOld || emailStatusNew != IsTrue(emailStatusOld.GetRef())) {
            settingsData->SetField("verified_mail", ::ToString(emailStatusNew));
            auto session = BuildTx<NSQL::Writable>();
            if (!!settingsTag.GetTagId()) {
                if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(settingsTag, permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            } else {
                if (!DriveApi->GetTagsManager().GetUserTags().AddTag(settingsTag.GetData(), permissions->GetUserId(), permissions->GetUserId(), Server, session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        }
    }

    if (CreateYandexAccount(permissions)) {
        auto eg = g.BuildEventGuard("CreateYandexAccount");
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetUserAccounts(permissions->GetUserId(), TInstant::Zero());
        TMaybe<NDrive::NBilling::TYandexAccountRecord> internalYandexAccount;
        for(const auto& acc : accounts) {
            if (acc->GetType() == NDrive::NBilling::EAccount::YAccount) {
                auto yRecordPtr = acc->GetRecordAs<NDrive::NBilling::TYandexAccountRecord>();
                if (yRecordPtr) {
                    internalYandexAccount = *yRecordPtr;
                    break;
                }
            }
        }

        if (!internalYandexAccount || !internalYandexAccount->GetPaymethodId()) {
            auto yAccount = DriveApi->GetBillingManager().GetYandexAccount(permissions->GetUserId(), true);
            if (yAccount && yAccount->GetBalance() > 0) {
                auto description = DriveApi->GetBillingManager().GetAccountsManager().GetDescriptionByName(::ToString(NDrive::NBilling::EAccount::YAccount), TInstant::Zero());
                if (description) {
                    auto session = BuildTx<NSQL::Writable>();
                    auto account = DriveApi->GetBillingManager().GetAccountsManager().GetOrCreateAccount(permissions->GetUserId(), *description, permissions->GetUserId(), session);
                    if (account) {
                        auto yRecordPtr = account->GetRecordAs<NDrive::NBilling::TYandexAccountRecord>();
                        if (yRecordPtr && !yRecordPtr->GetPaymethodId() && !account->PatchAccountData(NJson::TMapBuilder("paymethod_id", yAccount->GetId()), permissions->GetUserId(), session)) {
                            session.DoExceptionOnFail(ConfigHttpStatus);
                        }
                        Y_UNUSED(session.Commit());
                    }
                }
            }
        }
    }

    if (CreateMobilePaymentAccount(permissions)) {
        auto eg = g.BuildEventGuard("CreateMobilePaymentAccount");
        auto accounts = DriveApi->GetBillingManager().GetAccountsManager().GetUserAccounts(permissions->GetUserId(), TInstant::Zero());
        TMaybe<NDrive::NBilling::TMobilePaymentAccountRecord> internalMobilePaymentAccount;
        for (const auto& acc : accounts) {
            if (acc->GetType() == NDrive::NBilling::EAccount::MobilePayment) {
                if (auto recordPtr = acc->GetRecordAs<NDrive::NBilling::TMobilePaymentAccountRecord>()) {
                    internalMobilePaymentAccount = *recordPtr;
                    break;
                }
            }
        }
        if (!internalMobilePaymentAccount) {
            auto description = DriveApi->GetBillingManager().GetAccountsManager().GetDescriptionByName(::ToString(NDrive::NBilling::EAccount::MobilePayment), TInstant::Zero());
            if (description) {
                auto session = BuildTx<NSQL::Writable>();
                auto account = DriveApi->GetBillingManager().GetAccountsManager().GetOrCreateAccount(permissions->GetUserId(), *description, permissions->GetUserId(), session);
                if (account) {
                    if (!account->GetRecordAs<NDrive::NBilling::TMobilePaymentAccountRecord>()) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                    Y_UNUSED(session.Commit());
                    NDrive::TEventLog::Log("MobilePaymentAccount: new account created", NJson::TMapBuilder("user_id", permissions->GetUserId()));
                }
            }
        }
    }
}

void TBaseProcessor::Parse(const TCgiParameters& cgi) {
    Y_UNUSED(cgi);
}

void TBaseProcessor::Parse(const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
}

void TCommonServiceAppProcessorBase::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const auto requestData = GetRequestData();
    const auto& payloadPatch = requestData["payload_patch"];
    if (payloadPatch.IsString()) {
        TStringBuf payloadPatchRaw = payloadPatch.GetString();
        NJson::TJsonValue payloadPatchJson;
        if (!NJson::ReadJsonTree(payloadPatchRaw, &payloadPatchJson, true)) {
            ERROR_LOG << "payload_patch is specified in request, but we can't deserializer it: " << payloadPatchRaw << Endl;
        } else {
            SetPayloadPatch(std::move(payloadPatchJson));
        }
    }
    Parse(requestData);
    ProcessServiceRequest(g, permissions, requestData);
}

void TBaseProcessor::ProcessException(TJsonReport::TGuard& g, const TCodedException& exception) const {
    TBase::ProcessException(g, exception);
    auto notifierName = GetHandlerSetting<TString>("process_exception_notifier");
    auto notifier = notifierName ? Server->GetNotifier(*notifierName) : nullptr;

    auto notifierSkippedErrorCodes = notifier ? GetHandlerSetting<TString>("process_exception_skipped_error_codes") : Nothing();
    auto skippedErrorCodes = static_cast<TSet<TString>>(
        StringSplitter(notifierSkippedErrorCodes.GetOrElse(TString{})).SplitBySet(" ,").SkipEmpty()
    );
    if (notifier && !skippedErrorCodes.contains(exception.GetErrorCode())) {
        auto user = Server->GetDriveDatabase().GetUserManager().GetCachedObject(UserId);
        auto report = TStringBuilder()
            << "ErrorCode: " << exception.GetCode() << Endl
            << "ErrorMessage: " << exception.GetLocalizedMessage() << Endl
            << "ErrorTitle: " << exception.GetLocalizedTitle() << Endl
            << "DebugInfo: " << exception.GetReport().GetStringRobust() << Endl
            << "Handler: " << TypeName << Endl
            << "ReqId: " << g.GetReqId() << Endl
            << "Timestamp: " << Context->GetRequestStartTime() << Endl
            << "User: " << (user ? user->GetHRReport() : UserId) << Endl
        ;
        auto result = notifier->Notify(report);
        g.AddEvent(NJson::TMapBuilder
            ("event", "ProcessExceptionNotifyResult")
            ("result", result ? result->SerializeToJson() : NJson::JSON_NULL)
        );
    } else if (notifierName) {
        g.AddEvent(NJson::TMapBuilder
            ("event", "ProcessExceptionNotifierMissing")
            ("name", *notifierName)
        );
    }
}

TAtomicSharedPtr<IChatUserContext> TBaseProcessor::BuildChatContext(TUserPermissions::TPtr permissions) const {
    THolder<IChatUserContext> ctx = MakeHolder<TChatUserContext>(permissions, Server);
    ctx->SetApplicationId(GetApplicationId());
    ctx->SetPlatform(GetUserApp());
    ctx->SetAppBuild(GetAppBuild());
    ctx->SetDeviceId(GetDeviceId());
    ctx->SetClientIp(GetClientIp());
    ctx->SetLocale(GetLocale());
    ctx->SetOrigin(GetOrigin());
    ctx->SetReplyContext(Context);
    TMaybe<TGeoCoord> userLocation = GetUserLocation();
    if (userLocation) {
        ctx->SpecifyGeo(userLocation.GetRef());
    }
    return ctx.Release();
}

bool TBaseProcessor::CheckUserCards(TUserPermissions::TPtr permissions, ICommonOffer::TPtr offer) const {
    bool checkCards = GetHandlerSettingDef<bool>("billing.book.check_cards", true);
    if (checkCards) {
        auto cards = DriveApi->GetUserPaymentMethodsSync(*permissions, *Server, false);
        auto cardsList = TBillingManager::GetUserPaymentCards(cards.Get(), false);
        if (!cardsList.Defined()) {
            return false;
        }
        NDrive::TInfoEntitySession error;
        error.SetCode(ConfigHttpStatus.PaymentRequiredState);
        error.SetLocalizedMessageKey(NDrive::TLocalization::NoCardsProblems());
        if (cardsList->empty()) {
            error.SetErrorInfo("CheckUserCards", "no cards", NDrive::MakeError("card_required"), EDriveSessionResult::IncorrectRequest);
            return false;
        }

        if (offer && offer->GetSelectedCreditCard() && offer->GetDeposit()) {
            bool found = false;
            for (const auto& card : *cardsList) {
                if (card.Check(offer->GetSelectedCreditCard())) {
                    found = true;
                    if (card.OptionalPayerInfo() && card.OptionalPayerInfo()->OptionalFamilyInfo()) {
                        auto& familyInfo = card.OptionalPayerInfo()->GetFamilyInfoRef();
                        auto balance = Max<i64>(0, familyInfo.GetLimit() - familyInfo.GetExpenses());
                        auto limit = familyInfo.GetLimit();
                        if (limit < offer->GetDeposit()) {
                            error.SetErrorInfo("CheckUserCards", "small limit", NDrive::MakeError("small_limit"), EDriveSessionResult::IncorrectRequest);
                            return false;
                        }
                        if (balance < offer->GetDeposit()) {
                            error.SetErrorInfo("CheckUserCards", "small balance", NDrive::MakeError("small_balance"), EDriveSessionResult::IncorrectRequest);
                            return false;
                        }
                    }
                    break;
                }
            }
            if (!found) {
                error.SetErrorInfo("CheckUserCards", "unknown card", NDrive::MakeError("unknown_card"), EDriveSessionResult::IncorrectRequest);
                return false;
            }
        }
        return true;
    }
    return false;
}
