#pragma once

#include "access_verification.h"
#include "config.h"

#include <drive/backend/auth/common/auth.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_devices/manager.h>

#include <util/system/type_name.h>

class IChatUserContext;

enum class EDriveLocalizationCodes {
    NoPermissions = 0 /* "no_permissions" */,
    UserNotFound = 1 /* "user_not_found" */,
    InternalServerError = 2 /* "internal_server_error" */,
    SyntaxUserError = 3 /* "syntax_user_error" */,
    ObjectNotFound = 4 /* "object_not_found" */,
    IncorrectFuelingOffer = 5 /* "incorrect_fueling_offer" */,
    FuelingOfferProblem = 6 /* "fueling_offer_problem" */,
    CannotCopyOnEvolution = 7 /* "cannot_copy_on_evolve" */,
    NoOffers = 8 /* "no_offers" */,
    IncorrectPosition = 9 /* "incorrect_position" */,
    FuelingOfferExists = 10 /* "fueling_offer_exists" */,
    Conflict = 11 /* "conflict" */,
    UnswitchableOffer = 12 /* "unswitchable_offer" */,
    IncorrectPromoCode = 13 /* "incorrect_promo_code" */,
};

enum class EPhoneMatchPolicy {
    Disabled = 0 /* "disabled" */,
    Force = 1 /* "force" */,
    Try = 2 /* "try" */,
    Multiuser = 4 /* "multiuser" */,
};

class TBaseProcessor: public TAuthRequestProcessor {
private:
    using TBase = TAuthRequestProcessor;

private:
    R_FIELD(ENewDeviceStatus, NewDeviceStatus, ENewDeviceStatus::Verified);
    R_READONLY(TString, DeviceId);
    R_READONLY(TString, ClientIp);
    R_READONLY(TString, OriginatorId);
    R_READONLY(TString, UserId);
    R_OPTIONAL(NJson::TJsonValue, PayloadPatch);
    R_OPTIONAL(TString, Comment);

protected:
    void FillInfoEntitySession(NDrive::TInfoEntitySession& result, const TCgiParameters& cgi) const;
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildTx() const {
        auto timeout = Context->GetRequestDeadline() - Now();
        auto lockTimeout = TDuration::Zero();
        auto statementTimeout = timeout;
        auto result = DriveApi->GetTagsManager().GetDeviceTags().BuildTx<traits>(lockTimeout, statementTimeout);
        FillInfoEntitySession(result, Context->GetCgiParameters());
        return result;
    }
    NDrive::TEntitySession BuildChatSession(bool readOnly = false, bool repeatableRead = false) const;
    template<NSQL::TTransactionTraits traits>
    NDrive::TEntitySession BuildYdbTx(const TString& objectName) const {
        const auto useYDB = GetHandlerSetting<bool>("ydb." + objectName + ".use_ydb").GetOrElse(true);
        if (useYDB) {
            auto lockTimeout = TDuration::Zero();
            auto statementTimeout = Context->GetRequestDeadline() - Now();
            return DriveApi->BuildYdbTx<traits>(objectName, Server, statementTimeout, lockTimeout);
        }
        return NDrive::TEntitySession();
    }

    bool CheckAdmActions(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TAdministrativeAction::EEntity entity, const TString& instance = {}, const TMaybe<TSet<TString>>& instanceTags = {}) const;
    void ReqCheckAdmActions(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TAdministrativeAction::EEntity entity, const TString& instance = {}, const TMaybe<TSet<TString>>& instanceTags = {}) const;
    void ReqCheckCondition(bool checkValue, ui32 code, const TString& errorId, const TString& additionalMessage = "") const;
    void ReqCheckCondition(bool checkValue, ui32 code, EDriveLocalizationCodes errorId) const;
    void CheckOrganizationsAccess(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TSet<ui64> parentIds) const;
    TExpected<bool, TSet<TString>> CheckWalletsAccess(TUserPermissions::TPtr permissions, TAdministrativeAction::EAction action, TSet<TString> accountNames) const;

    bool CheckUserGeneralAccessRights(const TString& userId, TUserPermissions::TPtr permissions, const NAccessVerification::TAccessVerificationTraits& traits = NAccessVerification::VerifyAll, bool doThrow = false, bool forceShowObjectsWithoutTags = false) const;

    TString GetApplicationId() const;
    TString GetExternalUserId() const;
    TString GetFallbackUserId() const;
    TString GetNewUserPolicy() const;
    virtual TString GetOrigin() const;

    bool CheckAdvertisingHeaders() const;
    bool CreateYandexAccount(TUserPermissions::TPtr permissions) const;
    bool CreateMobilePaymentAccount(TUserPermissions::TPtr permissions) const;
    virtual bool AddCustomActionsToPermissions() const;

    TAtomicSharedPtr<const ISession> GetCurrentUserSession(TUserPermissions::TPtr permissions, TInstant actuality = TInstant::Zero()) const;
    TAtomicSharedPtr<IChatUserContext> BuildChatContext(TUserPermissions::TPtr permissions) const;

    bool CheckUserCards(TUserPermissions::TPtr permissions, TAtomicSharedPtr<ICommonOffer> offer) const;

public:
    TBaseProcessor(const TCommonAppConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const IServerBase* server)
        : TBase(config, context, authModule, server)
        , Server(server->GetAsPtrSafe<NDrive::IServer>())
        , DriveApi(Server->GetDriveAPI())
        , CommonAppConfig(config)
    {
        CHECK_WITH_LOG(DriveApi);
    }

    const NDrive::IServer* GetServer() const {
        return Server;
    }

    const TDriveAPI* GetDriveApi() const {
        return DriveApi;
    }

    static NDrive::TScheme GetCgiParametersScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */ = {}) {
        return NDrive::TScheme();
    }

    static NDrive::TScheme GetRequestDataScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */ = {}) {
        return NDrive::TScheme();
    }

    virtual void DoAuthProcess(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) override final;

protected:
    virtual void Parse(const TCgiParameters& cgi);
    virtual void Parse(const NJson::TJsonValue& requestData);

    virtual void Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) = 0;
    virtual void ProcessException(TJsonReport::TGuard& g, const TCodedException& exception) const override;

    NJson::TJsonValue GetRequestData() const;

    bool CheckAccessCount(TVector<TDBTag>& dbTags, const TString& eventTagName, const TString& eventName, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const;
    bool UpdateAccessCount(TVector<TDBTag>& dbTags, const TString& eventTagName, const TString& eventName, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) const;

    void SetOriginatorId(const TString& value) {
        OriginatorId = value;
    }

private:
    TString GetUserByPhone(
        TJsonReport::TGuard& g,
        const TUsersDB& usersData,
        const TString& phone,
        const TString& uid,
        EPhoneMatchPolicy matchByPhone,
        NDrive::TEntitySession& tx
    );

protected:
    const NDrive::IServer* Server;
    const TDriveAPI* DriveApi;

private:
    const TCommonAppConfig& CommonAppConfig;
};

class TCommonServiceAppProcessorBase: public TBaseProcessor {
private:
    using TBase = TBaseProcessor;

public:
    using TBase::TBase;

protected:
    void Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) final;

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) = 0;
};

template <typename THandler>
class TSchemeReportingProcessor: public TCommonServiceAppProcessorBase {
private:
    using TBase = TCommonServiceAppProcessorBase;

public:
    using TBase::TBase;

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) override {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Scheme);

        NDrive::TScheme scheme;
        scheme.Add<TFSStructure>("cgi").SetStructure(THandler::GetCgiParametersScheme(Server, Context->GetCgiParameters()));
        scheme.Add<TFSStructure>("request_data").SetStructure(THandler::GetRequestDataScheme(Server, Context->GetCgiParameters()));

        g.MutableReport().SetExternalReport(scheme.SerializeToJson());
        g.SetCode(HTTP_OK);
    }
};

template <typename THandler, typename THandlerConfigImpl, typename TConfigImpl = TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReportingProcessor<THandler>>>
class TAppCommonRegistrator {
public:
    using THandlerConfig = TConfigImpl;

protected:
    const THandlerConfig& Config;

public:
    TAppCommonRegistrator(const THandlerConfig& config)
        : Config(config)
    {
        DEBUG_LOG << Config.Registrator.GetName() << " registered" << Endl;
    }

private:
    const THandlerConfig Registrator = { "dummy" };
};

template <typename THandler,
          typename THandlerConfigImpl = TEmptyConfig,
          typename TBaseHandler = TCommonServiceAppProcessorBase,
          typename TConfigImpl = TContextAwareConfig<THandler, THandlerConfigImpl, TSchemeReportingProcessor<THandler>>>
class TAppCommonProcessor: public TBaseHandler, public TAppCommonRegistrator<THandler, THandlerConfigImpl, TConfigImpl> {
private:
    // NB. No need to parameterize common app registrator as base handler is already a template parameter
    //  and there is no need to inherit from a custom base handler directly
    using TBaseRegistrator = TAppCommonRegistrator<THandler, THandlerConfigImpl, TConfigImpl>;
    using TSelf = TAppCommonProcessor<THandler, THandlerConfigImpl, TBaseHandler, TConfigImpl>;

public:
    using TFactory = typename TBaseHandler::TFactory;

    TAppCommonProcessor(const TConfigImpl& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBaseHandler(config, context, authModule, server)
        , TBaseRegistrator(config)
    {
    }

private:
    static const IRequestProcessor::TFactory::template TRegistrator<TSelf> HandlerRegistrator;
};

template <class THandler, class THandlerConfigImpl, class TBaseHandler, typename TConfigImpl>
const IRequestProcessor::TFactory::template TRegistrator<TAppCommonProcessor<THandler, THandlerConfigImpl, TBaseHandler, TConfigImpl>> TAppCommonProcessor<THandler, THandlerConfigImpl, TBaseHandler, TConfigImpl>::HandlerRegistrator(THandler::GetTypeName());
