#include "processor.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/sessions/manager/billing.h>

#include <library/cpp/yson/node/node_io.h>
#include <library/cpp/json/yson/json2yson.h>

#include <tcmalloc/malloc_extension.h>
#include <execinfo.h>

TTagHistoryEvents GetEventsFromYt(const TString& ytPath, const TString& tagId, const ITagsHistoryContext* hContext) {
    TTagHistoryEvents events;
    auto ytClient = NYT::CreateClient("hahn");
    R_ENSURE(ytClient, HTTP_INTERNAL_SERVER_ERROR, "cannot create yt client");
    auto reader = ytClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(ytPath + "[\"" + tagId + "\"]"));
    for (; reader->IsValid(); reader->Next()) {
        NYT::TNode inputRow = reader->GetRow();
        TStringStream stream;
        NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

        NJson::TJsonValue jsonRow;
        NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);

        TCarTagHistoryEvent event;
        R_ENSURE(TBaseDecoder::DeserializeFromJson(event, jsonRow, hContext), HTTP_INTERNAL_SERVER_ERROR, "cannot parse " + jsonRow.GetStringRobust());
        events.emplace_back(std::move(event));
    }
    return events;
}

void TCreateCompiledRidingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    bool shouldCommit = GetValue<bool>(requestData, "commit", false).GetOrElse(false);
    auto tagId = GetString(requestData, "tag_id");
    auto tx = BuildTx<NSQL::Writable>();
    auto locale = GetLocale();

    const auto& sessionManager = DriveApi->GetSessionManager();
    IEventsSession<TCarTagHistoryEvent>::TConstPtr session;
    if (auto ytPath = GetString(requestData, "yt_path", false)) {
        session = sessionManager.ConsumeEvents(nullptr, GetEventsFromYt(ytPath, tagId, &(DriveApi->GetTagsManager().GetContext())));
    } else {
        auto optionalSession = sessionManager.GetTagSession(tagId, tx);
        R_ENSURE(optionalSession, HTTP_INTERNAL_SERVER_ERROR, "cannot acquire session for tag_id " << tagId, tx);
        session = *optionalSession;
    }
    R_ENSURE(session, HTTP_NOT_FOUND, "cannot find session for tag_id " << tagId, tx);
    g.AddReportElement("session", session->GetReport(locale, Server, nullptr));

    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);

    TBillingSession::TBillingCompilation billingCompilation;
    R_ENSURE(session->FillCompilation(billingCompilation), HTTP_INTERNAL_SERVER_ERROR, "cannot FillCompilation", tx);
    auto sessionId = billingCompilation.GetSessionId();
    g.AddReportElement("session_id", sessionId);

    TMaybe<TPaymentsData> payments;
    bool ignorePayments = GetValue<bool>(requestData, "ignore_payments", false).GetOrElse(false);
    if (!ignorePayments) {
        payments = DriveApi->GetBillingManager().GetPaymentsManager().GetFinishedPayments(sessionId, tx);
        R_ENSURE(payments, HTTP_INTERNAL_SERVER_ERROR, "cannot acquire payments for session_id " << sessionId, tx);
    }

    auto compiledRiding = billingSession->BuildCompiledRiding(Server, billingCompilation, payments.Get());
    R_ENSURE(compiledRiding, HTTP_INTERNAL_SERVER_ERROR, "cannot BuildCompiledRiding for session_id " << sessionId, tx);
    TObjectEvent<TFullCompiledRiding> objEvent(*compiledRiding, EObjectHistoryAction::Add, session->GetLastTS(), session->GetUserId(), permissions->GetUserId(), "");
    g.AddReportElement("compiled", compiledRiding->GetReport(locale, NDriveSession::ReportAll, *Server));

    if (shouldCommit) {
        R_ENSURE(DriveApi->GetMinimalCompiledRides().CreateCompiledRides({ objEvent }, *Server, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot AddHistory", tx);
        R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot Commit", tx);
    }

    g.SetCode(HTTP_OK);
}

void TActualSessionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requestData*/) {
    auto builder = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing", Now());
    R_ENSURE(builder, HTTP_INTERNAL_SERVER_ERROR, "cannot GetSessionBuilder billing");

    auto tx = BuildTx<NSQL::ReadOnly>();
    auto optionalSessions = Server->GetDriveDatabase().GetSessionManager().GetCurrentSessions(tx);
    R_ENSURE(optionalSessions, {}, "cannot GetCurrentSessions", tx);

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    auto sessions = builder->GetSessionsActual();
    for (auto&& session : sessions) {
        if (!session) {
            g.AddEvent("NullSession");
            continue;
        }

        NJson::TJsonValue sessionReport;
        sessionReport["session_id"] = session->GetSessionId();
        sessionReport["closed"] = session->GetClosed();
        auto lastEvent = session->GetLastEvent();
        if (lastEvent) {
            sessionReport["last_event_id"] = lastEvent->GetHistoryEventId();
            sessionReport["last_event_tag"] = lastEvent.GetRef()->GetName();
        }
        auto tagInfo = optionalSessions->find(session->GetSessionId());
        if (tagInfo != optionalSessions->end()) {
            sessionReport["tag_id"] = tagInfo->second.TagId;
            sessionReport["tag_entity_type"] = ToString(tagInfo->second.TagEntityType);
            optionalSessions->erase(tagInfo);
        }
        report.AppendValue(std::move(sessionReport));
    }
    for (auto&& [sessionId, tagInfo] : *optionalSessions) {
        NJson::TJsonValue sessionReport;
        sessionReport["session_id"] = sessionId;
        sessionReport["tag_id"] = tagInfo.TagId;
        sessionReport["tag_entity_type"] = ToString(tagInfo.TagEntityType);
        report.AppendValue(std::move(sessionReport));
    }

    g.MutableReport().AddReportElement("sessions", std::move(report));
    g.SetCode(HTTP_OK);
}

void TCreateCompiledRefundProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    auto sessionId = GetString(requestData, "session_id");
    auto sum = GetValue<ui32>(requestData, "sum").GetOrElse(0);
    R_ENSURE(sum, HTTP_BAD_REQUEST, "incorrect sum");

    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto bills = DriveApi->GetBillingManager().GetCompiledBills().GetBillsFromDB(sessionId, tx);
    R_ENSURE(bills, {}, "cannot fetch bills", tx);
    ui32 refundSum = 0;
    for (auto&& bill : Reversed(*bills)) {
        if (refundSum >= sum) {
            break;
        }
        TCompiledRefund refundField;
        refundField.SetBill(bill.GetBill());
        refundField.SetSessionId(bill.GetSessionId());
        refundField.SetBillingType(bill.GetBillingType());
        refundField.SetRealSessionId(bill.GetRealSessionId());
        refundField.SetComment("manual_refund");

        NDrive::NBilling::TFiscalDetails details;
        for (auto item : bill.GetDetails().GetItems()) {
            item.SetTransactionId(NUtil::CreateUUID());
            details.AddItem(std::move(item));
        }

        refundField.SetDetails(std::move(details));

        if (!DriveApi->GetBillingManager().GetCompiledRefunds().AddHistory(refundField, bill.GetUserId(), EObjectHistoryAction::Add, tx)) {
            tx.DoExceptionOnFail(ConfigHttpStatus);
        }
        refundSum += bill.GetBill();
    }
    R_ENSURE(refundSum >= sum, HTTP_BAD_REQUEST, "incorrect sum", tx);
    R_ENSURE(tx.Commit(), {}, "cannot commit", tx);
    g.MutableReport().AddReportElement("refund_sum", refundSum);
    g.SetCode(HTTP_OK);
}

void TCloseSessionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    bool shouldCommit = GetValue<bool>(requestData, "commit", false).GetOrElse(false);
    auto entity = GetValue<NEntityTagsManager::EEntityType>(requestData, "entity").GetRef();
    auto tagId = GetString(requestData, "tag_id");
    auto tx = BuildTx<NSQL::ReadOnly>();
    auto locale = GetLocale();

    const auto& sessionManager = DriveApi->GetSessionManager();
    IEventsSession<TCarTagHistoryEvent>::TConstPtr session;
    if (auto ytPath = GetString(requestData, "yt_path", false)) {
        switch (entity)
        {
        case NEntityTagsManager::EEntityType::Car: {
            session = sessionManager.ConsumeEvents(nullptr, GetEventsFromYt(ytPath, tagId, &(DriveApi->GetTagsManager().GetContext())));
            break;
        }
        default:
            R_ENSURE(false, HTTP_BAD_REQUEST, "entity " << entity << " is not supported", tx);
        }
    } else {
        switch (entity)
        {
        case NEntityTagsManager::EEntityType::Car: {
            auto optionalSession = sessionManager.GetTagSession(tagId, tx);
            R_ENSURE(optionalSession, {}, "cannot GetTagSession " << tagId, tx);
            session = *optionalSession;
            break;
        }
        case NEntityTagsManager::EEntityType::User: {
            session = CreateOfferHolderSession(tagId, {}, DriveApi->GetTagsManager().GetUserTags(), tx);
            R_ENSURE(session, {}, "cannot CreateOfferHolderSession", tx);
            break;
        }
        default:
            R_ENSURE(false, HTTP_BAD_REQUEST, "entity " << entity << " is not supported", tx);
        }
    }

    R_ENSURE(session, HTTP_NOT_FOUND, "cannot find session for tag_id " << tagId, tx);
    g.AddReportElement("session", session->GetReport(locale, Server, nullptr));
    R_ENSURE(session->GetClosed(), HTTP_NOT_IMPLEMENTED, "session " << session->GetSessionId() << " is not closed", tx);

    const auto& userId = session->GetUserId();
    const auto userPermissions = DriveApi->GetUserPermissions(userId);
    R_ENSURE(userPermissions, HTTP_INTERNAL_SERVER_ERROR, "cannot create user permissions for " << userId, tx);

    if (shouldCommit) {
        const TBillingManager& billingManager = DriveApi->GetBillingManager();
        auto optionalActiveTask = billingManager.GetActiveTasksManager().GetTask(session->GetSessionId(), tx);
        R_ENSURE(optionalActiveTask && *optionalActiveTask, HTTP_INTERNAL_SERVER_ERROR, "cannot GetBillingTask for " << session->GetSessionId(), tx);
        if (optionalActiveTask->GetState() == "active") {
            auto wTx = billingManager.BuildSession(false);
            R_ENSURE(billingManager.FinishingBillingTask(session->GetSessionId(), wTx) && wTx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "Can't finishing billing task " << session->GetSessionId(), wTx);
        }

        TChargableTag::CloseSession(session, *userPermissions, *Server);
    }
    g.SetCode(HTTP_OK);
}

void TSequentialTableInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto tableName = GetString(cgi, "table", true);
    const auto table = IBaseSequentialTableImpl::Instance(tableName);
    R_ENSURE(table, HTTP_INTERNAL_SERVER_ERROR, "cannot get SequentialTable " << tableName);
    const auto timeout = Context->GetRequestDeadline() - Now();
    {
        auto tx = BuildTx<NSQL::Writable>();
        auto maxEventId = table->GetMaxEventIdOrThrow(tx);
        g.AddReportElement("max_event_id", NJson::ToJson(maxEventId));
    }
    {
        auto maxLockedEventId = table->GetLockedMaxEventId(timeout);
        g.AddReportElement("max_locked_event_id", maxLockedEventId);
    }
    {
        auto expectedMaxLockedEventId = table->TryGetLockedMaxEventId();
        g.AddReportElement("try_max_locked_event_id", expectedMaxLockedEventId
            ? *expectedMaxLockedEventId
            : NJson::GetExceptionInfo(expectedMaxLockedEventId.GetError())
        );
    }
    g.SetCode(HTTP_OK);
}

void TMemoryInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const auto& stats = tcmalloc::MallocExtension::GetStats();
    g.AddReportElement("stats", stats);
    g.SetCode(HTTP_OK);
}

NJson::TJsonValue ProfileToJson(const tcmalloc::Profile& profile) {
    NJson::TJsonValue result;
    result.SetValueByPath("type", static_cast<int>(profile.Type()));
    result.SetValueByPath("period", profile.Period());
    NJson::TJsonValue samples = NJson::JSON_ARRAY;
    profile.Iterate([&samples](const tcmalloc::Profile::Sample& sample){
        NJson::TJsonValue s;
        s.SetValueByPath("allocated_size", sample.allocated_size);
        s.SetValueByPath("requested_size", sample.requested_size);
        s.SetValueByPath("count", sample.count);
        s.SetValueByPath("requested_alignment", sample.requested_alignment);
        s.SetValueByPath("sum", sample.sum);
        s.SetValueByPath("user_data", reinterpret_cast<size_t>(sample.user_data));
        s.SetValueByPath("depth", sample.depth);

        NJson::TJsonValue stack = NJson::JSON_ARRAY;
        std::unique_ptr<char*, decltype(&free)> symbols(backtrace_symbols(sample.stack, sample.depth), free);
        for (auto i= 0; i < sample.depth; ++i) {
            stack.AppendValue(symbols.get()[i]);
        }

        s.SetValueByPath("stack", stack);
        samples.AppendValue(std::move(s));
    });

    result.SetValueByPath("samples", std::move(samples));
    return result;
}

void TMemorySnapshotProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& kind = GetValue<int>(cgi, "kind", true);

    const auto& type = static_cast<tcmalloc::ProfileType>(kind.GetOrElse(0));
    const auto& profile = tcmalloc::MallocExtension::SnapshotCurrent(type);

    g.MutableReport().AddReportElement("profile", ProfileToJson(profile));
    g.SetCode(HTTP_OK);
}

static tcmalloc::MallocExtension::AllocationProfilingToken ActiveProfilingToken;
void TMemoryProfileProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& op = GetString(cgi, "op", true);
    if (op == "start") {
        ActiveProfilingToken = tcmalloc::MallocExtension::StartAllocationProfiling();
        g.SetCode(HTTP_OK);
        return;
    }

    if (op == "stop") {
        const auto& profile = std::move(ActiveProfilingToken).Stop();
        g.MutableReport().AddReportElement("profile", ProfileToJson(profile));
        g.SetCode(HTTP_OK);
        return;
    }

    g.MutableReport().AddReportElement("reason", "unknown op");
    g.MutableReport().AddReportElement("op", op);
    g.SetCode(HTTP_BAD_REQUEST);
}

void TMemoryReleaseProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& num_bytes = GetValue<size_t>(cgi, "bytes", true);

    tcmalloc::MallocExtension::ReleaseMemoryToSystem(num_bytes.GetRef());
    g.SetCode(HTTP_OK);
}
