#include "processor.h"

#include <drive/library/cpp/compression/simple.h>

#include <library/cpp/json/json_reader.h>

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoCar& object) {
    auto result = object.Raw;
    result["number"] = object.Number;
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoFeature& object) {
    auto result = object.Raw;
    result["index"] = object.Id;
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoFilter& object) {
    auto result = object.Raw;
    result["id"] = object.Id;
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoModel& object) {
    auto result = object.Raw;
    result["code"] = object.Code;
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoPatch& object) {
    auto result = object.Raw;
    result["index"] = object.Index;
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TDemoCarList& object) {
    auto result = object.Raw;
    result["cars"] = NJson::ToJson(object.Cars);
    result["sf"] = NJson::ToJson(object.Features);
    result["filters"] = NJson::ToJson(object.Filters);
    result["models"] = NJson::ToJson(NJson::Dictionary(object.Models));
    result["property_patches"] = NJson::ToJson(object.Patches);
    result["visibility"] = NJson::ToJson(object.Visibility);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoCar& result) {
    result.Raw = value;
    return
        NJson::ParseField(value["sf"], result.Features) &&
        NJson::ParseField(value["filters"], result.Filters) &&
        NJson::ParseField(value["model_id"], result.ModelId) &&
        NJson::TryFromJson(value["number"], result.Number);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoFeature& result) {
    result.Raw = value;
    return
        NJson::TryFromJson(value["index"], result.Id);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoFilter& result) {
    result.Raw = value;
    return
        NJson::TryFromJson(value["id"], result.Id);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoModel& result) {
    result.Raw = value;
    return
        NJson::TryFromJson(value["code"], result.Code);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoPatch& result) {
    result.Raw = value;
    return
        NJson::TryFromJson(value["index"], result.Index);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TDemoCarList& result) {
    result.Raw = value;
    return
        NJson::ParseField(value["sf"], result.Features) &&
        NJson::ParseField(value["filters"], result.Filters) &&
        //NJson::ParseField(value["models"], NJson::Dictionary(result.Models)) &&
        //NJson::ParseField(value["property_patches"], result.Patches) &&
        NJson::ParseField(value["visibility"], result.Visibility) &&
        NJson::TryFromJson(value["cars"], result.Cars);
}

void TDemoCarListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);

    const auto& cgi = Context->GetCgiParameters();
    const auto& selectedCarNumbers = MakeSet(GetStrings(cgi, "car_number", false));
    const auto& settings = Server->GetSettings();
    auto key = GetHandlerSetting<TString>("demo.car_list_key").GetOrElse("user_app.static.user_app_car_list");
    auto optionalData = permissions->GetSetting<TString>(settings, key);
    R_ENSURE(optionalData, ConfigHttpStatus.UnknownErrorStatus, "key " << key << " is not defined");
    auto data = std::move(*optionalData);
    auto decompress = permissions->GetSetting<bool>(settings, key + ".decompress").GetOrElse(false);
    if (decompress) {
        data = NDrive::Decompress(data).GetRef();
    }

    auto value = NJson::ReadJsonFastTree(data);
    auto carList = NJson::FromJson<NDrive::TDemoCarList>(value);
    auto selectedFeatures = TMaybe<TSet<ui64>>();
    auto selectedFilters = TMaybe<TSet<ui64>>();
    auto selectedModels = TMaybe<TSet<TString>>();
    auto selectedPatches = TMaybe<TSet<ui64>>();
    if (!selectedCarNumbers.empty()) {
        selectedFeatures.ConstructInPlace();
        selectedFilters.ConstructInPlace();
        selectedModels.ConstructInPlace();
        selectedPatches.ConstructInPlace();
        auto cars = NDrive::TDemoCars();
        for (auto&& car : carList.Cars) {
            if (selectedCarNumbers.contains(car.Number)) {
                selectedFeatures->insert(car.Features.begin(), car.Features.end());
                selectedFilters->insert(car.Filters.begin(), car.Filters.end());
                selectedModels->insert(car.ModelId);
                cars.push_back(std::move(car));
            }
        }
        carList.Cars = std::move(cars);
        carList.Visibility.clear();
        g.AddEvent(NJson::TMapBuilder
            ("event", "SelectedData")
            ("features", NJson::ToJson(selectedFeatures))
            ("filters", NJson::ToJson(selectedFilters))
            ("models", NJson::ToJson(selectedModels))
            ("patches", NJson::ToJson(selectedPatches))
        );
    }
    if (selectedFeatures) {
        auto features = NDrive::TDemoFeatures();
        for (auto&& feature : carList.Features) {
            if (selectedFeatures->contains(feature.Id)) {
                features.push_back(std::move(feature));
            }
        }
        carList.Features = std::move(features);
    }
    if (selectedFilters) {
        auto filters = NDrive::TDemoFilters();
        for (auto&& filter : carList.Filters) {
            if (selectedFilters->contains(filter.Id)) {
                filters.push_back(std::move(filter));
            }
        }
        carList.Filters = std::move(filters);
    }
    if (selectedModels) {
        auto models = NDrive::TDemoModels();
        for (auto&& [code, model] : carList.Models) {
            if (selectedModels->contains(model.Code)) {
                models.emplace(std::move(code), std::move(model));
            }
        }
        carList.Models = std::move(models);
    }
    if (selectedPatches) {
        auto patches = NDrive::TDemoPatches();
        for (auto&& patch : carList.Patches) {
            if (selectedPatches->contains(patch.Index)) {
                patches.push_back(std::move(patch));
            }
        }
        carList.Patches = std::move(patches);
    }
    g.SetExternalReport(NJson::ToJson(carList));
    g.SetCode(HTTP_OK);
}
