#include "processor.h"

#include <drive/backend/database/drive/landing.h>
#include <drive/backend/distributing_block_storage/common.h>
#include <drive/backend/distributing_block_storage/database.h>
#include <drive/backend/offers/actions/distributing_block.h>


namespace {
    void PatchButton(NJson::TJsonValue& button, const TString& distributingBlockType) {
        if (button["button_type"].GetString() == "hide") {
            button["button_type"] = "backend_handler";
            button["button_meta"] = NJson::TMapBuilder
                ("url", "/api/yandex/distributing_block/hide")
                ("body_params", NJson::TMapBuilder("id", distributingBlockType))
            ;
        }
    }

    void PatchIntroscreenBody(NJson::TJsonValue& value, const TString& distributingBlockType) {
        PatchButton(value, distributingBlockType);
        if (value["other_buttons"].IsArray()) {
            auto size = value["other_buttons"].GetArray().size();
            for (decltype(size) i = 0; i < size; ++i) {
                PatchButton(value["other_buttons"][i], distributingBlockType);
            }
        }
    }
}

void TShowDistributingBlockProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TString distributingBlockType = GetString(requestData, "id");
    if (distributingBlockType.empty()) {
        distributingBlockType = "empty";
    }
    TString userId = permissions->GetUserId();
    TInstant timestamp = Now();
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto event = MakeAtomicShared<TDistributingBlockEvent>();
    event->SetType(distributingBlockType);
    event->SetUserId(userId);
    event->SetTimestamp(timestamp);

    auto storage = Server->GetDistributingBlockEventsStorage();
    R_ENSURE(storage, ConfigHttpStatus.UnknownErrorStatus, "DistributingBlockEventsStorage is not configured");
    auto asyncStore = storage->Store(event, &session);
    R_ENSURE(asyncStore.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async DistributingBlockEventsStorage::Store");
    R_ENSURE(session.Commit(), {}, "cannot commit", session);
    g.SetCode(HTTP_OK);
}

void TGetDistributingBlockProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    TUserOfferContext uoc(Server, permissions, Context);
    uoc.SetLocale(GetLocale());
    uoc.SetNeedHistoryFreeTimeFees(false);
    uoc.SetNeedFinishArea(false);
    uoc.SetNeedGeoFeatures(false);
    uoc.SetNeedUserDestinationSuggest(false);
    uoc.SetNeedUserFeatures(false);
    uoc.SetNeedTaxiPrice(false);
    uoc.SetFilterAccounts(false);
    uoc.SetNeedDefaultAccount(false);
    uoc.SetNeedYandexPaymentMethod(false);
    uoc.SetNeedDistributingBlockShowsRestriction(uoc.GetSetting<bool>(TDistributingBlockAction::EnableShowsRestrictionSetting).GetOrElse(false));
    uoc.SetAllowDistributingBlock(uoc.GetSetting<bool>(TDistributingBlockAction::AllowDistributingBlockInMainScreen).GetOrElse(true));
    ReqCheckCondition(uoc.Prefetch(), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_prefetch_error");

    TOffersBuildingContext offersBuildingContext(std::move(uoc));
    const TVector<IOfferReport::TPtr> fakeOffers;
    const TDistributingBlockAction* bestAction = nullptr;
    for (const auto& action : permissions->GetActionsActual()) {
        auto distributingBlockAction = action.GetAs<TDistributingBlockAction>();
        if (distributingBlockAction
            && distributingBlockAction->IsRelevant(offersBuildingContext, fakeOffers, TDistributingBlockAction::ReportInMainScreen)) {
            if (!bestAction || distributingBlockAction->GetPriority() < bestAction->GetPriority()) {
                bestAction = distributingBlockAction;
            }
        }
    }
    if (const auto* mapObjectAction = dynamic_cast<const TMapObjectDistributingBlockAction*>(bestAction)) {
        const auto& landingsDB = *Yensured(Yensured(DriveApi)->GetLandingsDB());
        auto maybeWarning = landingsDB.GetObject(mapObjectAction->GetWarningId());
        if (maybeWarning) {
            auto introscreenBody = maybeWarning->GetJsonLanding();
            PatchIntroscreenBody(introscreenBody, mapObjectAction->GetName());
            g.AddReportElement("introscreens", NJson::TMapBuilder(mapObjectAction->GetWarningId(), std::move(introscreenBody)));
            g.AddReportElement("maps_promo", NJson::TArrayBuilder(mapObjectAction->BuildJsonReport(*Server, offersBuildingContext, fakeOffers, TDistributingBlockAction::EPlaceReportTraits::ReportInMainScreen)));
        }
    } else if (bestAction) {
        g.AddReportElement("distributing_block", bestAction->BuildJsonReport(*Server, offersBuildingContext, fakeOffers, TDistributingBlockAction::EPlaceReportTraits::ReportInMainScreen));
    }
    ui32 defaultRequestDelaySeconds = 5 * 60;
    auto requestDelaySeconds = offersBuildingContext.GetUserHistoryContextRef().GetSetting<ui32>("offers.distributing_block.main_screen_request_delay_s").GetOrElse(defaultRequestDelaySeconds);
    // sanity check
    if (defaultRequestDelaySeconds < 30) {
        ALERT_LOG << "Request delay on main screen for distributing block is " << defaultRequestDelaySeconds << "s, which is too short" << Endl;
        requestDelaySeconds = defaultRequestDelaySeconds;
    }
    g.AddReportElement("request_delay", requestDelaySeconds);
    g.SetCode(HTTP_OK);
}

void THideDistributingBlockProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TString distributingBlockType = GetString(requestData, "id");
    R_ENSURE(!distributingBlockType.empty() && distributingBlockType != "empty", HTTP_BAD_REQUEST, "do not hide empty distributing block");

    const auto* storage = Server->GetDistributingBlockEventsStorage();

    R_ENSURE(dynamic_cast<const NDrive::TDBDistributingBlockEventsStorage*>(storage), ConfigHttpStatus.UnknownErrorStatus, "DistributingBlockEventsStorage is configured incorrectly");

    TString userId = permissions->GetUserId();
    TInstant timestamp = Now();
    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto event = MakeAtomicShared<TDistributingBlockEvent>();
    event->SetType(distributingBlockType);
    event->SetUserId(userId);
    event->SetTimestamp(timestamp);
    event->SetShowsCount(Max<ui32>());
    auto asyncStore = storage->Store(event, &session);
    R_ENSURE(asyncStore.Initialized(), ConfigHttpStatus.UnknownErrorStatus, "uninitialized async DistributingBlockEventsStorage::Store");
    std::ignore = asyncStore.GetValueSync();
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "can't commit", session);
    g.SetCode(HTTP_OK);
}
