#include "processor.h"

#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/uuid.h>

void TGetDocumentsListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");
    const TDocumentsManager& manager = *(Server->GetDocumentsManager());

    const bool isActive = GetValue<bool>(Context->GetCgiParameters(), "is_active", false).GetOrElse(true);
    const bool forQueue = GetValue<bool>(Context->GetCgiParameters(), "for_queue", false).GetOrElse(false);
    if (!isActive) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Documents);
    }

    TVector<TDocumentDescriptionPtr> documents;
    bool fetchIsCorrect = false;
    if (forQueue) {
        fetchIsCorrect = manager.GetRegisteredDocuments(documents, permissions->GetAvailableDocuments(), isActive);
    } else {
        fetchIsCorrect = manager.GetRegisteredDocuments(documents, isActive);
    }
    R_ENSURE(fetchIsCorrect, ConfigHttpStatus.UnknownErrorStatus, "Cannot fetch queue documents from cache");

    NJson::TJsonValue documentsArray(NJson::JSON_ARRAY);
    for (auto&& doc : documents) {
        documentsArray.AppendValue(doc->GetReport());
    }
    g.MutableReport().AddReportElement("documents", std::move(documentsArray));
    g.SetCode(HTTP_OK);
}

void TGetDocumentHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");
    const TDocumentsManager& manager = *(Server->GetDocumentsManager());

    const TString name = GetString(Context->GetCgiParameters(), "document_name", false);
    const TInstant startInstant = GetTimestamp(Context->GetCgiParameters(), "start_ts", TInstant::Zero());
    TDocumentsDescriptionHistoryManager::TEventPtrs events;

    bool fetchIsCorrect = false;
    if (name.empty()) {
        fetchIsCorrect = manager.GetDescriptionsHistoryManager().GetEventsAll(startInstant, events, TInstant::Zero());
    } else {
        fetchIsCorrect = manager.GetDescriptionsHistoryManager().GetIndexByName().GetEvents(name, startInstant, events, TInstant::Zero());
        g.MutableReport().AddReportElement("document", name);
    }
    R_ENSURE(fetchIsCorrect, ConfigHttpStatus.UnknownErrorStatus, "Cannot fetch history from history manager");

    NJson::TJsonValue historyArray(NJson::JSON_ARRAY);
    for (auto&& event : events) {
        historyArray.AppendValue(event->SerializeToJson());
    }
    g.MutableReport().AddReportElement("history", std::move(historyArray));
    g.SetCode(HTTP_OK);
}

void TUpsertDocumentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    TMessagesCollector errors;
    auto description = TDocumentDescription::BuildFromJson(requestData, *Server, &errors);
    R_ENSURE(description, ConfigHttpStatus.SyntaxErrorStatus, errors.GetStringReport());

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDocumentsManager()->UpsertDocumentDescription(description, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TRemoveDocumentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const auto documentName = GetString(requestData, "id", true);

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDocumentsManager()->RemoveDocumentDescription(documentName, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TDocumentPreviewProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    TMessagesCollector errors;
    TBlob result = Server->GetDocumentsManager()->BuildDocumentPreview(requestData, *Server, permissions, errors);
    R_ENSURE(!result.Empty(), ConfigHttpStatus.EmptySetStatus, errors.GetStringReport());

    TStringBuf resultStr(result.AsCharPtr(), result.Size());
    g.MutableReport().AddReportElement("preview", resultStr);
    g.SetCode(HTTP_OK);
}

void TScheduleDocumentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");
    R_ENSURE(DriveApi->GetUsersData()->IsSearchEnabled() && DriveApi->GetCarsData()->IsSearchEnabled(), ConfigHttpStatus.UnknownErrorStatus, "Car search was not enabled in config, therefore no index was built");

    NJson::TJsonValue inputParameters = requestData;
    {
        TString documentName = GetString(inputParameters, "document_name", false);
        if (!documentName) {
            documentName = GetString(Context->GetCgiParameters(), "document_name", true);
            inputParameters.InsertValue("document_name", documentName);
        }
    }
    {
        auto storages = GetStrings(inputParameters, "storage", false);
        if (storages.empty()) {
            storages = GetStrings(Context->GetCgiParameters(), "storage", true);
            inputParameters.InsertValue("storage", JoinSeq(",", storages));
        }
    }
    {
        NJson::TJsonValue objectsReport;
        if (!ChangeEntity(NEntityTagsManager::EEntityType::Car, DriveApi->GetTagsManager().GetDeviceTags(), *permissions, *DriveApi->GetCarsData(), inputParameters, &objectsReport)) {
            g.MutableReport().AddReportElement("available_cars", std::move(objectsReport));
            g.SetCode(HTTP_BAD_REQUEST);
            return;
        }
    }
    {
        NJson::TJsonValue objectsReport;
        if (!ChangeEntity(NEntityTagsManager::EEntityType::User, DriveApi->GetTagsManager().GetUserTags(), *permissions, *DriveApi->GetUsersData(), inputParameters, &objectsReport)) {
            g.MutableReport().AddReportElement("available_users", std::move(objectsReport));
            g.SetCode(HTTP_BAD_REQUEST);
            return;
        }
    }

    TQueuedDocument newDocument;
    TString id = NUtil::CreateUUID();
    R_ENSURE(newDocument.SetId(id).SetStatus(EAssemblerStage::NotReady).SetAuthor(permissions->GetUserId()).SetInputParameters(inputParameters), ConfigHttpStatus.SyntaxErrorStatus, "incorrect input");

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDocumentsManager()->UpsertQueuedDocument(newDocument, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
    g.MutableReport().AddReportElement("id", std::move(id));
}

void TGetDocumentStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const TString documentId = GetString(requestData, "id", true);

    auto document = Server->GetDocumentsManager()->GetQueuedDocument(documentId, Context->GetRequestStartTime());
    R_ENSURE(document, ConfigHttpStatus.EmptySetStatus, "Undefined document id");
    R_ENSURE(document->GetAuthor() == permissions->GetUserId(), ConfigHttpStatus.SyntaxErrorStatus, "No permissions");

    switch (document->GetStatus()) {
    case EAssemblerStage::Done: {
        auto result = document->GetResultOutput();
        g.MutableReport().AddReportElement("content", std::move(result));
        g.SetCode(HTTP_OK);
        break;
    }
    case EAssemblerStage::NotReady:
        g.SetCode(HTTP_NO_CONTENT);
        break;

    case EAssemblerStage::Error: {
        auto error = document->GetError();
        g.MutableReport().AddReportElement("error", std::move(error));
        g.SetCode(ConfigHttpStatus.IncompleteStatus);
        break;
    }
    default:
        g.SetCode(ConfigHttpStatus.EmptySetStatus);
        break;
    }
}

void TCancelQueueDocumentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const TString documentId = GetString(requestData, "id", true);

    auto document = Server->GetDocumentsManager()->GetQueuedDocument(documentId, Context->GetRequestStartTime());
    R_ENSURE(document, ConfigHttpStatus.EmptySetStatus, "Undefined document id");

    document->SetStatus(EAssemblerStage::Canceled);

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDocumentsManager()->UpsertQueuedDocument(document.GetRef(), permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TSetQueueDocumentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const TString documentId = GetString(requestData, "id", true);
    const TString queue = GetString(requestData, "queue", true);

    auto document = Server->GetDocumentsManager()->GetQueuedDocument(documentId, Context->GetRequestStartTime());
    R_ENSURE(document, ConfigHttpStatus.EmptySetStatus, "Undefined document id");

    document->SetDocumentQueue(queue);

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDocumentsManager()->UpsertQueuedDocument(document.GetRef(), permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TQueuedDocumentListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const bool all = GetValue<bool>(Context->GetCgiParameters(), "all", false).GetOrElse(false);
    const auto status = GetValue<EAssemblerStage>(Context->GetCgiParameters(), "status", false);

    bool fetchIsCorrect = false;
    TVector<TQueuedDocument> documents;
    if (all) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Documents);
        fetchIsCorrect = Server->GetDocumentsManager()->GetQueuedDocuments(documents, status);
    } else {
        fetchIsCorrect = Server->GetDocumentsManager()->GetQueuedDocuments(documents, status, permissions->GetUserId());
    }
    R_ENSURE(fetchIsCorrect, ConfigHttpStatus.UnknownErrorStatus, "Cannot fetch queued documents from cache");

    TSet<TString> users;
    for (const auto& doc : documents) {
        users.insert(doc.GetAuthor());
    }

    auto gUsers = DriveApi->GetUsersData()->FetchInfo(users);

    NJson::TJsonValue result;
    for (const auto& doc : documents) {
        NJson::TJsonValue report = doc.GetReport();
        if (!!doc.GetAuthor()) {
            const TDriveUserData* userInfo = gUsers.GetResultPtr(doc.GetAuthor());
            if (!!userInfo) {
                report.InsertValue("author_details", userInfo->GetPublicReport());
            }
        }
        result.AppendValue(std::move(report));
    }

    g.MutableReport().AddReportElement("documents", std::move(result));
    g.SetCode(HTTP_OK);
}

void TGetDocumentQueueHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Documents);
    R_ENSURE(Server->GetDocumentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Undefined DocumentManager");

    const TDocumentsManager& manager = *(Server->GetDocumentsManager());
    const TString userId = GetUUID(Context->GetCgiParameters(), "user", false);
    const TString name = GetString(Context->GetCgiParameters(), "document_name", false);
    const TInstant startInstant = GetTimestamp(Context->GetCgiParameters(), "start_ts", TInstant::Zero());
    TDocumentsQueueHistoryManager::TEventPtrs events;

    bool fetchIsCorrect = false;
    if (userId.empty()) {
        if (name.empty()) {
            fetchIsCorrect = manager.GetQueueHistoryManager().GetEventsAll(startInstant, events, TInstant::Zero());
        } else {
            fetchIsCorrect = manager.GetQueueHistoryManager().GetIndexByDocument().GetEvents(name, startInstant, events, TInstant::Zero());
        }
    } else {
        fetchIsCorrect = manager.GetQueueHistoryManager().GetIndexByUser().GetEvents(userId, startInstant, events, TInstant::Zero());
        g.MutableReport().AddReportElement("user", userId);
        if (name) {
            TDocumentsQueueHistoryManager::TEventPtrs objectEvents;
            for (const auto& ev : events) {
                if (ev->GetDocumentName() == name) {
                    objectEvents.emplace_back(ev);
                }
            }
            events = objectEvents;
        }
    }
    R_ENSURE(fetchIsCorrect, ConfigHttpStatus.UnknownErrorStatus, "Cannot fetch queued documents from history manager");

    NJson::TJsonValue historyArray(NJson::JSON_ARRAY);
    for (auto&& event : events) {
        historyArray.AppendValue(event->SerializeToJson());
    }
    g.MutableReport().AddReportElement("history", std::move(historyArray));
    g.SetCode(HTTP_OK);
}
