#pragma once

#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/database/entity/search_index.h>
#include <drive/backend/registrar/ifaces.h>

class TGetDocumentsListProcessor: public TAppCommonProcessor<TGetDocumentsListProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TGetDocumentsListProcessor, TEmptyConfig>;
public:
    TGetDocumentsListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_documents_list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGetDocumentHistoryProcessor : public TAppCommonProcessor<TGetDocumentHistoryProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TGetDocumentHistoryProcessor, TEmptyConfig>;
public:
    TGetDocumentHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_document_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUpsertDocumentProcessor: public TAppCommonProcessor<TUpsertDocumentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUpsertDocumentProcessor, TEmptyConfig>;
public:
    TUpsertDocumentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "upsert_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TRemoveDocumentProcessor : public TAppCommonProcessor<TRemoveDocumentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveDocumentProcessor, TEmptyConfig>;
public:
    TRemoveDocumentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "remove_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TDocumentPreviewProcessor: public TAppCommonProcessor<TDocumentPreviewProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TDocumentPreviewProcessor, TEmptyConfig>;
public:
    TDocumentPreviewProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "preview_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TScheduleDocumentProcessor : public TAppCommonProcessor<TScheduleDocumentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TScheduleDocumentProcessor, TEmptyConfig>;
public:
    TScheduleDocumentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "schedule_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

private:
    template <class TObjectsContainer, class TTagsManager>
    bool ChangeEntity(const NEntityTagsManager::EEntityType entityType, const TTagsManager& tagsManager, const TUserPermissions& permissions, const TObjectsContainer& objectsDB, NJson::TJsonValue& parameters, NJson::TJsonValue* errorReport) {
        const auto entityStr = ::ToString(entityType);
        if (!parameters.Has(entityStr + "_template") && !parameters.Has(entityStr + "_document_template")) {
            return true;
        }

        auto ignoredStatuses = MakeSet(SplitString(GetHandlerSettingDef<TString>("ignore_statuses", "onboarding,screening,external"), ","));

        TVector<TString> documentFiltred;
        auto documentParams = GetStrings(parameters, entityStr + "_document_template", false);
        if (documentParams.size() && entityType == NEntityTagsManager::EEntityType::User && Server->GetUserRegistrationManager()) {
            TMap<TString, ui32> matchedIds;
            for (const auto& param : documentParams) {
                auto hash = Server->GetUserRegistrationManager()->GetDocumentNumberHash(param);
                for (auto&& id : DriveApi->GetUsersData()->GetDocumentOwnersByHash(hash)) {
                    auto it = matchedIds.find(id);
                    if (it == matchedIds.end()) {
                        matchedIds[id] = 1;
                    } else {
                        matchedIds[id]++;
                    }
                }
            }

            auto session = BuildTx<NSQL::ReadOnly>();
            auto users = DriveApi->GetUsersData()->FetchInfo(MakeSet(NContainer::Keys(matchedIds)), session);
            if (!users) {
                session.Check();
            }

            for(auto&& id : matchedIds) {
                if (id.second == documentParams.size()) {
                    auto user = users.GetResultPtr(id.first);
                    if (user && !ignoredStatuses.contains(user->GetStatus())) {
                        documentFiltred.emplace_back(std::move(id.first));
                    }
                }
            }
            if (documentFiltred.empty()) {
                if (errorReport) {
                    errorReport->SetType(NJson::JSON_ARRAY);
                }
                return false;
            }
        }

        auto requiredParams = GetStrings(parameters, entityStr + "_template", false);
        TVector<TString> requiredIds;
        if (requiredParams.size()) {
            auto searchRequest = TSearchRequest();

            for (auto param : requiredParams) {
                searchRequest.AddMatchingCondition(TMatchCondition(param, true));
            }
            searchRequest.SetLimit(GetHandlerSettingDef(entityStr + "_limit", 5));

            auto session = BuildTx<NSQL::ReadOnly>();
            auto entityFilter = [&](const TString& id) -> bool {
                auto taggedObject = tagsManager.GetCachedOrRestoreObject(id, session);
                if (taggedObject) {
                    if (permissions.GetVisibility(*taggedObject, entityType) == TUserPermissions::EVisibility::NoVisible) {
                        return false;
                    }

                    if (entityType == NEntityTagsManager::EEntityType::User) {
                        auto users = DriveApi->GetUsersData()->FetchInfo(id, session);
                        if (!users) {
                            session.Check();
                        }
                        auto user = users.GetResultPtr(id);
                        if (!user || ignoredStatuses.contains(user->GetStatus())) {
                            return false;
                        }
                    }
                } else {
                    session.Check();
                }
                return true;
            };
            requiredIds = objectsDB.GetMatchingIds(searchRequest, entityFilter);
        }

        TVector<TString> matchedIds;
        if (documentFiltred.size()) {
            if (requiredParams) {
                auto it = documentFiltred.begin();
                for(auto&& id : requiredIds) {
                    if (AdvanceSimple(it, documentFiltred.end(), id)) {
                        matchedIds.emplace_back(std::move(id));
                    }
                }
            } else {
                matchedIds = std::move(documentFiltred);
            }
        } else {
            matchedIds = std::move(requiredIds);
        }

        if (matchedIds.empty()) {
            if (errorReport) {
                errorReport->SetType(NJson::JSON_ARRAY);
            }
            return false;
        }

        auto buildMultiUserError = [&]() {
            if (errorReport) {
                auto matchedObjects = objectsDB.FetchInfo(matchedIds);
                errorReport->SetType(NJson::JSON_ARRAY);
                for (auto&& item : matchedObjects.GetResult()) {
                    errorReport->AppendValue(item.second.GetSearchReport());
                }
            }
        };

        if (auto duplicateDocument = GetHandlerSettingDef<TString>("duplicate_" + entityStr + "_document", "") ) {
            if (entityType == NEntityTagsManager::EEntityType::User && GetHandlerSettingDef("check_user_duplicate", true)) {
                auto tx = BuildTx<NSQL::ReadOnly>();
                auto check = DriveApi->GetUsersData()->CheckDuplicates(matchedIds, tx);
                if (!check) {
                    tx.Check();
                    return false;
                }
                if (!check.GetRef()) {
                    buildMultiUserError();
                    return false;
                }
            }
            auto& duplicateParameters = parameters["document_parameters_" + duplicateDocument];

            if (duplicateParameters.IsArray()) {
                NJson::TJsonValue newDuplicateParameters(NJson::JSON_ARRAY);
                for (auto value : duplicateParameters.GetArray()) {
                    for (const auto& id : matchedIds) {
                        value[entityStr + "_id"] = id;
                        newDuplicateParameters.AppendValue(value);
                    }
                }
                duplicateParameters = newDuplicateParameters;
            } else {
                for (const auto& id : matchedIds) {
                    duplicateParameters.AppendValue(NJson::TMapBuilder(entityStr + "_id", id));
                }
            }
            return true;
        }

        if (matchedIds.size() == 1) {
            parameters.InsertValue(entityStr + "_id", *matchedIds.begin());
            return true;
        }

        buildMultiUserError();
        return false;
    }
};

class TGetDocumentStatusProcessor : public TAppCommonProcessor<TGetDocumentStatusProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TGetDocumentStatusProcessor, TEmptyConfig>;
public:
    TGetDocumentStatusProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "document_status";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TCancelQueueDocumentProcessor : public TAppCommonProcessor<TCancelQueueDocumentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCancelQueueDocumentProcessor, TEmptyConfig>;
public:
    TCancelQueueDocumentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "cancel_queue_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TSetQueueDocumentProcessor : public TAppCommonProcessor<TSetQueueDocumentProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TSetQueueDocumentProcessor, TEmptyConfig>;
public:
    TSetQueueDocumentProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "set_queue_document";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TQueuedDocumentListProcessor : public TAppCommonProcessor<TQueuedDocumentListProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TQueuedDocumentListProcessor, TEmptyConfig>;
public:
    TQueuedDocumentListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_queued_documents_list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TGetDocumentQueueHistoryProcessor : public TAppCommonProcessor<TGetDocumentQueueHistoryProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TGetDocumentQueueHistoryProcessor, TEmptyConfig>;
public:
    TGetDocumentQueueHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "get_queued_documents_history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};
