#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>

namespace {
struct TTariffFilters {
    TSet<TString> CategoriesFilter;
    TSet<TString> ExteriorTypesFilter;
    TSet<TString> TransmissionsFilter;
    TSet<TString> FuelTypesFilter;
    TSet<TString> AirConditionsFilter;

    NJson::TJsonArray JModels;
    NJson::TJsonArray JCategories;
    NJson::TJsonArray JExteriorTypes;
    NJson::TJsonArray JTransmissions;
    NJson::TJsonArray JFuelTypes;
    NJson::TJsonArray JAirConditions;
};

void fillFilter(const TDriveModelSpecification& specification, TSet<TString>& keyFilter, NJson::TJsonArray& jFilter,  ELocalization locale) {
    if (!keyFilter.contains(specification.GetValue())) {
        NJson::TJsonValue jObject;
        jObject["id"] = specification.GetValue();
        jObject["name"] = specification.GetLocalizedValue(locale);
        jFilter.AppendValue(jObject);
        keyFilter.insert(specification.GetValue());
    }
}

void fillFiltersFromSpecifications(const TDriveModelSpecifications& specifications, TTariffFilters& filters, ELocalization locale) {
    for (const auto& specification: specifications.GetSpecifications()) {
        if (NDriveModelSpecification::EModelSpecificationType::Category == specification.GetType()) {
            fillFilter(specification, filters.CategoriesFilter, filters.JCategories, locale);
        } else if (NDriveModelSpecification::EModelSpecificationType::ExteriorType == specification.GetType()) {
            fillFilter(specification, filters.ExteriorTypesFilter, filters.JExteriorTypes, locale);
        } else if (NDriveModelSpecification::EModelSpecificationType::Transmission == specification.GetType()) {
            fillFilter(specification, filters.TransmissionsFilter, filters.JTransmissions, locale);
        } else if (NDriveModelSpecification::EModelSpecificationType::FuelType == specification.GetType()) {
            fillFilter(specification, filters.FuelTypesFilter, filters.JFuelTypes, locale);
        } else if (NDriveModelSpecification::EModelSpecificationType::AirConditioning == specification.GetType()) {
            fillFilter(specification, filters.AirConditionsFilter, filters.JAirConditions, locale);
        }
    }
}
}

void TTariffFiltersListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Models);

    auto tx = BuildTx<NSQL::ReadOnly>();

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);
    auto modelCodes = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::Models, permissions);

    auto modelInfo = Server->GetDriveAPI()->GetModelsData()->FetchInfo(modelCodes, tx);
    R_ENSURE(
          modelInfo
        , HTTP_INTERNAL_SERVER_ERROR
        , "cannot fetch models"
        , NDrive::MakeError("car_model.read_model_error")
        , tx);

    const auto locale = GetLocale();

    TTariffFilters filters;
    for (auto&& carModel : modelInfo.GetResult()) {
        if (!carModel.second.HasDeprecated() || !carModel.second.GetDeprecatedRef()) {
            NJson::TJsonValue jObject;
            jObject["id"] = carModel.second.GetCode();
            jObject["name"] = carModel.second.GetName();
            filters.JModels.AppendValue(jObject);
            fillFiltersFromSpecifications(carModel.second.GetSpecifications(), filters, locale);
        }
    }
    TSet<TString> visibleCarIds;

    {
        TDeviceTagsManager::TCurrentSnapshot devicesSnapshot;
        R_ENSURE(
              Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCurrentSnapshot(devicesSnapshot)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot get current snapshot"
            , tx);
            for (const auto& [carId, taggedObject]: devicesSnapshot) {
                if (permissions->GetVisibility(*taggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible) {
                    visibleCarIds.insert(carId);
                }
            }
    }

    auto carsInfo = Server->GetDriveAPI()->GetCarsData()->FetchInfo(visibleCarIds);
    for (const auto& [id, info]: carsInfo.GetResult()) {
        fillFiltersFromSpecifications(info.GetSpecifications(), filters, locale);
    }

    NJson::TJsonValue jFilters;
    jFilters["model"] = std::move(filters.JModels);
    jFilters["category"] = std::move(filters.JCategories);
    jFilters["exterior_type"] = std::move(filters.JExteriorTypes);
    jFilters["transmission"] = std::move(filters.JTransmissions);
    jFilters["fuel_type"] = std::move(filters.JFuelTypes);
    jFilters["air_conditioning"] = std::move(filters.JAirConditions);

    g.MutableReport().AddReportElement("filters", std::move(jFilters));
    g.SetCode(HTTP_OK);
}
