#include "processor.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>
#include <rtline/util/types/uuid.h>

void TCarModelCreateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Models);
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    R_ENSURE(
          requestData.Has("models") && requestData["models"].IsArray() && !requestData["models"].GetArray().empty()
        , HTTP_INTERNAL_SERVER_ERROR
        , "incorrect json, must be array 'object'"
        ,  NDrive::MakeError("car_model.incorrect_request")
        , tx
    );
    auto modelsAlreadyExists = GetValue<bool>(Context->GetCgiParameters(), "models_already_exists", false).GetOrElse(false);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    auto jArr = requestData["models"].GetArray();
    TVector<TString> modelCodes;
    for (auto& jObj: jArr) {
        TDriveModelData modelData;
        TString code;
        if (!jObj.Has("code")) {
            code = NUtil::CreateUUID();
            SubstGlobal(code, "-", "");
            jObj["code"] = code;
        } else {
            code = GetString(jObj, "code");
        }
        modelCodes.push_back(code);
        R_ENSURE(
              modelCodes.back()
            , ConfigHttpStatus.SyntaxErrorStatus
            , "empty code"
            , NDrive::MakeError("car_model.incorrect_request")
            , tx);
        if (!modelsAlreadyExists) {
            R_ENSURE(
                  modelData.Patch(jObj)
                , ConfigHttpStatus.SyntaxErrorStatus
                , "unable to parse"
                , NDrive::MakeError("car_model.incorrect_request")
                , tx);
            modelData.SetDeprecated(false);
            R_ENSURE(
                  Server->GetDriveAPI()->GetModelsData()->Insert(modelData, tx)
                , HTTP_INTERNAL_SERVER_ERROR
                , "cannot write car model to database"
                , NDrive::MakeError("car_model.write_model_error")
                , tx);
        }
    }

    TMessagesCollector errors;
    for (const auto& modelCode: modelCodes) {
        R_ENSURE(
              aclCompany->AddEntityObject(GetEntityType(), modelCode, permissions, errors)
            , HTTP_FORBIDDEN
            , "cannot add object to company, " << errors.GetStringReport()
            , NDrive::MakeError("acl.add")
            , tx
        );
    }
    R_ENSURE(
          Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx)
        , {}
        , "cannot update acl tag"
        , NDrive::MakeError("acl.update")
        , tx
    );
    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

    g.SetCode(HTTP_OK);
}
