#include "processor.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>

void TCarModelUpdateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Models);
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    R_ENSURE(
          requestData.Has("models") && requestData["models"].IsArray() && !requestData["models"].GetArray().empty()
        , HTTP_INTERNAL_SERVER_ERROR
        , "incorrect json, must be array 'object'"
        ,  NDrive::MakeError("car_model.update.incorrect_request")
        , tx
    );

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    auto jArr = requestData["models"].GetArray();
    TVector<TString> modelCodes;
    auto existingModelCodes = aclCompany->GetEntityObjects(GetEntityType(), permissions);
    for (const auto& jObj: jArr) {
        TString modelCode = GetString(jObj, "code");
        modelCodes.push_back(modelCode);
        R_ENSURE(
              modelCode
            , ConfigHttpStatus.SyntaxErrorStatus
            , "empty code"
            , NDrive::MakeError("car_model.incorrect_request")
            , tx);
        R_ENSURE(
              existingModelCodes.contains(modelCode)
            , HTTP_INTERNAL_SERVER_ERROR
            , "code has not found in acl tag"
            , NDrive::MakeError("car_model.model_not_found")
            , tx);
        auto modelData = Server->GetDriveAPI()->GetModelsData()->Fetch(modelCode, tx);
        R_ENSURE(
              modelData
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot fetch models"
            , NDrive::MakeError("car_model.read_model_error")
            , tx);

        R_ENSURE(
              modelData->Patch(jObj)
            , ConfigHttpStatus.SyntaxErrorStatus
            , "unable to parse"
            , NDrive::MakeError("car_model.incorrect_request")
            , tx);
        R_ENSURE(
              Server->GetDriveAPI()->GetModelsData()->Upsert(*modelData, tx)
            , HTTP_INTERNAL_SERVER_ERROR
            , "cannot write car model to database"
            , NDrive::MakeError("car_model.write_model_error")
            , tx);
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

    g.SetCode(HTTP_OK);
}
