#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/rental/rental_offer_holder_tag.h>
#include <drive/backend/data/rental/rental_service_mode_tag.h>
#include <drive/backend/data/rental/timetable_builder.h>
#include <drive/backend/database/config.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/raw_text/phone_number.h>

namespace {
constexpr TDuration year = TDuration::Days(365);
const TString userAlreadyExists = "rental.book.user_already_exists";
}

void TOffersBookRentalProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TEventsGuard egProcess(g.MutableReport(), "rental/book");

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::User);
    const TString offerId = GetString(requestData, "offer_id", true);
    TString clientId = GetString(requestData, "client_id", false);

    auto session = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    EUserIdStatus clientIdStatus = EUserIdStatus::Created;
    if (clientId.empty()) {
        std::tie(clientIdStatus, clientId) = GetClientId(permissions, session, requestData);
    }

    R_ENSURE(clientIdStatus != EUserIdStatus::Exists, HTTP_BAD_REQUEST, clientId, NDrive::MakeError(userAlreadyExists), session);

    if (!TUserProblemTag::EnsureNotBlocked(clientId, *Server, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (!TRentalOfferHolderTag::LockUser(clientId, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    auto offer = GetOffer(g, permissions, offerId, session);

    if (!TRentalOfferHolderTag::LockCar(offer, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    const auto& userTagManager = Server->GetDriveDatabase().GetTagsManager().GetUserTags();
    auto [userTagNames, carTagNames] = TRentalOfferHolderTag::GetRentalTagNamesByPermissions(*permissions);

    TTagEventsManager::TQueryOptions queryOptions;
    queryOptions.SetTags(userTagNames);
    queryOptions.SetOrderBy(TVector<TString>{"history_timestamp"});

    const auto userTagEvents = userTagManager.GetEvents({}, {Context->GetRequestStartTime() - year}, session, queryOptions);
    R_ENSURE(userTagEvents, ConfigHttpStatus.UnknownErrorStatus, "cannot get events", session);

    queryOptions.SetTags(carTagNames);
    const auto carTagEvents = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetEvents({}, {Context->GetRequestStartTime() - year}, session, queryOptions);
    R_ENSURE(carTagEvents, ConfigHttpStatus.UnknownErrorStatus, "cannot get events", session);

    TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
    R_ENSURE(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable,
                                                                                                     *userTagEvents,
                                                                                                     *carTagEvents,
                                                                                                     offer->GetSince(),
                                                                                                     offer->GetUntil(),
                                                                                                     *permissions,
                                                                                                     Server,
                                                                                                     session),
                                                                                                     ConfigHttpStatus.UnknownErrorStatus,
                                                                                                     "cannot build timetable",
                                                                                                     session);

    const auto conflictStatus = TTimetableBuilder::Instance().HasBookingTimetableConflicts(carsTimetable, offer->GetObjectId(), offer->GetSince(), Server, session, clientId);
    R_ENSURE(!conflictStatus.first, HTTP_CONFLICT, conflictStatus.second, NDrive::MakeError(conflictStatus.second), session);

    auto client = DriveApi->GetUserManager().RestoreUser(clientId, session);
    R_ENSURE(client, {}, "cannot restore user: " << clientId, session);
    auto clientRoles = DriveApi->GetUserManager().GetRoles().RestoreUserRoles(clientId, session);
    R_ENSURE(clientRoles, {}, "cannot restore user roles: " << clientId, session);
    auto clientTags = DriveApi->GetTagsManager().GetUserTags().RestoreObject(clientId, session);
    R_ENSURE(clientTags, {}, "cannot restore tagged user: " << clientId, session);

    auto clientPermissions = DriveApi->GetRolesManager()->BuildUsersPermissions(
        *client,
        *clientRoles,
        *clientTags,
        DriveApi->GetTagsManager(),
        DriveApi->GetConfig().GetRolesFeaturesConfig()
    );
    R_ENSURE(clientPermissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << clientId, session);

    TChargableTag::TBookOptions bookOptions;
    bookOptions.DeviceId = GetDeviceId();
    bookOptions.Futures = false;
    bookOptions.CheckBlocked = false;
    bookOptions.MultiRent = true;

    offer->SetUserId(clientId);

    auto bookedTag = TChargableTag::Book(offer, *clientPermissions, *Server, session, bookOptions);
    R_ENSURE(bookedTag, {}, "cannot book offer", session);
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.AddReportElement("client_id", clientId);
    g.SetCode(HTTP_OK);
}

TAtomicSharedPtr<TRentalOffer> TOffersBookRentalProcessor::GetOffer(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions,
                                                                    const TString& offerId, NDrive::TEntitySession& session) {
    auto asyncOffer = Server->GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session);
    if (!asyncOffer.Initialized()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    {
        TEventsGuard eg(g.MutableReport(), "WaitRestoreOffer");
        R_ENSURE(asyncOffer.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "RestoreOffer timeout");
    }
    auto baseOffer = asyncOffer.GetValue();
    R_ENSURE(baseOffer, ConfigHttpStatus.EmptySetStatus, "null offer restored");

    auto offer = std::dynamic_pointer_cast<TRentalOffer>(baseOffer);
    R_ENSURE(offer, ConfigHttpStatus.EmptySetStatus, "wrong-typed offer restored");

    return offer;
}

namespace {
    using TUserId = TString;
}

std::pair<TOffersBookRentalProcessor::EUserIdStatus, TUserId> TOffersBookRentalProcessor::GetClientId(
    TUserPermissions::TPtr permissions, NDrive::TEntitySession& session, const NJson::TJsonValue& requestData) {
    auto phoneNumber = GetString(requestData, "phone_number", true);
    auto firstName = GetString(requestData, "first_name", true);
    auto lastName = GetString(requestData, "last_name", false);
    auto email = GetString(requestData, "email", false);
    const bool validatePhoneNumber = GetHandlerSetting<bool>("validate_phone_number").GetOrElse(true);
    const bool forceShowObjectsWithoutTags = GetHandlerSetting<bool>("force_show_objects_without_tags").GetOrElse(false);

    if (validatePhoneNumber) {
        R_ENSURE(
            phoneNumber.StartsWith('+'),
            HTTP_BAD_REQUEST,
            "phone_number " << phoneNumber << " should start with +",
            NDrive::MakeError("incorrect_phone_number_format_plus_required"),
            session
        );
        TPhoneNormalizer phoneNormalizer;
        phoneNumber = phoneNormalizer.TryNormalize(phoneNumber);
    }

    const auto& api = *Yensured(Server->GetDriveAPI());

    auto optionalUsers = api.GetUsersData()->GetUsersByPhone(phoneNumber, session);
    R_ENSURE(optionalUsers, ConfigHttpStatus.UnknownErrorStatus, "cannot get users", session);

    TUsersDB::TOptionalUser optionalUser;

    auto findUser = [&api, &optionalUsers, &permissions, &session, &forceShowObjectsWithoutTags]()->TUsersDB::TOptionalUser {
        if (!optionalUsers->empty()) {
            for (const auto& user: *optionalUsers) {
                auto taggedObject = api.GetTagsManager().GetUserTags().GetCachedOrRestoreObject(user.GetUserId(), session);
                R_ENSURE(taggedObject, {}, "cannot GetCachedOrRestoreObject " << user.GetUserId(), session);
                if (permissions->GetVisibility(*taggedObject, NEntityTagsManager::EEntityType::User, nullptr, forceShowObjectsWithoutTags) == TUserPermissions::EVisibility::Visible) {
                    return user;
                }
            }
        }
        return {};
    };

    optionalUser = findUser();

    if (!optionalUser && !email.empty()) {
        optionalUsers = api.GetUsersData()->GetUsersByEmail(email, session);
        R_ENSURE(optionalUsers, ConfigHttpStatus.UnknownErrorStatus, "cannot get users", session);
        optionalUser = findUser();
    }

    if (!optionalUser) {
        NDrive::TExternalUser externalUser;
        externalUser
            .SetFirstName(firstName)
            .SetLastName(lastName)
            .SetEmail(email)
            .SetPhone(phoneNumber)
        ;
        optionalUser = api.GetUsersData()->RegisterExternalUser(permissions->GetUserId(), session, externalUser);

        R_ENSURE(optionalUser && *optionalUser, ConfigHttpStatus.UnknownErrorStatus, "cannot create user", session);

        auto commonTagNamesString = GetHandlerSetting<TString>("common_tag_names");
        auto commonActionsIdsStr = GetHandlerSetting<TString>("common_actions_ids");

        TVector<TString> tagNames;
        if (commonTagNamesString) {
            tagNames = StringSplitter(*commonTagNamesString).SplitBySet(", ").SkipEmpty();
        }

        TMap<TString, TVector<TString>> dmRoleToCommonActionsIds;
        if (commonActionsIdsStr) {
            NJson::TJsonValue jsonValue;
            R_ENSURE(NJson::ReadJsonFastTree(*commonActionsIdsStr, &jsonValue), HTTP_INTERNAL_SERVER_ERROR, "can't parse common actions ids");
            R_ENSURE(NJson::ParseField(jsonValue, NJson::Dictionary(dmRoleToCommonActionsIds)), HTTP_INTERNAL_SERVER_ERROR, "common actions ids have wrong format");
        }

        NDrivematics::TUserOrganizationAffiliationTag::AddRoleAndTagsToUser(*optionalUser, tagNames, dmRoleToCommonActionsIds, permissions, Server, session);
    } else {
        return {EUserIdStatus::Exists, optionalUser->GetUserId()};
    }

    return {EUserIdStatus::Created, optionalUser->GetUserId()};
}
