#include "processor.h"

#include "callback.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/condition.h>
#include <drive/backend/chat_robots/robot_manager.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/taxi/chat/client.h>

#include <rtline/library/json/cast.h>

ui32 TExternalChatMessageProcessor::TExternalMessage::GetMessageTraits() const {
    switch (MessageType) {
    case (TExternalChatMessageProcessor::EExternalMessageType::FromUser):
        return 0;
    case (TExternalChatMessageProcessor::EExternalMessageType::Comment):
        return (ui32) NDrive::NChat::TMessage::EMessageTraits::StaffOnly;
    default:
        return 0;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TExternalChatMessageProcessor::TExternalMessage& result) {
    TString text;
    TString id;
    ui32 type;
    if (!NJson::ParseField(value["text"], text, false)
        || !NJson::ParseField(value["id"], id, true)
        || !NJson::ParseField(value["timestamp"], result.MutableSendTime(), true)
        || !NJson::ParseField(value["type"], type, true)
        || !NJson::ParseField(value["attachments_id"], result.MutableAttachmentIds(), false))
    {
        return false;
    }
    NDrive::NChat::TMessage message;
    result.SetMessageType(TExternalChatMessageProcessor::EExternalMessageType(type));
    message.SetType(NDrive::NChat::IMessage::EMessageType::Plaintext);
    message.SetTraits(result.GetMessageTraits());
    message.SetText(std::move(text));
    message.SetExternalId(std::move(id));
    message.SetExternalStatus(NDrive::NChat::TMessage::EExternalStatus::Sent);
    result.SetMessage(std::move(message));
    return true;
}

void TExternalChatMessageProcessor::Parse(const NJson::TJsonValue& requestData) {
    Provider = GetString(requestData, "provider", true);
    ExternalChatId = GetString(requestData, "provider_chat_id", true);
    Closed = GetValue<bool>(requestData, "closed", false).GetOrElse(false);
    ReadMessagesUntil = GetTimestamp(requestData, "read_messages_ts", TInstant::Zero());
    R_ENSURE(NJson::ParseField(requestData["messages"], ExternalMessages, false), ConfigHttpStatus.SyntaxErrorStatus, "can't read messages data");
    if (requestData.Has("provider_meta")) {
        ChatMeta = requestData["provider_meta"];
    }
}

void TExternalChatMessageProcessor::FillHandlerSettingsValues() {
    ChatTagName = GetHandlerSettingDef<TString>(Provider + ".chat_tag_name", "support_tag_external");
    ChatId = GetHandlerSettingDef<TString>(Provider + ".chat_id", "support");
    OperatorId = GetHandlerSettingDef<TString>(Provider + ".operator_id", "robot-frontend");
    InitialNode = GetHandlerSettingDef<TString>(Provider + ".chat_initial_node", "intro");
    R_ENSURE(ChatTagName.size(), ConfigHttpStatus.SyntaxErrorStatus, "chat tag name not provided");
}

TMaybe<std::pair<TDBTag, TString>> TExternalChatMessageProcessor::GetSupportChatTagAndTopicLink(const TString& userId, const TString& topic) const {
    TVector<TDBTag> tags;
    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {}, tags, session), ConfigHttpStatus.SyntaxErrorStatus, "Can't restore tags " + session.GetStringReport());
    for (auto&& tag : tags) {
        auto tagImpl = tag.GetTagAs<TSupportChatTag>();
        if (tagImpl && tagImpl->GetExternalChatInfo().MakeChatTopic() == topic) {
            return std::make_pair(std::move(tag), tagImpl->GetTopicLink());
        }
        auto containerTag = tag.GetTagAs<TUserContainerTag>();
        if (containerTag) {
            ITag::TPtr containerImpl = containerTag->RestoreTag();
            auto supportChatTagImpl = dynamic_cast<TSupportChatTag*>(containerImpl.Get());
            if (supportChatTagImpl && supportChatTagImpl->GetExternalChatInfo().MakeChatTopic() == topic) {
                return std::make_pair(std::move(tag), supportChatTagImpl->GetTopicLink());
            }
        }
    }
    return {};
}

void AcceptResourceAndSendMessage(const TVector<NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment>>& futures, const TSimpleUnencryptedMediaStorage* mediaStorage, IChatRobot::TPtr chatRobot, const TString& userId, const TString& topic, IServerReportBuilder::TPtr report) {
    TString messageText;
    TMap<TString, TVector<ITaxiChatClient::TTaxiChatAttachment>> attachmentByMessageId;
    TMap<TString, TString> attachmentMessageByMessageId;
    {
        auto session = Yensured(chatRobot)->BuildChatEngineSession();
        for (auto&& future : futures) {
            auto attachmentResult = future.GetValue();
            if (!Yensured(mediaStorage)->RegisterResource(userId, attachmentResult.GetId(), attachmentResult.GetContentType(), false, session)) {
                throw yexception() << "Can't register resource: " << attachmentResult.GetId() << ". Session report: " << session.GetStringReport();
            }
            attachmentByMessageId[attachmentResult.GetExternalMessageId()].emplace_back(attachmentResult);
            if (attachmentMessageByMessageId.contains(attachmentResult.GetExternalMessageId())) {
                attachmentMessageByMessageId[attachmentResult.GetExternalMessageId()] += "," + attachmentResult.GetId();
            } else {
                attachmentMessageByMessageId[attachmentResult.GetExternalMessageId()] = attachmentResult.GetId();
            }
        }
        if (!session.Commit()) {
            throw yexception() << "Can't commit register resources session " << session.GetStringReport();
        }
    }
    for (const auto& [messageId, attachments] : attachmentByMessageId) {
        TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback(new TExternalChatAttachmentCallback(report, chatRobot, attachments.size(), attachmentMessageByMessageId[messageId], userId, topic, messageId));
        for (auto&& attach : attachments) {
            Yensured(mediaStorage)->UploadResource(userId, attach.GetId(), attach.GetContentType(), attach.GetContent(), callback);
        }
    }
}

void TExternalChatMessageProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = permissions->GetUserId();
    FillHandlerSettingsValues();
    TString topic = TSupportChatTag::TExternalChatInfo::MakeChatTopic(ExternalChatId, Provider);
    auto tagWithTopic = GetSupportChatTagAndTopicLink(userId, topic);
    bool externalChatTagFound = tagWithTopic.Defined();
    if (externalChatTagFound) {
        auto& [tag, topicLink] = *tagWithTopic;
        auto session = BuildTx<NSQL::Writable>();
        if (Closed) {
            R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple({ tag.GetTagId() }, userId, session, false) && session.Commit(), ConfigHttpStatus.UnknownErrorStatus, session.GetStringReport());
        } else {
            if (!IChatScriptAction::MatchingUndefer(tag, topicLink, TSupportChatTag::TypeName, false, Server, session) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            IChatRobot::ParseTopicLink(topicLink, ChatId, topic);
        }
    }

    if (Closed) {
        g.SetCode(HTTP_OK);
        return;
    }

    R_ENSURE(!ReadMessagesUntil || externalChatTagFound, ConfigHttpStatus.SyntaxErrorStatus, "no chat found to mark messages read");

    if (!externalChatTagFound) {
        R_ENSURE(ChatId && InitialNode && OperatorId, ConfigHttpStatus.SyntaxErrorStatus, "chat parameters not configured");
        TString comment = GetHandlerSettingDef<TString>(Provider + ".default_comment", "");
        if (ExternalMessages.size() && ExternalMessages.begin()->GetMessageType() == EExternalMessageType::Comment) {
            comment = ExternalMessages.begin()->GetMessage().GetText();
            ExternalMessages.erase(ExternalMessages.begin());
        }
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(ChatTagName, comment);
        R_ENSURE(tag, ConfigHttpStatus.SyntaxErrorStatus, "can not create chat tag " + ChatTagName);
        auto tagImpl = dynamic_cast<TSupportChatTag*>(tag.Get());
        R_ENSURE(tagImpl, ConfigHttpStatus.SyntaxErrorStatus, "tag is not support chat tag");
        tagImpl->SetExternalChatInfo(TSupportChatTag::TExternalChatInfo(ExternalChatId, Provider));
        tagImpl->SetTopicLink(ChatId + "." + topic);
        tagImpl->SetInitialNode(InitialNode);
        tagImpl->SetMeta(ChatMeta);
        auto session = BuildTx<NSQL::Writable>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, OperatorId, userId, Server, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
    }

    auto chatRobot = Server->GetChatRobot(ChatId);
    R_ENSURE(chatRobot, ConfigHttpStatus.SyntaxErrorStatus, "unknown 'chat_id'");
    R_ENSURE(!chatRobot->GetChatConfig().GetIsStatic(), ConfigHttpStatus.SyntaxErrorStatus, "can't send messages to static chat");

    if (ReadMessagesUntil) {
        auto session = BuildChatSession(false);
        auto messageId = chatRobot->GetFirstEventIdByTime(userId, topic, ReadMessagesUntil, true, session);
        if (!messageId || (*messageId > 0 && !chatRobot->UpdateLastViewedMessageId(userId, topic, userId, *messageId, session))) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    TMap<TString, TString> attachmentIds;
    if (ExternalMessages.size()) {
        TVector<NDrive::NChat::TMessage> messages;
        for (auto extMessage : ExternalMessages) {
            if (extMessage.GetMessage().GetText()) {
                messages.push_back(std::move(extMessage.GetMessage()));
            }
            for (auto&& attachmentId : extMessage.GetAttachmentIds()) {
                attachmentIds[attachmentId] = extMessage.GetMessage().GetExternalId();
            }
        }
        NDrive::TEntitySession session = chatRobot->BuildChatEngineSession(false);
        R_ENSURE(chatRobot->AcceptMessages(userId, topic, messages, userId, session), ConfigHttpStatus.SyntaxErrorStatus, "can't accept messages");
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
    }

    if (attachmentIds.size()) {
        auto taxiChatClient = Server->GetTaxiChatClient();
        R_ENSURE(taxiChatClient, ConfigHttpStatus.SyntaxErrorStatus, "taxi client not configured");
        auto mediaStorage = Server->GetChatRobotsManager()->GetChatRobotsMediaStorage();
        R_ENSURE(mediaStorage, ConfigHttpStatus.ServiceUnavailable, "chat media storage not configured");

        TVector<NThreading::TFuture<ITaxiChatClient::TTaxiChatAttachment>> futures;
        for (auto&& [attachmentId, externalId] : attachmentIds) {
            futures.push_back(taxiChatClient->GetAttachment(ExternalChatId, attachmentId, externalId));
        }

        auto report = g.GetReport();
        g.Release();
        NThreading::WaitExceptionOrAll(futures).Subscribe([futures, report, mediaStorage, chatRobot, userId, topic](const NThreading::TFuture<void>& waiter) {
            TMaybe<TString> error;
            if (waiter.HasValue()) {
                try {
                    AcceptResourceAndSendMessage(futures, mediaStorage, chatRobot, userId, topic, report);
                } catch (const yexception& e) {
                    error = e.what();
                }
            } else if (waiter.HasException()) {
                error = NThreading::GetExceptionMessage(waiter);
            } else {
                error = "WaitExceptionOrAll future has no value";
            }
            if (error) {
                TJsonReport::TGuard g(report, HTTP_OK);
                g.AddReportElement("error", *error);
                g.SetCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            }
        });
    } else {
        g.SetCode(HTTP_OK);
    }
}
