#include "fetchers.h"

#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/processor/processor.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/maps_router/router.h>
#include <drive/library/cpp/threading/future_cast.h>

bool TNearestCarsDetector::Initialize(const TGeoCoord& source, const IRequestProcessor* processor) {
    Source = source;
    Deadline = processor->GetContext()->GetRequestDeadline();
    Sparcify = processor->GetHandlerSettingDef("sparcity.enabled", false);
    SparcifyProximityThreshold = processor->GetHandlerSettingDef<double>("sparcity.proximity_threshold", 50);

    TVector<NExternalOfferImpl::TCarInfo> cars;
    const TCgiParameters& cgi = processor->GetContext()->GetCgiParameters();
    TMaybe<TGeoRect> bbox = processor->GetValue<TGeoRect>(cgi, "bbox", false);
    if (bbox) {
        auto bboxGrowDistance = processor->GetHandlerSetting<float>("external_offer.bbox_grow_distance").GetOrElse(100);
        bbox->GrowDistance(bboxGrowDistance);
    } else {
        auto fetchCarsDistance = processor->GetHandlerSetting<float>("external_offer.fetch_cars_distance").GetOrElse(1000.f);
        bbox = Source;
        bbox->GrowDistance(fetchCarsDistance);
    }
    {
        WalkingDistanceLimit = processor->GetHandlerSettingDef<double>("external_offer.walking_distance_limit", 500);
        TGeoRect limitingBoundingBox = Source;
        limitingBoundingBox.GrowDistance(WalkingDistanceLimit);
        TGeoRect finalBoundingBox;
        if (bbox->Cross(limitingBoundingBox, finalBoundingBox)) {
            FinalBoundingBox = finalBoundingBox;
        }
    }
    StringSplitter(processor->GetHandlerSettingDef<TString>("external_offer.status_filter", "")).SplitBySet(", ").SkipEmpty().Collect(&StatusFilterSet);
    return true;
}

NExternalOfferImpl::TCarsInfo TNearestCarsDetector::FetchCars(const TGeoRect& rect) const {
    TEventsGuard egWalkingRoutes(Report, "fetching_cars");
    NExternalOfferImpl::TCarsInfo result;
    TSet<TString> carIds;
    {
        auto geoHash = Server->GetSnapshotsManager().GetGeoHash();
        if (geoHash) {
            TEventsGuard eg(Report, "geo_hash_lookup");
            auto actor = [&](const TDevicesSnapshotManager::THashedDevice& d) {
                if (carIds.insert(d.CarId).second) {
                    result.push_back({
                        d.CarId,
                        d.ModelId,
                        d.Location,
                        TDuration::Max(),
                        Nothing()
                        });
                }
                return true;
            };
            geoHash->FindObjects(rect, actor);
        } else if (Report) {
            Report->AddEvent("GeoHash is missing");
        }
    }
    if (carIds.empty()) {
        return {};
    }
    egWalkingRoutes.AddEvent(NJson::TMapBuilder("cars", NJson::ToJson(carIds)));

    if (!StatusFilterSet.empty()) {
        TEventsGuard eg(Report, "status_filtering");
        auto statuses = Server->GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
        auto survivors = NExternalOfferImpl::TCarsInfo();
        for (auto&& carInfo : result) {
            const auto& id = carInfo.Id;
            const auto& status = statuses[id];
            if (StatusFilterSet.contains(status)) {
                survivors.push_back(std::move(carInfo));
            } else if (Report) {
                Report->AddEvent(TStringBuilder() << "filter out " << id << " due to status " << status);
            }
        }
        result = std::move(survivors);
    }

    return result;
}

bool TNearestCarsDetector::GetCars(const ui32 carsLimit, TVector<NExternalOfferImpl::TCarInfo>& result) const {
    TVector<NExternalOfferImpl::TCarInfo> cars;
    if (FinalBoundingBox) {
       cars = FetchCars(*FinalBoundingBox);
    }
    bool useStubRouter = Server ? Server->GetSettings().GetValue<bool>("stub_router.enabled").GetOrElse(false) : false;
    if (!useStubRouter && UseRoutingMatrix) {
        TEventsGuard eg(Report, "routing_matrix");
        auto from = NContainer::Scalar(Source);
        size_t countLimit = Server ? Permissions.GetSetting<size_t>("pedestrian_router.count_limit", 100) : 100;
        if (cars.size() > countLimit) {
            Report->AddEvent(NJson::TMapBuilder
                ("event", "DroppingCars")
                ("from_count", NJson::ToJson(cars.size()))
                ("to_count", NJson::ToJson(countLimit))
            );
            for (size_t i = 0; i < cars.size(); i++) {
                cars[i].WalkingDistance = Source.GetLengthTo(cars[i].Location.GetCoord());
            }
            std::sort(cars.begin(), cars.end(), [](const NExternalOfferImpl::TCarInfo& lhs, const NExternalOfferImpl::TCarInfo& rhs) {
                return lhs.WalkingDistance < rhs.WalkingDistance;
            });
            cars.erase(cars.begin() + countLimit, cars.end());
        }
        auto to = TVector<TGeoCoord>();
        for (auto&& car : cars) {
            to.push_back(car.Location.GetCoord());
        }
        if (to.empty()) {
            result = {};
            return true;
        }
        auto router = Server->GetPedestrianRouter();
        if (!router) {
            Report->AddEvent("PedestrianRoutingMissing");
            return false;
        }
        NThreading::TFuture<NGraph::TBaseRouter::TRoutingMatrix> asyncRoutingMatrix = router->GetMatrix(from, to);

        if (!asyncRoutingMatrix.Wait(Deadline) || !asyncRoutingMatrix.HasValue()) {
            Report->AddEvent(NJson::TMapBuilder
                ("event", "GetRoutingMatrixError")
                ("async_routing_matrix", NJson::ToJson(asyncRoutingMatrix))
                ("from", NJson::ToJson(from))
                ("to", NJson::ToJson(to))
            );
            return false;
        }
        auto routingMatrix = asyncRoutingMatrix.ExtractValue();

        NExternalOfferImpl::TCarsInfo filtered;
        for (size_t i = 0; i < cars.size(); ++i) {
            auto& car = cars[i];
            auto route = routingMatrix.GetElement(0, i);
            if (!route) {
                eg.AddEvent(TStringBuilder() << car.Id << " filtered out by RoutingMatrix");
                continue;
            }
            if (route->Length > WalkingDistanceLimit) {
                eg.AddEvent(TStringBuilder() << car.Id << " filtered out by distance: " << route->Length);
                continue;
            }
            car.WalkingDistance = route->Length;
            car.WalkingTime = TDuration::Seconds(route->Length);
            eg.AddEvent(TStringBuilder() << "car: " << car.Id << ":" << car.ModelId << ":" << *car.WalkingDistance << ":" << car.Location.Longitude << ' ' << car.Location.Latitude);
            filtered.push_back(std::move(car));
        }
        std::sort(filtered.begin(), filtered.end(), [](const NExternalOfferImpl::TCarInfo& left, const NExternalOfferImpl::TCarInfo& right) {
            return left.WalkingDistance < right.WalkingDistance;
        });
        cars = std::move(filtered);
    } else {
        for (auto&& car : cars) {
            car.WalkingDistance = car.Location.GetCoord().GetLengthTo(Source);
            car.WalkingTime = TDuration::Seconds(*car.WalkingDistance);
        }
        std::sort(cars.begin(), cars.end(), [](const NExternalOfferImpl::TCarInfo& left, const NExternalOfferImpl::TCarInfo& right) {
            return left.WalkingDistance < right.WalkingDistance;
        });
    }
    if (Sparcify) {
        TEventsGuard eg(Report, "sparcify");

        TSet<TString> modelIds;
        TSet<TString> objectIds;
        NExternalOfferImpl::TCarsInfo rearranged;
        rearranged.reserve(cars.size());
        for (auto&& object : cars) {
            if (modelIds.contains(object.ModelId)) {
                continue;
            }
            bool proximity = false;
            for (auto&& i : rearranged) {
                if (i.Location.GetCoord().GetLengthTo(object.Location.GetCoord()) < SparcifyProximityThreshold) {
                    proximity = true;
                    break;
                }
            }
            if (proximity) {
                continue;
            }

            rearranged.push_back(object);
            objectIds.insert(object.Id);
            modelIds.insert(object.ModelId);

            if (rearranged.size() > carsLimit) {
                break;
            }
        }
        for (auto&& object : cars) {
            if (!objectIds.contains(object.Id)) {
                rearranged.push_back(std::move(object));
            }
        }
        cars = std::move(rearranged);
    }
    result = cars;
    return true;
}
