#include "processor.h"

#include "fetchers.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/taxi/suggest/client.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>
#include <rtline/util/logging/tskv_log.h>

#include <util/generic/utility.h>
#include <util/string/builder.h>

void TExternalOfferProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const ILocalization* localization = Server->GetLocalization();
    const ISettings& settings = Server->GetSettings();
    const bool fullReport = GetHandlerSetting<bool>("external_offer.full_report").GetOrElse(true);
    const auto locale = GetLocale();

    auto& report = g.MutableReport();
    auto applicationLink = GetHandlerSetting<TString>("external_offer.application_link").GetOrElse(settings.GetValueDef<TString>("external_offer.application_link", ""));
    report.AddReportElement("app_link", applicationLink);
    bool isRegistred = permissions ? permissions->IsRegistered() : false;
    report.AddReportElement("is_registred", isRegistred);

    auto tx = BuildTx<NSQL::Writable | NSQL::Deferred>();

    auto offersStorage = Server->GetOffersStorage();
    R_ENSURE(offersStorage, ConfigHttpStatus.UnknownErrorStatus, "OffersStorage is missing");

    auto previousOfferIds = GetStrings(requestData, "previous_offer_ids", false);
    auto previousOffers = TMap<TString, NThreading::TFuture<ICommonOffer::TPtr>>();
    for (auto&& offerId : previousOfferIds) {
        auto asyncOffer = offersStorage->RestoreOffer(offerId, permissions->GetUserId(), tx);
        if (asyncOffer.Initialized()) {
            previousOffers.emplace(offerId, std::move(asyncOffer));
        } else {
            g.AddEvent(NJson::TMapBuilder
                ("event", "RestoreOfferFailure")
                ("errors", tx.GetReport())
                ("offer_id", offerId)
            );
        }
        tx.ClearErrors();
    }

    auto originalPermissions = permissions;
    if (!permissions->GetUserFeatures().GetIsFallbackUser() && permissions->GetStatus() == NDrive::UserStatusOnboarding) {
        TString fallbackId;
        if (!fallbackId) {
            fallbackId = GetHandlerSetting<TString>("external_offer.onboarding_fallback_user_id").GetOrElse(TString());
        }
        if (!fallbackId) {
            fallbackId = GetSettings().GetValue<TString>("external_offer.onboarding_fallback_user_id").GetOrElse(TString());
        }
        if (fallbackId) {
            permissions = Server->GetDriveAPI()->GetUserPermissions(fallbackId, permissions->GetUserFeatures(), TInstant::Zero(), Context);
            R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "Failed to get user permissions for fallback user " << fallbackId, tx);
            SetSettings(permissions->SettingGetter());
        } else {
            report.AddReportElement("offers", NJson::JSON_ARRAY);
            g.SetCode(HTTP_OK);
            return;
        }
    }

    TUserOfferContext uhc(Server, permissions, Context);
    ReqCheckCondition(uhc.FillData(this, requestData), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_fill_data_error");
    const auto& cgi = Context->GetCgiParameters();
    uhc.SetCommonSettingsPrefix(GetTypeName());
    uhc.SetNeedTaxiPrice(false);
    uhc.SetFilterAccounts(originalPermissions->GetSetting<bool>(settings, "billing.filter_accounts").GetOrElse(true));
    uhc.SetExternalUserId(GetExternalUserId());
    uhc.SetLocale(locale);
    uhc.SetOrigin(GetOrigin());
    uhc.SetNeedPaymentMethods(GetHandlerSettingDef("need_payment_methods", true));

    {
        auto userLocation = GetUserLocation();
        auto externalUserLocation = GetUserLocation("src");
        R_ENSURE(userLocation || externalUserLocation, ConfigHttpStatus.UserErrorState, "either src or UserLocation should be specified");
        if (externalUserLocation) {
            uhc.SetExternalUserPosition(*externalUserLocation);
        }
        uhc.SetUserPosition(userLocation ? *userLocation : *externalUserLocation);
    }
    const TGeoCoord source = uhc.OptionalExternalUserPosition().GetOrElse(uhc.GetUserPositionUnsafe());
    const auto destinationCountLimit = GetValue<ui32>(cgi, "destination_count_limit", false).GetOrElse(
        GetHandlerSetting<ui32>("external_offer.taxi_suggest.limit").GetOrElse(3)
    );
    auto offerCountLimit = GetValue<ui32>(cgi, "offer_count_limit", false).GetOrElse(
        GetHandlerSetting<ui32>("external_offer.offers_number").GetOrElse(1)
    );
    const auto fast = GetValue<bool>(cgi, "fast", false).GetOrElse(false);
    const auto reportActiveSession = GetValue<bool>(cgi, "report_active_session", false).GetOrElse(
        GetHandlerSetting<bool>("external_offer.report_active_session").GetOrElse(false)
    );
    if (fast) {
        uhc.SetNeedFinishArea(false);
        uhc.SetNeedGeoFeatures(false);
        uhc.SetNeedUserFeatures(false);
        uhc.SetNeedHistoryFreeTimeFees(false);
    }

    if (Config.GetSuggestOfferReport()) {
        offerCountLimit = GetHandlerSettingDef<ui32>("offers_suggest.fetch_cars_limit", 100);
        uhc.SetNeedCarSuggest(true);
        uhc.ClearUserDestination();
    }

    if (!Config.GetUseCustomOfferReport() && !fast) {
        uhc.SetNeedYandexPaymentMethod(true);
    }

    g.AddEvent(NJson::TMapBuilder
        ("event", "QueryOptions")
        ("fast", fast)
        ("destination_count_limit", destinationCountLimit)
        ("offer_count_limit", offerCountLimit)
        ("report_active_session", reportActiveSession)
    );

    uhc.SetEnableTaxiSuggest(GetHandlerSetting<bool>("external_offer.taxi_suggest.enabled").GetOrElse(false));
    uhc.SetDestinationsCountLimit(destinationCountLimit);
    uhc.SetDeadline(Context->GetRequestDeadline());
    uhc.SetLocale(locale);
    uhc.SetOfferCountLimit(offerCountLimit);

    ReqCheckCondition(uhc.Prefetch(), ConfigHttpStatus.UnknownErrorStatus, "user_offer_context_prefetch_error");
    const auto encodeDeeplink = GetValue<bool>(cgi, "encode_deeplink", /*required=*/false).GetOrElse(false);
    const auto options = NUtil::TTSKVRecordParser::Parse<';', '='>(GetString(cgi, "options", false));
    if (permissions != originalPermissions) {
        g.AddEvent(NJson::TMapBuilder
            ("current_user_id", permissions->GetUserId())
            ("original_user_id", originalPermissions->GetUserId())
        );
    }

    if (reportActiveSession) {
        auto eg = g.BuildEventGuard("report_active_session");
        auto sessionBuilder = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
        auto sessions = sessionBuilder ? sessionBuilder->GetUserSessions(permissions->GetUserId()) : TVector<IEventsSession<TTagHistoryEvent>::TPtr>();

        NJson::TJsonValue activeSessionIds = NJson::JSON_ARRAY;
        for (auto&& session : sessions) {
            if (session && !session->GetClosed()) {
                activeSessionIds.AppendValue(session->GetSessionId());
            }
        }
        g.AddReportElement("active_session_ids", std::move(activeSessionIds));
    }

    NExternalOfferImpl::TCarsInfo cars;
    {
        THolder<TNearestCarsDetector> detector(new TNearestCarsDetector(Server, *permissions, &report));
        detector->SetUseRoutingMatrix(!fast);
        R_ENSURE(detector->Initialize(source, this), ConfigHttpStatus.UnknownErrorStatus, "cannot Initialize detector");
        R_ENSURE(detector->GetCars(offerCountLimit, cars), ConfigHttpStatus.UnknownErrorStatus, "cannot GetCars");
    }

    TString reason;
    if (reason.empty() && cars.empty()) {
        reason = "no_cars";
    }

    NExternalOfferImpl::TOffersInfo offers;
    NExternalOfferImpl::TOffersInfo complementary;
    if (Config.GetSuggestOfferReport()) {
        const auto kind = GetValue<TString>(cgi, "kind", false).GetOrElse("");
        // We should use special logic for destination_* suggests.
        auto isDestinationKind = kind.StartsWith("destination_");
        auto suggestOffers = BuildOffers(cars, uhc, complementary, tx, report);
        TSet<TString> stdOfferCars;
        for (auto report : suggestOffers) {
            auto offer = report ? report->GetOfferPtrAs<IOffer>() : nullptr;
            if (offer && offer->GetTypeName() == TStandartOffer::GetTypeNameStatic()) {
                stdOfferCars.insert(offer->GetObjectId());
            }
        }
        auto modelName = GetHandlerSettingDef<TString>("offers_suggest." + kind + ".offer_score_model", "");
        NDrive::TOfferModelConstPtr model = {};
        if (modelName) {
            auto modelsStorage = Server->GetModelsStorage();
            model = modelsStorage ? modelsStorage->GetOfferModel(modelName) : nullptr;
            R_ENSURE(
                model, ConfigHttpStatus.UnknownErrorStatus,
                "Car suggest model " << modelName << " does not exists"
            );
        }
        TVector<size_t> suggestPerm(suggestOffers.size());
        TVector<double> suggestScores(suggestOffers.size());
        TVector<size_t> suggestRanks(suggestOffers.size());
        const auto& hintedDestinations = uhc.GetHintedDestinations();
        for (size_t i = 0; i < suggestOffers.size(); i++) {
            auto offer = suggestOffers[i]->GetOfferAs<TFullPricesContext>();
            R_ENSURE(offer, ConfigHttpStatus.UnknownErrorStatus, "Invalid offer type");
            suggestPerm[i] = i;
            if (model) {
                suggestScores[i] = model->Calc(offer->MutableFeatures());
            }
            if (isDestinationKind && hintedDestinations) {
                suggestRanks[i] = hintedDestinations.size();
                if (auto fpOffer = suggestOffers[i]->GetOfferAs<TFixPointOffer>()) {
                    for (size_t j = 0; j < hintedDestinations.size(); j++) {
                        auto&& destination = hintedDestinations[j];
                        if (fpOffer->GetDestinationName() != destination.GetName()) {
                            continue;
                        }
                        suggestRanks[i] = j;
                        break;
                    }
                }
            }
        }
        if (model) {
            std::sort(
                suggestPerm.begin(), suggestPerm.end(),
                [&suggestOffers, &suggestScores, &suggestRanks](size_t lhs, size_t rhs) {
                    const auto& lhsOfferReport = suggestOffers[lhs];
                    const auto& rhsOfferReport = suggestOffers[rhs];
                    const auto lhsOffer = lhsOfferReport->GetOfferPtrAs<IOffer>();
                    const auto rhsOffer = rhsOfferReport->GetOfferPtrAs<IOffer>();
                    if (lhsOffer && rhsOffer && lhsOffer->GetObjectId() == rhsOffer->GetObjectId()) {
                        return
                            std::tuple(lhsOfferReport->GetListPriorityDef(0), lhsOfferReport->GetInternalPriorityDef(0)) <
                            std::tuple(rhsOfferReport->GetListPriorityDef(0), rhsOfferReport->GetInternalPriorityDef(0));
                    }
                    return
                        std::tuple(suggestRanks[lhs], -suggestScores[lhs]) <
                        std::tuple(suggestRanks[rhs], -suggestScores[rhs]);
                }
            );
        }
        TSet<TString> objectIds;
        offers.clear();
        auto eg = g.BuildEventGuard("car_suggest_offers");
        const size_t offersLimit = GetHandlerSettingDef<size_t>("offers_suggest." + kind + ".offers_limit", 5);
        for (size_t i = 0; i < suggestOffers.size(); i++) {
            auto offerReport = suggestOffers[suggestPerm[i]];
            auto offer = offerReport ? offerReport->GetOfferPtrAs<IOffer>() : nullptr;
            if (!offer) {
                continue;
            }
            // Ignore cars without standartOffer.
            if (!isDestinationKind && stdOfferCars.find(offer->GetObjectId()) == stdOfferCars.end()) {
                continue;
            }
            if (objectIds.find(offer->GetObjectId()) == objectIds.end() && objectIds.size() < offersLimit) {
                objectIds.insert(offer->GetObjectId());
            }
            if (objectIds.find(offer->GetObjectId()) == objectIds.end()) {
                eg.AddEvent(NJson::TMapBuilder
                    ("event", "SkippedOffer")
                    ("offer_id", offer->GetOfferId())
                    ("score", suggestScores[suggestPerm[i]])
                    ("rank", suggestRanks[suggestPerm[i]])
                    ("model", modelName)
                );
                continue;
            }
            offers.push_back(std::move(offerReport));
            eg.AddEvent(NJson::TMapBuilder
                ("event", "SelectedOffer")
                ("offer_id", offer->GetOfferId())
                ("score", suggestScores[suggestPerm[i]])
                ("rank", suggestRanks[suggestPerm[i]])
                ("model", modelName)
            );
        }
    } else {
        auto destinationOffers = BuildOffers(cars, uhc, complementary, tx, report);
        {
            auto eg = g.BuildEventGuard("remove_dup_offers");
            SortBy(destinationOffers, [](const NExternalOfferImpl::TOfferPtr& offer) -> ui32 {
                auto fixPointOffer = offer ? offer->GetOfferAs<TFixPointOffer>() : nullptr;
                if (fixPointOffer) {
                    return fixPointOffer->GetPackPrice();
                }
                auto standardOffer = offer ? offer->GetOfferAs<TStandartOffer>() : nullptr;
                if (standardOffer) {
                    return standardOffer->GetRiding().GetPrice();
                }
                return 0;
            });
            TSet<TString> objectIds;
            for (auto&& offerReport : destinationOffers) {
                auto offer = offerReport ? offerReport->GetOfferPtrAs<IOffer>() : nullptr;
                if (!offer) {
                    continue;
                }
                if (objectIds.insert(offer->GetObjectId()).second) {
                    offers.push_back(std::move(offerReport));
                } else {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "remove_dup_offer")
                        ("offer", NJson::ToJson(offer))
                    );
                }
            }
            destinationOffers = std::move(offers);
        }
        {
            TEventsGuard eg(report, "sort_offers");
            const auto priceWeight = GetHandlerSetting<float>("external_offer.price_weight").GetOrElse(1);
            const auto walkingWeight = GetHandlerSetting<float>("external_offer.walking_weight").GetOrElse(0);
            const auto ridingWeight = GetHandlerSetting<float>("external_offer.riding_weight").GetOrElse(0);
            SortBy(destinationOffers, [&cars, priceWeight, walkingWeight, ridingWeight](const NExternalOfferImpl::TOfferPtr& offer) {
                auto car = std::find_if(cars.begin(), cars.end(), [&offer](const NExternalOfferImpl::TCarInfo& c) {
                    return c.Id == offer->GetOfferPtrAs<IOffer>()->GetObjectId();
                });
                R_ENSURE(car != cars.end(), HTTP_INTERNAL_SERVER_ERROR, "cannot find car object " << offer->GetOfferPtrAs<IOffer>()->GetObjectId());

                if (auto fixPointOffer = offer->GetOfferAs<TFixPointOffer>()) {
                    return
                        priceWeight * fixPointOffer->GetPackPrice() +
                        ridingWeight * fixPointOffer->GetRouteDuration().Minutes() +
                        walkingWeight * car->WalkingTime.Minutes();
                }
                if (auto standardOffer = offer->GetOfferAs<TStandartOffer>()) {
                    return
                        priceWeight * standardOffer->GetRiding().GetPrice() +
                        walkingWeight * car->WalkingTime.Minutes();
                }
                return 1.f * car->WalkingTime.Minutes();
            });
        }
        offers = std::move(destinationOffers);
        if (!complementary.empty()) {
            TEventsGuard eg(report, "filter_complementary");
            auto coffers = std::move(complementary);
            for (auto&& offer : offers) {
                auto p = std::find_if(coffers.begin(), coffers.end(), [&offer](const auto& coffer) {
                    return offer && coffer && offer->GetConstructionId() == coffer->GetConstructionId();
                });
                if (p == coffers.end()) {
                    continue;
                }
                complementary.push_back(std::move(*p));
            }
        }
    }
    for (auto&& [offerId, asyncOffer] : previousOffers) {
        auto eg = g.BuildEventGuard("restore_offer:" + offerId);
        if (asyncOffer.Wait(uhc.GetDeadline()) && asyncOffer.HasValue() && asyncOffer.GetValue()) {
            auto offerReport = MakeAtomicShared<TStandartOfferReport>(asyncOffer.GetValue(), /*priceOfferConstructor=*/nullptr);
            auto objectId = offerReport->GetOfferPtrAs<IOffer>()->GetObjectId();
            for (auto&& i : offers) {
                auto offer = i ? i->GetOfferPtrAs<IOffer>() : nullptr;
                if (offer && offer->GetObjectId() == objectId && i->HasWalkingDuration()) {
                    offerReport->SetWalkingDuration(i->OptionalWalkingDuration());
                    break;
                }
            }
            offers.push_back(std::move(offerReport));
        } else {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "RestoreOfferFailure")
                ("async_offer", NJson::ToJson(asyncOffer))
            );
        }
    }
    if (reason.empty() && offers.empty()) {
        ui64 top = 0;
        for (auto&& [error, count] : ErrorCounts) {
            if (top < count) {
                top = count;
                reason = error;
            }
        }
    }

    TSet<TString> existingCars;
    for (auto&& offer : offers) {
        if (!offer) {
            continue;
        }
        auto vehicleOffer = offer->GetOfferPtrAs<IOffer>();
        if (!vehicleOffer) {
            continue;
        }
        existingCars.insert(vehicleOffer->GetObjectId());
    }

    bool forceStoreOffers = permissions->GetSetting<bool>(settings, "offer.default.force_store").GetOrElse(false);
    if (forceStoreOffers || isRegistred) {
        TEventsGuard eg(report, "store_offers");
        auto storedOffers = offersStorage->StoreOffers(offers, tx);
        if (!storedOffers.Initialized()) {
            eg.AddEvent(NJson::TMapBuilder
                ("event", "StoreOffersFailed")
                ("error", tx.GetReport())
            );
        }
    }

    auto deviceReportTraits = NDeviceReport::ReportExternal |
        NDeviceReport::ReportLocationCourse;
    deviceReportTraits &= permissions->GetDeviceReportTraits();
    const auto reportTraitsCgi = GetValue<TString>(cgi, "report", false).GetOrElse("all");
    const auto traits = GetValues<NDriveSession::EReportTraits>(cgi, "traits", false);
    NDriveSession::TReportTraits sessionReportTraits = NDriveSession::GetReportTraits(reportTraitsCgi);
    for (auto&& trait : traits) {
        sessionReportTraits |= trait;
    }

    TCarsFetcher fetcher(*Server, deviceReportTraits);
    fetcher.SetLocale(locale);
    {
        TEventsGuard eg(report, "fetch_data");
        fetcher.SetCheckVisibility(false);
        fetcher.SetIsRealtime(true);
        if (!fetcher.FetchData(permissions, existingCars)) {
            report.AddEvent("CarsFetcher::FetchData is unsuccessful");
        }
    }

    if (fullReport && (deviceReportTraits & NDeviceReport::ReportCars)) {
        TEventsGuard eg(report, "report_cars");
        report.AddReportElementString("cars", fetcher.GetAvailableCarsReportSafe(permissions->GetFilterActions()));
    } else {
        report.AddReportElement("cars", NJson::JSON_ARRAY);
    }
    if (fullReport && (deviceReportTraits & NDeviceReport::ReportModels)) {
        TEventsGuard eg(report, "report_models");
        report.AddReportElement("models", fetcher.GetModelsReportSafe());
    } else {
        report.AddReportElement("models", NJson::JSON_ARRAY);
    }
    if (fullReport && (deviceReportTraits & NDeviceReport::ReportViews)) {
        TEventsGuard eg(report, "report_views");
        report.AddReportElement("views", fetcher.GetViewsReportSafe());
    } else {
        report.AddReportElement("views", NJson::JSON_ARRAY);
    }
    if (fullReport && (deviceReportTraits & NDeviceReport::ReportPatches)) {
        TEventsGuard eg(report, "report_patches");
        report.AddReportElement("patches", fetcher.GetPatchesReportSafe());
    } else {
        report.AddReportElement("patches", NJson::JSON_ARRAY);
    }
    {
        TEventsGuard eg(report, "report_offers");
        auto deeplinkStub = GetHandlerSetting<TString>("external_offer.offer_deeplink_stub").GetOrElse("yandexdrive://nothing");
        auto deeplinkTemplate = GetHandlerSetting<TString>("external_offer.offer_deeplink_template").GetOrElse({});
        auto formatPriceSeparator = GetHandlerSetting<TString>("format_price_separator").GetOrElse({});
        auto partialReportRoudingPrecision = GetHandlerSetting<ui32>("external_offer.rounding_precision").GetOrElse(50 * 100);
        auto useWalkingDurationFromPin = GetHandlerSetting<bool>("external_offer.walking_duration_from_pin").GetOrElse(false);

        auto deeplinkTemplateOriginal = deeplinkTemplate;
        for (auto&& [key, value] : options) {
            auto macro = ToUpperUTF8(key);
            SubstGlobal(deeplinkTemplate, macro, value);
        }
        g.AddEvent(NJson::TMapBuilder
            ("event", "ReportOptions")
            ("deeplink_encode", encodeDeeplink)
            ("deeplink_stub", deeplinkStub)
            ("deeplink_template", deeplinkTemplate)
            ("deeplink_template_original", deeplinkTemplateOriginal)
            ("format_price_separator", formatPriceSeparator)
            ("locale", ToString(uhc.GetLocale()))
            ("partial_report_rounding_precision", partialReportRoudingPrecision)
            ("use_walking_duration_from_pin", useWalkingDurationFromPin)
        );

        auto accounts = uhc.GetUserAccounts();
        auto cards = uhc.GetPaymentMethods();
        auto paymentCards = TBillingManager::GetUserPaymentCards(cards.Get(), false);
        auto yandexAccount = permissions->UseYandexPaymentMethod(settings) ? TBillingManager::GetYandexAccount(cards.Get()) : Nothing();

        auto makeOfferReport = [&](const IOfferReport::TPtr offer) -> NJson::TJsonValue {
            const TString& objectId = offer->GetOfferPtrAs<IOffer>()->GetObjectId();
            const TString& offerId = offer->GetOffer()->GetOfferId();

            const TDriveCarInfo* carInfo = fetcher.GetCarInfo(objectId);
            if (!carInfo) {
                report.AddEvent("cannot find CarInfo for " + objectId);
                return {};
            }
            NJson::TJsonValue element;
            if (Config.GetUseCustomOfferReport()) {
                const TDriveModelData* modelData = fetcher.GetModelInfo(carInfo->GetModel());
                const TString& number = carInfo->GetNumber();
                if (fullReport) {
                    element["model_id"] = carInfo->GetModel();
                    element["number"] = number;
                    element["offer_type"] = offer->GetOffer()->GetTypeName();
                }
                if (fullReport && isRegistred) {
                    element["offer_id"] = offerId;
                }
                TString deeplink;
                if (fullReport && deeplinkTemplate) {
                    deeplink = deeplinkTemplate;
                    SubstGlobal(deeplink, "NUMBER", encodeDeeplink ? CGIEscapeRet(number) : number);
                    if (isRegistred) {
                        SubstGlobal(deeplink, "OFFER_ID", offerId);
                    } else {
                        SubstGlobal(deeplink, "offer_id=OFFER_ID", "");
                    }
                } else if (fullReport && isRegistred) {
                    deeplink = TStringBuilder() << carInfo->GetDeepLink() << "?offer_id=" << offerId;
                } else if (fullReport) {
                    deeplink = carInfo->GetDeepLink();
                } else {
                    deeplink = deeplinkStub;
                }

                auto car = std::find_if(cars.begin(), cars.end(), [&objectId](const NExternalOfferImpl::TCarInfo& c) {
                    return c.Id == objectId;
                });
                auto carWalkingTime = car != cars.end() ? car->WalkingTime : TDuration::Zero();
                TDuration walkingDuration = useWalkingDurationFromPin ? carWalkingTime : offer->GetWalkingDurationDef(carWalkingTime);
                element["deeplink"] = deeplink;
                element["model"] = modelData ? modelData->GetName() : TString();
                element["walking_duration"] = walkingDuration.Seconds();

                auto standardOffer = offer->GetOfferAs<TStandartOffer>();
                if (standardOffer) {
                    auto freeReservationDuration = standardOffer->GetFreeDuration(TChargableTag::Reservation);
                    element["free_reservation_duration"] = freeReservationDuration.Seconds();
                    if (localization) {
                        if (freeReservationDuration.Minutes()) {
                            element["localized_free_reservation_duration"] = localization->FormatDuration(locale, freeReservationDuration);
                        } else {
                            element["localized_free_reservation_duration"] = localization->GetLocalString(locale, "units.no");
                        }
                    }
                }

                if (auto fixPointOffer = offer->GetOfferAs<TFixPointOffer>()) {
                    auto cashbackPrediction = fixPointOffer->GetEffectiveCashback(*Server);
                    if (cashbackPrediction && !fixPointOffer->GetHiddenCashback()) {
                        element["cashback_prediction"] = NJson::ToJson(cashbackPrediction);
                    }
                    ui32 price = fixPointOffer->GetPublicDiscountedPrice(fixPointOffer->GetPackPrice(), "", 100);
                    if (!fullReport) {
                        price = ICommonOffer::RoundPrice(price, partialReportRoudingPrecision);
                        element["price_rounded"] = true;
                    }
                    element["price"] = price;
                    element["riding_duration"] = fixPointOffer->GetRouteDuration().Seconds();
                    if (fullReport) {
                        element["price_undiscounted"] = fixPointOffer->GetPublicOriginalPackPrice({}, 100);
                    }
                    if (localization) {
                        element["localized_price"] = localization->FormatPrice(locale, price, { "units.short." + fixPointOffer->GetCurrency() }, formatPriceSeparator);
                        element["localized_riding_duration"] = localization->FormatDuration(locale, fixPointOffer->GetRouteDuration());
                    }
                } else if (standardOffer) {
                    auto cashbackPercent = standardOffer->GetEffectiveCashbackPercent(*Server);
                    if (cashbackPercent && !standardOffer->GetHiddenCashback()) {
                        element["cashback_percent"] = NJson::ToJson(cashbackPercent);
                    }
                    ui32 acceptanceCost = standardOffer->GetPublicDiscountedAcceptance();
                    ui32 ridingPrice = standardOffer->GetPublicDiscountedRiding();
                    ui32 parkingPrice = standardOffer->GetPublicDiscountedParking();
                    if (!fullReport) {
                        ridingPrice = ICommonOffer::RoundPrice(ridingPrice, partialReportRoudingPrecision);
                        parkingPrice = ICommonOffer::RoundPrice(parkingPrice, partialReportRoudingPrecision);
                        element["price_rounded"] = true;
                    }
                    element["riding_price"] = ridingPrice;
                    element["parking_price"] = parkingPrice;
                    if (acceptanceCost) {
                        element["acceptance_cost"] = acceptanceCost;
                    }
                    if (fullReport) {
                        element["riding_price_undiscounted"] = standardOffer->GetPublicOriginalRiding();
                        element["parking_price_undiscounted"] = standardOffer->GetPublicOriginalParking();
                    }
                    if (acceptanceCost && localization) {
                        element["localized_acceptance_cost"] = localization->FormatPrice(locale, acceptanceCost, { "units.short." + standardOffer->GetCurrency(), "/", "units.short.minutes" }, formatPriceSeparator);
                        element["localized_acceptance_cost_short"] = localization->FormatPrice(locale, acceptanceCost, { "units.short." + standardOffer->GetCurrency() }, formatPriceSeparator);
                    }
                    if (localization) {
                        element["localized_riding_price"] = localization->FormatPrice(locale, ridingPrice, { "units.short." + standardOffer->GetCurrency(), "/", "units.short.minutes" }, formatPriceSeparator);
                        element["localized_riding_price_short"] = localization->FormatPrice(locale, ridingPrice, { "units.short." + standardOffer->GetCurrency() }, formatPriceSeparator);
                        element["localized_parking_price"] = localization->FormatPrice(locale, parkingPrice, { "units.short." + standardOffer->GetCurrency(), "/", "units.short.minutes" }, formatPriceSeparator);
                        element["localized_parking_price_short"] = localization->FormatPrice(locale, parkingPrice, { "units.short." + standardOffer->GetCurrency() }, formatPriceSeparator);
                    }
                }

                if (localization) {
                    auto resourceId = isRegistred
                        ? GetHandlerSetting<TString>("external_offer.maps_book_button").GetOrElse("external_offer.maps_book_button")
                        : GetHandlerSetting<TString>("external_offer.maps_register_button").GetOrElse("external_offer.maps_register_button");
                    auto roundedWalkingDuration = std::max(walkingDuration, TDuration::Minutes(1));
                    element["button_text"] = localization->GetLocalString(locale, resourceId);
                    element["localized_walking_duration"] = localization->FormatDuration(locale, roundedWalkingDuration);
                }
            } else {
                element = offer->BuildJsonReport(locale, sessionReportTraits, *Server, *permissions);
                element["car_number"] = carInfo->GetNumber();
                if (sessionReportTraits & NDriveSession::EReportTraits::ReportPaymentMethods) {
                    element.InsertValue("payment_methods", TBillingManager::GetPaymentMethodsReport(
                        permissions->GetSetting<bool>("show_bonus_payment_method").GetOrElse(false),
                        locale,
                        offer->GetOffer()->GetTimestamp(),
                        accounts,
                        MakeSet(offer->GetOffer()->GetChargableAccounts()),
                        paymentCards,
                        yandexAccount,
                        *Server,
                        permissions->GetMobilePaymentMethod(Server->GetSettings())
                    ));
                }
            }
            if (Config.GetSuggestOfferReport() && GetHandlerSettingDef<bool>("offers_suggest.offer_car_info", true)) {
                // Fixes bug on client app.
                if (!fetcher.BuildOneCarInfoReport(objectId, permissions->GetFilterActions(), element["car_info"])) {
                    report.AddEvent("cannot build car info for " + objectId);
                    return {};
                }
            }
            return element;
        };
        NJson::TJsonValue offrs = NJson::JSON_ARRAY;
        for (auto&& offer : offers) {
            auto element = makeOfferReport(offer);
            if (!element.IsDefined()) {
                continue;
            }
            offrs.AppendValue(std::move(element));
        }
        report.AddReportElement("offers", std::move(offrs));

        NJson::TJsonValue complemntry = NJson::JSON_ARRAY;
        for (auto&& offer : complementary) {
            auto element = makeOfferReport(offer);
            if (!element.IsDefined()) {
                continue;
            }
            complemntry.AppendValue(std::move(element));
        }
        if (!uhc.GetComplementaryInsuranceTypes().empty()) {
            report.AddReportElement("complementary", std::move(complemntry));
        }
    }

    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);

    if (reason && offers.empty()) {
        auto resourceId = TStringBuilder() << "external_offer.reason." << reason << ".message";
        auto message = localization ? localization->GetLocalString(locale, resourceId) : resourceId;
        report.AddReportElement("reason_code", reason);
        report.AddReportElement("reason", std::move(message));
    }

    bool isServiceAvailable = !offers.empty();
    if (!isServiceAvailable) {
        TEventsGuard eg(report, "determine_service_availability");
        auto locationTags = DriveApi->GetAreasDB()->GetTagsInPoint(source);
        isServiceAvailable = HasIntersection(locationTags, NDrive::ServiceIsAvailableLocationTags);
    }
    report.AddReportElement("is_service_available", isServiceAvailable);
    g.SetCode(HTTP_OK);
}

NExternalOfferImpl::TOffersInfo TExternalOfferProcessor::BuildOffers(
    const NExternalOfferImpl::TCarsInfo& cars,
    const TUserOfferContext& uhc,
    NExternalOfferImpl::TOffersInfo& complementary,
    NDrive::TInfoEntitySession& session,
    TJsonReport& report
) {
    if (!uhc.HasUserPosition()) {
        return {};
    }

    TEventsGuard egFetchReport(report, "prepare_offers");
    const TUserPermissions& permissions = *Yensured(uhc.GetUserPermissions());
    const auto& builders = permissions.GetOfferBuilders();
    const auto& correctors = permissions.GetOfferCorrections();

    TVector<std::pair<
        TAtomicSharedPtr<const IOfferBuilderAction>,
        TAtomicSharedPtr<TOffersBuildingContext>
    >> selected;
    TSet<TString> selectedCars;
    const auto& cgi = Context->GetCgiParameters();
    auto checkCarsNumber = GetHandlerSetting<ui32>("external_offer.check_cars_number").GetOrElse(5);
    auto offersNumber = uhc.GetOfferCountLimit();
    auto types = GetHandlerSettingDef<TString>("external_offer.offer_builder_type", TFixPointOfferConstructor::GetTypeName());
    auto kind = GetValue<TString>(cgi, "kind", false).GetOrElse("");
    bool needHintsOnly = true;
    if (Config.GetSuggestOfferReport()) {
        needHintsOnly = kind.StartsWith("destination_");
        types = GetHandlerSettingDef<TString>(
            "offers_suggest." + kind + ".offer_builder_types",
            TFixPointOfferConstructor::GetTypeName()
        );
        checkCarsNumber = offersNumber;
    }
    auto fast = GetValue<bool>(cgi, "fast", false).GetOrElse(false);

    const TSet<TString> offerBuilderTypes = StringSplitter(types).Split(',');

    const auto carsLimit = std::min<ui32>(checkCarsNumber, cars.size());
    const auto offersLimit = offersNumber;
    for (size_t i = 0; i < carsLimit && selectedCars.size() < offersLimit; ++i) {
        if (Now() > uhc.GetDeadline()) {
            report.AddEvent("breaking prepare loop by deadline");
            report.SetTerminated(true);
            break;
        }

        const auto& objectId = cars[i].Id;
        TEventsGuard egPrepareOffer(report, "prepare_offer_" + objectId);
        auto context = MakeAtomicShared<TOffersBuildingContext>(MakeCopy(uhc));
        context->SetCarId(objectId);
        context->SetNeedOfferedHintsSeparately();
        if (needHintsOnly) {
            context->SetNeedOfferedHintsOnly();
        }
        if (fast) {
            context->SetNeedGeoFeatures(false);
            context->SetUseHaversine(true);
        }

        auto before = selected.size();
        for (auto&& b : builders) {
            if (!b || !offerBuilderTypes.contains(b->GetType())) {
                continue;
            }
            auto constructor = std::dynamic_pointer_cast<const IOfferBuilderAction>(b);
            if (!constructor) {
                continue;
            }

            TEventsGuard egPrefetchOfferCheck(report, "prefetch_offer_check_" + objectId + "_" + b->GetName());
            auto checkResult = constructor->CheckOfferConditions(*context, permissions);
            if (checkResult == EOfferCorrectorResult::Success) {
                TEventsGuard egPrefetchOfferStart(report, "prefetch_offer_start_" + objectId + "_" + b->GetName());
                context->Prefetch();
                selected.emplace_back(constructor, context);
                selectedCars.insert(objectId);
            } else if (checkResult == EOfferCorrectorResult::Problems) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
        auto after = selected.size();
        if (after == before) {
            report.AddEvent(NJson::TMapBuilder
                ("event", "prefetch_context_errors")
                ("errors", NJson::ToJson(NJson::Dictionary(context->GetOfferConstructionProblems())))
                ("object_id", objectId)
            );
        }
    }

    if (selected.empty()) {
        ErrorCounts["no_cars"] = 1;
        return {};
    }

    NExternalOfferImpl::TOffersInfo res;
    for (auto&& [builder, context]: selected) {
        if (Now() > uhc.GetDeadline()) {
            report.AddEvent("breaking build loop by deadline");
            ErrorCounts["break_loop_by_deadline"] = 1;
            break;
        }

        TVector<IOfferReport::TPtr> offers;
        TEventsGuard egPrefetchOfferCheck(report, "build_offer_" + context->GetCarIdDef("unknown") + "_" + builder->GetName());
        auto offerBuildResult = builder->BuildOffers(permissions, correctors, offers, *context, Server, session);
        if (offerBuildResult == EOfferCorrectorResult::Problems) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        } else if (offerBuildResult == EOfferCorrectorResult::Success && offers.size()) {
            for (auto&& offer : offers) {
                auto o = offer ? offer->GetOfferPtrAs<IOffer>() : nullptr;
                if (o) {
                    res.push_back(offer);
                    egFetchReport.AddEvent(TStringBuilder() << "created offer:" << builder->GetName() << ':' << o->GetObjectId() << ':' << o->GetOfferId());
                } else {
                    egFetchReport.AddEvent(TStringBuilder() << "non-vehicle or null offer created by " << builder->GetName());
                }
            }
        } else {
            egFetchReport.AddEvent(TStringBuilder() << "unable to generate offer: "
                                                    << ToUnderlying(offerBuildResult));
        }
        if (offers.empty()) {
            for (auto&& [name, error] : context->GetOfferConstructionProblems()) {
                ErrorCounts[error] += 1;
            }
            report.AddEvent(NJson::TMapBuilder
                ("event", "build_context_errors")
                ("errors", NJson::ToJson(NJson::Dictionary(context->GetOfferConstructionProblems())))
                ("object_id", NJson::ToJson(context->OptionalCarId()))
            );
        }
    }
    for (auto&& [builder, context]: selected) {
        if (Now() > uhc.GetDeadline()) {
            report.AddEvent("breaking build loop by deadline");
            ErrorCounts["break_loop_by_deadline"] = 1;
            break;
        }
        auto complementaryInsuranceTypes = context->GetComplementaryInsuranceTypes();
        if (complementaryInsuranceTypes) {
            for (auto&& insuranceType : *complementaryInsuranceTypes) {
                TEventsGuard eg(report, "build_complementary_" + context->GetCarIdDef("unknown") + "_" + builder->GetName() + "_" + insuranceType);
                TVector<IOfferReport::TPtr> offers;
                context->SetInsuranceType(insuranceType);
                auto offerBuildResult = builder->BuildOffers(permissions, correctors, offers, *context, Server, session);
                if (offerBuildResult == EOfferCorrectorResult::Problems) {
                    session.Check();
                }
                for (auto&& offer : offers) {
                    auto o = offer ? offer->GetOfferPtrAs<IOffer>() : nullptr;
                    if (o) {
                        complementary.push_back(offer);
                        report.AddEvent(TStringBuilder() << "created complementary:" << builder->GetName() << ':' << o->GetObjectId() << ':' << o->GetOfferId());
                    }
                }
            }
        }
    }
    return res;
}
