#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/fines/autocode/entries.h>
#include <drive/backend/fines/constructors/fine_attachment_constructor.h>
#include <drive/backend/fines/constructors/fine_constructor.h>
#include <drive/backend/fines/constructors/tag_constructor.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/major/client.h>
#include <drive/backend/processor/processor.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/autocode/client.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/string/cast.h>

void TCheckDriverLicenseProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasAutocodeClient(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    const auto& autocodeClient = Server->GetDriveAPI()->GetAutocodeClient();
    R_ENSURE(autocodeClient.HasDriverLicenseClient(), ConfigHttpStatus.ServiceUnavailable, "Driver license client is not configured");

    const TCgiParameters& cgi = Context->GetCgiParameters();

    TString userId = GetUUID(cgi, "user_id", false);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);

    TString licenseNumber = GetString(cgi, "license_number", false);

    auto issueDate = GetTimestamp(cgi, "issue_timestamp", false);
    if (!issueDate && cgi.Has("issue_date")) {
        issueDate.ConstructInPlace();
        R_ENSURE(TInstant::TryParseIso8601(GetString(cgi, "issue_date"), *issueDate), ConfigHttpStatus.SyntaxErrorStatus, "Invalid issue date format");
    }

    R_ENSURE(userId || (!!licenseNumber && !!issueDate), ConfigHttpStatus.SyntaxErrorStatus, "Either user id or both license number and issue date must be specified");

    if (!licenseNumber || !issueDate) {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto userPtr = Server->GetDriveAPI()->GetUsersData()->RestoreUser(userId, session);
        R_ENSURE(userPtr, ConfigHttpStatus.EmptySetStatus, "user not found", session);

        auto isDeleted = Server->GetDriveAPI()->GetUsersData()->IsDeletedByTags(userId, session);
        R_ENSURE(isDeleted, {}, "Error checking driver tags", session);
        R_ENSURE(!*isDeleted, ConfigHttpStatus.EmptySetStatus, "user not found ()");

        TUserDrivingLicenseData drivingLicenseData;
        // use async call?
        R_ENSURE(
            Server->GetDriveAPI()->GetPrivateDataClient().GetDrivingLicenseSync(*userPtr, userPtr->GetDrivingLicenseDatasyncRevision(), drivingLicenseData),
            ConfigHttpStatus.UnknownErrorStatus,
            "unable to fetch user license data",
            session
        );

        licenseNumber = drivingLicenseData.GetNumber();
        issueDate = drivingLicenseData.GetIssueDate();
    }

    NDrive::NAutocode::TDriverLicenseInfo licenseInfo;
    TMessagesCollector errors;
    R_ENSURE(autocodeClient.GetDriverLicenseInfo(licenseNumber, *issueDate, licenseInfo, errors), ConfigHttpStatus.UnknownErrorStatus, "Error checking driver license info: " + errors.GetStringReport());

    auto report = licenseInfo.SerializeToJson();
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TResendFineEmailProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasFinesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Send, TAdministrativeAction::EEntity::Fine);

    const auto& finesManager = Server->GetDriveAPI()->GetFinesManager();

    const TString userId = permissions->GetUserId();

    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString fineId = GetUUID(cgi, "fine_id", /* required = */ true);

    auto tx = BuildTx<NSQL::Writable>();
    NDrive::NFine::TAutocodeFineEntry fine;
    {
        auto dbFine = finesManager.GetFineById(fineId, tx);
        R_ENSURE(dbFine, ConfigHttpStatus.SyntaxErrorStatus, "fine " + fineId + " is not found", tx);
        fine = std::move(*dbFine);
    }

    TString recepientUserId = fine.GetUserId();
    if (cgi.Has("override_recepient_user_id")) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::SendForce, TAdministrativeAction::EEntity::Fine);
        recepientUserId = GetUUID(cgi, "override_recepient_user_id", /* required = */ true);

        auto userData = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(recepientUserId);
        R_ENSURE(userData, ConfigHttpStatus.EmptyRequestStatus, "cannot found recepient user " + recepientUserId);
    }

    fine.SetChargeEmailSentAt(ModelingNow());  // however do not add tag id to related ones

    NDrive::NFine::TFineFetchContextConfig contextConfig;
    R_ENSURE(contextConfig.DeserializeFromJson(requestData["context_config"]),
                   ConfigHttpStatus.SyntaxErrorStatus,
                   "cannot construct fine fetch context");

    TMessagesCollector errors;

    NDrive::NFine::TFineFetchContext context(contextConfig, *Server, fine);
    R_ENSURE(context.GetPrefetchedPhotos(errors, &tx), ConfigHttpStatus.SyntaxErrorStatus, "Fail to fetch fine photos", tx);

    NJson::TJsonValue tagConfigJson = requestData["tag_config"];
    if (!tagConfigJson.Has("tag_name_template")) {
        tagConfigJson.InsertValue("tag_name_template", NDrive::NFine::IFineContextFetcher::GetPlaceholderName<NDrive::NFine::TFineMailTagNameContextFetcher>(context));
    }

    NDrive::NFine::TFineMailTagConfig tagConfig;
    R_ENSURE(tagConfig.DeserializeFromJson(tagConfigJson),
                   ConfigHttpStatus.SyntaxErrorStatus,
                   "cannot construct fine mail tag");

    ITag::TPtr tagPtr = tagConfig.CreateTag(context, errors);
    R_ENSURE(tagPtr != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "Cannot create tag " + errors.GetStringReport());

    auto mailTagPtr = dynamic_cast<TUserMailTag*>(tagPtr.Get());
    R_ENSURE(mailTagPtr != nullptr, ConfigHttpStatus.UnknownErrorStatus, "Tag created is not a mail tag: " + errors.GetStringReport());

    if (mailTagPtr->GetAttachments().empty()) {
        TString violationDetailedDocumentUrl;
        if (fine.GetCachedViolationDetailedDocumentUrl(violationDetailedDocumentUrl, "") && !!violationDetailedDocumentUrl) {
            TUserMailTag::TAttachInfo attachment;
            NJson::TJsonValue attachmentJson = NJson::TMapBuilder("mime_type", "application/pdf")("filename", "documents.pdf")("url", violationDetailedDocumentUrl);
            R_ENSURE(attachment.DeserializeFromJson(attachmentJson), ConfigHttpStatus.UnknownErrorStatus, "Cannot construct mail tag attachment");
            mailTagPtr->MutableAttachments().push_back(std::move(attachment));
        }
    }

    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
    R_ENSURE(tagsManager.GetUserTags().AddTags({ tagPtr }, userId, recepientUserId, Server, tx), ConfigHttpStatus.UnknownErrorStatus, "error adding tags", tx);
    R_ENSURE(finesManager.UpsertFine(fine, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot upsert fine", tx);

    g.SetCode(HTTP_OK);
}

void TUpsertUserFinesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasFinesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Fine);

    const auto& finesManager = Server->GetDriveAPI()->GetFinesManager();
    auto carsDB = Server->GetDriveAPI()->GetCarsData();
    auto usersDB = Server->GetDriveAPI()->GetUsersData();

    R_ENSURE(requestData.Has("data") && requestData["data"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "malformed data: 'data' field is absent or it's not an array");

    TVector<NDrive::NFine::TAutocodeFineEntry> fines;
    for (auto&& fineData : requestData["data"].GetArray()) {
        NDrive::NFine::TAutocodeFineEntry fine;
        R_ENSURE(fine.DeserializeFromJson(fineData, /* useCents = */ true), ConfigHttpStatus.SyntaxErrorStatus, "malformed fine data: cannot deserialize fine " + fineData.GetStringRobust());

        auto carInfo = carsDB->FetchInfo(fine.GetCarId(), TInstant::Zero());
        auto carPtr = carInfo.GetResultPtr(fine.GetCarId());
        R_ENSURE(carPtr, ConfigHttpStatus.SyntaxErrorStatus, "malformed fine car data: no car with id " + fine.GetCarId());

        if (!fine.GetViolationDocumentNumber()) {
            fine.SetViolationDocumentNumber(ToString(carPtr->GetRegistrationID()));
        }

        if (!!fine.GetUserId()) {
            auto userInfo = usersDB->GetCachedObject(fine.GetUserId());
            R_ENSURE(userInfo, ConfigHttpStatus.SyntaxErrorStatus, "malformed fine user data: no user with id " + fine.GetUserId());
        }

        fines.push_back(std::move(fine));
    }

    auto session = BuildTx<NSQL::Writable>();
    for (auto&& fine : fines) {
        R_ENSURE(finesManager.UpsertFine(fine, session),
                       ConfigHttpStatus.UnknownErrorStatus,
                       "cannot upsert fine: constraint violations are possible - ensure that fine ruling number is unique");
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TReAssignUserFinesProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    Y_UNUSED(server);
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("fine_id", "Идентификатор штрафа").SetVisual(TFSString::EVisualType::UUID).SetRequired(true);
    scheme.Add<TFSBoolean>("do_refund", "Производить ли возврат средств").SetRequired(true);
    scheme.Add<TFSString>("refund_comment", "Комментарий к возврату");
    scheme.Add<TFSBoolean>("require_session_rebound", "Найти нарушителя автоматически").SetDefault(false);
    scheme.Add<TFSString>("target_user_id", "Идентификатор обновленного нарушителя (если есть)").SetVisual(TFSString::EVisualType::UUID);
    scheme.Add<TFSString>("target_session_id", "Идентификатор обновленной сессии нарушения (если есть)");

    // internal fields
    // scheme.Add<TFSNumeric>("target_skipped").SetMin(0).SetDefault(0);  // actually not used
    // scheme.Add<TFSStructure>("refund_tag_config").SetStructure(std::remove_reference_t<decltype(*RefundTagConfig)>::GetScheme(*server));
    // scheme.Add<TFSStructure>("context_config").SetStructure(std::remove_reference_t<decltype(*ContextConfig)>::GetScheme());

    return scheme;
}

void TReAssignUserFinesProcessor::Parse(const NJson::TJsonValue& data) {
    ParseDataUUIDField(data, "fine_id", FineId, /* required = */ true);

    ParseDataField(data, "do_refund", DoRefund, /* required = */ true);
    ParseDataField(data, "refund_comment", RefundComment, /* required = */ false);

    ParseDataField(data, "require_session_rebound", RequireSessionRebound, /* required = */ false);

    ParseDataUUIDField(data, "target_user_id", TargetUserId, /* required = */ false);
    ParseDataField(data, "target_session_id", TargetSessionId, /* required = */ false);
    ParseDataField(data, "target_skipped", TargetSkipped, /* required = */ false);

    IsTargetUserSet = CheckIsTargetUserSet(TargetUserId, TargetSessionId);

    R_ENSURE(RefundTagConfig->DeserializeFromJson(data["refund_tag_config"]),
                   ConfigHttpStatus.SyntaxErrorStatus,
                   "cannot construct fine refund tag");

    R_ENSURE(ContextConfig->DeserializeFromJson(data["context_config"]),
                   ConfigHttpStatus.SyntaxErrorStatus,
                   "cannot construct fine fetch context");
}

bool TReAssignUserFinesProcessor::CheckIsTargetUserSet(const TString& targetUserId, const TString& targetSessionId) const {
    if (!!targetUserId || !!targetSessionId) {
        R_ENSURE(targetUserId && !!targetSessionId, ConfigHttpStatus.SyntaxErrorStatus, "both target user id and target session id must be provided");
        {
            auto userInfo = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(targetUserId);
            R_ENSURE(userInfo, ConfigHttpStatus.SyntaxErrorStatus, "target user id does not exist: " + targetUserId);
        }
        return true;
    }
    return false;
}

void TReAssignUserFinesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasFinesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Fine);

    const auto& finesManager = Server->GetDriveAPI()->GetFinesManager();
    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();

    const TString performerId = permissions->GetUserId();

    auto tx = BuildTx<NSQL::Writable>();
    NDrive::NFine::TAutocodeFineEntry fine;
    {
        auto dbFine = finesManager.GetFineById(FineId, tx);
        R_ENSURE(dbFine, ConfigHttpStatus.SyntaxErrorStatus, "fine " + FineId + " is not found", tx);
        fine = std::move(*dbFine);
    }

    NJson::TJsonValue chargeTagIdJson = fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::ChargeTagId);
    TString chargeTagId = chargeTagIdJson.GetString();


    TString objectId = fine.GetUserId();
    if (!!objectId) {
        auto userInfo = Server->GetDriveAPI()->GetUsersData()->FetchInfo(objectId, tx);
        auto userPtr = userInfo.GetResultPtr(objectId);
        R_ENSURE(userPtr, ConfigHttpStatus.SyntaxErrorStatus, "user fine has been charged from does not exist now: " + objectId);
    }

    ITag::TPtr refundTagPtr = nullptr;
    TString refundTagId;

    if (GetDoRefund()) {
        R_ENSURE(
            !!fine.GetChargedAt() && !!chargeTagId && !!objectId,
            ConfigHttpStatus.SyntaxErrorStatus,
            "cannot refund fine " << fine.GetId() << " as it has not been charged or charge information is incorrect"
        );
        refundTagPtr = PrepareRefundTag(fine, tx);
    }

    // NB. Refund may be provided apart from the current reassignment (e.g. manually)

    if (GetDoRefund()) {
        auto optionalAddedTags = tagsManager.GetUserTags().AddTag(refundTagPtr, performerId, objectId, Server, tx);
        R_ENSURE(optionalAddedTags && optionalAddedTags->size() == 1, ConfigHttpStatus.UnknownErrorStatus, "cannot add refund tag", tx);
        refundTagId = optionalAddedTags->front().GetTagId();
    }

    R_ENSURE(UpdateFineRelatedTagIds(fine, { chargeTagId, refundTagId }), ConfigHttpStatus.UnknownErrorStatus, "cannot update fine related tag ids");

    // NB. Current binding can be empty, however it's useful to provide session drop to store history and clear all related fields
    R_ENSURE(fine.DropSessionBinding(performerId), ConfigHttpStatus.UnknownErrorStatus, "cannot drop session binding");

    if (GetIsTargetUserSetUnsafe() || GetRequireSessionRebound()) {
        if (GetIsTargetUserSetUnsafe()) {
            fine.SetUserId(GetTargetUserId());
            fine.SetSessionId(GetTargetSessionId());
            fine.SetSkipped(GetTargetSkipped());
        }

        auto fineConstructorPtr = NDrive::NFine::TFineConstructor::ConstructDefault(*Server);
        R_ENSURE(fineConstructorPtr && fineConstructorPtr->UpdateDependentFields(fine, GetRequireSessionRebound()), ConfigHttpStatus.UnknownErrorStatus, "Cannot update dependent fine fields: check fine charge status");
    }

    R_ENSURE(finesManager.UpsertFine(fine, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot upsert fine: constraint violations are possible or data is outdated", tx);

    g.SetCode(HTTP_OK);
}

ITag::TPtr TReAssignUserFinesProcessor::PrepareRefundTag(const NDrive::NFine::TAutocodeFineEntry& fine, NDrive::TEntitySession& session) const {
    NDrive::NFine::TFineFetchContext::TDynamicContext dynamicContext = { { "refund_comment", GetRefundComment() } };

    TMessagesCollector errors;
    NDrive::NFine::TFineFetchContext context(*ContextConfig, *Server, fine, dynamicContext);
    R_ENSURE(context.GetPrefetchedPhotos(errors, &session), ConfigHttpStatus.SyntaxErrorStatus, "Fail to fetch fine photos", session);

    ITag::TPtr tagPtr = RefundTagConfig->CreateTag(context, errors);
    R_ENSURE(tagPtr != nullptr, ConfigHttpStatus.SyntaxErrorStatus, "Cannot create tag " + errors.GetStringReport());
    return tagPtr;
}

bool TReAssignUserFinesProcessor::UpdateFineRelatedTagIds(NDrive::NFine::TAutocodeFineEntry& fine, const TVector<TString>& tagIds) const {
    auto relatedTagIds = NJson::FromJsonWithDefault<TSet<TString>>(fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::RelatedTagIds), {});
    for (const auto& tagId : tagIds) {
        if (!!tagId) {
            relatedTagIds.insert(tagId);
        }
    }
    fine.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::RelatedTagIds, NJson::ToJson(relatedTagIds));
    return true;
}

NDrive::TScheme TListUserFinesProcessor::GetCgiParametersScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;

    // scheme.Add<TFSBoolean>("extended_info").SetDefault(false);  // internal

    scheme.Add<TFSString>("fine_id", "Фильтр по идентификатору штрафа").SetVisual(TFSString::EVisualType::UUID);
    scheme.Add<TFSString>("user_id", "Фильтр по идентификатору пользователя").SetVisual(TFSString::EVisualType::UUID);
    scheme.Add<TFSString>("car_id", "Фильтр по идентификатору автомобиля").SetVisual(TFSString::EVisualType::UUID);
    scheme.Add<TFSBoolean>("charged_only", "Только штрафы к списанию").SetDefault(true);
    scheme.Add<TFSBoolean>("use_cents", "Возвращать суммы в копейках").SetDefault(true);

    scheme.Add<TFSNumeric>("since").SetMin(0).SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("until").SetMin(0).SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("offset").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("limit").SetMin(1).SetMax(MaxLimit).SetDefault(MaxLimit);

    return scheme;
}

void TListUserFinesProcessor::Parse(const TCgiParameters& cgi) {
    if (cgi.Has("extended_info")) {
        ExtendedInfo = ParseValue<bool>(GetString(cgi, "extended_info"));
    }

    if (cgi.Has("use_cents")) {
        UseCents = ParseValue<bool>(GetString(cgi, "use_cents"));
    }

    FineId = GetUUID(cgi, "fine_id", false);
    UserId = GetUUID(cgi, "user_id", false);
    CarId = GetUUID(cgi, "car_id", false);

    if (cgi.Has("charged_only")) {
        ChargedOnly = ParseValue<bool>(GetString(cgi, "charged_only"));
    }

    Since = GetTimestamp(cgi, "since", TInstant::Zero());
    Until = GetTimestamp(cgi, "until", TInstant::Zero());

    if (cgi.Has("offset")) {
        Offset = ParseValue<size_t>(GetString(cgi, "offset"));
    }

    if (cgi.Has("limit")) {
        Limit = std::min(ParseValue<size_t>(GetString(cgi, "limit")), MaxLimit);
    }
}

void TListUserFinesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasFinesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);

    const auto& finesManager = Server->GetDriveAPI()->GetFinesManager();

    const auto locale = GetLocale();
    const bool extendedInfo = GetExtendedInfo();
    const bool useCents = GetUseCents();

    NDrive::NFine::TFineFilterGroup filters = { MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>() };

    TString fineId = GetFineId();  // fine will be shown either if user has observe permissions or the fine belongs to the user

    TString userId;
    if (extendedInfo) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Fine);
        userId = GetUserId();
    } else {
        userId = permissions->GetUserId();
    }

    if (!!userId) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineUserFilter>(userId));
    }

    TString carId = GetCarId();
    if (!!carId) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineCarFilter>(carId));
    }

    if (!!GetSince() || !!GetUntil()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineViolationTimeFilter>(GetSince(), GetUntil()));
    }

    if (GetChargedOnly()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineChargedFilter>());
    }

    size_t offset = GetOffset();
    size_t limit = GetLimit();

    TVector<NDrive::NFine::TAutocodeFineEntry> fines;
    {
        size_t localOffset = offset;
        size_t localLimit = Max(limit, MaxLimit);
        auto session = BuildTx<NSQL::ReadOnly>();
        while (fines.size() < localLimit) {
            NSQL::TQueryOptions options(localLimit, /* descending = */ true, localOffset);
            options.SetOrderBy({"violation_time"});
            if (fineId) {
                options.AddGenericCondition("id", fineId);
            }
            auto finesDB = finesManager.GetFines(filters, session, options);
            R_ENSURE(finesDB, ConfigHttpStatus.UnknownErrorStatus, "error collecting fines", session);
            if (finesDB->empty()) {
                break;
            }
            for (auto&& fine : *finesDB) {
                if (fines.size() >= limit) {
                    break;
                }
                fines.emplace_back(std::move(fine));
            }
            localOffset += localLimit;
        }
    }


    NJson::TJsonValue report(NJson::JSON_ARRAY);
    NDrive::NFine::NFineTraits::TReportTraits reportTraits = (extendedInfo) ? NDrive::NFine::NFineTraits::ReportAll : NDrive::NFine::NFineTraits::NoTraits;

    if (useCents) {
        reportTraits |= NDrive::NFine::NFineTraits::UseCents;
    }

    auto gCars = Server->GetDriveAPI()->GetCarsData();
    auto gUsers = Server->GetDriveAPI()->GetUsersData();
    auto gModelsData = Server->GetDriveAPI()->GetModelsData();

    size_t reportedFinesCount = 0;

    TMap<TString, TVector<NDrive::NFine::TAutocodeFineAttachmentEntry>> attachmans;
    {
        TSet<TString> fineIds;
        Transform(fines.begin(), fines.end(), std::inserter(fineIds, fineIds.begin()), [](const auto& fine) {
            return fine.GetId();
        });
        auto session = BuildTx<NSQL::ReadOnly>();
        auto photos = finesManager.GetFineAttachments(fineIds, NDrive::NFine::TFinesManager::EFineAttachmentType::Photo, session);
        R_ENSURE(photos, ConfigHttpStatus.UnknownErrorStatus, "fail to get photos", session);
        for (auto&& photo : * photos) {
            attachmans[photo.GetFineId()].emplace_back(std::move(photo));
        }
    }

    for (const auto& fine : fines) {
        if (reportedFinesCount >= limit) {
            break;
        }

        auto currentFineReport = fine.BuildReport(reportTraits);

        auto& currentFinePhotosReport = currentFineReport.InsertValue("photos", NJson::JSON_ARRAY);
        if (auto ptr = attachmans.FindPtr(fine.GetId())) {
            for (const auto& finePhotoEntry : *ptr) {
                currentFinePhotosReport.AppendValue(finePhotoEntry.BuildReport());
                currentFinePhotosReport.AppendValue(finePhotoEntry.BuildReport());
            }
        }

        {
            auto carFetchResult = gCars->FetchInfo(fine.GetCarId(), TInstant::Zero());
            auto carPtr = carFetchResult.GetResultPtr(fine.GetCarId());
            R_ENSURE(carPtr, ConfigHttpStatus.UnknownErrorStatus, "undefined car data");

            NJson::TJsonValue carReport = NJson::JSON_NULL;

            if (!extendedInfo) {
                carReport = carPtr->GetReport(locale, NDeviceReport::EReportTraits::ReportCarId);

                NJson::TJsonValue modelReport = NJson::JSON_MAP;
                modelReport["code"] = carPtr->GetModel();

                carReport.InsertValue("model", std::move(modelReport));
            } else {
                carReport = carPtr->GetReport(locale,
                                              NDeviceReport::EReportTraits::ReportCarId |
                                              NDeviceReport::EReportTraits::ReportVIN |
                                              NDeviceReport::EReportTraits::ReportIMEI |
                                              NDeviceReport::EReportTraits::ReportRegistrationId);

                auto modelData = gModelsData->FetchInfo(carPtr->GetModel(), TInstant::Zero());
                auto modelPtr = modelData.GetResultPtr(carPtr->GetModel());
                R_ENSURE(modelPtr, ConfigHttpStatus.UnknownErrorStatus, "undefined car model data");

                carReport.InsertValue("model", modelPtr->GetReport(locale, NDriveModelReport::ReportAll));
            }

            currentFineReport.InsertValue("car", std::move(carReport));
        }

        if (extendedInfo) {
            NJson::TJsonValue userReport = NJson::JSON_NULL;

            if (!!fine.GetUserId()) {
                auto userPtr = gUsers->GetCachedObject(fine.GetUserId());
                R_ENSURE(userPtr, ConfigHttpStatus.UnknownErrorStatus, "undefined user data");

                userReport = userPtr->GetReport(NUserReport::ReportPublicId);
            }

            currentFineReport.InsertValue("user", std::move(userReport));
        }

        {
            TString articleCode = fine.GetArticleCode();
            TString readableArticle;

            auto articleMatcherPtr = finesManager.GetFineArticleMatcherPtr(/* ensureActual = */ false);
            if (!!articleMatcherPtr) {
                if (!articleCode) {
                    articleMatcherPtr->DetermineArticle(fine.GetArticleKoap(), articleCode);
                }

                readableArticle = articleMatcherPtr->GetReadableArticleDefault(fine.GetArticleKoap(), articleCode);
            }

            currentFineReport.InsertValue("article_code", articleCode);
            currentFineReport.InsertValue("article_koap_human_readable", readableArticle);
        }

        report.AppendValue(std::move(currentFineReport));
        ++reportedFinesCount;
    }

    g.MutableReport().SetExternalReport(std::move(report));

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TFineDecreeInfoProcessor::GetCgiParametersScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("fine_id", "Фильтр по идентификатору штрафа").SetVisual(TFSString::EVisualType::UUID).SetRequired(true);
    return scheme;
}

void TFineDecreeInfoProcessor::Parse(const TCgiParameters& cgi) {
    FineId = GetUUID(cgi, "fine_id", true);
}

void TFineDecreeInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    ReqCheckCondition(!!Server->GetDriveAPI() && Server->GetDriveAPI()->HasFinesManager(), ConfigHttpStatus.ServiceUnavailable, EDriveLocalizationCodes::InternalServerError);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Fine);

    const auto& finesManager = Server->GetDriveAPI()->GetFinesManager();

    NDrive::NFine::TAutocodeFineEntry fine;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto fineAttachments = finesManager.GetFineAttachments(GetFineId(), NDrive::NFine::TFinesManager::EFineAttachmentType::Decree, tx);
        R_ENSURE(fineAttachments, ConfigHttpStatus.UnknownErrorStatus, "fail to get decree", tx);
        if (!fineAttachments->empty()) {
            g.MutableReport().SetExternalReport(fineAttachments->front().BuildReport());  // only one decree is supposed to be or all are equal so get any
            g.SetCode(HTTP_OK);
            return;
        }
        auto dbFine = finesManager.GetFineById(FineId, tx);
        R_ENSURE(dbFine, ConfigHttpStatus.SyntaxErrorStatus, "fine " + FineId + " is not found", tx);
        fine = std::move(*dbFine);
    }

    const bool hasDecree = fine.GetHasDecree(/* defaultValue = */ true);
    R_ENSURE(hasDecree, ConfigHttpStatus.EmptySetStatus, "no decree available");

    const auto& majorClient = Server->GetDriveAPI()->GetMajorClient();

    TMessagesCollector errors;

    NMajorClient::TCarPenaltyDecreeRequest::TCarPenaltyDecree penaltyDecree;
    R_ENSURE(majorClient.GetCarPenaltyDecree(fine.GetRulingNumber(), penaltyDecree, errors), ConfigHttpStatus.EmptySetStatus, "no decree available (empty response)");

    auto fineAttachmentConstructorConfig = NDrive::NFine::TFineAttachmentConstructorConfig::Construct();
    R_ENSURE(fineAttachmentConstructorConfig, ConfigHttpStatus.UnknownErrorStatus, "cannot construct attachment constructor config");

    auto fineAttachmentConstructor = NDrive::NFine::TFineAttachmentConstructor(*fineAttachmentConstructorConfig, *Server);

    NDrive::NFine::TAutocodeFineAttachmentEntry fineAttachmentEntry;
    R_ENSURE(fineAttachmentConstructor.ConstructAutocodeFineAttachmentEntry(GetFineId(), penaltyDecree, fineAttachmentEntry, errors), ConfigHttpStatus.UnknownErrorStatus, "cannot construct attachment from obtained decree or upload it");

    auto tx = finesManager.BuildTx<NSQL::Writable>();
    R_ENSURE(finesManager.UpsertFineAttachment(fineAttachmentEntry, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot upsert decree", tx);

    g.MutableReport().SetExternalReport(fineAttachmentEntry.BuildReport());
    g.SetCode(HTTP_OK);
}
