#include "common.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/fueling_manager/fueling_manager.h>

EFuelType NDrive::FuelTypeFromDriveFuelType(const TString& ft) {
    EFuelType type = EFuelType::Undefined;
    if (ft == "92") {
        return EFuelType::A92;
    } else if (ft == "95") {
        return EFuelType::A95;
    } else if (ft == "98") {
        return EFuelType::A98;
    } else if (ft == "diesel") {
        return EFuelType::Diesel;
    } else if (TryFromString(ft, type)) {
        return type;
    } else {
        return EFuelType::Undefined;
    }
}

EFuelClientType TFuelingCommon::GetFuelClientType(const TCgiParameters& cgi) const {
    TVector<TString> result;
    const TString param = "fueling_client";
    auto range = cgi.equal_range(param);
    for (auto i = range.first; i != range.second; ++i) {
        R_ENSURE(result.empty(), ServerCommon->GetHttpStatusManagerConfig().SyntaxErrorStatus, "multiple " << param << " are specified");
        result.emplace_back(i->second);
    }
    EFuelClientType type;
    if (!result.empty() && TryFromString<EFuelClientType>(result.front(), type)) {
        return type;
    }
    return EFuelClientType::Default;
}

EFuelClientType TFuelingCommon::GetFuelClientType(const NJson::TJsonValue& requestData) const {
    EFuelClientType result = EFuelClientType::Default;
    const TString param = "fueling_client";
    if (NJson::ParseField(requestData, param, NJson::Stringify(result))) {
        return result;
    }
    return EFuelClientType::Default;
}

TFuelingCommon::EPostpayCheckToOrder TFuelingCommon::CheckToOrder(TUserFuelingTag& tag, const double allowedError) const {
    if (!tag.HasFuelType()) {
        NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(ServerCommon->GetLocalization());
    }
    if (!tag.GetOrderId().empty()) {
        NDrive::NMessages::UserHaveFuelingOrder.DoException(ServerCommon->GetLocalization());
    }

    TFuelGroups fuelGroups;
    fuelGroups.InitFromSettings(ServerCommon->GetSettings());

    if (!tag.HasActualLiters() || !tag.HasActualFuelType()) {
        NDrive::NMessages::PollingIsNotCompleted.DoException(ServerCommon->GetLocalization());
    }
    auto status = tag.GetCurrentState();
    if (status != EFuelingStatus::FuelingCompleted) {
        NDrive::NMessages::PollingIsNotCompleted.DoException(ServerCommon->GetLocalization());
    }
    if (tag.GetActualLitersRef() > tag.GetLiters() + allowedError) {
        status = EFuelingStatus::WrongFuelLiters;
    } else if (!fuelGroups.FuelTypeCmp(tag.GetFuelTypeRef(), tag.GetActualFuelTypeRef())) {
        status = EFuelingStatus::WrongFuelType;
    }
    if (status != tag.GetCurrentState()) {
        tag.SetCurrentState(status);
        return EPostpayCheckToOrder::TagUpdated;
    }
    return EPostpayCheckToOrder::NeedOrder;
}

TFuelingManager::TOrderInfo BuildOrderInfo(const TUserFuelingTag& tag, const bool postPay, const TMaybe<TGeoCoord> coord,const NDrive::IServer& server) {
    TFuelingManager::TOrderInfo order;
    order.OrderId = tag.GetPreOrderId();
    order.StationId = tag.GetStationId();
    order.ColumnId = tag.GetColumnId();
    order.CarId = tag.GetObjectId();
    order.Coord = coord;
    if (postPay && (!tag.HasActualLiters() || !tag.HasActualFuelType())) {
        NDrive::NMessages::PollingIsNotCompleted.DoException(server.GetLocalization());
    }
    order.FuelInfo.FuelId = postPay ? tag.GetActualFuelTypeRef() : tag.GetFuelTypeRef();
    order.FuelInfo.Liters = postPay ? tag.GetActualLitersRef() : tag.GetLiters();
    order.ClientType = tag.GetFuelClientTypeDef(EFuelClientType::Default);
    return order;
}

bool TFuelingCommon::CreateOrder(TUserFuelingTag& tag, const bool postPay, TMessagesCollector& errors, const TMaybe<TGeoCoord> coord /* = {} */) const {
    auto orderInfo = BuildOrderInfo(tag, postPay, coord, *ServerCommon);
    if (!ServerCommon->GetFuelingManager()->CreateOrder(orderInfo, errors)) {
        return false;
    }
    tag.SetOrderId(orderInfo.OrderId);
    tag.SetCurrentState(EFuelingStatus::AcceptOrder);
    tag.SetComment(GetFuelingStatusInfo(ServerCommon->GetLocalization(), EFuelingStatus::AcceptOrder));
    return true;
}

bool TFuelingCommon::RestoreCurrentState(TUserPermissions::TPtr permissions, TAtomicSharedPtr<const ISession> userSession, TString& objectId, TString& orderId, TDBTag* userTagResult) {
    auto session = DriveApiCommon->template BuildTx<NSQL::ReadOnly>();
    TVector<TDBTag> userTags;
    if (!DriveApiCommon->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), {TUserFuelingTag::GetTypeName()}, userTags, session)) {
        return false;
    }
    for (auto&& tag : userTags) {
        if (tag.Is<TUserFuelingTag>()) {
            if (!userSession || userSession->GetObjectId() == tag.GetTagAs<TUserFuelingTag>()->GetObjectId()) {
                if (userTagResult) {
                    *userTagResult = tag;
                }
                orderId = tag.GetTagAs<TUserFuelingTag>()->GetOrderId();
                DEBUG_LOG << "CURRENT FUELING ORDER: " << orderId << Endl;
                break;
            }
        }
    }
    if (userSession) {
        objectId = userSession->GetObjectId();
    } else {
        TVector<TDBTag> deviceTags;
        if (!DriveApiCommon->GetTagsManager().GetDeviceTags().RestorePerformerTags({permissions->GetUserId()}, deviceTags, session)) {
            return false;
        }
        for (auto&& tag : deviceTags) {
            if (tag.Is<TChargableTag>()) {
                if (!!objectId && objectId != tag.GetObjectId()) {
                    return false;
                }
                objectId = tag.GetObjectId();
            }
        }
    }

    return !!objectId;
}

EFuelingCondition TFuelingCommon::CheckConditions(const TString& objectId, TUserPermissions::TPtr permissions, TFuelingContext& resultContext, const bool tanker) {
    auto& [fuelInfo, station, columns, coord, session, columnId, patchContext, modelFuelType] = resultContext;
    auto& [fType, liters] = fuelInfo;
    auto& [patches, exactPatch, ignoreModel] = patchContext;
    fType = EFuelType::Undefined;
    if (!objectId) {
        return EFuelingCondition::IncorrectPosition;
    }
    const TRTDeviceSnapshot ds = ServerCommon->GetSnapshotsManager().GetSnapshot(objectId);

    NDrive::TLocation location;
    if (!ds.GetLocation(location)) {
        session.Fail(NDrive::NMessages::UndefinedCarLocation);
        return EFuelingCondition::Problems;
    }

    double d;
    coord = location.GetCoord();
    const bool isCustomSensor = exactPatch && !patches.empty() && patches.front().GetSensor();
    if (tanker || isCustomSensor) {
        ui16 sensor = isCustomSensor ? patches.front().GetSensor() : NDrive::NVega::AuxFuelLevel<1>();
        auto dut = ds.GetSensor(sensor);
        auto configurationTag = DriveApiCommon->GetTagsManager().GetDeviceTags().GetTagFromCache(objectId, TTelematicsConfigurationTag::Type(), TInstant::Zero());
        auto configuration = configurationTag ? configurationTag->GetTagAs<TTelematicsConfigurationTag>() : nullptr;
        if (!dut || !configuration || !configuration->GetCalibrators().contains(dut->Id) || configuration->GetCalibrators().at(dut->Id).GetTable().empty()) {
            session.Fail(NDrive::NMessages::UndefinedFuelLevel);
            return EFuelingCondition::Problems;
        }
        const auto& calibrator = configuration->GetCalibrators().at(dut->Id);
        auto currentLiters = calibrator.Get(*dut);
        auto maxElem = MaxElement(calibrator.GetTable().begin(), calibrator.GetTable().end());
        liters = maxElem->Calibrated - currentLiters;
        d = currentLiters / maxElem->Calibrated;
    } else {
        // NOTE: ignore sensors from tags
        if (!ds.GetFuelLevel(d)) {
            session.Fail(NDrive::NMessages::UndefinedFuelLevel);
            return EFuelingCondition::Problems;
        }
    }

    auto gCar = DriveApiCommon->GetCarsData()->GetCachedOrFetch(objectId);
    const TDriveCarInfo* carInfo = gCar.GetResultPtr(objectId);
    if (!carInfo) {
        session.Fail(NDrive::NMessages::CannotRestoreCarInfo);
        return EFuelingCondition::Problems;
    }
    auto gModel = DriveApiCommon->GetModelsData()->GetCached(carInfo->GetModel());
    const TDriveModelData* modelInfo = gModel.GetResultPtr(carInfo->GetModel());
    if (!modelInfo) {
        session.Fail(NDrive::NMessages::CannotRestoreModelInfo);
        return EFuelingCondition::Problems;
    }
    if (!tanker && !isCustomSensor) {
        liters = NDrive::GetNeedFuelingLiters(d, carInfo, modelInfo);
    }

    modelFuelType = NDrive::FuelTypeFromDriveFuelType(modelInfo->GetFuelType());
    if (modelFuelType == EFuelType::Undefined) {
        session.Fail(NDrive::NMessages::UndefinedModelFuelType);
        return EFuelingCondition::Problems;
    }

    const ui64 percLimit = tanker
        ? TUserFuelingTag::GetTankerFuelingPercentLimit(carInfo->GetModel(), permissions, *ServerCommon)
        : TUserFuelingTag::GetFuelingPercentLimit(carInfo->GetModel(), permissions, *ServerCommon);
    const double litersLimit = tanker
        ? TUserFuelingTag::GetTankerFuelingVolumeLimit(carInfo->GetModel(), permissions, *ServerCommon)
        : TUserFuelingTag::GetFuelingVolumeLimit(carInfo->GetModel(), permissions, *ServerCommon);

    if (liters < litersLimit || d > percLimit || liters < 1e-5) {
        session.Finish(NDrive::NMessages::TankIsFull);
        return EFuelingCondition::NotApplicable;
    }

    if (!ServerCommon->GetFuelingManager()->GetStationInfo(location.GetCoord(), station)) {
        TGeoRect carRect(coord);
        const auto distance = permissions->GetSetting<ui32>("fueling.station_search_meters", 0);
        carRect.GrowDistance(distance);
        const auto stationSearchResult = distance ? ServerCommon->GetFuelingManager()->GetStationInfo(carRect, station) : EStationSearchResult::NotInStation;
        if (stationSearchResult != EStationSearchResult::Ok) {
            session.Finish(NDrive::NMessages::IncorrectLocationForFueling);
            return stationSearchResult == EStationSearchResult::MoreThanOneStation ? EFuelingCondition::MoreThanOneStation : EFuelingCondition::IncorrectPosition;
        }
    }

    if (station.GetPostPay() && !permissions->GetSetting<bool>("fueling.enable_post_pay", false)) {
        session.Finish(NDrive::NMessages::IncorrectStationForPollingType);
        return EFuelingCondition::NotApplicable;
    }

    TMaybe<EFuelType> optType;
    if (!patches.empty()) {
        ui32 pos = Max<ui32>();
        auto processColumn = [&optType, &pos](const auto& column, const auto& types, auto& columns) mutable {
            for (ui32 t = 0; t < types.size(); t++ ) {
                auto type = types[t];
                if (column.GetFuels().contains(type)) {
                    columns.emplace(column.GetId(), column);
                    if (t < pos) {
                        pos = t;
                        optType = type;
                    }
                }
            }
        };
        for (auto&& i : station.GetColumns()) {
            if (columnId && i.GetId() != columnId) {
                continue;
            }
            if (!exactPatch) {
                for (const auto& patch : patches) {
                    if (patch.GetSensor()) {
                        continue;
                    }
                    processColumn(i, patch.GetPatchTypes(), columns);
                }
            } else {
                processColumn(i, patches.front().GetPatchTypes(), columns);
            }
        }
        if (optType) {
            fType = *optType;
        }
    }

    bool noNeedModelFType = isCustomSensor;
    noNeedModelFType |= ignoreModel;
    noNeedModelFType |= !!optType;
    if (!noNeedModelFType) {
        for (auto&& i : station.GetColumns()) {
            if (i.GetFuels().contains(modelFuelType)) {
                columns.emplace(i.GetId(), i);
            }
        }
        fType = modelFuelType;
    }

    if (columns.empty()) {
        session.Finish(NDrive::NMessages::IncorrectStationForFuelType);
        return EFuelingCondition::NotApplicable;
    }

    if (fType == EFuelType::Undefined) {
        session.Fail(NDrive::NMessages::UndefinedModelFuelType);
        return EFuelingCondition::Problems;
    }

    return EFuelingCondition::OK;
}
