#pragma once
#include <drive/backend/processors/common_app/processor.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/fueling_manager/fueling_manager.h>

#include <library/cpp/http/misc/httpcodes.h>

enum class EFuelingCondition {
    OK,
    IncorrectPosition,
    NotApplicable,
    Problems,
    MoreThanOneStation,
};

namespace NDrive {
    class TProcessingResult {
    private:
        R_FIELD(HttpCodes, Code, HttpCodes::HTTP_OK);
        R_FIELD(TString, Id);
        R_FIELD(TString, Title);
        R_FIELD(TString, Message);

        std::pair<TString, TString> GetLocalizationTexts(const ILocalization* localization) {
            if (localization) {
                TString title = localization->GetLocalString(ELocalization::Rus, Id + ".title", Title);
                TString message = localization->GetLocalString(ELocalization::Rus, Id + ".message", Message);
                return {title, message};
            }
            return {Title, Message};
        }

    public:
        TProcessingResult(const HttpCodes code, const TString& id, const TString& message)
            : Code(code)
            , Id(id)
            , Message(message)
        {
        }

        TProcessingResult(const HttpCodes code, const TString& id, const TString& title, const TString& message)
            : Code(code)
            , Id(id)
            , Title(title)
            , Message(message)
        {
        }

        void DoException(const TMessagesCollector& collector, const ILocalization* localization) {
            CHECK_WITH_LOG(Code != HTTP_OK);
            if (collector.HasMessages()) {
                auto&& [title, message] = GetLocalizationTexts(localization);
                ythrow TCodedException(Code)
                    .AddInfo("session_info", collector.GetReport())
                    .SetLocalizedTitle(title)
                    .SetLocalizedMessage(message);
            } else {
                DoException(localization);
            }
        }

        void DoException(const ILocalization* localization, const TString& errorCode = "", const TString& additionalInfo = "") {
            auto&& [title, message] = GetLocalizationTexts(localization);
            if (additionalInfo) {
                NJson::TJsonValue info = additionalInfo;
                ythrow TCodedException(Code)
                    .SetLocalizedTitle(title)
                    .SetLocalizedMessage(message)
                    .SetErrorCode(errorCode)
                    .AddPublicInfo("additional_info", info);
            }
            ythrow TCodedException(Code)
                .SetLocalizedTitle(title)
                .SetLocalizedMessage(message)
                .SetErrorCode(errorCode);
        }
    };

    namespace NMessages {
        static TProcessingResult UserHaveFuelingOrder(HttpCodes::HTTP_BAD_REQUEST, "user_have_fueling_order", "У Вас уже происходит заправка");
        static TProcessingResult FuelingCannotBeStopped(HttpCodes::HTTP_BAD_REQUEST, "fueling_cannot_be_stopped", "Заправка не может быть прервана");
        static TProcessingResult NoCurrentFuelingOrder(HttpCodes::HTTP_BAD_REQUEST, "no_current_fueling_order", "Не определена актуальная сессия заправки");
        static TProcessingResult InternalFuelingServiceProblems(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "internal_fueling_service_problems", "Проблемы сервиса заправки");
        static TProcessingResult UndefinedCarForUser(HttpCodes::HTTP_BAD_REQUEST, "undefined_car_for_user", "Не получается определить автомобиль пользователя");
        static TProcessingResult UndefinedFuelLevel(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "undefined_fuel_level", "Не получается определить уровень топлива");
        static TProcessingResult UndefinedCarLocation(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "undefined_car_location", "Не получается определить локацию автомобиля");
        static TProcessingResult TankIsFull(HttpCodes::HTTP_CONFLICT, "tank_is_full", "Полный бак. Заправка не требуется.");
        static TProcessingResult CannotRestoreCarInfo(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "cannot_restore_car_info", "Не определяются данные по машине");
        static TProcessingResult CannotRestoreModelInfo(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "cannot_restore_model_info", "Не определяются данные по модели");
        static TProcessingResult UndefinedModelFuelType(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "undefined_model_fuel_type", "Неизвестные данные топлива по модели");
        static TProcessingResult IncorrectLocationForFueling(HttpCodes::HTTP_BAD_REQUEST, "incorrect_location_for_fueling", "Машина не находится на заправке");
        static TProcessingResult IncorrectStationForFuelType(HttpCodes::HTTP_BAD_REQUEST, "incorrect_station_for_fuel_type", "Нет подходящего типа топлива");
        static TProcessingResult InternalUserTagsHistoryRestoreProblem(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "internal_user_tags_history_restore_problem", "Не получается определить данные по истории заправок");
        static TProcessingResult InternalUserTagsRestoreProblem(HttpCodes::HTTP_INTERNAL_SERVER_ERROR, "internal_user_tags_restore_problem", "Не получается определить данные заправок по пользователю");
        static TProcessingResult TooFrequentFueling(HttpCodes::HTTP_BAD_REQUEST, "too_frequent_fueling", "Слишком частые заправки");
        static TProcessingResult IncorrectStationForPollingType(HttpCodes::HTTP_BAD_REQUEST, "incorrect_station_for_polling_type", "Станция недоступна для заправки");
        static TProcessingResult PollingIsNotCompleted(HttpCodes::HTTP_BAD_REQUEST, "polling_is_not_completed", "Подача топлива не завершена");
        static TProcessingResult IncorrectColumnForPollingType(HttpCodes::HTTP_BAD_REQUEST, "incorrect_column_for_polling_type", "Эта колонка пока занята", "Попробуйте ещё раз или выберите другую");
    }

    enum class EInfoSessionStatus {
        InProgress = 0,
        Failed = 1,
        Finished = 2,
    };

    template <class ECustomStatus>
    class TInfoSessionImpl {
    private:
        using TSelf = TInfoSessionImpl<ECustomStatus>;

    private:
        R_READONLY(EInfoSessionStatus, Status, EInfoSessionStatus::InProgress);
        R_FIELD(TString, OriginatorId);
        R_OPTIONAL(TProcessingResult, Result);
        R_FIELD(ECustomStatus, CustomStatus);
        R_FIELD(TMessagesCollector, Infos);

    private:
        void Stop(const TProcessingResult& result, const ECustomStatus cs, const EInfoSessionStatus status) {
            Stop(result, status);
            CustomStatus = cs;
        }

        void Stop(const TProcessingResult& result, const EInfoSessionStatus status) {
            CHECK_WITH_LOG(Status == EInfoSessionStatus::InProgress);
            CHECK_WITH_LOG(status != EInfoSessionStatus::InProgress);
            Status = status;
            Result = result;
        }

    public:
        void DoException(const ILocalization* localization) {
            CHECK_WITH_LOG(!!Result);
            Result->DoException(Infos, localization);
        }

        void Fail(const TProcessingResult& result, const ECustomStatus cs) {
            Stop(result, EInfoSessionStatus::Failed, cs);
        }

        void Fail(const TProcessingResult& result) {
            Stop(result, EInfoSessionStatus::Failed);
        }

        void Finish(const TProcessingResult& result, const ECustomStatus cs) {
            Stop(result, EInfoSessionStatus::Finished, cs);
        }

        void Finish(const TProcessingResult& result) {
            Stop(result, EInfoSessionStatus::Finished);
        }
    };

    using TInfoSession = TInfoSessionImpl<TNull>;

    EFuelType FuelTypeFromDriveFuelType(const TString& ft);
}

struct TFuelPatchContext {
    TVector<TPatchFuelingTag::TDescription> Patches;
    bool ExactPatch = false;
    bool IgnoreModel = false;
};


struct TFuelingContext {
    TFuelingManager::TFuelInfo FuelInfo;
    TStation Station;
    TMap<TString, TColumn> Columns;
    TGeoCoord Coord;
    NDrive::TInfoSession Session;
    TString ColumnId;
    TFuelPatchContext Patches;
    EFuelType ModelFuelId;
};

class TFuelingCommon {
private:
    const NDrive::IServer* ServerCommon;
    const TDriveAPI* DriveApiCommon;
public:
    TFuelingCommon(const NDrive::IServer* server)
        : ServerCommon(server)
        , DriveApiCommon(ServerCommon->GetDriveAPI())
    {

    }

    enum class EPostpayCheckToOrder {
        TagUpdated,
        NeedOrder,
    };

    EPostpayCheckToOrder CheckToOrder(TUserFuelingTag& tag, const double allowedError) const;

    bool CreateOrder(TUserFuelingTag& tag, const bool postPay, TMessagesCollector& errors, const TMaybe<TGeoCoord> coord = {}) const;
    bool RestoreCurrentState(TUserPermissions::TPtr permissions, TAtomicSharedPtr<const ISession> userSession, TString& objectId, TString& orderId, TDBTag* userTagResult = nullptr);

    EFuelingCondition CheckConditions(const TString& objectId, TUserPermissions::TPtr permissions, TFuelingContext& resultContext, const bool tanker = false);
    EFuelClientType GetFuelClientType(const TCgiParameters& cgi) const;
    EFuelClientType GetFuelClientType(const NJson::TJsonValue& requestData) const;
};
