#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/fueling_manager/fueling_manager.h>


TMaybe<TMap<TString, TFuelPatchContext>> GetPatches(const IDriveTagsManager& tagsManager, const TSet<TString>& objectIds, NDrive::TEntitySession& tx, const TString& tagName = "") {
    auto descriptions = tagsManager.GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { TPatchFuelingTag::GetTypeName() });
    auto tags = tagName
        ? tagsManager.GetDeviceTags().RestoreTagsRobust(objectIds, { tagName }, tx)
        : tagsManager.GetDeviceTags().RestoreTagsRobust(objectIds, MakeVector(NContainer::Keys(descriptions)), tx);
    if (!tags) {
        return {};
    }
    TMap<TString, TFuelPatchContext> result;
    Transform(objectIds.begin(), objectIds.end(), std::inserter(result, result.begin()), [](const TString& id) -> std::pair<TString, TFuelPatchContext> { return { id, TFuelPatchContext() }; });
    for (auto&& tag : *tags) {
        if (tagName && tag->GetName() != tagName) {
            continue;
        }
        auto desc = descriptions.FindPtr(tag->GetName());
        if (!desc) {
            continue;
        }
        auto patchDesc = dynamic_cast<const TPatchFuelingTag::TDescription*>(desc->Get());
        if (!patchDesc) {
            continue;
        }
        const TString objectId = tag.GetObjectId();
        if (auto context = result.FindPtr(objectId)) {
            context->ExactPatch = !tagName.empty();
            context->IgnoreModel |= patchDesc->IsOverrideModel() && (!patchDesc->GetSensor() || tagName);
            context->Patches.push_back(*patchDesc);
        }
    }
    return result;
}

void TFuelingMapProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    TVector<TDBTag> actualTags;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        R_ENSURE(DriveApi->GetTagsManager().GetDeviceTags().RestorePerformerTags({permissions->GetUserId()}, actualTags, tx), ConfigHttpStatus.UnknownErrorStatus, "cannot restore tags");
    }
    TSet<TString> carIds;
    for (auto&& i : actualTags) {
        if (i.GetTagAs<TChargableTag>()) {
            carIds.emplace(i.TConstDBTag::GetObjectId());
        }
    }

    TCarsDB::TFetchResult cars;
    TMaybe<TMap<TString, TFuelPatchContext>> patches;
    {
        auto tx = BuildTx<NSQL::ReadOnly>();
        cars = DriveApi->GetCarsData()->FetchInfo(carIds, tx);
        R_ENSURE(cars, ConfigHttpStatus.UnknownErrorStatus, "Fail to fetch cars", tx);
        patches = GetPatches(DriveApi->GetTagsManager(), carIds, tx);
        R_ENSURE(patches, ConfigHttpStatus.UnknownErrorStatus, "Fail to fetch patches", tx);
    }
    TSet<TString> modelIds;
    for (auto&& [carId, car] : cars.GetResult()) {
        if (auto patch = patches->FindPtr(carId); patch && !patch->IgnoreModel) {
            modelIds.insert(car.GetModel());
        }
    }

    TSet<EFuelType> modelFuelTypes;
    auto gModels = DriveApi->GetModelsData()->FetchInfo(modelIds, TInstant::Zero());
    for (auto&& i : gModels.GetResult()) {
        auto fType = NDrive::FuelTypeFromDriveFuelType(i.second.GetFuelType());
        if (fType != EFuelType::Undefined) {
            modelFuelTypes.emplace(fType);
        } else {
            ERROR_LOG << "Cannot detect fuel type: " << i.second.GetFuelType() << Endl;
        }
    }
    TSet<EFuelType> fuelTypes = modelFuelTypes;
    for (const auto& context : NContainer::Values(*patches)) {
        for (const auto& patch : context.Patches) {
            fuelTypes.insert(patch.GetPatchTypes().begin(), patch.GetPatchTypes().end());
        }
    }

    TMaybe<TGeoRect> carRect;
    if (const auto distance = permissions->GetSetting<ui32>("fueling.station_visibility_meters", 0)) {
        if (auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime())) {
            const auto objectId = userSession->GetObjectId();
            const TRTDeviceSnapshot ds = Server->GetSnapshotsManager().GetSnapshot(objectId);
            NDrive::TLocation location;
            if (!ds.GetLocation(location)) {
                NDrive::NMessages::UndefinedCarLocation.DoException(Server->GetLocalization());
            }
            carRect = TGeoRect(location.GetCoord());
            carRect->GrowDistance(distance);
        }
    }

    g.MutableReport().AddReportElement("map", Server->GetFuelingManager()->GetJsonMapReport(&fuelTypes, permissions->GetSetting<bool>("fueling.enable_post_pay", false), carRect));
    NJson::TJsonValue tankers = NJson::JSON_ARRAY;
    TSet<TString> reported;
    for (auto&& context : NContainer::Values(*patches)) {
        for (const auto& patch : context.Patches) {
            if (!reported.insert(patch.GetName()).second) {
                continue;
            }
            auto& tanker = tankers.AppendValue(NJson::JSON_MAP);
            NJson::InsertField(tanker, "tag_name", patch.GetName());
            NJson::InsertField(tanker, "tag_hr_name", patch.GetDisplayName());
            NJson::InsertField(tanker, "fuel_types", patch.GetPatchTypes());
        }
    }
    if (!modelFuelTypes.empty()) {
        auto& tanker = tankers.AppendValue(NJson::JSON_MAP);
        auto locale = GetLocale();
        NJson::InsertField(tanker, "tag_hr_name", Server->GetLocalization()->GetLocalString(locale, "fueling.model_fuel.title", "Бак машины"));
        NJson::InsertField(tanker, "tag_name", TString("default"));
        NJson::InsertField(tanker, "fuel_types", modelFuelTypes);
    }
    g.MutableReport().AddReportElement("fuel_types", std::move(tankers));
    g.SetCode(HTTP_OK);
}

TMaybe<TFuelPatchContext> GetPatches(const IDriveTagsManager& tagsManager, const TString& objectId, NDrive::TEntitySession& tx, const TString& tagName = "") {
    auto subResult = GetPatches(tagsManager, TSet<TString>{ objectId }, tx, tagName);
    if (!subResult) {
        return {};
    }
    if (auto result = subResult->FindPtr(objectId)) {
        Sort(result->Patches.begin(), result->Patches.end(), [](const auto& l, const auto& r) { return l.GetPatchTypes().size() > r.GetPatchTypes().size(); });
        return std::move(*result);
    }
    return TFuelPatchContext();
}

TSet<EFuelType> GetFuelTypes(const TFuelPatchContext& context, const EFuelType modelType) {
    if (context.ExactPatch && !context.Patches.empty()) {
        return MakeSet(context.Patches.front().GetPatchTypes());
    }
    TSet<EFuelType> result;
    for (const auto& patch : context.Patches) {
        if (!patch.GetSensor()) {
            result.insert(patch.GetPatchTypes().begin(), patch.GetPatchTypes().end());
        }
    }
    if (!context.IgnoreModel && modelType != EFuelType::Undefined) {
        result.insert(modelType);
    }
    return result;
}

void TFuelingInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    TSet<EFuelingStatus> allowedPostPayRetry;
    StringSplitter(permissions->GetSetting<TString>("fueling.post_pay.allowed_retry", "")).Split(',').SkipEmpty().ParseInto(&allowedPostPayRetry);

    const TString fillUpKey = "filled_up";
    if (GetValue<bool>(Context->GetCgiParameters(), fillUpKey, /* required = */ false).GetOrElse(false) || GetValue<bool>(requestData, fillUpKey, /* required = */ false).GetOrElse(false)) {
        TVector<TDBTag> userTags;
        {
            auto session = BuildTx<NSQL::ReadOnly>();
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), {TUserFuelingTag::GetTypeName()}, userTags, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            if (userTags.empty()) {
                NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
            }
        }
        TUserFuelingTag* tag;
        {
            TStation station;
            tag = userTags.front().MutableTagAs<TUserFuelingTag>();
            if (!tag || !Server->GetFuelingManager()->GetStationInfo(tag->GetStationId(), station)) {
                NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
            }
            if (!station.GetPostPay()) {
                NDrive::NMessages::IncorrectStationForPollingType.DoException(Server->GetLocalization());
            }
            const TString retryKey = "retry";
            if ((GetValue<bool>(Context->GetCgiParameters(), retryKey, /* required = */ false).GetOrElse(false) || GetValue<bool>(requestData, retryKey, /* required = */ false).GetOrElse(false))
                && allowedPostPayRetry.contains(tag->GetCurrentState()))
            {
                tag->SetCurrentState(EFuelingStatus::FuelingCompleted);
                tag->SetOrderId("");
                tag->SetPreOrderId("");
            }
        }
        const double allowedError = permissions->GetSetting<double>("fueling.allowed_litres_error", 5);
        switch (CheckToOrder(*tag, allowedError)) {
            case EPostpayCheckToOrder::TagUpdated:
            {
                auto session = BuildTx<NSQL::Writable>();
                if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(userTags.front(), permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                g.MutableReport().AddReportElement("current_state", tag->GetPublicReport(Server->GetLocalization(), Server->GetFuelingManager(), allowedPostPayRetry));
                g.SetCode(HTTP_OK);
                return;
            }
            case EPostpayCheckToOrder::NeedOrder:
            {
                if (!tag->GetPreOrderId()) {
                    tag->SetPreOrderId(TFuelingManager::GenerateOrderId());
                    auto session = BuildTx<NSQL::Writable>();
                    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(userTags.front(), permissions->GetUserId(), session) || !session.Commit()) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                }
                TMessagesCollector errors;
                if (!CreateOrder(*tag, /* postPay = */ true, errors)) {
                    NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
                }
                auto session = BuildTx<NSQL::Writable>();
                if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(userTags.front(), permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                g.MutableReport().AddReportElement("current_state", tag->GetPublicReport(Server->GetLocalization(), Server->GetFuelingManager(), allowedPostPayRetry));
                g.SetCode(HTTP_OK);
                return;
            }
        }
    }

    TString objectId;
    TDBTag userFuelingTag;
    {
        TString orderId;
        auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
        if (!RestoreCurrentState(permissions, userSession, objectId, orderId, &userFuelingTag)) {
            NDrive::NMessages::UndefinedCarForUser.DoException(Server->GetLocalization());
        }
    }
    if (auto tag = userFuelingTag.MutableTagAs<TUserFuelingTag>()) {
        switch (tag->GetCurrentState()) {
            case EFuelingStatus::TriedOrderCreate:
            {
                const TString orderId = tag->GetPreOrderId() ? tag->GetPreOrderId() : tag->GetOrderId();
                if (!orderId) {
                    break;
                }
                TString comment;
                TMessagesCollector errors;
                auto status = Server->GetFuelingManager()->GetStatus(orderId, comment, tag->GetFuelClientTypeDef(EFuelClientType::Default), errors);
                if (status == EFuelingStatus::ServerProblems) {
                    NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
                }
                if (status == EFuelingStatus::Unknown) {
                    break;
                }
                tag->SetCurrentState(status);
                tag->SetOrderId(orderId);
                NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
                if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(userFuelingTag, permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
            case EFuelingStatus::OrderCreated:
            case EFuelingStatus::AcceptOrder:
            case EFuelingStatus::Fueling:
            case EFuelingStatus::Completed:
            case EFuelingStatus::UserCanceled:
            case EFuelingStatus::Expire:
            case EFuelingStatus::PaymentInProgress:
            case EFuelingStatus::ErrorPayment:
            case EFuelingStatus::WaitingRefueling:
            case EFuelingStatus::StationCanceled:
            case EFuelingStatus::Free:
            case EFuelingStatus::FuelingCompleted:
            case EFuelingStatus::WrongFuelType:
            case EFuelingStatus::WrongFuelLiters:
            case EFuelingStatus::WaitCancel:
                g.MutableReport().AddReportElement("current_state", tag->GetPublicReport(Server->GetLocalization(), Server->GetFuelingManager(), allowedPostPayRetry));
                g.SetCode(HTTP_OK);
                return;
            default:
                g.MutableReport().AddReportElement("current_state", tag->GetPublicReport(Server->GetLocalization(), Server->GetFuelingManager(), allowedPostPayRetry));
                g.SetCode(ConfigHttpStatus.UnknownErrorStatus);
                return;
        }
    }

    TFuelingContext fuelingContex;
    {
        TString patchName = GetString(Context->GetCgiParameters(), "tag_name", false);
        if (!patchName) {
            patchName = GetString(requestData, "tag_name", false);
        }
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto patches = GetPatches(DriveApi->GetTagsManager(), objectId, tx, patchName == "default" ? "" : patchName);
        R_ENSURE(patches, ConfigHttpStatus.UnknownErrorStatus, "Fail to fetch car tags", tx);
        R_ENSURE(!patches->ExactPatch || !patches->Patches.empty(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect tag_name");
        fuelingContex.Patches = std::move(*patches);
    }
    auto& station = fuelingContex.Station;
    auto& fType = fuelingContex.FuelInfo.FuelId;
    auto constructInfo = [this] (EFuelingStatus status, EFuelType fuelType, NJson::TJsonValue&& stationReport = NJson::JSON_UNDEFINED) {
        NJson::TJsonValue stationInfo;
        if (stationReport.IsDefined()) {
            stationInfo.InsertValue("station", stationReport);
        }
        stationInfo.InsertValue("status", ToString(status));
        stationInfo.InsertValue("status_info", GetFuelingStatusInfo(Server->GetLocalization(), status));
        stationInfo.InsertValue("fueling_message", Server->GetFuelingManager()->GetFuelingTypeMessage(fuelType));
        return stationInfo;
    };
    const bool tanker = GetValue<bool>(Context->GetCgiParameters(), "tanker", false).GetOrElse(false);
    switch (CheckConditions(objectId, permissions, fuelingContex, tanker)) {
        case EFuelingCondition::Problems:
        case EFuelingCondition::NotApplicable:
            fuelingContex.Session.DoException(Server->GetLocalization());
        case EFuelingCondition::MoreThanOneStation:
            g.MutableReport().AddReportElement("status", ToString(EFuelingStatus::MoreThanOneStation));
            g.MutableReport().AddReportElement("current_state", constructInfo(EFuelingStatus::MoreThanOneStation, fType));
            break;
        case EFuelingCondition::IncorrectPosition:
            g.MutableReport().AddReportElement("status", ToString(EFuelingStatus::NotInStation));
            g.MutableReport().AddReportElement("current_state", constructInfo(EFuelingStatus::NotInStation, fType));
            break;
        case EFuelingCondition::OK:
            R_ENSURE(!fuelingContex.Patches.ExactPatch || !fuelingContex.Patches.Patches.empty(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect tag_name");
            const TSet<EFuelType> fTypes = GetFuelTypes(fuelingContex.Patches, fuelingContex.ModelFuelId);
            g.MutableReport().AddReportElement("current_state", constructInfo(EFuelingStatus::ReadyForStart, fType, station.GetJsonMapReport(&fTypes)));
    }
    g.SetCode(HTTP_OK);
}

THolder<TUserFuelingTag> BuildFuelingTag(const TStartFuelingProcessor::TOrdererInfo& ordererInfo, const TFuelingContext& fuelingContex) {
    auto tag = MakeHolder<TUserFuelingTag>(fuelingContex.Station.GetId(), ordererInfo.PickedColumnId);
    tag->SetObjectId(ordererInfo.CarId);
    tag->SetFuelType(fuelingContex.FuelInfo.FuelId);
    tag->SetLiters(fuelingContex.FuelInfo.Liters);
    if (!!ordererInfo.SessionId) {
        tag->SetSessionId(ordererInfo.SessionId);
    }
    if (ordererInfo.ClientType != EFuelClientType::Default) {
        tag->SetFuelClientType(ordererInfo.ClientType);
    }
    return tag;
}

void TStartFuelingProcessor::StartPrePayFueling(TDBTag& userFuelingTag, const TOrdererInfo& ordererInfo, TFuelingContext& fuelingContex, const TMaybe<double> litersUpperBound) const {
    if (litersUpperBound && fuelingContex.FuelInfo.Liters > *litersUpperBound) {
        fuelingContex.FuelInfo.Liters = *litersUpperBound;
    }
    bool needNewTag = true;
    bool needOrder = true;
    if (auto oldTag = userFuelingTag.MutableTagAs<TUserFuelingTag>(); oldTag && oldTag->GetPreOrderId()) {
        TMessagesCollector errors;
        TString comment;
        auto status = Server->GetFuelingManager()->GetStatus(oldTag->GetPreOrderId(), comment, ordererInfo.ClientType, errors);
        if (status == EFuelingStatus::ServerProblems) {
            NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
        }
        needNewTag = status == EFuelingStatus::Unknown;
        if (!needNewTag) {
            oldTag->SetCurrentState(status);
            oldTag->SetOrderId(oldTag->GetPreOrderId());
            needOrder = false;
        }
    }
    if (needNewTag) { // rewrite or add new
        auto tag = BuildFuelingTag(ordererInfo, fuelingContex);
        tag->SetCurrentState(EFuelingStatus::TriedOrderCreate);
        tag->SetPreOrderId(TFuelingManager::GenerateOrderId());
        NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
        auto tags = DriveApi->GetTagsManager().GetUserTags().AddTag(tag.Release(), ordererInfo.UserId, ordererInfo.UserId, Server, session, EUniquePolicy::Rewrite);
        if (!tags || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (tags->empty() || !tags->front().Is<TUserFuelingTag>()) {
            NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
        }
        userFuelingTag = std::move(tags->front());
    }
    if (needOrder) {
        TMessagesCollector errors;
        if (!CreateOrder(*userFuelingTag.MutableTagAs<TUserFuelingTag>(), /* postPay = */ false, errors, fuelingContex.Coord)) {
            NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
        }
    }
    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(userFuelingTag, ordererInfo.UserId, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
}

void TStartFuelingProcessor::StartPostPayFueling(const TOrdererInfo& ordererInfo, const TFuelingContext& fuelingContex, const double allowedError) const {
    TMaybe<TFuelingManager::TStatusInfo> statusInfo;
    TMessagesCollector errors;
    auto status = Server->GetFuelingManager()->GetPostStatus(fuelingContex.Station.GetId(), ordererInfo.PickedColumnId, statusInfo, ordererInfo.ClientType, errors);
    if (status == EFuelingStatus::ServerProblems) {
        NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
    }
    static const TSet<EFuelingStatus> availableStatuses = {EFuelingStatus::Free, EFuelingStatus::Fueling, EFuelingStatus::FuelingCompleted};
    if (!availableStatuses.contains(status)) {
        NDrive::NMessages::IncorrectColumnForPollingType.DoException(Server->GetLocalization());
    }
    auto tag = BuildFuelingTag(ordererInfo, fuelingContex);
    tag->SetCurrentState(status);
    INFO_LOG << "Status in start: " << status << " " << tag->GetFuelTypeDef(EFuelType::Undefined) << " " << tag->GetActualFuelTypeDef(EFuelType::Undefined) << Endl;
    if (statusInfo && status != EFuelingStatus::Free) {
        tag->SetActualLiters(statusInfo->Liters);
        tag->SetActualFuelType(statusInfo->FuelId);
        if (status == EFuelingStatus::FuelingCompleted) {
            CheckToOrder(*tag, allowedError);
        }
    }
    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetTagsManager().GetUserTags().AddTag(tag.Release(), ordererInfo.UserId, ordererInfo.UserId, Server, session, EUniquePolicy::Rewrite) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
}

void TStartFuelingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    TString columnId;
    if (!columnId) {
        columnId = GetString(Context->GetCgiParameters(), "column_id", false);
    }
    if (!columnId) {
        columnId = GetString(requestData, "column_id", false);
    }
    R_ENSURE(columnId, ConfigHttpStatus.UserErrorState, "parameter column_id is missing");
    {
        auto eventsHistory = permissions->GetSetting<TDuration>("fueling.events_deep", TDuration::Minutes(10));
        auto session = BuildTx<NSQL::ReadOnly>();
        auto optionalEvents = DriveApi->GetTagsManager().GetUserTags().GetEventsByObject(permissions->GetUserId(), session, 0, Now() - eventsHistory);
        R_ENSURE(optionalEvents, {}, "cannot GetEventsByObject for " << permissions->GetUserId(), session);

        for (auto&& i : *optionalEvents) {
            auto fTag = i.GetTagAs<TUserFuelingTag>();
            if (fTag && fTag->GetOrderId() && (fTag->GetCurrentState() == EFuelingStatus::Completed || fTag->GetCurrentState() == EFuelingStatus::Fueling)) {
                NDrive::NMessages::TooFrequentFueling.DoException(Server->GetLocalization());
            }
        }
    }

    TString objectId;
    TString orderId;
    TDBTag userFuelingTag;
    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    if (!RestoreCurrentState(permissions, userSession, objectId, orderId, &userFuelingTag)) {
        NDrive::NMessages::UndefinedCarForUser.DoException(Server->GetLocalization());
    }

    {
        const TDuration eventsHistory = permissions->GetSetting<TDuration>("fueling.events_deep_car", TDuration::Hours(6));
        const TInstant since = Context->GetRequestStartTime() - eventsHistory;
        auto session = BuildTx<NSQL::ReadOnly>();
        auto optionalEvents = Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetEventsByObject(objectId, session, 0, since);
        if (!optionalEvents) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "GetEventsByObjectError")
                ("errors", session.GetReport())
            );
            NDrive::NMessages::InternalUserTagsHistoryRestoreProblem.DoException(Server->GetLocalization());
        }

        for (auto&& i : *optionalEvents) {
            auto fTag = i.GetTagAs<TUserFuelingTag>();
            if (fTag && (fTag->GetCurrentState() == EFuelingStatus::Completed || (fTag->GetCurrentState() == EFuelingStatus::Fueling && fTag->GetOrderId()))) {
                NDrive::NMessages::TooFrequentFueling.DoException(Server->GetLocalization());
            }
        }
    }

    if (!!orderId) {
        NDrive::NMessages::UserHaveFuelingOrder.DoException(Server->GetLocalization());
    }

    TFuelingContext fuelingContex;
    fuelingContex.ColumnId = columnId;
    {
        TString patchName = GetString(Context->GetCgiParameters(), "tag_name", false);
        if (!patchName) {
            patchName = GetString(requestData, "tag_name", false);
        }
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto patches = GetPatches(DriveApi->GetTagsManager(), objectId, tx, patchName == "default" ? "" : patchName);
        R_ENSURE(patches, ConfigHttpStatus.UnknownErrorStatus, "Fail to fetch car tags", tx);
        R_ENSURE(!patches->ExactPatch || !patches->Patches.empty(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect tag_name");
        fuelingContex.Patches = std::move(*patches);
    }
    {
        const bool tanker = GetValue<bool>(Context->GetCgiParameters(), "tanker", false).GetOrElse(false);
        if (CheckConditions(objectId, permissions, fuelingContex, tanker) != EFuelingCondition::OK) {
            fuelingContex.Session.DoException(Server->GetLocalization());
        }
    }
    R_ENSURE(fuelingContex.Columns.contains(columnId), ConfigHttpStatus.SyntaxErrorStatus, "incorrect column_id", EDriveSessionResult::IncorrectRequest);
    const auto clientType = GetFuelClientType(Context->GetCgiParameters());
    R_ENSURE(Server->GetFuelingManager()->HasFuelClientKey(clientType), ConfigHttpStatus.UnknownErrorStatus, "Incorrect fuel configuration");
    TOrdererInfo ordererInfo = {
        permissions->GetUserId(),
        /* CarId = */ objectId,
        (userSession ? userSession->GetSessionId() : ""),
        clientType,
        /* PickedColumnId = */ columnId
    };
    if (fuelingContex.Station.GetPostPay()) {
        if (userFuelingTag.Is<TUserFuelingTag>()) {
            NDrive::NMessages::UserHaveFuelingOrder.DoException(Server->GetLocalization());
        }
        const double allowedError = permissions->GetSetting<double>("fueling.allowed_litres_error", 5);
        StartPostPayFueling(ordererInfo, fuelingContex, allowedError);
    } else {
        const auto litersUpperBound = permissions->GetSetting<double>("fueling.pre_pay.liters_upper_bound");
        StartPrePayFueling(userFuelingTag, ordererInfo, fuelingContex, litersUpperBound);
    }

    g.SetCode(HTTP_OK);
}

void TCancelFuelingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    TString reason;
    if (!reason) {
        reason = GetString(Context->GetCgiParameters(), "reason", false);
    }
    if (!reason) {
        reason = GetString(requestData, "reason", false);
    }
    R_ENSURE(reason, ConfigHttpStatus.UserErrorState, "parameter reason is missing");

    TString objectId;
    TString orderId;
    TDBTag userTagForRemove;
    auto userSession = GetCurrentUserSession(permissions, Context->GetRequestStartTime());
    if (!RestoreCurrentState(permissions, userSession, objectId, orderId, &userTagForRemove)) {
        NDrive::NMessages::UndefinedCarForUser.DoException(Server->GetLocalization());
    }
    auto ufTag = userTagForRemove.MutableTagAs<TUserFuelingTag>();
    if (!ufTag) {
        NDrive::NMessages::NoCurrentFuelingOrder.DoException(Server->GetLocalization(), "no_tag_data");
    }

    EFuelingStatus status;
    TMessagesCollector errors;
    EFuelClientType clientType = ufTag->GetFuelClientTypeDef(EFuelClientType::Default);
    if (!orderId && ufTag->GetPreOrderId()) {
        orderId = ufTag->GetPreOrderId();
    }
    if (!orderId) {
        status = ufTag->GetCurrentState();
    } else {
        TString comment;
        status = Server->GetFuelingManager()->GetStatus(orderId, comment, clientType, errors);
        if (status == EFuelingStatus::ServerProblems) {
            NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
        }
    }

    switch (status) {
        case EFuelingStatus::Unknown:
        case EFuelingStatus::Completed:
        case EFuelingStatus::UserCanceled:
        case EFuelingStatus::Expire:
        case EFuelingStatus::StationCanceled:
        case EFuelingStatus::Free:
        case EFuelingStatus::Unavailable:
            break;
        case EFuelingStatus::WrongFuelType:
        case EFuelingStatus::WrongFuelLiters:
        case EFuelingStatus::FuelingCompleted:
        case EFuelingStatus::Fueling:
            if (!orderId) {
                ufTag->SetCurrentState(EFuelingStatus::WaitCancel);
                auto session = BuildTx<NSQL::Writable>();
                if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(userTagForRemove, permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                g.SetCode(HTTP_OK);
                return;
            }
        case EFuelingStatus::WaitCancel:
        case EFuelingStatus::WaitingRefueling:
            NDrive::NMessages::FuelingCannotBeStopped.DoException(Server->GetLocalization());
        case EFuelingStatus::OrderCreated:
        case EFuelingStatus::AcceptOrder:
        case EFuelingStatus::PaymentInProgress:
            if (!Server->GetFuelingManager()->CancelOrder(orderId, reason, clientType, errors)) {
                NDrive::NMessages::FuelingCannotBeStopped.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
            }
            break;
        case EFuelingStatus::ErrorPayment:
            if (!ufTag->HasActualFuelType() && !ufTag->HasActualLiters()) {
                break;
            }
        default:
            g.MutableErrors().SetCode(ConfigHttpStatus.UnknownErrorStatus);
            g.MutableErrors().SetErrorCode(ToString(status));
            return;
    }

    {
        auto session = BuildTx<NSQL::Writable>();
        {
            TString comment;
            TMaybe<TFuelingManager::TStatusInfo> info;
            status = orderId
                ? Server->GetFuelingManager()->GetStatus(orderId, comment, clientType, errors)
                : Server->GetFuelingManager()->GetPostStatus(ufTag->GetStationId(), ufTag->GetColumnId(), info, clientType, errors);
            if (status == EFuelingStatus::ServerProblems) {
                g.AddEvent(NJson::TMapBuilder
                    ("event", "FuelingStatusAfterCancelError")
                    ("error", errors.GetStringReport())
                    ("order_id", orderId)
                    ("station_id", ufTag->GetStationId())
                    ("column_id", ufTag->GetColumnId())
                    ("client_type", ToString(clientType))
                );
            }
            if (status != EFuelingStatus::ServerProblems && ufTag->GetCurrentState() != status) {
                ufTag->SetCurrentState(status);
                if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(userTagForRemove, permissions->GetUserId(), session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        }
        if (!DriveApi->GetTagsManager().GetUserTags().RemoveTag(userTagForRemove, permissions->GetUserId(), Server, session, true)
            || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    g.SetCode(HTTP_OK);
}
