#include "support.h"

#include <drive/backend/common/localization.h>
#include <drive/backend/fueling_manager/fueling_manager.h>
#include <drive/backend/tags/tags_manager.h>


TDBTag TSupportCreateFuelingOrderProcessor::UpsertFuelingTag(TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) const {
    TDBTags userTags;
    auto session = BuildTx<NSQL::Writable>();
    const TString userId = GetString(requestData, "user_id", true);
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {TUserFuelingTag::GetTypeName()}, userTags, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TUserFuelingTag* tag;
    const bool needNewTag = userTags.empty();
    const TString columnId = GetString(requestData, "column_id", /* required = */ needNewTag);
    auto clientType = GetFuelClientType(requestData);
    if (needNewTag) {
        ReqCheckCondition(!!columnId, ConfigHttpStatus.SyntaxErrorStatus, "column id is required for new tag");
        const TString stationId = GetString(requestData, "station_id", /* required = */ true);
        ReqCheckCondition(!!stationId, ConfigHttpStatus.SyntaxErrorStatus, "station id is required for new tag");
        auto newTag = MakeAtomicShared<TUserFuelingTag>(stationId, columnId);
        const TString carId = GetString(requestData, "car_id", /* required = */ true);
        ReqCheckCondition(!!carId, ConfigHttpStatus.SyntaxErrorStatus, "car id is required for new tag");
        newTag->SetObjectId(carId);
        newTag->SetFuelClientType(clientType);
        newTag->SetCurrentState(EFuelingStatus::FuelingCompleted);
        auto optionalTags = DriveApi->GetTagsManager().GetUserTags().AddTags({newTag}, permissions->GetUserId(), userId, Server, session, EUniquePolicy::SkipIfExists);
        if (!optionalTags) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        userTags = std::move(*optionalTags);
        if (userTags.empty()) {
            NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
        }
        tag = userTags.front().MutableTagAs<TUserFuelingTag>();
    } else {
        tag = userTags.front().MutableTagAs<TUserFuelingTag>();
        if (tag->GetPreOrderId()) {
            TString comment;
            TMessagesCollector errors;
            auto status = Server->GetFuelingManager()->GetStatus(tag->GetPreOrderId(), comment, tag->GetFuelClientTypeDef(clientType), errors);
            if (status == EFuelingStatus::ServerProblems) {
                NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
            }
            if (status != EFuelingStatus::Unknown) {
                tag->SetOrderId(tag->GetPreOrderId());
                auto session = BuildTx<NSQL::Writable>();
                if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(userTags.front(), permissions->GetUserId(), session) || !session.Commit()) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
                throw TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "Order already created";
            }
        }
    }
    if (!tag) {
        NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
    }
    ReqCheckCondition(tag->GetOrderId().empty(), ConfigHttpStatus.SyntaxErrorStatus, "tag already has order " + tag->GetOrderId());
    {
        TStation station;
        ReqCheckCondition(Server->GetFuelingManager()->GetStationInfo(tag->GetStationId(), station), ConfigHttpStatus.SyntaxErrorStatus, "fail to get station data");
        if (!tag || !Server->GetFuelingManager()->GetStationInfo(tag->GetStationId(), station)) {
            NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
        }
        if (!station.GetPostPay()) {
            NDrive::NMessages::IncorrectStationForPollingType.DoException(Server->GetLocalization());
        }
        if (columnId) {
            tag->SetColumnId(columnId);
        }
        if (tag->HasFuelClientType()) {
            clientType = tag->GetFuelClientTypeRef();
        }
    }
    auto liters = GetValue<double>(requestData, "liters", /* required = */ true);
    ReqCheckCondition(!!liters, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect liters value");
    tag->SetActualLiters(*liters);
    tag->SetLiters(*liters);
    auto fuelType = GetValue<EFuelType>(requestData, "fuel_type", /* required = */ true);
    ReqCheckCondition(!!fuelType, ConfigHttpStatus.SyntaxErrorStatus, "Incorrect fuel type value");
    tag->SetActualFuelType(*fuelType);
    tag->SetFuelType(*fuelType);
    tag->SetCurrentState(EFuelingStatus::FuelingCompleted);
    tag->SetPreOrderId(TFuelingManager::GenerateOrderId());

    if (!DriveApi->GetTagsManager().GetUserTags().UpdateTagData(userTags.front(), permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    return userTags.front();
}

void TSupportCreateFuelingOrderProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Tag, TUserFuelingTag::GetTypeName());
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");

    auto dbTag = UpsertFuelingTag(permissions, requestData);
    TUserFuelingTag* tag = dbTag.MutableTagAs<TUserFuelingTag>();
    if (!tag) {
        NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
    }

    TMessagesCollector errors;
    if (!CreateOrder(*tag, /* postPay = */ true, errors)) {
        NDrive::NMessages::InternalFuelingServiceProblems.DoException(Server->GetLocalization(), /* errorCode = */ "", /* additionalInfo = */ errors.GetStringReport());
    }
    {
        auto session = BuildTx<NSQL::Writable>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(dbTag, permissions->GetUserId(), session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TSupportCreateFuelingOrderProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("user_id", "Пользователь").SetVisual(TFSString::EVisualType::ObjectId).SetRequired(true);
    scheme.Add<TFSString>("car_id", "Машина (обязательно, если нет тега на пользователе)").SetVisual(TFSString::EVisualType::ObjectId);
    const TFuelingManager* manager = server && server->GetAsSafe<NDrive::IServer>().GetFuelingManager() ? server->GetAsSafe<NDrive::IServer>().GetFuelingManager() : nullptr;
    if (manager) {
        TVector<TFSVariants::TCompoundVariant> variants;
        for(auto&& [id, addr] : manager->GetStationsAddresses()) {
            variants.emplace_back(id, addr);
        }
        scheme.Add<TFSVariants>("station_id", "Станция (обязательно, если нет тега на пользователе)").SetCompoundVariants(variants);
    } else {
        scheme.Add<TFSString>("station_id", "Станция (обязательно, если нет тега на пользователе)");
    }
    scheme.Add<TFSString>("column_id", "Колонка (обязательно, если нет тега на пользователе)");
    scheme.Add<TFSString>("liters", "Литры").SetRequired(true);
    scheme.Add<TFSVariants>("fuel_type", "Тип топлива").SetVariants(GetEnumAllValues<EFuelType>()).SetRequired(true);
    TVector<TString> types;
    if (manager) {
        for (auto&& type : GetEnumAllValues<EFuelClientType>()) {
            if (manager->HasFuelClientKey(type)) {
                types.emplace_back(ToString(type));
            }
        }
    } else {
        types.emplace_back(ToString(EFuelClientType::Default));
    }
    scheme.Add<TFSVariants>("fuel_client_type", "Ключ API заправок").SetVariants(types).SetDefault(ToString(EFuelClientType::Default)).SetRequired(true);
    return scheme;
}

void TSupportGetFuelStationColumnsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /* permissions */, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(Server->GetFuelingManager(), ConfigHttpStatus.UnknownErrorStatus, "Incorrect server configuration");
    TString stationId;
    if (TString userId = GetString(Context->GetCgiParameters(), "user_id", /* required = */ false)) {
        TDBTags userTags;
        auto session = BuildTx<NSQL::ReadOnly>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {TUserFuelingTag::GetTypeName()}, userTags, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        if (!userTags.empty()) {
            auto tag = userTags.front().MutableTagAs<TUserFuelingTag>();
            if (!tag) {
                NDrive::NMessages::InternalUserTagsRestoreProblem.DoException(Server->GetLocalization());
            }
            stationId = tag->GetStationId();
        }
    }
    if (!stationId) {
        stationId = GetString(Context->GetCgiParameters(), "station_id", /* required = */ true);
    }
    TStation station;
    ReqCheckCondition(Server->GetFuelingManager()->GetStationInfo(stationId, station), ConfigHttpStatus.SyntaxErrorStatus, "fail to get station data");
    if (!station.GetPostPay()) {
        g.SetExternalReport(NJson::TMapBuilder("post_pay", false)("columns", NJson::TJsonArray()));
        g.SetCode(HTTP_OK);
        return;
    }
    g.AddReportElement("post_pay", true);
    NThreading::TFutures<NJson::TJsonValue> futures;
    for (auto& column : station.GetColumns()) {
        futures.push_back(Server->GetFuelingManager()->GetColumnStatus(stationId, column.GetId()));
    }
    auto report = g.GetReport();
    g.Release();
    NThreading::WaitAll(futures).Subscribe([futures = std::move(futures), report](const NThreading::TFuture<void>& /* waiter */) {
        TJsonReport::TGuard g(report, HTTP_OK);
        NJson::TJsonArray columns;
        for (auto future : futures) {
            if (future.HasValue()) {
                columns.AppendValue(future.ExtractValue());
            } else {
                NJson::TJsonValue errorInfo = NJson::TMapBuilder
                    ("error", NThreading::GetExceptionMessage(future));
                columns.AppendValue(std::move(errorInfo));
            }
        }
        g.AddReportElement("columns", std::move(columns));
    });
}

NDrive::TScheme TSupportGetFuelStationColumnsProcessor::GetCgiParametersScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("user_id", "Пользователь").SetVisual(TFSString::EVisualType::ObjectId);
    const TFuelingManager* manager = server && server->GetAsSafe<NDrive::IServer>().GetFuelingManager() ? server->GetAsSafe<NDrive::IServer>().GetFuelingManager() : nullptr;
    if (manager) {
        TVector<TFSVariants::TCompoundVariant> variants;
        for(auto&& [id, addr] : manager->GetStationsAddresses()) {
            variants.emplace_back(id, addr);
        }
        scheme.Add<TFSVariants>("station_id", "Станция (обязательно, если нет тега на пользователе)").SetCompoundVariants(variants);
    } else {
        scheme.Add<TFSString>("station_id", "Станция (обязательно, если нет тега на пользователе)");
    }
    return scheme;
}
