#include "processor.h"

#include <drive/backend/car_attachments/hardware/hardware.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/server/location/names.h>

namespace {
    TString GetBeaconSearchKey(const TString& imei) {
        return "telematics.settings.imei:" + imei + ":enable_beacon_search";
    }
}

void TBeaconStateProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(requestData);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto deadline = Context->GetRequestDeadline();

    auto objectId = GetString(cgi, { "car_id"sv, "object_id"sv }, true);
    auto object = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);
    R_ENSURE(object, HTTP_NOT_FOUND, "cannot find object" << objectId);

    auto invisibilityInfo = TUserPermissions::TUnvisibilityInfoSet(0);
    auto visibility = permissions->GetVisibility(*object, NEntityTagsManager::EEntityType::Car, &invisibilityInfo);
    R_ENSURE(visibility != TUserPermissions::EVisibility::NoVisible, HTTP_FORBIDDEN, "cannot access " << objectId << ": " << TUserPermissions::ExplainInvisibility(invisibilityInfo));

    TVector<TAtomicSharedPtr<TCarHardwareBeacon>> beacons;
    {
        auto eg = g.BuildEventGuard("GetBeacons");
        auto tx = BuildTx<NSQL::ReadOnly>();
        auto optionalAttachments = Server->GetDriveDatabase().GetCarAttachmentAssignments().GetAttachmentOfType(objectId, EDocumentAttachmentType::CarHardwareBeacon, tx);
        R_ENSURE(optionalAttachments, {}, "cannot GetAttachmentOfType", tx);
        for (auto&& attachment : *optionalAttachments) {
            auto beacon = std::dynamic_pointer_cast<TCarHardwareBeacon>(attachment.GetImpl());
            R_ENSURE(beacon, HTTP_INTERNAL_SERVER_ERROR, "cannot cast " << attachment.GetId() << " as CarHardwareBeacon", tx);
            beacons.push_back(std::move(beacon));
        }
    }

    const TMap<NDrive::TSensorId, TString> sensorNames = {
        { VEGA_GSM_SIGNAL_LEVEL, "gsm_level" },
        { VEGA_INT_TEMP, "internal_temperature" },
        { VEGA_OPERATION_MODE, "operation_mode" },
        { VEGA_TOTAL_SAT_INVIEW, "satellites_inview" },
        { CAN_BATTERY_CHARGE_LEVEL, "battery" }
    };
    const TVector<NDrive::TSensorId> requiredIds = MakeVector(NContainer::Keys(sensorNames));

    auto sensorClient = Server->GetSensorApi();
    R_ENSURE(sensorClient, HTTP_INTERNAL_SERVER_ERROR, "cannot find SensorApi");
    TMap<TString, NThreading::TFuture<TMaybe<NDrive::TLocation>>> locationsGps;
    TMap<TString, NThreading::TFuture<TMaybe<NDrive::TLocation>>> locationsLbs;
    TMap<TString, NThreading::TFuture<NDrive::TMultiSensor>> sensors;
    for (auto&& beacon : beacons) {
        if (!beacon) {
            g.AddEvent("NullBeacon0");
            continue;
        }

        const auto& imei = beacon->GetIMEI();
        locationsGps[imei] = sensorClient->GetLocation(imei);
        locationsLbs[imei] = sensorClient->GetLocation(imei, NDrive::LBSLocationName);
        sensors[imei] = sensorClient->GetSensors(imei, requiredIds);
    }

    NJson::TJsonValue beaconReports = NJson::JSON_ARRAY;
    for (auto&& beacon : beacons) {
        if (!beacon) {
            g.AddEvent("NullBeacon1");
            continue;
        }
        const auto& imei = beacon->GetIMEI();

        NJson::TJsonValue beaconReport;
        beaconReport.InsertValue("imei", imei);
        beaconReport.InsertValue("serial_number", beacon->GetSerialNumber());

        auto beaconLocationGps = locationsGps[imei];
        auto beaconLocationLbs = locationsLbs[imei];
        auto beaconSensors = sensors[imei];
        R_ENSURE(beaconLocationGps.Initialized(), HTTP_INTERNAL_SERVER_ERROR, "uninitialized gps locations for " << imei);
        R_ENSURE(beaconLocationLbs.Initialized(), HTTP_INTERNAL_SERVER_ERROR, "uninitialized lbs locations for " << imei);
        {
            auto eg = g.BuildEventGuard("WaitGpsLocations:" + imei);
            R_ENSURE(beaconLocationGps.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "location wait timeout for " << imei);
        }
        {
            auto eg = g.BuildEventGuard("WaitLbsLocations:" + imei);
            R_ENSURE(beaconLocationLbs.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "location wait timeout for " << imei);
        }
        {
            auto eg = g.BuildEventGuard("WaitSensors:" + imei);
            R_ENSURE(beaconSensors.Wait(deadline), HTTP_GATEWAY_TIME_OUT, "sensors wait timeout for " << imei);
        }

        auto& locationGps = beaconLocationGps.GetValue();
        auto& locationLbs = beaconLocationLbs.GetValue();

        TMaybe<NDrive::TLocation> resultLocation;
        if (locationLbs.Defined() && locationGps.Defined()) {
            resultLocation = locationGps.Get()->Timestamp >= locationLbs->Timestamp ? locationGps : locationLbs;
        } else if (locationGps.Defined()) {
            resultLocation = locationGps;
        } else if (locationLbs.Defined()) {
            resultLocation = locationLbs;
        }

        beaconReport.InsertValue("location", NJson::ToJson(resultLocation.GetRef()));

        for (auto&& sensor : beaconSensors.GetValue()) {
            beaconReport.InsertValue(sensorNames.Value(sensor.Id, ""), sensor.ConvertTo<double>());
        }

        auto key = GetBeaconSearchKey(imei);
        TString value;
        if (Server->GetSettings().GetValue(key, value)) {
            beaconReport.InsertValue("beacon_search_state", IsTrue(value));
        } else {
            beaconReport.InsertValue("beacon_search_state", false);
        }

        beaconReports.AppendValue(std::move(beaconReport));
    }
    g.AddReportElement("beacons", std::move(beaconReports));
    g.SetCode(HTTP_OK);
}

void TBeaconSearchProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    auto objectId = GetString(cgi, { "car_id"sv, "object_id"sv }, true);
    auto object = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetObject(objectId);
    R_ENSURE(object, HTTP_NOT_FOUND, "cannot find object" << objectId);

    auto invisibilityInfo = TUserPermissions::TUnvisibilityInfoSet(0);
    auto visibility = permissions->GetVisibility(*object, NEntityTagsManager::EEntityType::Car, &invisibilityInfo);
    R_ENSURE(visibility != TUserPermissions::EVisibility::NoVisible, HTTP_FORBIDDEN, "cannot access " << objectId << ": " << TUserPermissions::ExplainInvisibility(invisibilityInfo));
    bool searchState = GetValue<bool>(cgi, "search_state", false).GetOrElse(false);

    auto imeis = GetStrings(requestData, "imeis", true);

    TVector<TSetting> settings;
    for (auto&& imei : imeis) {
        TSetting setting;
        auto key = GetBeaconSearchKey(imei);

        setting.SetKey(key);
        setting.SetValue(ToString(searchState));

        settings.emplace_back(setting);
    }

    R_ENSURE(Server->GetSettings().SetValues(settings, permissions->GetUserId()), ConfigHttpStatus.UnknownErrorStatus, "cannot add imeis");

    g.SetCode(HTTP_OK);
}
