#include "processor.h"

#include <drive/backend/cars/hardware.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/head/head_account.h>

#include <rtline/util/types/accessor.h>

void TRegisterHeadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    TString headId = requestData["head_id"].GetString();
    R_ENSURE(headId, ConfigHttpStatus.SyntaxErrorStatus, "Need 'head_id' info");
    R_ENSURE(!DriveApi->GetHeadAccountManager().IsRegistered(headId), ConfigHttpStatus.HeadAppError, "");

    TString publicKey = requestData["public_key"].GetString();
    R_ENSURE(publicKey, ConfigHttpStatus.SyntaxErrorStatus, "Need 'public_key' info");

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetHeadAccountManager().RegisterHead(headId, publicKey, headId, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}


void TGetSessionKeyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    TString userId(Context->GetRequestData().HeaderInOrEmpty("UserId"));
    if (!userId) {
        userId = Context->GetRequestData().HeaderInOrEmpty("DeviceId");
    }
    R_ENSURE(userId, ConfigHttpStatus.SyntaxErrorStatus, "Need 'UserId' header");

    auto sessionInfo = DriveApi->GetShortSessionsManager().GetUserSession(userId);
    if (!sessionInfo.Defined()) {
        sessionInfo = DriveApi->GetShortSessionsManager().GetUserSession("default");
    }
    R_ENSURE(sessionInfo.Defined(), ConfigHttpStatus.HeadAppError, "No session");
    g.MutableReport().AddReportElement("session_key", sessionInfo->GetSessionKey());
    g.SetCode(HTTP_OK);
}


void THeadInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Car);

    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto objectIds = GetUUIDs(cgi, "car_id", false);

    TMap<TString, TString> heads;
    bool acquired = DriveApi->GetHeadAccountManager().GetHeadIds(objectIds, heads);
    R_ENSURE(acquired, ConfigHttpStatus.UnknownErrorStatus, "cannot acquire headIds");

    NJson::TJsonValue objects(NJson::JSON_ARRAY);
    for (auto&&[objectId, headId] : heads) {
        NJson::TJsonValue object;
        object["car_id"] = objectId;
        object["head_id"] = headId;
        objects.AppendValue(std::move(object));
    }
    g.MutableReport().AddReportElement("objects", std::move(objects));
    if (objectIds.size() == 1) {
        const TString& objectId = objectIds.front();
        auto p = heads.find(objectId);
        R_ENSURE(p != heads.end(), ConfigHttpStatus.EmptySetStatus, "cannot find head_id for " << objectId);
        g.MutableReport().AddReportElement("head_id", p->second);
    }
    g.SetCode(HTTP_OK);
}

void TDropPublicKeyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Car);
    TVector<TString> headIds;
    auto session = BuildTx<NSQL::Writable>();
    const TString& carId = Context->GetCgiParameters().Get("car_id");
    if (!!carId) {
        TString headId = DriveApi->GetHeadAccountManager().GetHeadIdByCar(carId);
        R_ENSURE(headId, ConfigHttpStatus.UserErrorState, "Can't find head_id for car");
        headIds.push_back(headId);
    } else {
        TVector<TString> carsNumbers = GetStrings(Context->GetCgiParameters(), "cars", false);
        if (carsNumbers.empty()) {
            headIds = GetStrings(Context->GetCgiParameters(), "head_ids", true);
        }
        auto carIds = DriveApi->GetCarIdByNumbers(carsNumbers);
        R_ENSURE(carIds.size() == carsNumbers.size(), ConfigHttpStatus.ServiceUnavailable, "inconsistent");
        for (ui32 i = 0; i < carIds.size(); ++i) {
            TString headId = DriveApi->GetHeadAccountManager().GetHeadIdByCar(carIds[i]);
            R_ENSURE(headId, ConfigHttpStatus.UserErrorState, "Can't find head_id for car " + carsNumbers[i]);
            headIds.push_back(headId);
        }
    }
    for (auto&& headId : headIds) {
        if (!DriveApi->GetShortSessionsManager().DropPublicKey(headId, permissions->GetUserId(), session)) {
            g.SetCode(ConfigHttpStatus.ConflictRequest);
            return;
        }
    }
    if (!session.Commit()) {
        g.SetCode(ConfigHttpStatus.ConflictRequest);
        return;
    }
    g.SetCode(HTTP_OK);
}

void TRegisterNewHeadForCarProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Heads);

    TString headId = GetString(requestData, "head_id", true);
    TString carId = GetUUID(requestData, "car_id", true);

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetHeadAccountManager().UpdateCarHeadId(carId, headId, permissions->GetUserId(), session, Server) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TGetSessionForHeadProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Heads);
    TString headId = GetString(Context->GetCgiParameters(), "device_id", false);
    if (!headId) {
        headId = GetString(Context->GetCgiParameters(), "head_id", true);
    }
    TString carId;
    {
        TEventsGuard egProcess(g.MutableReport(), "restore_car_id");
        carId = DriveApi->GetCarAttachmentAssignments().GetAttachmentOwnerByServiceAppSlug(headId);
    }
    R_ENSURE(carId, ConfigHttpStatus.UserErrorState, "unknown head_id " << headId, NDrive::MakeError("unknown_head_id"));

    TVector<TDBTag> sessionTags;
    {
        TEventsGuard egProcess(g.MutableReport(), "restore_car_tags");
        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> actualTags;
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreEntityTags({ carId }, {}, actualTags, session),
            ConfigHttpStatus.UnknownErrorStatus,
            "Can't restore tags");

        for (auto&& i : actualTags) {
            if (i.GetTagAs<TChargableTag>()) {
                sessionTags.emplace_back(i);
            }
        }
    }
    bool noSessions = true;
    for (auto&& tag : sessionTags) {
        if (tag->GetPerformer()) {
            noSessions = false;
        }
    }
    if (noSessions || sessionTags.size() != 1) {
        g.SetCode(ConfigHttpStatus.EmptySetStatus);
        return;
    }

    {
        TEventsGuard egProcess(g.MutableReport(), "restore_car_sessions");
        auto builder = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
        R_ENSURE(builder, ConfigHttpStatus.UnknownErrorStatus, "Incorrect session builder 'billing'");
        auto currentSession = builder->GetLastObjectSession(carId);

        if (!currentSession || !currentSession->GetLastEvent()) {
            g.SetCode(ConfigHttpStatus.EmptySetStatus);
            return;
        }

        const auto sessionLastEvent = currentSession->GetLastEvent();
        const TConstDBTag& sessionTag = sessionLastEvent.GetRef();
        const TConstDBTag& currentTag = sessionTags.front();
        if (currentTag.GetTagId() != sessionTag.GetTagId() || currentTag->GetName() != sessionTag->GetName()) {
            // Refresh
            builder = DriveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Context->GetRequestStartTime());
            R_ENSURE(builder, ConfigHttpStatus.UnknownErrorStatus, "Incorrect session builder 'billing'");
            currentSession = builder->GetLastObjectSession(carId);
        }

        if (!!currentSession && !currentSession->GetClosed() && !!currentSession->GetLastEvent() && currentSession->GetLastEvent().GetRef()->GetName() == "old_state_riding") {
            g.MutableReport().AddReportElement("session_id", currentSession->GetSessionId());
            g.SetCode(HTTP_OK);
        } else {
            g.SetCode(ConfigHttpStatus.EmptySetStatus);
        }
    }
}
