#include "processor.h"

#include <drive/backend/auth/blackbox2/blackbox.h>
#include <drive/backend/base/server.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/network/data/data.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/string/cast.h>

class TIDMException : public yexception {
public:
    enum ESeverity {
        Warning,
        Error,
        Fatal,
    };

public:
    TIDMException(ESeverity severity)
        : Severity(severity)
    {
    }

    ESeverity GetSeverity() const {
        return Severity;
    }
    TStringBuf GetSeverityString() const {
        switch (Severity) {
        case Warning: return "warning"sv;
        case Error: return "error"sv;
        case Fatal: return "fatal"sv;
        }
    }

private:
    const ESeverity Severity;
};

constexpr TStringBuf DriveSlug = "drive_role";

const TDriveAPI& TIDMCommonProcessor::GetAPI() const {
    const auto server = BaseServer->GetAs<NDrive::IServer>();
    Y_ENSURE_EX(server, TIDMException(TIDMException::Fatal) << "internal error: server not found");
    const auto api = server->GetDriveAPI();
    Y_ENSURE_EX(api, TIDMException(TIDMException::Fatal) << "internal error: api not found");
    return *api;
}

const TRolesManager& TIDMCommonProcessor::GetRolesManager() const {
    auto rolesManager = GetAPI().GetRolesManager();
    Y_ENSURE_EX(rolesManager, TIDMException(TIDMException::Fatal) << "internal error: roles manager not found");
    return *rolesManager;
}

TUsersDB& TIDMCommonProcessor::GetUsersDB() const {
    auto usersDb = GetAPI().GetUsersData();
    Y_ENSURE_EX(usersDb, TIDMException(TIDMException::Fatal) << "internal error: users data not found");
    return *usersDb;
}

TIDMCommonProcessor::TModifyRoleRequest TIDMCommonProcessor::ParseRequest(const TString& operatorUserId, NDrive::TEntitySession& session) const {
    const TBlob& data = Context->GetBuf();
    TStringBuf post(data.AsCharPtr(), data.Size());
    TCgiParameters form(post);

    const TUsersDB& users = GetUsersDB();
    TModifyRoleRequest result;

    const TString& login = form.Get("login");
    Y_ENSURE_EX(login, TIDMException(TIDMException::Error) << "cannot find login field");

    TVector<TString> userIds;
    Y_ENSURE_EX(users.SelectUsers("username", { login }, userIds, session), TIDMException(TIDMException::Error) << "cannot select users with login " << login);
    {
        auto usersFetchResult = users.FetchInfo(userIds, session);
        TVector<TString> filteredUserIds;
        for (auto&& id : userIds) {
            auto info = usersFetchResult.GetResultPtr(id);
            if (info && !info->IsStaffAccount()) {
                continue;
            }
            filteredUserIds.push_back(id);
        }
        userIds = std::move(filteredUserIds);
    }

    if (userIds.empty() && BaseServer) {
        auto blackboxAuthModuleName = BaseServer->GetSettings().GetValue<TString>("idm.blackbox_auth_module").GetOrElse("blackbox_staff");
        auto blackboxAuthModuleConfig = BaseServer->GetAuthModuleInfo(blackboxAuthModuleName);
        auto blackboxAuthModule = blackboxAuthModuleConfig ? blackboxAuthModuleConfig->ConstructAuthModule(BaseServer) : nullptr;
        auto authModule = std::dynamic_pointer_cast<TBlackbox2AuthModule>(std::move(blackboxAuthModule));
        auto blackboxClient = authModule ? authModule->GetClient() : nullptr;
        if (blackboxClient) {
            auto asyncResponse = blackboxClient->LoginInfoRequest(login, NUtil::GetClientIp(Context->GetRequestData()));
            Y_ENSURE_EX(asyncResponse.Wait(Context->GetRequestDeadline()), TIDMException(TIDMException::Error) << "blackbox wait timeout");
            auto response = asyncResponse.ExtractValue();
            Y_ENSURE_EX(response, TIDMException(TIDMException::Error) << "null blackbox response");
            auto info = blackboxClient->Parse(*response);
            auto user = GetAPI().GetUsersData()->RegisterNewUser(operatorUserId, info.PassportUid, info.Login, session, info.DefaultPhone, info.DefaultEmail);
            R_ENSURE(user, ConfigHttpStatus.UnknownErrorStatus, "cannot register new user", session);
            userIds.push_back(user->GetUserId());
        }
    }
    Y_ENSURE_EX(!userIds.empty(), TIDMException(TIDMException::Error) << "no users with login " << login);

    auto usersFetchResult = users.FetchInfo(userIds, session);
    {
        TVector<TString> filteredUserIds;
        for (auto&& id : userIds) {
            auto info = usersFetchResult.GetResultPtr(id);
            if (info && !info->IsStaffAccount()) {
                continue;
            }
            filteredUserIds.push_back(id);
        }
        userIds = std::move(filteredUserIds);
    }

    Y_ENSURE_EX(!userIds.empty(), TIDMException(TIDMException::Error) << "no users with login " << login);
    Y_ENSURE_EX(userIds.size() == 1, TIDMException(TIDMException::Error) << "multiple users with login " << login);
    result.User = userIds[0];

    const TString& roleField = form.Get("role");
    Y_ENSURE_EX(roleField, TIDMException(TIDMException::Error) << "cannot find role field");
    NJson::TJsonValue roleDescription;
    Y_ENSURE_EX(NJson::ReadJsonFastTree(roleField, &roleDescription), TIDMException(TIDMException::Error) << "cannot parse role field");
    result.Role = roleDescription[DriveSlug].GetStringRobust();
    Y_ENSURE_EX(result.Role, TIDMException(TIDMException::Error) << "empty role field");

    return result;
}

void TIDMCommonProcessor::DoAuthProcess(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    TJsonReport& report = g.MutableReport();
    try {
        Y_ENSURE_EX(authInfo, TIDMException(TIDMException::Error) << "cannot authenticate IDM");
        Process(g, authInfo);
        report.AddReportElement("code", 0);
    } catch (const TIDMException& e) {
        report.AddReportElement("code", 1);
        report.AddReportElement(ToString(e.GetSeverityString()), e.AsStrBuf());
    } catch (const std::exception& e) {
        report.AddReportElement("code", 1);
        report.AddReportElement("fatal", FormatExc(e));
    }
    g.SetCode(HTTP_OK);
}

void TIDMInfoProcessor::Process(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    Y_UNUSED(authInfo);
    const TRolesManager& manager = GetRolesManager();

    NJson::TJsonValue roles;
    roles["slug"] = DriveSlug;
    roles["name"]["en"] = "user role in Yandex.Drive";
    roles["name"]["ru"] = "Роль в Яндекс.Драйв";
    NJson::TJsonValue& value = roles.InsertValue("values", NJson::JSON_MAP);
    for (auto&& i : manager.GetRoles(TInstant::Zero())) {
        if (!i.GetIsIDM()) {
            continue;
        }

        NJson::TJsonValue role;
        role["name"]["en"] = i.GetName();
        role["name"]["ru"] = i.GetName();
        role["help"]["ru"] = i.GetDescription();
        role["help"]["en"] = i.GetDescription();
        value.InsertValue(i.GetName(), std::move(role));
    }
    g.MutableReport().AddReportElement("roles", std::move(roles));
}

void TIDMAddRoleProcessor::Process(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    Y_UNUSED(g);
    const TRolesManager& manager = GetRolesManager();
    const auto& roles = manager.GetRoles(TInstant::Zero());

    auto operatorUserId = authInfo ? authInfo->GetUserId() : "idm";
    auto session = manager.BuildSession();
    const TModifyRoleRequest& request = ParseRequest(operatorUserId, session);

    auto p = std::find_if(roles.begin(), roles.end(), [&request](const TDriveRoleHeader& header) {
        return header.GetName() == request.Role;
    });
    Y_ENSURE_EX(p != roles.end(), TIDMException(TIDMException::Fatal) << "role " << request.Role << " is not found");
    Y_ENSURE_EX(p->GetIsIDM(), TIDMException(TIDMException::Fatal) << "role " << request.Role << " is not public");

    TUserRole role;
    role.SetUserId(request.User);
    role.SetRoleId(request.Role);

    Y_ENSURE_EX(manager.UpsertRoleForUser(role, authInfo->GetUserId(), session), TIDMException(TIDMException::Error) << "cannot upsert role: " << session.GetStringReport());
    Y_ENSURE_EX(session.Commit(), TIDMException(TIDMException::Error) << "cannot commit transaction: " << session.GetStringReport());
}

void TIDMRemoveRoleProcessor::Process(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    Y_UNUSED(g);
    const TRolesManager& manager = GetRolesManager();
    auto operatorUserId = authInfo ? authInfo->GetUserId() : "idm";
    auto session = manager.BuildSession();

    const TModifyRoleRequest& request = ParseRequest(operatorUserId, session);
    Y_ENSURE_EX(manager.RemoveRoleForUser(request.User, request.Role, authInfo->GetUserId(), session), TIDMException(TIDMException::Error) << "cannot remove role: " << session.GetStringReport());
    Y_ENSURE_EX(session.Commit(), TIDMException(TIDMException::Error) << "cannot commit transaction: " << session.GetStringReport());
}

void TIDMGetAllRolesProcessor::Process(TJsonReport::TGuard& g, IAuthInfo::TPtr authInfo) {
    Y_UNUSED(authInfo);
    const TRolesManager& rolesManager = GetRolesManager();
    TUsersDB& usersManager = GetUsersDB();

    TSet<TString> privateRoles;
    for (auto&& i : rolesManager.GetRoles(TInstant::Zero())) {
        if (!i.GetIsIDM()) {
            continue;
        }
        privateRoles.insert(i.GetName());
    }

    TMap<TString, TSet<TString>> userRoles;
    {
        TVector<TString> ids;
        TSet<TString> roles;
        for (auto&& role : privateRoles) {
            roles = { role };
            ids.clear();
            Y_ENSURE_EX(usersManager.GetRoles().GetUsersWithRoles(roles, ids), TIDMException(TIDMException::Error) << "cannot receive users by roles");
            for (auto&& id : ids) {
                userRoles[id].insert(role);
            }
        }
    }

    TSet<TString> userIds;
    for (auto&&[id, roles] : userRoles) {
        userIds.insert(id);
    }
    auto usersFetchResult = usersManager.FetchInfo(userIds);

    NJson::TJsonValue users(NJson::JSON_ARRAY);
    for (auto&&[id, info] : usersFetchResult) {
        if (!info.IsStaffAccount()) {
            continue;
        }

        NJson::TJsonValue user;
        user["login"] = info.GetLogin();
        NJson::TJsonValue& roles = user.InsertValue("roles", NJson::JSON_ARRAY);
        for (auto&& i : userRoles[id]) {
            NJson::TJsonValue role;
            role.InsertValue(DriveSlug, i);
            roles.AppendValue(std::move(role));
        }
        users.AppendValue(std::move(user));
    }
    g.MutableReport().AddReportElement("users", std::move(users));
}

TSimpleProcessorRegistrator<TIDMInfoProcessor> IDMInfoProcessorRegistrator("idm_info");
TSimpleProcessorRegistrator<TIDMAddRoleProcessor> IDMAddRoleProcessorRegistrator("idm_add_role");
TSimpleProcessorRegistrator<TIDMRemoveRoleProcessor> IDMRemoveRoleProcessorRegistrator("idm_remove_role");
TSimpleProcessorRegistrator<TIDMGetAllRolesProcessor> IDMGetAllRolesProcessorRegistrator("idm_get_all_roles");
