#include "processor.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/images/database.h>
#include <drive/backend/images/image.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags.h>

#include <drive/library/cpp/image_transformation/resize.h>

#include <rtline/library/json/merge.h>
#include <rtline/util/algorithm/ptr.h>

#include <library/cpp/digest/md5/md5.h>

#include <util/generic/algorithm.h>

void TRegisterPhotoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TString tagName = GetString(requestData, "tag_name", false);

    const auto entity = GetValue<NEntityTagsManager::EEntityType>(requestData, "entity", false).GetOrElse(NEntityTagsManager::EEntityType::Car);
    const IEntityTagsManager& tagsManager = DriveApi->GetEntityTagsManager(entity);

    TVector<TDBTag> tags;
    bool updateHandler = permissions->GetSetting<bool>("user.new_photos_register_handler", false);

    if (tagName) {
        if (!updateHandler) {
            R_ENSURE(permissions->GetTagNamesByAction(TTagAction::ETagAction::UpdateOnPerform).contains(tagName), ConfigHttpStatus.PermissionDeniedStatus, "not permitted tag");
        }
        auto carId = GetString(requestData, { TStringBuf("car_id"), TStringBuf("object_id") }, true);
        {
            NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
            if (!tagsManager.RestoreTags({ carId }, { tagName }, tags, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }

            if (tags.empty()) {
                auto tag = DriveApi->GetTagsManager().GetTagsMeta().CreateTag(tagName, TypeName);
                if (!tagsManager.AddTag(tag, permissions->GetUserId(), carId, Server, session, EUniquePolicy::SkipIfExists)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }

            if (!tagsManager.RestoreTags({ carId }, { tagName }, tags, session) || !session.Commit()) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    } else {
        auto tagIds = GetUUIDs(requestData, "tag_ids", false);
        {
            NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
            if (!tagsManager.RestoreTags(MakeSet(tagIds), tags, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!updateHandler) {
            if (Config.GetRequireTagPerform()) {
                for (auto&& tag : tags) {
                    R_ENSURE(
                        tag->GetPerformer() == permissions->GetUserId(),
                        ConfigHttpStatus.PermissionDeniedStatus,
                        "have not permissions to work with tag " + tag.GetTagId(),
                        EDriveSessionResult::NoUserPermissions
                    );
                }
            } else {
                for (auto&& tag : tags) {
                    R_ENSURE(permissions->GetTagNamesByAction(TTagAction::ETagAction::Update).contains(tag->GetName()), ConfigHttpStatus.PermissionDeniedStatus, "no permissions to update " << tag->GetName() << " tag");
                }
            }
        }
    }

    if (updateHandler) {
        for (auto&& tag : tags) {
            TTagAction::ETagAction tagAction;
            if (!tag->GetPerformer()) {
                tagAction = TTagAction::ETagAction::Update;
            } else if (tag->GetPerformer() == permissions->GetUserId()) {
                tagAction = TTagAction::ETagAction::UpdateOnPerform;
            } else {
                tagAction = TTagAction::ETagAction::UpdateOnPerformByAnother;
            }
            R_ENSURE(
                permissions->GetTagNamesByAction(tagAction).contains(tag->GetName()),
                ConfigHttpStatus.PermissionDeniedStatus,
                "no permissions to make " << tagAction  << " on " << tag->GetName() << " tag");
        }
    }
    R_ENSURE(!tags.empty(), ConfigHttpStatus.UserErrorState, "no_tags");
    //R_ENSURE(tags.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "multiple tags deprecated");

    auto& tagToProcess = tags.front();

    auto description = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagToProcess->GetName());
    R_ENSURE(description, ConfigHttpStatus.SyntaxErrorStatus, "unknow tag description");

    auto groupingTags = description->GetGrouppingTags();
    const TString source = groupingTags.empty() ? "default" : *groupingTags.begin();

    NJson::TJsonValue extraMetaData = GetTagDerivedMetaData(entity, tagToProcess);

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();

    const auto& imagesStorage = DriveApi->GetImagesDB();
    auto snapshot = imagesStorage.BuildSnapshot(requestData, tagToProcess, entity, source, permissions->GetUserId(), Config.GetGeneratePreviewPath(), session, {}, extraMetaData);
    R_ENSURE(snapshot, ConfigHttpStatus.SyntaxErrorStatus, "snapshot construction fails");

    const auto complaintId = GetString(requestData, "complaint_id", false);
    if (!complaintId.empty()) {
        R_ENSURE(DriveApi->HasMDSClient(), HTTP_INTERNAL_SERVER_ERROR, "No MDS configured available", session);
        const auto& mdsClient = DriveApi->GetMDSClient();

        auto complaintConfig = GetComplaintConfig(complaintId, requestData);
        R_ENSURE(complaintConfig, ConfigHttpStatus.SyntaxErrorStatus, "Bad complaint config id", session);

        auto topicLink = GetString(requestData, "chat_id", false);
        auto carComplaintTag = MakeHolder<TCarComplaintTag>(complaintConfig->TagName);
        R_ENSURE(carComplaintTag, ConfigHttpStatus.UnknownErrorStatus, "cannot construct complaint tag", session);
        TMap<TString, THolder<TCarComplaintTag::TComplaintSource>> originToComplaintSource;

        if (const TImagesSnapshot* imagesSnapshot = dynamic_cast<const TImagesSnapshot*>(snapshot.Get())) {
            TSet<TString> imagePaths;
            for (auto&& image : imagesSnapshot->GetImages()) {
                const auto& origin = image.Origin;

                if (complaintConfig->LostItemType != TCarComplaintTag::ELostItemType::None) {
                    auto source = MakeHolder<TCarComplaintTag::TLostItemSource>();
                    source->SetLostItemType(complaintConfig->LostItemType);
                    source->SetDocument(complaintConfig->Document);
                    source->SetItemDescription(complaintConfig->ItemDescription);
                    originToComplaintSource[origin] = std::move(source);
                } else {
                    originToComplaintSource[origin] = MakeHolder<TCarComplaintTag::TComplaintSource>();
                }

                auto& complaintSource = originToComplaintSource[origin];
                auto imagePath = TCommonImageData::BuildUrl(image.Path, mdsClient);
                complaintSource->MutableImagePaths().insert(imagePath);
                complaintSource->SetSourceName(origin);
                complaintSource->SetUserId(permissions->GetUserId());
                complaintSource->SetUserAvailable(complaintConfig->IsUserAvailable);
                complaintSource->SetLocation(complaintConfig->Location);
                complaintSource->SetAddedWeight(complaintConfig->Weight);
                complaintSource->SetTopicLink(topicLink);
                if (imagesSnapshot->HasComment()) {
                    complaintSource->SetComment(imagesSnapshot->GetCommentRef());
                }
                complaintSource->SetCreatedAt(Context->GetRequestStartTime());
            }
        }
        carComplaintTag->SetComplaintSources(std::move(originToComplaintSource));

        R_ENSURE(tagToProcess->GetName() == carComplaintTag->GetName(), ConfigHttpStatus.UserErrorState, "tags\' type can\'t be changed unless evolved");
        if (!tagsManager.UpdateTagData(tagToProcess, std::move(carComplaintTag), permissions->GetUserId(), Server, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!tagsManager.AddSnapshot(tagToProcess, snapshot, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (Config.GetReportResultSnapshot()) {
        g.MutableReport().SetExternalReport(snapshot->SerializeToJson());
    }
    g.SetCode(HTTP_OK);
}

TMaybe<TCarComplaintConfig> TRegisterPhotoProcessor::GetComplaintConfig(const TString& complaintId, const NJson::TJsonValue& requestData) const {
    TCarComplaintConfig result;
    if (NJson::ParseField(requestData, result, false)) {
        return result;
    }
    auto complaintConfigs = Server->GetSettings().GetJsonValue("car_complaints.config");
    for (auto&& complaintConfigJson : complaintConfigs.GetArray()) {
        if (NJson::ParseField(complaintConfigJson, result, false) && result.Id == complaintId) {
            return result;
        }
    }
    return {};
}

NJson::TJsonValue TRegisterPhotoProcessor::GetTagDerivedMetaData(const NEntityTagsManager::EEntityType entityType, const TDBTag& tag) const {
    NJson::TJsonValue metaData;

    auto vinDependentTagsJson = Server->GetSettings().GetJsonValue("images.derived_meta_info.vin_dependent_tags");
    const bool requireVinInfo = AnyOf(vinDependentTagsJson.GetArray(), [&tag](const auto& tagNameJson){ return tag->GetName() == tagNameJson.GetString(); });
    if (requireVinInfo && entityType == NEntityTagsManager::EEntityType::Car) {
        auto carId = tag.GetObjectId();
        auto gCars = DriveApi->GetCarsData()->FetchInfo(carId, TInstant::Zero());
        auto carDataPtr = gCars.GetResultPtr(carId);
        metaData.InsertValue("vin", ((carDataPtr) ? carDataPtr->GetVin() : ""));
    }

    auto mileageDependentTagsJson = Server->GetSettings().GetJsonValue("images.derived_meta_info.mileage_dependent_tags");
    const bool requireMileageInfo = AnyOf(mileageDependentTagsJson.GetArray(), [&tag](const auto& tagNameJson){ return tag->GetName() == tagNameJson.GetString(); });
    if (requireMileageInfo && entityType == NEntityTagsManager::EEntityType::Car) {
        auto carId = tag.GetObjectId();
        auto snapshotPtr = Server->GetSnapshotsManager().GetSnapshotPtr(carId);
        auto deviceSnapshotPtr = std::dynamic_pointer_cast<TRTDeviceSnapshot>(snapshotPtr);
        metaData.InsertValue("mileage", ((deviceSnapshotPtr) ? deviceSnapshotPtr->GetMileage().GetOrElse(0.0) : 0.0));
    }

    auto documentsFlagDependentTagsJson = Server->GetSettings().GetJsonValue("images.derived_meta_info.documents_flag_dependent_tags");
    const bool requireHasDocumentsFlag = AnyOf(documentsFlagDependentTagsJson.GetArray(), [&tag](const auto& tagNameJson){ return tag->GetName() == tagNameJson.GetString(); });
    if (requireHasDocumentsFlag && entityType == NEntityTagsManager::EEntityType::Car) {
        metaData.InsertValue("has_documents", true);
    }

    return metaData;
}

void TUploadServicePhotoProcessor::Process(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    Y_UNUSED(permissions);
    R_ENSURE(DriveApi->HasMDSClient(), HTTP_INTERNAL_SERVER_ERROR, "No MDS configured available");
    R_ENSURE(Config.GetPreviewMaxSideSize(), HTTP_INTERNAL_SERVER_ERROR, "Invalid image preview size configured");

    TBlob image = Context->GetBuf();
    R_ENSURE(!image.Empty(), ConfigHttpStatus.UserErrorState, "no_image");

    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString objectId = GetString(cgi, { TStringBuf("object_id"), TStringBuf("car_id") }, true);
    TString photoId = GetString(cgi, "photo_id", true);
    const auto entity = GetValue<NEntityTagsManager::EEntityType>(cgi, "entity", false).GetOrElse(NEntityTagsManager::EEntityType::Car);
    const IEntityTagsManager& entityTags = DriveApi->GetEntityTagsManager(entity);

    TMaybe<TImageToUploadDescription> imageDescription;
    if (Config.GetEnableServiceSessions()) {
        auto acceptanceTs = ModelingNow() - Config.GetAcceptanceInterval();
        imageDescription = FetchImageDescriptionFromTagsHistory(entityTags, objectId, photoId, acceptanceTs);
    } else {
        imageDescription = FetchImageDescriptionFromTags(entityTags, objectId, photoId);
    }

    R_ENSURE(imageDescription.Defined() && imageDescription->ImageId, ConfigHttpStatus.UserErrorState, "unknown_photo");

    const auto checkDataHash = BaseServer->GetSettings().GetValueDef<bool>("photos.check_data_hash", true);
    if (!imageDescription->MD5.empty() && checkDataHash) {
        TString dataHash = MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size()));
        R_ENSURE(dataHash == imageDescription->MD5, ConfigHttpStatus.UserErrorState, "md5_check_fails");
    }

    TMessagesCollector errors;
    R_ENSURE(DriveApi->GetMDSClient().UploadBlob(Config.GetBucketName(), imageDescription->Path, image, errors), ConfigHttpStatus.UnknownErrorStatus, "upload_fails:" + errors.GetStringReport());

    if (imageDescription->PreviewPath) {
        auto resizer = NImageTransformation::TResize(Config.GetPreviewMaxSideSize());

        TBlob preview;
        if (!resizer.Transform(image, preview, errors /* jpg */) && !resizer.Transform(image, preview, errors, "png")) {
            if (BaseServer->GetSettings().GetValueDef<bool>("photos.skip_failed_transformation", false)) {
                g.SetCode(HTTP_OK);
                return;
            }
            R_ENSURE(false, ConfigHttpStatus.UserErrorState, "preview_making_error:" << errors.GetStringReport());
        }

        R_ENSURE(DriveApi->GetMDSClient().UploadBlob(Config.GetBucketName(), imageDescription->PreviewPath, preview, errors), ConfigHttpStatus.UnknownErrorStatus, "preview_upload_fails:" + errors.GetStringReport());
    }

    g.SetCode(HTTP_OK);
}

TMaybe<TUploadServicePhotoProcessor::TImageToUploadDescription> TUploadServicePhotoProcessor::FetchImageDescriptionFromTags(const IEntityTagsManager& entityTags, const TString& objectId, const TString& photoId) const {
    auto session = DriveApi->template BuildTx<NSQL::ReadOnly>();
    auto expectedTags = entityTags.RestoreEntityTags(objectId, {}, session);
    R_ENSURE(expectedTags, ConfigHttpStatus.UnknownErrorStatus, "cannot RestoreEntityTags", session);

    for (auto&& expectedTag : *expectedTags) {
        if (expectedTag) {
            const auto& tag = expectedTag;
            auto snapshot = tag->GetObjectSnapshotAs<TImagesSnapshot>();
            if (snapshot) {
                for (auto&& image : snapshot->GetImages()) {
                    if (image.ExternalId == photoId) {
                        return TImageToUploadDescription(*image.ImageId, image.Path, image.PreviewPath, image.MD5);  // TImagesSnapshot::TImage
                    }
                }
            }
        }
    }

    return {};
}

TMaybe<TUploadServicePhotoProcessor::TImageToUploadDescription> TUploadServicePhotoProcessor::FetchImageDescriptionFromTagsHistory(const IEntityTagsManager& entityTags, const TString& objectId, const TString& photoId, const TInstant acceptanceTs) const {
    auto session = DriveApi->template BuildTx<NSQL::ReadOnly>();
    auto tagEvents = entityTags.GetEventsByObject(objectId, session, 0, acceptanceTs, TTagEventsManager::TQueryOptions()
        .SetActions({ EObjectHistoryAction::AddSnapshot })
    );
    R_ENSURE(tagEvents, HTTP_INTERNAL_SERVER_ERROR, "cannot GetEventsByObject " << objectId, session);

    for (auto&& event: *tagEvents) {
        if (event.GetHistoryAction() == EObjectHistoryAction::AddSnapshot) {
            auto snapshot = event->GetObjectSnapshotAs<TImagesSnapshot>();
            if (snapshot) {
                for (auto&& image : snapshot->GetImages()) {
                    if (image.ExternalId == photoId) {
                        return TImageToUploadDescription(*image.ImageId, image.Path, image.PreviewPath, image.MD5);  // TImagesSnapshot::TImage
                    }
                }
            }
        }
    }

    return {};
}

void TGetServicePhotoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    // NB. Same as TGetMarkedPhotosProcessor, however it process tags AddSnapshot events and group or filter photos by tags
    // NB. TGetMarkedPhotosProcessor is preferred to be used further if no tags info nedded
    const TCgiParameters& cgi = Context->GetCgiParameters();

    auto entity = GetValue<NEntityTagsManager::EEntityType>(cgi, "entity", false).GetOrElse(NEntityTagsManager::EEntityType::Car);
    TString objectId = GetUUID(cgi, TVector<TStringBuf>{TStringBuf{"car_id"}, TStringBuf{"object_id"}}, true);

    TInstant since = GetTimestamp(cgi, "since", TInstant::Now() - TDuration::Days(15));
    TInstant until = GetTimestamp(cgi, "until", TInstant::Now());

    TString sessionId = GetString(cgi, "session_id", false);

    // specific filters; incident related photos could be filtered by origin and specific marker
    TString marker = GetString(cgi, "marker", false);
    TString origin = GetString(cgi, "origin", false);

    auto tags = GetStrings(cgi, "tag_names", false);

    auto session = BuildTx<NSQL::ReadOnly>();
    {
        auto taggedDevice = DriveApi->GetEntityTagsManager(entity).RestoreObject(objectId, session);
        ReqCheckCondition(taggedDevice.Defined(), ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::ObjectNotFound);
        ReqCheckCondition(permissions->GetVisibility(*taggedDevice, entity) != TUserPermissions::EVisibility::NoVisible, ConfigHttpStatus.PermissionDeniedStatus, EDriveLocalizationCodes::NoPermissions);
    }

    TMap<ui64, TCommonImageData> imagesInfoMapping;
    {
        auto optionalImages = DriveApi->GetImagesDB().Get(objectId, {since, until}, session);
        if (!optionalImages) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        for (auto&& image : *optionalImages) {
            imagesInfoMapping.emplace(image.GetImageId(), std::move(image));
        }
    }

    auto hostByImageSourceMapping = DriveApi->HasMDSClient()
        ? TCommonImageData::GetHostByImageSourceMapping(DriveApi->GetMDSClient())
        : TMap<TString, TString>();

    auto optionalEvents = DriveApi->GetEntityTagsManager(entity).GetEventsByObject(objectId, session, 0, since);
    if (!optionalEvents) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    auto requestedTags = MakeSet(tags);
    auto permittedTags = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);

    TSet<TString> collectedTags;

    NJson::TJsonValue report;
    for (auto&& ev : *optionalEvents) {
        if (!ev) {
            continue;
        }
        if (ev.GetHistoryInstant() >= until) {
            break;
        }
        if (ev.GetHistoryAction() != EObjectHistoryAction::AddSnapshot) {
            continue;
        }

        const TString& tagName = ev->GetName();
        if (requestedTags && !requestedTags.contains(tagName)) {
            continue;
        }

        if (!permittedTags.contains(tagName)) {
            continue;
        }

        auto snapshot = ev->GetObjectSnapshotAs<TImagesSnapshot>();
        if (!snapshot) {
            continue;
        }

        NJson::TJsonValue rawImages(NJson::JSON_ARRAY);
        for (auto&& image: snapshot->GetImages()) {
            auto imageInfoPtr = imagesInfoMapping.FindPtr(*image.ImageId);

            if ((sessionId && imageInfoPtr && sessionId != imageInfoPtr->GetSessionId()) ||
                (marker && marker != image.Marker) ||
                (origin && origin != image.Origin)
            ) {
                continue;
            }

            auto imageReport = NJson::ToJson(image);
            if (imageInfoPtr) {
                NJson::MergeJson(imageInfoPtr->BuildReport(hostByImageSourceMapping), imageReport);
            }
            rawImages.AppendValue(std::move(imageReport));
        }

        if (rawImages.GetArray().empty()) {
            continue;
        }

        collectedTags.emplace(tagName);

        NJson::TJsonValue imageSnapshotReport;
        imageSnapshotReport["tag"] = tagName;
        imageSnapshotReport["tag_id"] = ev.GetTagId();
        imageSnapshotReport["timestamp"] = ev.GetHistoryInstant().Seconds();
        imageSnapshotReport["history_user_id"] = ev.GetHistoryUserId();
        imageSnapshotReport["images"] = std::move(rawImages);

        report[tagName].AppendValue(std::move(imageSnapshotReport));
    }
    g.MutableReport().AddReportElement("images", std::move(report));

    NJson::TJsonValue displayTagNamesMapping(NJson::JSON_MAP);
    {
        const ITagsMeta& tagsMeta = DriveApi->GetTagsManager().GetTagsMeta();
        for (auto&& [tagName, tagDescriptionPtr]: tagsMeta.GetRegisteredTags()) {
            if (collectedTags.contains(tagName)) {
                TString displayTagName = (tagDescriptionPtr) ? tagDescriptionPtr->GetDisplayName() : tagName;
                displayTagNamesMapping.InsertValue(tagName, displayTagName);
            }
        }
    }
    g.MutableReport().AddReportElement("display_tag_names", std::move(displayTagNamesMapping));
    g.SetCode(HTTP_OK);
}

void TGetMarkedPhotosProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    // NB. Same as TGetServicePhotoProcessor, however it does not process tags AddSnapshot events and do not group or filter photos by tags
    // NB. It's preferred to be used further instead of others
    const TCgiParameters& cgi = Context->GetCgiParameters();

    auto entity = GetValue<NEntityTagsManager::EEntityType>(cgi, "entity", false).GetOrElse(NEntityTagsManager::EEntityType::Car);
    TString objectId = GetUUID(cgi, TVector<TStringBuf>{TStringBuf{"car_id"}, TStringBuf{"object_id"}}, false);

    TInstant since = GetTimestamp(cgi, "since", TInstant::Now() - TDuration::Days(15));
    TInstant until = GetTimestamp(cgi, "until", TInstant::Now());

    TString sessionId = GetString(cgi, "session_id", false);

    // specific filters; incident related photos could be filtered by origin and specific marker
    TString marker = GetString(cgi, "marker", false);
    TString origin = GetString(cgi, "origin", false);
    TString source = GetString(cgi, "source", false);

    bool reportDups = GetValue<bool>(cgi, "report_dups", false).GetOrElse(false);
    bool reportImages = GetValue<bool>(cgi, "report_images", false).GetOrElse(true);
    bool reportValidated = GetValue<bool>(cgi, "report_validated", false).GetOrElse(false);

    auto locale = GetLocale();

    auto tx = BuildTx<NSQL::ReadOnly>();
    if (!objectId && sessionId) {
        auto eg = g.BuildEventGuard("find_active_session");
        auto optionalSession = DriveApi->GetSessionManager().GetSession(sessionId, tx);
        R_ENSURE(optionalSession, {}, "cannot GetSession " << sessionId, tx);
        auto session = *optionalSession;
        if (session) {
            objectId = session->GetObjectId();
            since = session->GetStartTS();
            until = Now();
        }
    }
    if (!objectId && sessionId) {
        auto eg = g.BuildEventGuard("find_session");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly | NSQL::Deferred>("get_marked_photos");
        const auto& compiledManager = DriveApi->GetMinimalCompiledRides();
        auto optionalCompiledSessions = compiledManager.Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTx);
        R_ENSURE(optionalCompiledSessions, {}, "cannot GetMinimalCompiledRiding " << sessionId, tx);
        R_ENSURE(!optionalCompiledSessions->empty(), HTTP_NOT_FOUND, "cannot find session " << sessionId, tx);
        for (auto&& compiledSession : *optionalCompiledSessions) {
            objectId = compiledSession.GetObjectId();
            since = compiledSession.GetStartInstant();
            until = Now();
            break;
        }
    }
    R_ENSURE(objectId, HTTP_BAD_REQUEST, "cannot determine object_id", tx);
    {
        auto taggedDevice = Server->GetDriveAPI()->GetEntityTagsManager(entity).RestoreObject(objectId, tx);
        ReqCheckCondition(taggedDevice.Defined(), ConfigHttpStatus.UserErrorState, EDriveLocalizationCodes::ObjectNotFound);
        ReqCheckCondition(permissions->GetVisibility(*taggedDevice, entity) != TUserPermissions::EVisibility::NoVisible, ConfigHttpStatus.PermissionDeniedStatus, EDriveLocalizationCodes::NoPermissions);
    }

    const auto& imagesDB = DriveApi->GetImagesDB();
    auto optionalImages = imagesDB.Get(objectId, {since, until}, tx);
    R_ENSURE(optionalImages, {}, "cannot Get images", tx);

    TMap<TString, TVector<TCommonImageData>> dups;
    if (reportDups) {
        auto eg = g.BuildEventGuard("build_dups");
        auto allImages = imagesDB.Get(objectId, {}, tx);
        R_ENSURE(allImages, {}, "cannot Get all images", tx);
        eg.AddEvent("fetched_all_photos");
        for (auto&& image : *allImages) {
            for (auto&& markup : image.GetMarkUpList()) {
                const auto& description = markup.GetDescription();
                if (description == UnknownCarDamageDescription) {
                    continue;
                }
                if (markup.IsDiscarded()) {
                    continue;
                }
                dups[description].push_back(image);
            }
        }
    }

    auto hostByImageSourceMapping = DriveApi->HasMDSClient()
        ? TCommonImageData::GetHostByImageSourceMapping(DriveApi->GetMDSClient())
        : TMap<TString, TString>();

    if (reportValidated) {
        auto eg = g.BuildEventGuard("validated_images");
        auto validatedImages = imagesDB.GetCachedValidatedImages(objectId, Context->GetRequestDeadline(), GetHandlerSetting<TDuration>("car.damage.cache_lifetime"));
        const auto validatedDetails = GetHandlerSettingDef<bool>("car.damage.validated_details", false);
        if (validatedDetails) {
            validatedImages->FetchImagesDetails(tx);
        }
        StringSplitter(GetHandlerSettingDef<TString>("car.damage.allowed_support_verdicts", "")).Split(',').SkipEmpty().ParseInto(&validatedImages->MutableReportContext().MutableSupportAllowedVerdicts());
        validatedImages->MutableReportContext().SetSkipIdenticalDups(GetHandlerSetting<bool>("car.damage.skip_dups"));
        validatedImages->MutableReportContext().SetReportMarkups(GetHandlerSettingDef<bool>("car.damage.report_markups", false));
        validatedImages->MutableReportContext().SetSkipElementDups(GetHandlerSettingDef<bool>("car.damage.skip_element_dups", true));
        validatedImages->MutableReportContext().SetReportValidatedDetails(validatedDetails);
        g.AddReportElement("validated_images", validatedImages->GetReport(locale, *Server, hostByImageSourceMapping));
    }

    if (reportImages) {
        auto eg = g.BuildEventGuard("images");
        NJson::TJsonValue report;
        for (auto&& image : *optionalImages) {
            if ((sessionId && sessionId != image.GetSessionId()) || (marker && marker != image.GetMarker()) || (origin && origin != image.GetOrigin())) {
                continue;
            }
            if (source && source != image.GetSource()) {
                continue;
            }
            auto imageReport = image.BuildReport(hostByImageSourceMapping);
            if (reportDups) {
                auto& dupsReport = imageReport.InsertValue("dups", NJson::JSON_ARRAY);
                for (auto&& markup : image.GetMarkUpList()) {
                    if (markup.IsDiscarded()) {
                        continue;
                    }
                    for (auto&& dupImage : dups[markup.GetDescription()]) {
                        if (dupImage.GetImageId() == image.GetImageId()) {
                            continue;
                        }
                        dupsReport.AppendValue(dupImage.BuildReport(hostByImageSourceMapping));
                    }
                }
            }
            report.AppendValue(std::move(imageReport));
        }
        g.AddReportElement("images", std::move(report));
    }
    g.SetCode(HTTP_OK);
}

NDrive::TScheme TUpdatePhotoMetaDataProcessor::GetRequestDataScheme(const NDrive::IServer* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;
    scheme.Add<TFSNumeric>("image_id").SetRequired(true);
    scheme.Add<TFSJson>("meta_data").SetRequired(true);
    return scheme;
}

void TUpdatePhotoMetaDataProcessor::Parse(const NJson::TJsonValue& requestData) {
    ParseDataField(requestData, "image_id", ImageId, /* required = */ true);
    ParseDataField(requestData, "meta_data", MetaData, /* required = */ true);
}

void TUpdatePhotoMetaDataProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const auto& imagesDB = DriveApi->GetImagesDB();
    auto session = imagesDB.BuildSession(false);

    if (!imagesDB.UpdateMetaData(ImageId, MetaData, permissions->GetUserId(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TUpdatePhotoMarkUpProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto imageId = GetValue<ui64>(requestData, "image_id");
    bool replaceFlag = requestData["replace_flag"].GetBoolean();
    R_ENSURE(imageId.Defined(), ConfigHttpStatus.UserErrorState, "No \"image_id\"");

    TVector<TCarDamage> carDamages;
    TMessagesCollector errors;
    if (requestData.Has("mark_up_list") && requestData["mark_up_list"].IsArray()) {
        for (auto&& element : requestData["mark_up_list"].GetArray()) {
            TCarDamage carDamage;
            R_ENSURE(carDamage.ParseFromJson(element, errors), ConfigHttpStatus.UserErrorState, errors.GetStringReport());
            carDamages.emplace_back(std::move(carDamage));
        }
    } else {
        TCarDamage carDamage;
        R_ENSURE(carDamage.ParseFromJson(requestData, errors), ConfigHttpStatus.UserErrorState, errors.GetStringReport());
        carDamages.emplace_back(std::move(carDamage));
    }
    const auto& imagesDB = DriveApi->GetImagesDB();
    auto session = BuildTx<NSQL::Writable>();
    if (!imagesDB.UpdateMarkUp(*imageId, carDamages, permissions->GetUserId(), replaceFlag, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TInvalidateMarkUpProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto imageId = GetValue<ui64>(requestData, "image_id");
    auto descriptions = GetStrings(requestData, "description", false);
    auto markupIndexes = GetValues<ui64>(requestData, "markup_indexes", false);
    bool restoreFlag = requestData["restore_flag"].GetBoolean();
    R_ENSURE(imageId.Defined(), ConfigHttpStatus.UserErrorState, "No \"image_id\"");
    R_ENSURE(!descriptions.empty() || !markupIndexes.empty(), ConfigHttpStatus.UserErrorState, "No descriptions markup_indexes");

    const auto& imagesDB = DriveApi->GetImagesDB();
    auto session = BuildTx<NSQL::Writable>();
    TCommonImageData carDamages;
    if (!imagesDB.GetMarkUp(*imageId, carDamages, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    ui64 changes = 0;
    for (auto&& description : descriptions) {
        for (auto&& markup : carDamages.MutableMarkUpList()) {
            if (markup.GetDescription() == description) {
                markup.SetDiscarded(!restoreFlag);
                changes += 1;
            }
        }
    }
    for (auto index : markupIndexes) {
        R_ENSURE(index < carDamages.GetMarkUpList().size(), ConfigHttpStatus.UserErrorState, "Incorrect index");
        carDamages.MutableMarkUpList()[index].SetDiscarded(!restoreFlag);
        changes += 1;
    }
    R_ENSURE(changes > 0, HTTP_NOT_MODIFIED, "no markup has been changed", session);

    if (!imagesDB.UpdateMarkUp(*imageId, carDamages.GetMarkUpList(), permissions->GetUserId(), true, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TSelectMarkUpProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto imageIdPrev = GetValue<ui64>(requestData, "image_id_prev", false);

    auto imageIdNew = GetValue<ui64>(requestData, "image_id_new");
    EDamageArea area = GetValue<EDamageArea>(requestData, "area", true).GetOrElse(EDamageArea::Exterior);
    TString description = GetString(requestData, "description", true);

    const auto& imagesDB = DriveApi->GetImagesDB();
    auto session = BuildTx<NSQL::Writable>();

    TCommonImageData carDamagesNew;
    if (!imagesDB.GetMarkUp(*imageIdNew, carDamagesNew, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    ui64 indexNew = 0;
    for (;indexNew < carDamagesNew.GetMarkUpList().size(); ++indexNew) {
        if (carDamagesNew.GetMarkUpList()[indexNew].GetArea() == area && carDamagesNew.GetMarkUpList()[indexNew].GetDescription() == description) {
            break;
        }
    }

    R_ENSURE(indexNew < carDamagesNew.GetMarkUpList().size(), ConfigHttpStatus.UserErrorState, "Markup for new not found");
    carDamagesNew.MutableMarkUpList()[indexNew].SetIsTheBest(true);

    if (imageIdPrev.Defined()) {
        R_ENSURE(*imageIdNew != *imageIdPrev, ConfigHttpStatus.UserErrorState, "Same images ids");
        TCommonImageData carDamagesPrev;
        if (!imagesDB.GetMarkUp(*imageIdPrev, carDamagesPrev, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        ui64 indexPrev = 0;
        for (;indexPrev < carDamagesPrev.GetMarkUpList().size(); ++indexPrev) {
            if (carDamagesPrev.GetMarkUpList()[indexPrev].GetArea() == area && carDamagesPrev.GetMarkUpList()[indexPrev].GetDescription() == description) {
                break;
            }
        }
        R_ENSURE(indexPrev < carDamagesPrev.GetMarkUpList().size(), ConfigHttpStatus.UserErrorState, "Markup for prev not found");
        R_ENSURE(carDamagesPrev.GetMarkUpList()[indexPrev].GetIsTheBest(), ConfigHttpStatus.UserErrorState, "Not the best");

        R_ENSURE(carDamagesPrev.GetObjectId() == carDamagesNew.GetObjectId(), ConfigHttpStatus.UserErrorState, "Different objects in pair");

        carDamagesPrev.MutableMarkUpList()[indexPrev].SetIsTheBest(false);

        if (!imagesDB.UpdateMarkUp(*imageIdPrev, carDamagesPrev.GetMarkUpList(), permissions->GetUserId(), true, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!imagesDB.UpdateMarkUp(*imageIdNew, carDamagesNew.GetMarkUpList(), permissions->GetUserId(), true, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}
