#include "processor.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/scheme_adapters.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/snapshots/image.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/images/database.h>
#include <drive/backend/images/image.h>
#include <drive/backend/incident/manager.h>
#include <drive/backend/incident/states.h>
#include <drive/backend/incident/contexts/evacuation_ticket.h>
#include <drive/backend/incident/contexts/incident_ticket.h>
#include <drive/backend/incident/renins_claims/catalogue_entity_kasko.h>
#include <drive/backend/incident/transitions/make_evacuation_ticket.h>
#include <drive/backend/incident/transitions/make_incident_ticket.h>
#include <drive/backend/notifications/startrek/startrek.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/startrek/entity.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/algorithm.h>
#include <util/generic/guid.h>

NDrive::TScheme TInitiateIncidentProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme = NDrive::TIncidentTicketIncidentContext::GetScheme(*server, {}, nullptr);

    scheme.Add<TFSString>("car_id", "id автомобиля").SetVisual(TFSString::EVisualType::ObjectId);
    scheme.Add<TFSString>("user_id", "id пользователя").SetVisual(TFSString::EVisualType::ObjectId);
    scheme.Add<TFSString>("session_id", "id сессии").SetVisual(TFSString::EVisualType::GUID);

    return scheme;
}

void TInitiateIncidentProcessor::Parse(const NJson::TJsonValue& requestData) {
    // Consider scheme validation during parsing
    TMessagesCollector errors;
    auto instance = NDrive::TIncidentData::ConstructInitial(requestData, errors);
    R_ENSURE(instance, ConfigHttpStatus.SyntaxErrorStatus, "Invalid incident data: " << errors.GetStringReport());
    Instance = std::move(*instance);
}

void TInitiateIncidentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Incident);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Incident);

    const TString& performerId = permissions->GetUserId();

    const NDrive::TIncidentsManager& incidentManager = DriveApi->GetIncidentsManager();

    TMaybe<NDrive::TIncidentData> incident;
    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        incident = incidentManager.CreateIncident(Instance, performerId, session);
        if (!incident) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        // ticket creation result must be committed
        incident = incidentManager.PerformTransition(incident, NDrive::EIncidentTransition::MakeIncidentTicket, requestData, performerId, session);
        if (!incident) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        TVector<std::pair<NDrive::EIncidentTransition, bool>> transitions = { // transition_id, quiet perform
            { NDrive::EIncidentTransition::HandleIncidentObjectTags, false },
            { NDrive::EIncidentTransition::HandleAvarcomReninsTags, true },
            { NDrive::EIncidentTransition::HandleAvarcomIngosTags, true },
            { NDrive::EIncidentTransition::FetchReninsKaskoDefaults, true }, // conditional
        };
        for (auto&& [transitionId, quiet ] : transitions) {
            incident = incidentManager.PerformTransition(incident, transitionId, requestData, performerId, session, quiet);
            if (!incident) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.MutableReport().AddReportElement("incident_data", incident->BuildReport());

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TInitiateEvacuationProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme = NDrive::TEvacuationTicketIncidentContext::GetScheme(*server, {}, nullptr);

    scheme.Add<TFSString>("car_id", "id автомобиля").SetVisual(TFSString::EVisualType::ObjectId).SetRequired(true);
    scheme.Add<TFSString>("user_id", "id пользователя").SetVisual(TFSString::EVisualType::ObjectId).SetRequired(false);
    scheme.Add<TFSString>("session_id", "id сессии").SetVisual(TFSString::EVisualType::GUID).SetRequired(false);

    return scheme;
}

void TInitiateEvacuationProcessor::Parse(const NJson::TJsonValue& requestData) {
    // Consider scheme validation during parsing
    TMessagesCollector errors;
    R_ENSURE(Instance.DeserializeDataFromJson(requestData, errors), ConfigHttpStatus.SyntaxErrorStatus, "Invalid evacuation data: " << errors.GetStringReport());
}

void TInitiateEvacuationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Incident);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Incident);

    const TString& performerId = permissions->GetUserId();

    const NDrive::TIncidentsManager& incidentManager = DriveApi->GetIncidentsManager();

    TMaybe<NDrive::TIncidentData> incident;
    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        incident = incidentManager.CreateIncident(Instance, performerId, session);
        if (!incident) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        // ticket creation result must be committed
        incident = incidentManager.PerformTransition(incident, NDrive::EIncidentTransition::MakeEvacuationTicket, requestData, performerId, session);
        if (!incident) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        TVector<NDrive::EIncidentTransition> transitionIds = {
            NDrive::EIncidentTransition::HandleEvacuationObjectTags,
            NDrive::EIncidentTransition::Close,
        };
        for (auto&& transitionId: transitionIds) {
            incident = incidentManager.PerformTransition(incident, transitionId, requestData, performerId, session);
            if (!incident) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.MutableReport().AddReportElement("incident_data", incident->BuildReport());

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TUploadIncidentStartrekAttachmentProcessor::GetCgiParametersScheme(const IServerBase* server, const TCgiParameters& /* schemeCgi */) {
    R_ENSURE(server->GetStartrekClient(), HTTP_INTERNAL_SERVER_ERROR, "Startrek client is not configured");

    NDrive::TScheme scheme;
    scheme.Add<TFSString>("file_name", "Имя файла").SetRequired(true);
    scheme.Add<TFSString>("md5", "MD5 hash");
    return scheme;
}

void TUploadIncidentStartrekAttachmentProcessor::Parse(const TCgiParameters& cgi) {
    FileName = GetString(cgi, "file_name", true);
    R_ENSURE(FileName, ConfigHttpStatus.SyntaxErrorStatus, "file name must be provided");
    MD5 = GetString(cgi, "md5", false);
}

void TUploadIncidentStartrekAttachmentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(Server->GetStartrekClient(), ConfigHttpStatus.UnknownErrorStatus, "Startrek client is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Incident);

    const TBlob& content = Context->GetBuf();
    R_ENSURE(!content.Empty(), ConfigHttpStatus.UserErrorState, "no content to upload");

    if (MD5) {
        TString dataHash = MD5::Calc(TStringBuf(content.AsCharPtr(), content.Size()));
        R_ENSURE(dataHash == MD5, ConfigHttpStatus.UserErrorState, "md5 check failed");
    }

    TString contentStr(content.AsCharPtr(), content.Size());

    TStartrekClient::TAttachmentId attachmentId;
    TMessagesCollector errors;

    const TStartrekClient& client = DriveApi->GetStartrekClient();
    R_ENSURE(client.UploadAttachment(CGIEscapeRet(FileName), contentStr, attachmentId, errors), ConfigHttpStatus.UnknownErrorStatus, "Error uploading attachment: " << errors.GetStringReport());

    g.MutableReport().AddReportElement("attachment_id", attachmentId);

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TCreateIncidentProcessor::GetRequestDataScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    return NDrive::TIncidentData::GetInitialScheme();
}

void TCreateIncidentProcessor::Parse(const NJson::TJsonValue& requestData) {
    TMessagesCollector errors;
    auto instance = NDrive::TIncidentData::ConstructInitial(requestData, errors);
    R_ENSURE(instance, ConfigHttpStatus.SyntaxErrorStatus, "Invalid incident data: " << errors.GetStringReport());
    Instance = std::move(*instance);
}

void TCreateIncidentProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(DriveApi && DriveApi->HasIncidentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Incidents manager is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Incident);

    const NDrive::TIncidentsManager& incidentManager = DriveApi->GetIncidentsManager();

    TMaybe<NDrive::TIncidentData> incident;
    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        incident = incidentManager.CreateIncident(Instance, permissions->GetUserId(), session);
        if (!incident || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.MutableReport().AddReportElement("incident_data", incident->BuildReport());

    g.SetCode(HTTP_OK);
}

NDrive::TScheme TListIncidentsProcessor::GetCgiParametersScheme(const IServerBase* /* server */, const TCgiParameters& /* schemeCgi */) {
    NDrive::TScheme scheme;

    scheme.Add<TFSVariants>("incident_type", "Тип инцидента").InitVariants<NDrive::EIncidentType>().SetMultiSelect(true);
    scheme.Add<TFSVariants>("incident_status", "Статус инцидента").InitVariants<NDrive::EIncidentStatus>().SetMultiSelect(true);
    scheme.Add<TFSString>("incident_id", "Id инцидента").SetVisual(TFSString::EVisualType::UUID);
    scheme.Add<TFSString>("car_id", "Id автомобиля").SetVisual(TFSString::EVisualType::ObjectId);
    scheme.Add<TFSString>("user_id", "Id пользователя").SetVisual(TFSString::EVisualType::ObjectId);
    scheme.Add<TFSString>("session_id", "Id сессии").SetVisual(TFSString::EVisualType::GUID);

    // to be generalized
    scheme.Add<TFSNumeric>("since").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("until").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("offset").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("limit").SetMin(1).SetMax(MaxLimit).SetDefault(MaxLimit);

    return scheme;
}

void TListIncidentsProcessor::Parse(const TCgiParameters& cgi) {
    bool requiredConstraintSet = false;
    {
        TVector<NDrive::EIncidentType> incidentTypes = GetValues<NDrive::EIncidentType>(cgi, "incident_type", /* required = */ false);
        if (incidentTypes) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentTypeFilter>(incidentTypes));
        }
    }
    {
        TVector<NDrive::EIncidentStatus> incidentStatuses = GetValues<NDrive::EIncidentStatus>(cgi, "incident_status", /* required = */ false);
        if (incidentStatuses) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentStatusFilter>(incidentStatuses));
        }
    }
    {
        TString incidentId = GetUUID(cgi, "incident_id", /* required = */ false);
        if (incidentId) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentIdFilter>(incidentId));
            requiredConstraintSet = true;
        }
    }
    {
        TString carId = GetUUID(cgi, "car_id", /* required = */ false);
        if (carId) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentCarFilter>(carId));
            requiredConstraintSet = true;
        }
    }
    {
        TString userId = GetUUID(cgi, "user_id", /* required = */ false);
        if (userId) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentUserFilter>(userId));
            requiredConstraintSet = true;
        }
    }
    {
        TString sessionId = GetString(cgi, "session_id", /* required = */ false);
        if (sessionId) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentSessionFilter>(sessionId));
            requiredConstraintSet = true;
        }
    }
    {
        TInstant since = GetTimestamp(cgi, "since", /* required = */ false).GetOrElse(TInstant::Zero());
        TInstant until = GetTimestamp(cgi, "until", /* required = */ false).GetOrElse(TInstant::Zero());
        if (since || until) {
            Filters.Append(MakeAtomicShared<NDrive::TIncidentInitiationTimeFilter>(since, until));
        }
    }

    if (cgi.Has("offset")) {
        Offset = ParseValue<size_t>(GetString(cgi, "offset"));
    }

    if (cgi.Has("limit")) {
        Limit = std::min(ParseValue<size_t>(GetString(cgi, "limit")), MaxLimit);
    }
}

void TListIncidentsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(DriveApi && DriveApi->HasIncidentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Incidents manager is not configured");
    R_ENSURE(DriveApi && DriveApi->HasMDSClient(), ConfigHttpStatus.UnknownErrorStatus, "MDS client is required but not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Incident);

    const NDrive::TIncidentsManager& incidentManager = DriveApi->GetIncidentsManager();
    auto locale = GetLocale();
    auto session = incidentManager.BuildSession(/* readonly = */ true);

    TMaybe<TVector<NDrive::TIncidentData>> incidents;
    {
        incidents = incidentManager.GetIncidents(Filters, session);
        if (!incidents) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    // NB. The code below must be generalized and customized (ordering and comparer) and moved to filters
    {
        const auto comparer = [](const auto& lhs, const auto& rhs) {
            return -static_cast<i64>(lhs.GetInitiatedAt().GetValue()) < -static_cast<i64>(rhs.GetInitiatedAt().GetValue());
        };

        auto begin = (Offset < incidents->size()) ? incidents->begin() + Offset : incidents->end();
        auto end = (Offset + Limit < incidents->size()) ? incidents->begin() + Offset + Limit : incidents->end();

        if (begin != incidents->end()) {
            if (begin != incidents->begin()) {
                NthElement(incidents->begin(), begin, incidents->end(), comparer);
            }
            if (end != incidents->end()) {
                NthElement(begin, end, incidents->end(), comparer);
            }
        }

        if (end != incidents->end()) {
            incidents->erase(end, incidents->end());
        }
        if (begin != incidents->begin()) {
            incidents->erase(incidents->begin(), begin);
        }

        Sort(*incidents, comparer);
    }

    TSet<TString> relatedCarIds;
    TSet<TString> relatedUserIds;
    TSet<TString> relatedSessionIds;

    TSet<TString> relatedCarTagIds;
    TSet<TString> relatedUserTagIds;

    TVector<ui64> relatedImageIds;

    {
        NJson::TJsonValue serializedIncidents(NJson::JSON_ARRAY);

        for (const NDrive::TIncidentData& incident: *incidents) {
            relatedCarIds.insert(incident.GetCarId());
            if (incident.GetUserId()) {
                relatedUserIds.insert(incident.GetUserId());
            }
            if (incident.GetSessionId()) {
                relatedSessionIds.insert(incident.GetSessionId());
            }

            for (const auto& tagLink: incident.GetTagLinks()) {
                if (tagLink.GetTagEntityType() == NEntityTagsManager::EEntityType::Car) {
                    relatedCarTagIds.emplace(tagLink.GetTagId());
                } else if (tagLink.GetTagEntityType() == NEntityTagsManager::EEntityType::User) {
                    relatedUserTagIds.emplace(tagLink.GetTagId());
                }
            }

            for (const auto& photoLink: incident.GetPhotoLinks()) {
                relatedImageIds.emplace_back(photoLink.GetImageId());
            }

            serializedIncidents.AppendValue(incident.BuildReport());
        }

        g.MutableReport().AddReportElement("incidents", std::move(serializedIncidents));
    }

    // fetch extra data

    {
        TSet<TString> carModelCodes;

        NJson::TJsonValue serializedCars(NJson::JSON_ARRAY);

        const NDeviceReport::TReportTraits reportTraits = NDeviceReport::EReportTraits::ReportCarId;

        auto gCars = DriveApi->GetCarsData()->FetchInfo(relatedCarIds, session);
        for (auto&& carId: relatedCarIds) {
            auto carDataPtr = gCars.GetResultPtr(carId);
            if (carDataPtr) {
                carModelCodes.insert(carDataPtr->GetModel());

                serializedCars.AppendValue(carDataPtr->GetReport(locale, reportTraits));
            }
        }

        g.MutableReport().AddReportElement("cars", std::move(serializedCars));

        NJson::TJsonValue serializedCarModels(NJson::JSON_ARRAY);

        auto gModels = DriveApi->GetModelsData()->FetchInfo(carModelCodes, session);
        for (auto&& modelCode: carModelCodes) {
            auto carModelCodePtr = gModels.GetResultPtr(modelCode);
            if (carModelCodePtr) {
                serializedCarModels.AppendValue(carModelCodePtr->GetReport(locale, NDriveModelReport::UserReport));
            }
        }

        g.MutableReport().AddReportElement("car_models", std::move(serializedCarModels));
    }

    {
        NJson::TJsonValue serializedUsers(NJson::JSON_ARRAY);

        const NUserReport::TReportTraits reportTraits = (
            NUserReport::EReportTraits::ReportId | NUserReport::EReportTraits::ReportNames | NUserReport::EReportTraits::ReportLogin
        );

        auto gUsers = DriveApi->GetUsersData()->FetchInfo(relatedUserIds, session);
        for (auto&& userId: relatedUserIds) {
            auto userDataPtr = gUsers.GetResultPtr(userId);
            if (userDataPtr) {
                serializedUsers.AppendValue(userDataPtr->GetReport(reportTraits));
            }
        }

        g.MutableReport().AddReportElement("users", std::move(serializedUsers));
    }

    {
        NJson::TJsonValue serializedSessions(NJson::JSON_ARRAY);

        THistoryRidesContext sessionsContext(*Server);
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("list_incidents");
        if (relatedSessionIds && sessionsContext.InitializeSessions(MakeVector(relatedSessionIds), session, ydbTx)) {
            auto sessionsIterator = sessionsContext.GetIterator();

            THistoryRideObject sessionInfo;
            while (sessionsIterator.GetAndNext(sessionInfo)) {
                NJson::TJsonValue sessionReport = NJson::TMapBuilder
                    ("session_id", sessionInfo.GetSessionId())
                    ("last_ts", sessionInfo.GetLastTS().Seconds());
                serializedSessions.AppendValue(sessionReport);
            }
        }

        g.MutableReport().AddReportElement("sessions", std::move(serializedSessions));
    }

    {
        const auto& tagsManager = DriveApi->GetTagsManager();
        g.MutableReport().AddReportElement("car_tags", GetTagsDesciption(tagsManager.GetDeviceTags(), relatedCarTagIds, session));
        g.MutableReport().AddReportElement("user_tags", GetTagsDesciption(tagsManager.GetUserTags(), relatedUserTagIds, session));
    }

    {
        NJson::TJsonValue serializedImages(NJson::JSON_ARRAY);

        const auto& imagesStorage = DriveApi->GetImagesDB();
        auto images = imagesStorage.GetRecords(relatedImageIds, session);
        if (!images) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }

        auto hostByImageSourceMapping = TCommonImageData::GetHostByImageSourceMapping(DriveApi->GetMDSClient());
        for (auto&& image: *images) {
            serializedImages.AppendValue(image.BuildReport(hostByImageSourceMapping));
        }

        g.MutableReport().AddReportElement("images", std::move(serializedImages));
    }

    {
        NJson::TJsonValue serializedAttachmentCodeDescriptions(NJson::JSON_MAP);

        NDrive::NRenins::TKaskoDocumentTypeEntry availableDocuments;
        if (availableDocuments.LoadSettingValue(Server)) {
            serializedAttachmentCodeDescriptions = availableDocuments.SerializeToJson();
        }

        g.MutableReport().AddReportElement("attachment_code_description_mapping", std::move(serializedAttachmentCodeDescriptions));
    }

    g.SetCode(HTTP_OK);
}

NJson::TJsonValue TListIncidentsProcessor::GetTagsDesciption(const IEntityTagsManager& tagsManagerImpl, const TSet<TString>& tagIds, NDrive::TEntitySession& session) const {
    NJson::TJsonValue serializedTags(NJson::JSON_ARRAY);

    auto tags = tagsManagerImpl.RestoreTags(tagIds, session);
    if (!tags) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    for (auto&& tag: *tags) {
        if (!tag) {
            continue;
        }
        auto tagDescription = DriveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
        if (!tagDescription) {
            continue;
        }

        serializedTags.AppendValue(
            NJson::TMapBuilder
            ("tag_id", tag.GetTagId())
            ("object_id", tag.GetObjectId())
            ("tag_name", tag->GetName())
            ("display_name", tagDescription->GetDisplayName())
        );
    }

    return serializedTags;
}

NDrive::TScheme TPerformIncidentTransitionProcessor::GetRequestDataScheme(const IServerBase* server, const TCgiParameters& schemeCgi) {
    const NDrive::IServer& serverImpl = server->GetAsSafe<NDrive::IServer>();

    const THttpStatusManagerConfig& httpStatuses = serverImpl.GetHttpStatusManagerConfig();

    const TDriveAPI* driveApi = serverImpl.GetDriveAPI();
    R_ENSURE(driveApi && driveApi->HasIncidentsManager(), httpStatuses.UnknownErrorStatus, "Incidents manager is not configured");

    R_ENSURE(GetUuid(schemeCgi.Get("incident_id")), httpStatuses.SyntaxErrorStatus, "ill-formed or absent incident id");
    const TString& incidentId = schemeCgi.Get("incident_id");

    TMaybe<NDrive::TIncidentData> incident;
    NDrive::TIncidentsManager::TTransitions transitions;
    {
        const NDrive::TIncidentsManager& incidentManager = driveApi->GetIncidentsManager();

        auto session = incidentManager.BuildSession(/* readOnly = */ true);
        incident = incidentManager.GetIncident(incidentId, session);
        if (!incident) {
            session.DoExceptionOnFail(httpStatuses);
        }
        auto optTransitions = incidentManager.GetAvailableTransitions(*incident, session);
        if (!optTransitions) {
            session.DoExceptionOnFail(httpStatuses);
        }
        transitions = std::move(*optTransitions);
    }

    NDrive::TScheme scheme;
    scheme.Add<TFSString>("incident_id", "id инцидента").SetDefault(incidentId).SetRequired(true);

    {
        NDrive::TScheme transitionGroupScheme;

        NDrive::TIncidentStateContext context(&serverImpl, *incident, {});

        TSet<TString> hiddenTransitions;  // some transitions can be used for service purposes only and just hidden from the scheme
        if (!NJson::TryFromJson(serverImpl.GetSettings().GetJsonValue(NDrive::IncidentTransitionsSettingPrefix + ".hidden_type_names"), hiddenTransitions)) {
            ERROR_LOG << "Error parsing hidden transitions setting value" << Endl;  // not critical
        }

        auto& transitionScheme = transitionGroupScheme.Add<TFSVariable>("transition_id", "id перехода");
        transitionScheme.SetRequired(true);

        for (auto&& transitionPtr: transitions) {
            if (hiddenTransitions.contains(transitionPtr->GetType())) {
                continue;
            }
            auto transitionTypeAdapter = TLocalizedVariantAdapter::FromEnum(transitionPtr->GetTransitionType());
            transitionScheme.AddVariant(transitionTypeAdapter, transitionPtr->GetScheme(context));
        }

        scheme.Add<TFSStructure>("transition", "Контекст перехода").SetStructure(transitionGroupScheme);
    }

    return scheme;
}

void TPerformIncidentTransitionProcessor::Parse(const NJson::TJsonValue& requestData) {
    ParseDataField(requestData, "incident_id", IncidentId, /* required = */ true);

    TransitionData = requestData["transition"];
    ParseDataField(TransitionData, "transition_id", NJson::Stringify(TransitionId), /* required = */ true);
}

void TPerformIncidentTransitionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    R_ENSURE(DriveApi && DriveApi->HasIncidentsManager(), ConfigHttpStatus.UnknownErrorStatus, "Incidents manager is not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Incident);

    const NDrive::TIncidentsManager& incidentManager = DriveApi->GetIncidentsManager();

    TMaybe<NDrive::TIncidentData> incident;
    {
        auto session = incidentManager.BuildSession(/* readonly = */ false);

        incident = incidentManager.PerformTransition(IncidentId, TransitionId, TransitionData, permissions->GetUserId(), session);
        if (!incident || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    g.MutableReport().AddReportElement("incident_data", incident->BuildReport());

    g.SetCode(HTTP_OK);
}
