#include "processor.h"

#include <drive/backend/actions/evolution_policy.h>
#include <drive/backend/abstract/user_position_context.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/charset/wide.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/string.h>
#include <util/string/split.h>

void TLandingAcceptProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData["landings"].IsArray(), ConfigHttpStatus.SyntaxErrorStatus, "'landings' absent");

    const NJson::TJsonValue::TArray& arrLandings = requestData["landings"].GetArraySafe();

    TVector<TLandingAcceptance> landingAcceptances;
    TSet<TString> landingIds;
    for (auto&& i : arrLandings) {
        R_ENSURE(i["id"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "incorrect request");
        landingAcceptances.emplace_back(TLandingAcceptance().SetId(i["id"].GetString()).SetComment(i["comment"].GetString()).SetLastAcceptedAt(Context->GetRequestStartTime()));
        landingIds.emplace(i["id"].GetString());
    }

    TVector<TLanding> landings;
    R_ENSURE(Server->GetDriveAPI()->GetLandingsDB()->GetCustomObjectsFromCache(landings, landingIds, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

    NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->AcceptLandings(*permissions, landingAcceptances, Context->GetRequestStartTime(), session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    if (!landings.empty()) {
        TMap<TString, TString> eventTagNames;
        for (auto&& landing : landings) {
            if (!landing.GetEventTagName().empty()) {
                eventTagNames[landing.GetEventTagName()] = landing.GetId();
            }
        }
        if (!eventTagNames.empty()) {
            NJson::TJsonValue deviceDescription = TUserDevice::GenerateDescriptionJson(Context);
            auto eventTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions->GetUserId(), MakeVector(NContainer::Keys(eventTagNames)), session);
            if (!eventTags) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            bool updated = false;
            for (auto&& tag : *eventTags) {
                auto eventTag = tag.MutableTagAs<IEventTag>();
                if (eventTag) {
                    eventTag->AddEvent(eventTagNames[tag->GetName()], Context->GetRequestStartTime(), deviceDescription);
                    updated = true;
                }
            }
            if (updated && !Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagsData(*eventTags, permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TLandingInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& format = cgi.Has("format") ? cgi.Get("format") : "map";
    const bool reportNewLandings = GetValue<bool>(cgi, "report_new_landings", false).GetOrElse(false);
    const bool arrayFormat = (format == "array");
    const auto locale = GetLocale();

    TVector<TLanding> landings;

    const TVector<TString> ids = StringSplitter(Context->GetCgiParameters().Get("id")).SplitBySet(", ").SkipEmpty();
    TSet<TString> idsSet(ids.begin(), ids.end());
    TVector<TDBTag> skippedTags;
    {
        NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
        TUserPositionContext upContext(Context);
        if (!Server->GetDriveAPI()->GetLandingsForUser(Server, *permissions, upContext.GetUserPositionPtr(), landings, session, skippedTags, reportNewLandings)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (skippedTags.size()) {
        NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
        if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple(skippedTags, permissions->GetUserId(), session, true) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (reportNewLandings) {
        TVector<TLanding> chatLandings;
        TVector<TLanding> oldLandings;

        auto ctx = BuildChatContext(permissions);
        for (auto&& landing : landings) {
            if (!!landing.GetChatId()) {
                if (landing.GetAuditoryConditionRaw().IsDefined() && !landing.GetAuditoryCondition()) {
                    ERROR_LOG << "could not parse landing " << landing.GetId() << " auditory condition, skipping" << Endl;
                    continue;
                }
                TChatContext emptyContext;
                if (landing.GetAuditoryCondition() && !landing.GetAuditoryCondition()->IsMatching(ctx, emptyContext)) {
                    continue;
                }
                chatLandings.emplace_back(std::move(landing));
            } else {
                oldLandings.push_back(std::move(landing));
            }
        }

        const bool useStoriesFormat = permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Stories);

        landings = std::move(oldLandings);
        std::sort(chatLandings.begin(), chatLandings.end());
        NJson::TJsonValue chatLandingsReport(arrayFormat ? NJson::JSON_ARRAY : NJson::JSON_MAP);
        TVector<TLandingAcceptance> acceptances;
        if (chatLandings.size() && Server->GetChatEngine()) {
            for (auto&& landing : chatLandings) {
                auto chatSession = BuildChatSession();
                auto chatRobot = Server->GetChatRobot(landing.GetChatId());
                if (!chatRobot) {
                    continue;
                }
                if (!useStoriesFormat) {
                    if (!chatRobot->EnsureChat(permissions->GetUserId(), "", chatSession, false)) {
                        NDrive::TEventLog::Log("ChatLandingError", NJson::TMapBuilder
                            ("method", "EnsureChat")
                            ("message", chatSession.GetReport())
                        );
                        chatSession.ClearErrors();
                        continue;
                    }
                    if (!chatRobot->MoveToStep(permissions->GetUserId(), "", landing.GetChatMessagesGroup(), &chatSession, true)) {
                        NDrive::TEventLog::Log("ChatLandingError", NJson::TMapBuilder
                            ("method", "EnsureChat")
                            ("message", chatSession.GetReport())
                            ("step", landing.GetChatMessagesGroup())
                        );
                        chatSession.ClearErrors();
                        continue;
                    }
                }
                NJson::TJsonValue chatReport = landing.GetPublicReport(locale);
                chatReport["id"] = !useStoriesFormat ? landing.GetChatId() : (landing.GetChatId().StartsWith("static_") ? landing.GetChatId() : "static_" + landing.GetChatId()) + "." + landing.GetChatMessagesGroup();
                chatReport["title"] = chatRobot->GetChatConfig().GetTitle();
                NJson::InsertField(chatReport, "styles", chatRobot->GetChatConfig().GetTheme().GetStyles());
                if (arrayFormat) {
                    chatLandingsReport.AppendValue(std::move(chatReport));
                } else {
                    chatLandingsReport.InsertValue(landing.GetChatId(), std::move(chatReport));
                }
                acceptances.emplace_back(TLandingAcceptance().SetId(landing.GetId()).SetLastAcceptedAt(Context->GetRequestStartTime()));
                R_ENSURE(chatSession.Commit(), ConfigHttpStatus.UnknownErrorStatus, "could not commit chat session");
                break;
            }
        }
        if (acceptances.size()) {
            NDrive::TEntitySession session = BuildTx<NSQL::Writable>();
            if (!Server->GetDriveAPI()->AcceptLandings(*permissions, acceptances, Context->GetRequestStartTime(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "could not commit acceptance session");
        }
        g.MutableReport().AddReportElement("chat_landings", std::move(chatLandingsReport));
    }

    NJson::TJsonValue report(arrayFormat ? NJson::JSON_ARRAY : NJson::JSON_MAP);
    TUtf16String publicName = CharToWide(permissions->GetFirstName(), CODES_UTF8);
    if (!publicName) {
        publicName = CharToWide(permissions->GetLogin(), CODES_UTF8);
    }
    NJsonWriter::TBuf writer;
    writer.WriteString("<name>", NJsonWriter::HEM_DONT_ESCAPE_HTML);
    TUtf16String fs = CharToWide(writer.Str(), CODES_UTF8);
    fs = fs.substr(1, fs.size() - 2);
    std::sort(landings.begin(), landings.end());
    if (landings.size() && GetHandlerSettingDef("intros_limit_for_session", Max<ui32>()) != Max<ui32>()) {
        landings.resize(GetHandlerSettingDef("intros_limit_for_session", 1));
    }
    for (auto&& i : landings) {
        if (idsSet.empty() || idsSet.contains(i.GetId())) {
            TUtf16String text = CharToWide(i.GetJsonLanding().GetStringRobust(), CODES_UTF8);

            for (size_t pos = text.find(fs); pos != TString::npos; pos = text.find(fs, pos)) {
                text = text.replace(pos, fs.size(), publicName);
                pos = pos + (publicName.size());
            }

            const TString textForJson = WideToChar(text, CODES_UTF8);
            const TString localizedTextForJson = Server->GetLocalization()->ApplyResourcesForJson(textForJson, locale);

            NJson::TJsonValue jsonNew;
            if (!NJson::ReadJsonFastTree(localizedTextForJson, &jsonNew)) {
                ERROR_LOG << "Cannot parse landing: " << WideToChar(text, CODES_UTF8) << Endl;
                continue;
            }

            i.SetJsonLanding(std::move(jsonNew));

            if (arrayFormat) {
                report.AppendValue(i.GetPublicReport(locale));
            } else {
                report.InsertValue(i.GetId(), i.GetPublicReport(locale));
            }
        }
    }

    g.MutableReport().AddReportElement("landings", std::move(report));

    g.SetCode(HTTP_OK);
}

void TLandingsClearProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TString userId;
    if (Context->GetCgiParameters().Has("user_id")) {
        R_ENSURE(permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Delegation, TAdministrativeAction::EEntity::Landings, TypeName), ConfigHttpStatus.PermissionDeniedStatus, "no_permissions");
        userId = GetUUID(Context->GetCgiParameters(), "user_id", true);
    } else {
        userId = permissions->GetUserId();
    }
    const TVector<TString> ids = GetStrings(Context->GetCgiParameters(), "id", false);
    auto session = BuildTx<NSQL::Writable>();
    R_ENSURE(Server->GetDriveAPI()->CleanLandings(userId, ids, &session), ConfigHttpStatus.UnknownErrorStatus, "transaction_problems");
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TLandingUpsertProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Landings);

    TLanding landing;
    TMessagesCollector errors;
    R_ENSURE(landing.DesereializeRequestJson(requestData, errors), ConfigHttpStatus.SyntaxErrorStatus, errors.GetStringReport());

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->GetLandingsDB()->ForceUpsertObject(landing, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TLandingsGetProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Landings);

    const TSet<TString> ids = MakeSet(GetStrings(Context->GetCgiParameters(), "id", false));

    TVector<TLanding> landings;
    if (ids.size()) {
        ReqCheckCondition(Server->GetDriveAPI()->GetLandingsDB()->GetCustomObjectsFromCache(landings, ids, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    } else {
        ReqCheckCondition(Server->GetDriveAPI()->GetLandingsDB()->GetAllObjectsFromCache(landings, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
    }

    NJson::TJsonValue jsonResult;
    std::sort(landings.begin(), landings.end());
    for (auto&& l : landings) {
        jsonResult.AppendValue(l.GetAdminReport());
    }

    auto session = BuildTx<NSQL::ReadOnly>();

    auto optionalLandingPropositions = DriveApi->GetLandingsDB()->GetPropositions()->Get(session);
    R_ENSURE(optionalLandingPropositions, {}, "cannot get propositions", session);
    auto landingPropositions = *optionalLandingPropositions;

    NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
    TSet<TString> users;
    for (auto&& i : landingPropositions) {
        if (ids.size() && !ids.contains(i.second.GetId())) {
            continue;
        }
        propositionsJson.AppendValue(i.second.BuildJsonReport());
        users.emplace(i.second.GetPropositionAuthor());
    }

    auto userData = DriveApi->GetUsersData()->FetchInfo(users);
    NJson::TJsonValue usersJson = NJson::JSON_MAP;
    for (auto&& [_, i] : userData) {
        usersJson.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
    }

    g.MutableReport().AddReportElement("landings", std::move(jsonResult));
    g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));
    g.MutableReport().AddReportElement("users", std::move(usersJson));

    g.SetCode(HTTP_OK);
}

void TLandingRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Landings);

    const TSet<TString> ids = MakeSet(GetStrings(requestData, "id", true));

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->GetLandingsDB()->RemoveObject(ids, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TLandingProposeProcessor::THandlerConfig> Registrator1(TLandingProposeProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TLandingConfirmProcessor::THandlerConfig> Registrator2(TLandingConfirmProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TLandingRejectPropositionProcessor::THandlerConfig> Registrator3(TLandingRejectPropositionProcessor::GetTypeName() + "~");
}
