#include "add_cars.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/unicode/utf8_char/utf8_char.h>
#include <library/cpp/unicode/utf8_iter/utf8_iter.h>


template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TCarInfo& carInfo) {
    NJson::TJsonValue result;
    NJson::FieldsToJson(result, carInfo.GetFields());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TCarInfo& carInfo) {
    return NJson::TryFieldsFromJson(value, carInfo.GetFields());
}

namespace {
    NDrivematics::TCarInfo ParseCarInfo(const NJson::TJsonValue& jsonValue) {
        NDrivematics::TCarInfo result;
        TMessagesCollector collector;
        R_ENSURE(NJson::TryFieldsFromJson(jsonValue, result.GetFields(), &collector), HTTP_BAD_REQUEST, "errors while parsing car info: " << collector.GetStringReport());
        result.MutableManufacturer() = StripString(ToLowerUTF8(result.GetManufacturer()));
        result.MutableModel() = StripString(ToLowerUTF8(result.GetModel()));
        result.MutableNumber() = ToLowerUTF8(result.GetNumber());
        result.SetVIN(ToUpperUTF8(result.GetVIN()));
        return result;
    }

    TString Capitalize(TStringBuf s) {
        if (TUtfIterCode it = s) {
            wchar32 firstLetter = it.GetNext();
            firstLetter = ToUpper(firstLetter);
            return TString::Join(TUtf8Char(firstLetter), TStringBuf(it.Ptr(), s.end()));
        }
        return TString();
    }

    TSet<TString> RegisterLeasingTags(const ITagsMeta& tagsMeta, const NDrivematics::TCarInfo& carInfo, TUserPermissions::TPtr permissions, NDrive::TEntitySession& session) {
        ITagsMeta::TTagDescriptionsByName leasingTagsDescriptions = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { NDrivematics::TLeasingCompanyTag::TypeName });
        ITagsMeta::TTagDescriptionsByName taxiTagsDescriptions = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { NDrivematics::TTaxiCompanyTag::TypeName });
        ITagsMeta::TTagDescriptionsByName brandTagsDescriptions = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { NDrivematics::TBrandTag::TypeName });
        auto leasingTagDescIterator = std::find_if(leasingTagsDescriptions.begin(), leasingTagsDescriptions.end(), [&carInfo](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const NDrivematics::TLeasingCompanyTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag description");
            return tagDesc->GetLeasingCompanyName() == carInfo.GetLeasingCompanyName();
        });
        auto taxiTagDescIterator = std::find_if(taxiTagsDescriptions.begin(), taxiTagsDescriptions.end(), [&carInfo](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const NDrivematics::TTaxiCompanyTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag description");
            return tagDesc->GetTin() == carInfo.GetTaxiCompanyTin();
        });
        auto brandTagDescIterator = std::find_if(brandTagsDescriptions.begin(), brandTagsDescriptions.end(), [&carInfo](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const NDrivematics::TBrandTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag description");
            return carInfo.HasBrandId() && tagDesc->GetBrandId() == carInfo.GetBrandIdRef();
        });
        if (leasingTagDescIterator == leasingTagsDescriptions.end()) {
            TString newLeasingCompanyTagName = "leasing_company_tag_" + ToString(FnvHash<ui64>(carInfo.GetLeasingCompanyName()));

            auto tagDescription = MakeAtomicShared<NDrivematics::TLeasingCompanyTag::TDescription>();
            tagDescription->SetName(newLeasingCompanyTagName);
            tagDescription->SetType(NDrivematics::TLeasingCompanyTag::TypeName);
            tagDescription->SetDisplayName(TStringBuilder() << "Тег для " << carInfo.GetLeasingCompanyName());
            tagDescription->SetLeasingCompanyName(carInfo.GetLeasingCompanyName());

            R_ENSURE(
                tagsMeta.RegisterTag(tagDescription, permissions->GetUserId(), session),
                {},
                "cannot RegisterTag " << tagDescription->GetName(),
                session
            );

            bool result;
            std::tie(leasingTagDescIterator, result) = leasingTagsDescriptions.emplace(newLeasingCompanyTagName, tagDescription);
            R_ENSURE(result, HTTP_INTERNAL_SERVER_ERROR, "new tag description should not be present in leasingTagsDecriptions", session);

            NDrive::TEventLog::Log(
                "LeasingCarsAdd",
                NJson::TMapBuilder
                    ("event", "AutonomousLeasingCompanyTagCreate")
                    ("leasing_company_name", newLeasingCompanyTagName)
            );
        }
        if (brandTagDescIterator == brandTagsDescriptions.end() && carInfo.HasBrandId()) {
            TString newBrandTagName = "brand_tag_" + carInfo.GetBrandIdRef();

            auto tagDescription = MakeAtomicShared<NDrivematics::TBrandTag::TDescription>();
            tagDescription->SetName(newBrandTagName);
            tagDescription->SetType(NDrivematics::TBrandTag::TypeName);
            tagDescription->SetDisplayName(TStringBuilder() << "Тег для бренда " << carInfo.OptionalBrandName().GetOrElse(carInfo.GetBrandIdRef()));
            tagDescription->SetBrandName(carInfo.OptionalBrandName().GetOrElse(""));
            tagDescription->SetBrandId(carInfo.GetBrandIdRef());

            R_ENSURE(
                tagsMeta.RegisterTag(tagDescription, permissions->GetUserId(), session),
                {},
                "cannot RegisterTag " << tagDescription->GetName(),
                session
            );

            bool result;
            std::tie(brandTagDescIterator, result) = brandTagsDescriptions.emplace(newBrandTagName, tagDescription);
            R_ENSURE(result, HTTP_INTERNAL_SERVER_ERROR, "new tag description should not be present in brandTagsDecriptions", session);

            NDrive::TEventLog::Log(
                "LeasingCarsAdd",
                NJson::TMapBuilder
                    ("event", "AutonomousBrandTagCreate")
                    ("brand_name", newBrandTagName)
                    ("brand_id", carInfo.GetBrandIdRef())
            );
        }
        if (taxiTagDescIterator == taxiTagsDescriptions.end()) {
            TString newTaxiCompanyTagName = "taxi_company_tag_" + ToString(carInfo.GetTaxiCompanyTin());

            auto tagDescription = MakeAtomicShared<NDrivematics::TTaxiCompanyTag::TDescription>();
            tagDescription->SetName(newTaxiCompanyTagName);
            tagDescription->SetType(NDrivematics::TTaxiCompanyTag::TypeName);
            tagDescription->SetDisplayName(TStringBuilder() << "Тег для " << carInfo.GetTaxiCompanyName());
            tagDescription->SetCity(carInfo.GetCity());
            tagDescription->SetTaxiCompanyName(carInfo.GetTaxiCompanyName());
            tagDescription->SetTin(carInfo.GetTaxiCompanyTin());
            R_ENSURE(
                tagsMeta.RegisterTag(tagDescription, permissions->GetUserId(), session),
                {},
                "cannot RegisterTag " << tagDescription->GetName(),
                session
            );

            bool result;
            std::tie(taxiTagDescIterator, result) = taxiTagsDescriptions.emplace(newTaxiCompanyTagName, tagDescription);
            R_ENSURE(result, HTTP_INTERNAL_SERVER_ERROR, "new tag description should not be present in taxiTagsDecriptions", session);

            NDrive::TEventLog::Log(
                "LeasingCarsAdd",
                NJson::TMapBuilder
                    ("event", "AutonomousTaxiCompanyTagCreate")
                    ("taxi_company_name", newTaxiCompanyTagName)
            );
        }
        R_ENSURE(leasingTagDescIterator != leasingTagsDescriptions.end(), HTTP_INTERNAL_SERVER_ERROR, "description should be present in the map");
        R_ENSURE(taxiTagDescIterator != taxiTagsDescriptions.end(), HTTP_INTERNAL_SERVER_ERROR, "description should be present in the map");
        TSet<TString> tagNames{leasingTagDescIterator->first, taxiTagDescIterator->first};
        if (brandTagDescIterator != brandTagsDescriptions.end()) {
            tagNames.insert(brandTagDescIterator->first);
        }
        return tagNames;
    }
}

TDriveModelData NDrivematics::TAddLeasingCarsProcessor::GenerateModelData(TStringBuf registryManufacturer, TStringBuf registryModel) {
    TDriveModelData modelData;
    modelData.SetName(TString::Join(Capitalize(registryManufacturer), " ", Capitalize(registryModel)));
    modelData.SetShortName(Capitalize(registryModel));
    TString code = TString::Join(registryManufacturer, "_", registryModel);
    SubstGlobal(code, " ", "_");
    modelData.SetCode(code);
    modelData.SetManufacturer(Capitalize(registryManufacturer));
    modelData.SetRegistryManufacturer(registryManufacturer);
    modelData.SetRegistryModel(registryModel);
    return modelData;
}

void NDrivematics::TAddLeasingCarsProcessor::CreateModel(
    TStringBuf vin,
    TStringBuf manufacturer,
    TStringBuf model,
    TMap<std::pair<TString, TString>, TString>& mapping,
    const TDriveAPI& driveApi, NDrive::TEntitySession& session
) {
    auto modelData = NDrivematics::TAddLeasingCarsProcessor::GenerateModelData(manufacturer, model);
    bool upserted;
    std::tie(std::ignore, upserted) = mapping.emplace(std::make_pair(manufacturer, model), modelData.GetCode());
    R_ENSURE(upserted, HTTP_INTERNAL_SERVER_ERROR, "new model data should not be in modelCodesMapping", session);
    R_ENSURE(driveApi.GetModelsData()->Upsert(modelData, session), {}, "cannot upsert ModelData", session);
    NDrive::TEventLog::Log("TAddLeasingCarsProcessor::CreateModel",
    NJson::TMapBuilder
        ("event", "GenerateModel")
        ("vin", vin)
        ("model_code", modelData.GetCode())
    );
}

void NDrivematics::TAddLeasingCarsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(requestData.Has("car") && requestData["car"].IsMap(), HTTP_BAD_REQUEST, "no \"car\" dict in the root of the request body");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Tag, NDrivematics::TLeasingCompanyTag::TypeName);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Tag, NDrivematics::TTaxiCompanyTag::TypeName);
    const auto& carInfoJson = requestData["car"];
    auto carInfo = ParseCarInfo(carInfoJson);
    R_ENSURE(carInfo.GetVIN(), HTTP_BAD_REQUEST, "car has empty vin");
    const auto& api = *Yensured(DriveApi);
    TSet<TString> tagNamesToCreate;
    {
        // create new session, because in subsequent GetTagsMeta().RegisterTag() we must see the new tag description in another transaction
        auto session = BuildTx<NSQL::Writable>();
        tagNamesToCreate = RegisterLeasingTags(api.GetTagsManager().GetTagsMeta(), carInfo, permissions, session);
        R_ENSURE(session.Commit(), HTTP_INTERNAL_SERVER_ERROR, "can't commit", session);
    }
    if (carInfo.GetHasTelematics()) {
        tagNamesToCreate.insert(THasTelematicsTag::TypeName);
    }
    auto session = BuildTx<NSQL::Writable>();
    auto modelCodesMapping = api.GetModelsData()->GetRegistryModelCodesMapping();
    auto allVins = api.GetCarsData()->GetAllVins(session);

    auto gModelsData = api.GetModelsData()->FetchInfo(session);
    auto modelsData = gModelsData.GetResult();
    {
        TString carId = NUtil::CreateUUID();
        if (auto it = allVins.find(carInfo.GetVIN()); it != allVins.end()) {
            carId = it->second;
            NDrive::TEventLog::Log("LeasingCarsAdd",
            NJson::TMapBuilder
                ("event", "UpdateExistingCar")
                ("id", carId)
                ("vin", carInfo.GetVIN())
            );
        }

        const auto modelKey = std::make_pair(carInfo.GetManufacturer(), carInfo.GetModel());
        if (!modelCodesMapping.contains(modelKey)) {
            CreateModel(
                carInfo.GetVIN(),
                modelKey.first,
                modelKey.second,
                modelCodesMapping,
                api, session
            );
        }

        TString modelCode = modelCodesMapping[modelKey];
        NJson::TJsonValue minCarJson = NJson::TMapBuilder
            ("id", carId)
            ("vin", carInfo.GetVIN())
            ("model_code", modelCode)
            ("number", carInfo.GetNumber());

        R_ENSURE(api.GetCarsData()->UpdateCarFromJSON(minCarJson, carId, permissions->GetUserId(), session), {}, "cannot update car " << carId, session);
        session.Committed().Subscribe([minCarJson](const NThreading::TFuture<void>& f) {
            if (f.HasValue()) {
                NDrive::TEventLog::Log("UpdateCarFromJson", NJson::TMapBuilder("car", minCarJson));
            }
        });

        TVector<TDBTag> tagsToRemove;
        auto carObject = api.GetTagsManager().GetDeviceTags().RestoreObject(carId, session);
        R_ENSURE(carObject, {}, "cannot restore car tags " << carId, session);
        for (auto&& dbTag : carObject->GetTags()) {
            if (!dbTag.Is<TTaxiCompanyTag>()
                && !dbTag.Is<TLeasingCompanyTag>()
                && !dbTag.Is<TBrandTag>()
                && !dbTag.Is<THasTelematicsTag>()) {
                continue;
            }
            if (auto it = tagNamesToCreate.find(dbTag->GetName()); it != tagNamesToCreate.end()) {
                // do not recreate existing tag
                tagNamesToCreate.erase(it);
                continue;
            }
            if (dbTag.Is<THasTelematicsTag>()) {
                tagsToRemove.push_back(dbTag);
            }
        }

        R_ENSURE(
            api.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tagsToRemove, permissions->GetUserId(), session, false),
            {},
            "cannot delete tags from car " << carId,
            session
        );

        for (const auto& tagName : tagNamesToCreate) {
            ITag::TPtr tagData = api.GetTagsManager().GetTagsMeta().CreateTag(tagName, "", Now());
            R_ENSURE(tagData, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << tagName, session);
            if (auto uniqueTag = std::dynamic_pointer_cast<IUniqueTypeNameTag>(tagData)) {
                uniqueTag->SetCachedObject(std::move(*carObject));
            }
            R_ENSURE(
                api.GetTagsManager().GetDeviceTags().AddTag(tagData, permissions->GetUserId(), carId, Server, session),
                {},
                "cannot add tag " << tagData->GetName() << " to " << carId,
                session
            );
        }
    }

    R_ENSURE(
        session.Commit(),
        {},
        "cannot commit session",
        session
    );
    g.SetCode(HTTP_OK);
}
