#include "common.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>

#include <rtline/util/algorithm/container.h>

namespace NDrivematics {
    std::pair<TString, ELeasingTagType> GetFirstObservableTagName(TUserPermissions::TPtr permissions, const TDriveAPI& api) {
        const auto observeObjectTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::ObserveObject);
        auto leasingCompanyTagsNames = api.GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TLeasingCompanyTag::TypeName});
        auto brandTagsNames = api.GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TBrandTag::TypeName});
        auto observableLeasingCompanyTagsNames = MakeIntersection(observeObjectTagsNames, leasingCompanyTagsNames);
        auto observableBrandTagsNames = MakeIntersection(observeObjectTagsNames, brandTagsNames);

        R_ENSURE(!observableLeasingCompanyTagsNames.empty() || !observableBrandTagsNames.empty(), HTTP_FORBIDDEN, "neither leasing company nor brand tag are observable");
        if (observableLeasingCompanyTagsNames.size() + observableBrandTagsNames.size() != 1) {
            ALERT_LOG << "user " << permissions->GetUserId() << " has more than one leasing company and brand tags visible " << Endl;
        }
        if (!observableLeasingCompanyTagsNames.empty()) {
            return { *observableLeasingCompanyTagsNames.begin(), ELeasingTagType::LeasingCompany };
        }
        return { *observableBrandTagsNames.begin(), ELeasingTagType::Brand };
    }

    void CheckAccessToOrganizationMember(const TString& memberId, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        const auto& api = *Yensured(server.GetDriveAPI());
        auto affiliatedCompanyTagDesc = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, server, tx);
        auto optionalTaggedUser = api.GetTagsManager().GetUserTags().RestoreObject(memberId, tx);
        R_ENSURE(optionalTaggedUser, HTTP_INTERNAL_SERVER_ERROR, "can't restore user", tx);
        auto dbTag = optionalTaggedUser->GetFirstTagByClass<TUserOrganizationAffiliationTag>();
        R_ENSURE(dbTag, HTTP_FORBIDDEN, "user has no affiliated company tag");
        R_ENSURE(dbTag->GetName() == affiliatedCompanyTagDesc->GetName(), HTTP_FORBIDDEN, "user belongs to different organization");
    }

}
