#pragma once

#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/database/drive/named_filters.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/backend/processor/params_processor.h>
#include <library/cpp/http/misc/httpcodes.h>

class TDriveAPI;

namespace NDrivematics {
    template <typename StatsAggregator, typename Predicate>
    NDrivematics::TAggregatedLeasingStats CalculateStats(const TSet<TString>& carsIds, const TTaggedObjectsSnapshot& objectsSnapshot,
                                                        TMaybe<ui32> daySince, TMaybe<ui32> dayUntil, ui32 daysCount, ui32 noSHThreshold, StatsAggregator aggregator, Predicate pred) {
        NDrivematics::TLeasingStatsTagsRefs tagsRefs;
        ui32 freshIssueDateCarsCount = 0;
        for (const auto& carId : carsIds) {
            const auto* taggedObject = objectsSnapshot.Get(carId);
            R_ENSURE(taggedObject, HTTP_INTERNAL_SERVER_ERROR, "car " << carId << " wasn't restored");
            if (!pred(*taggedObject)) {
                continue;
            }
            auto leasingStatsDBTags = taggedObject->GetTagsByClass<NDrivematics::TLeasingStatsTag>();
            if (leasingStatsDBTags.empty()) {
                continue;
            }
            const auto& leasingStatsTag = *Yensured(leasingStatsDBTags[0].GetTagAs<NDrivematics::TLeasingStatsTag>());
            {
                auto maybeTag = taggedObject->GetTag(NDrivematics::FreshIssueDateTagName);
                if (maybeTag) {
                    ++freshIssueDateCarsCount;
                    continue;
                }
            }

            tagsRefs.push_back(std::ref(leasingStatsTag));
        }

        NDrivematics::TAggregatedLeasingStats aggregatedStats;
        if (daySince && dayUntil) {
            aggregatedStats = aggregator(tagsRefs, noSHThreshold, [since = *daySince, until = *dayUntil](const TLeasingStatsTag& tag){
                return tag.AggregateStats(since, until);
            });
        } else {
            aggregatedStats = aggregator(tagsRefs, noSHThreshold, [daysCount](const TLeasingStatsTag& tag){
                return tag.AggregateStats(daysCount);
            });
        }
        aggregatedStats.SetFreshIssueDateCarsCount(freshIssueDateCarsCount);
        return aggregatedStats;
    }

    enum class ELeasingTagType {
        LeasingCompany,
        Brand,
    };

    std::pair<TString, ELeasingTagType> GetFirstObservableTagName(TUserPermissions::TPtr permissions, const TDriveAPI& api);

    void CheckAccessToOrganizationMember(const TString& memberId, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);

    template<IterableContainer TObjectIds>
    [[nodiscard]] bool RestoreCompanyNotifier(TVector<TNotifierContainer>& restoredNotifiers, const TObjectIds& ids, const TString& type, const TNotifiersManager* notifiersManagerImpl) {
        auto action = [&restoredNotifiers, &type] (const auto& object) -> void {
            if (!type || object.GetNotifierConfigPtr()->GetTypeName() == type) {
                restoredNotifiers.push_back(object);
            }
        };
        return notifiersManagerImpl->ForObjectsList(action, Now(), &ids);
    }

    template<IterableContainer TObjectIds>
    [[nodiscard]] bool RestoreCompanyNotifier(TSet<TNotifierContainer>& restoredNotifiers, const TObjectIds& ids, const TString& type, const TNotifiersManager* notifiersManagerImpl) {
        auto action = [&restoredNotifiers, &type] (const auto& object) -> void {
            if (!type || object.GetNotifierConfigPtr()->GetTypeName() == type) {
                restoredNotifiers.insert(object);
            }
        };
        return notifiersManagerImpl->ForObjectsList(action, Now(), &ids);
    }
}
