#include "create_driver.h"

#include "common.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/raw_text/phone_number.h>
#include <rtline/library/json/adapters.h>

#include <library/cpp/http/misc/httpcodes.h>

namespace NDrivematics {
    void TRegisterDriverProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::User);
        auto tx = BuildTx<NSQL::Writable>();

        auto phoneNumber = GetString(requestData, "phone_number");
        auto firstName = GetString(requestData, "first_name");
        auto pName = GetString(requestData, "pn", false);
        auto lastName = GetString(requestData, "last_name");
        auto email = GetString(requestData, "email", false);
        TUserOrganizationAffiliationTag::TRoles roles{TUserOrganizationAffiliationTag::DriverRoleName}; 
        if (requestData.Has("roles")) {
            roles = MakeSet(GetStrings(requestData, "roles", false));
        }
        bool validatePhoneNumber = GetHandlerSetting<bool>("validate_phone_number").GetOrElse(true);
        if (validatePhoneNumber) {
            R_ENSURE(
                phoneNumber.StartsWith('+'),
                HTTP_BAD_REQUEST,
                "phone_number " << phoneNumber << " should start with +",
                NDrive::MakeError("incorrect_phone_number_format_plus_required"),
                tx
            );
            TPhoneNormalizer phoneNormalizer;
            phoneNumber = phoneNormalizer.TryNormalize(phoneNumber);
        }

        NDrive::TExternalUser externalUser;
        externalUser
            .SetFirstName(firstName)
            .SetPName(pName)
            .SetLastName(lastName)
            .SetEmail(email)
            .SetPhone(phoneNumber)
        ;

        auto driverUser = Server->GetDriveDatabase().GetUserManager().RegisterExternalUser(permissions->GetUserId(), tx, externalUser);
        R_ENSURE(driverUser, {}, "can't create user", tx);
        auto commonTagNamesString = GetHandlerSetting<TString>("common_tag_names");
        auto commonActionsIdsStr = GetHandlerSetting<TString>("common_actions_ids");

        TVector<TString> tagNames;
        if (commonTagNamesString) {
            tagNames = StringSplitter(*commonTagNamesString).SplitBySet(", ").SkipEmpty();
        }

        TMap<TString, TVector<TString>> dmRoleToCommonActionsIds;
        if (commonActionsIdsStr) {
            NJson::TJsonValue jsonValue;
            R_ENSURE(NJson::ReadJsonFastTree(*commonActionsIdsStr, &jsonValue), HTTP_INTERNAL_SERVER_ERROR, "can't parse common actions ids");
            R_ENSURE(NJson::ParseField(jsonValue, NJson::Dictionary(dmRoleToCommonActionsIds)), HTTP_INTERNAL_SERVER_ERROR, "common actions ids have wrong format");
        }

        TUserOrganizationAffiliationTag::AddRoleAndTagsToUser(*driverUser, tagNames, dmRoleToCommonActionsIds, permissions, Server, tx, roles);
        R_ENSURE(tx.Commit(), {}, "can't commit", tx);
        g.AddReportElement("user", driverUser->GetPublicReport());
        g.SetCode(HTTP_OK);
    }

    void TGetPossibleRolesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        TUserOrganizationAffiliationTag::TRoles roles{TUserOrganizationAffiliationTag::AdminRoleName, TUserOrganizationAffiliationTag::DriverRoleName};
        auto tx = BuildTx<NSQL::ReadOnly>();        
        if (auto affiliatedCompanyTagDesc = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, *Server, tx)) {
            roles = MakeSet(NContainer::Keys(affiliatedCompanyTagDesc->GetCompanyRoles()));
        }
        g.AddReportElement("roles", NJson::ToJson(roles));
        g.SetCode(HTTP_OK);
    }
}
