#include "create_signal.h"

#include <algorithm>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/backend/processors/leasing/common.h>
#include <drive/backend/rt_background/manager/manager.h>
#include <drive/backend/tags/tags_manager.h>


namespace NDrivematics {

    using EEntity = TAdministrativeAction::EEntity;

    namespace {
        void NamedFiltersProcess(
            const NJson::TJsonValue& signalConfigurationJson,
            ISignalConfiguration::TPtr configuration,
            TACLTag::TDescription::TConstPtrImpl aclCompany,
            TUserPermissions::TPtr permissions,
            NDrive::TEntitySession& tx
        ) {
            if (signalConfigurationJson.Has("named_filters")) {
                R_ENSURE(signalConfigurationJson["named_filters"].IsMap() && !signalConfigurationJson["named_filters"].GetMap().empty(), HTTP_BAD_REQUEST, "signal_configuration.named_filters must be not empty map");

                auto& namedFiltersJson = signalConfigurationJson["named_filters"];
                auto includeNamedFiltersIds = NJson::TryFromJson<TSet<TString>>(namedFiltersJson["include_dynamic_group"]).GetOrElse({});
                auto excludeNamedFiltersIds = NJson::TryFromJson<TSet<TString>>(namedFiltersJson["exclude_dynamic_group"]).GetOrElse({});
                R_ENSURE(!HasIntersection(includeNamedFiltersIds, excludeNamedFiltersIds), HTTP_BAD_REQUEST, "'include_dynamic_group' and 'exclude_dynamic_group' must not overlap", NDrive::MakeError("named_filters.incorrect_request"), tx);

                TSet<TNamedFilter::TId> companyNamedFiltersIds = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::NamedFilter, permissions);
                {
                    auto tempIntersection = MakeIntersection(includeNamedFiltersIds, companyNamedFiltersIds);
                    R_ENSURE(tempIntersection.size() == includeNamedFiltersIds.size(), HTTP_BAD_REQUEST, "'include_dynamic_group' not all found for company", NDrive::MakeError("named_filters.incorrect_request"), tx);
                    tempIntersection = MakeIntersection(excludeNamedFiltersIds, companyNamedFiltersIds);
                    R_ENSURE(tempIntersection.size() == excludeNamedFiltersIds.size(), HTTP_BAD_REQUEST, "'exclude_dynamic_group' not all found for company", NDrive::MakeError("named_filters.incorrect_request"), tx);
                }
                TCarsFilter* filterPtr;
                if (configuration->HasCarsFilter()) {
                    filterPtr = &configuration->MutableCarsFilterRef();
                } else {
                    filterPtr = &configuration->OptionalCarsFilter().ConstructInPlace();
                }
                filterPtr->MutableIncludeDynamicFilters() = std::move(includeNamedFiltersIds);
                filterPtr->MutableExcludeDynamicFilters() = std::move(excludeNamedFiltersIds);
            }
        }

        void ZoneProcess(
            const NJson::TJsonValue& signalConfigurationJson,
            ISignalConfiguration::TPtr configuration,
            TACLTag::TDescription::TConstPtrImpl aclCompany,
            TUserPermissions::TPtr permissions,
            const NDrive::IServer& server,
            NDrive::TEntitySession& tx
        ) {
            if (signalConfigurationJson.Has("zone")) {
                R_ENSURE(signalConfigurationJson["zone"].IsMap() && !signalConfigurationJson["zone"].GetMap().empty(), HTTP_BAD_REQUEST, "signal_configuration.zone must be not empty map");

                auto& zoneJson = signalConfigurationJson["zone"];
                auto includeZoneNames = NJson::TryFromJson<TSet<TString>>(zoneJson["included_group_ids"]).GetOrElse({});
                auto excludeZoneNames = NJson::TryFromJson<TSet<TString>>(zoneJson["excluded_group_ids"]).GetOrElse({});
                R_ENSURE(!HasIntersection(includeZoneNames, excludeZoneNames), HTTP_BAD_REQUEST, "'included_group_ids' and 'excluded_group_ids' must not overlap", NDrive::MakeError("zone.incorrect_request"), tx);

                NDrive::TZoneIds companyZoneIds = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::Zone, permissions);
                TMap<TString, NDrive::TZoneId> zoneNameToZoneId;
                auto action = [&zoneNameToZoneId, &includeZoneNames, &excludeZoneNames](const TZone& zone) -> void {
                    if (includeZoneNames.contains(zone.GetName()) || excludeZoneNames.contains(zone.GetName())) {
                        zoneNameToZoneId[zone.GetName()] = zone.GetInternalId();
                    }
                };
                R_ENSURE(server.GetDriveAPI()->GetZoneDB()->ForObjectsList(action, Now(), &companyZoneIds), HTTP_INTERNAL_SERVER_ERROR, "cannot restore zones", NDrive::MakeError("zone.acl.object_not_found"), tx);
                R_ENSURE(zoneNameToZoneId.size() == includeZoneNames.size() + excludeZoneNames.size(), HTTP_BAD_REQUEST, "not all zones found", NDrive::MakeError("zone.acl.object_not_found"), tx);

                NJson::TJsonValue zoneIdsJson;
                auto fillZoneIds = [&zoneNameToZoneId, &zoneIdsJson](TString&& filedName, TSet<TString>&& zoneNames) -> void {
                    for (auto&& zoneName : zoneNames) {
                        zoneIdsJson[filedName].AppendValue(zoneNameToZoneId[zoneName]);
                    }
                };
                fillZoneIds("included_group_ids", std::move(includeZoneNames));
                fillZoneIds("excluded_group_ids", std::move(excludeZoneNames));

                configuration->SetZoneContainer(ISignalConfiguration::TZoneContainer{});
                configuration->MutableZoneContainerRef().DeserializeFromJson(zoneIdsJson);
            }
        }

        ISignalConfiguration::TPtr BuildConfiguration(
            const NJson::TJsonValue& signalConfigurationJson,
            const TUserOrganizationAffiliationTag::TDescription& affiliatedCompanyTagDescription,
            const TACLTag::TDescription::TPtrImpl aclCompany,
            TUserPermissions::TPtr permissions,
            const NDrive::IServer& server,
            NDrive::TEntitySession& tx
        ) {
            TMessagesCollector errors;
            ISignalConfiguration::TPtr configuration = ISignalConfiguration::ConstructConfiguration(signalConfigurationJson, &errors, server);
            R_ENSURE(configuration, HTTP_BAD_REQUEST, "can't create signal configuration: " << errors.GetStringReport());
            configuration->SetSignalId(CreateGuidAsString());
            configuration->SetOwnerName(affiliatedCompanyTagDescription.GetCompanyName());
            configuration->SetOwnerId(affiliatedCompanyTagDescription.GetName());
            NamedFiltersProcess(signalConfigurationJson, configuration, aclCompany, permissions, tx);
            if (!configuration->HasCarsFilter()) {
                configuration->OptionalCarsFilter().ConstructInPlace();
            }
            configuration->MutableCarsFilterRef().SetIncludeTagsFilter(TTagsFilter::BuildFromString(affiliatedCompanyTagDescription.GetOwningCarTagName()));
            ZoneProcess(signalConfigurationJson, configuration, aclCompany, permissions, server, tx);
            return configuration;
        }
    }

    void TCreateSignalProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        auto tx = BuildTx<NSQL::Writable>();
        const auto& api = *Yensured(Server->GetDriveAPI());

        auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
        auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

        auto affiliatedCompanyTagDescription = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx);

        auto signalsConfiguraionsIds = aclCompany->GetEntityObjects(EEntity::SignalConfiguration, permissions);
        ISignalConfiguration::TPtr configuration = BuildConfiguration(requestData["signal_configuration"], *affiliatedCompanyTagDescription, aclCompany, permissions, *Server, tx);

        auto signalsConfigs = api.GetSignalsConfigurationsDB()->GetObjects(signalsConfiguraionsIds, tx);
        R_ENSURE(signalsConfigs, {}, "cannot restore signals", tx);
        for (auto&& signalConf : *signalsConfigs) {
            R_ENSURE(
                signalConf->GetDisplayName() != configuration->GetDisplayName(),
                HTTP_BAD_REQUEST,
                "signal " << configuration->GetDisplayName() << " already exists",
                tx
            );
        }
        const auto signalsCountLimit = permissions->GetSetting<ui32>("leasing.signals_count_limit", 10);
            R_ENSURE(
            signalsConfiguraionsIds.size() < signalsCountLimit,
            HTTP_BAD_REQUEST,
            "signals count limit has been reached: " << signalsCountLimit
        );
        NStorage::TObjectRecordsSet<TSignalConfigurationDB> records;
        TSignalConfigurationDB configurationDB(configuration);
        R_ENSURE(api.GetSignalsConfigurationsDB()->AddObjects(NContainer::Scalar(configurationDB), permissions->GetUserId(), tx, &records), {}, "", tx);
        R_ENSURE(!records.empty(), {}, "cannot add object", tx);

        R_ENSURE(configuration->CreateTags(permissions->GetUserId(), *Server, tx), {}, "can't create signal tags", tx);
        TMessagesCollector errors;
        R_ENSURE(aclCompany->AddEntityObject(EEntity::SignalConfiguration, records.front().GetInternalId(), permissions, errors), HTTP_FORBIDDEN, "cannot add object to company, " << errors.GetStringReport(), tx);
        R_ENSURE(api.GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx), {}, "cannot update object for company", tx);

        TVector<TRTBackgroundProcessContainer> containers = configuration->ConstructBackgroundProcesses();
        for (const auto& container : containers) {
            R_ENSURE(Server->GetRTBackgroundManager()->UpsertObject(container, permissions->GetUserId(), tx), {}, "can't upsert robot", tx);
        }
        R_ENSURE(tx.Commit(), {}, "can't commit", tx);
        g.SetCode(HTTP_OK);
    }
}
