#include "get_signals.h"

#include "common.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/drivematics/signal/manager.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/signalq/signals/helpers.h>
#include <drive/backend/signalq/signals/tag.h>
#include <drive/backend/tags/tags_search.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>

#include <library/cpp/iterator/iterate_keys.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>

#include <variant>

using NDrivematics::TSignalDescription;

namespace NDrive{
namespace {
    using TCarId = TString;
    using TUserId = TString;
    using TTagId = TString;
    using TSignalName = TString;
    using TSessionId = TString;

    using TMinimalCompiledRidingConstRef = std::reference_wrapper<const TMinimalCompiledRiding>;
    using TEventsSessionPtr = TSharedPtr<const IEventsSession<TObjectEvent<TConstDBTag>>, TAtomicCounter>;
    using TDriveCarInfoConstRef = std::reference_wrapper<const TDriveCarInfo>;
    using TSerialNumberEventIdToEventPresignedUrlsMap = NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse::TSerialNumberEventIdToEventPresignedUrlsMap;

    class TSignalFilter {
    private:
        R_OPTIONAL(TSet<TSignalDescription::ESource>, Sources);
        R_OPTIONAL(TSet<TString>, Types);
        R_OPTIONAL(NDrivematics::ISignalConfiguration::ESignalPriority, Priority);

    public:
        auto& ParseSources(TSet<TString>&& sources) {
            if (!sources.empty()) {
                Sources.ConstructInPlace();
                for (auto&& source : sources) {
                    TSignalDescription::ESource sourcesTemp;
                    if (TryFromString(source, sourcesTemp)) {
                        MutableSources()->insert(sourcesTemp);
                    }
                }
                if (GetSourcesRef().empty()) {
                    ClearSources();
                }
            }
            return *this;
        }

        auto& ParseTypes(TSet<TString>&& types) {
            if (!types.empty()) {
                TSet<TString> restoredTelematicsSignalTypes = ISignalConfiguration::TFactory::GetRegisteredKeys();
                SetTypes(MakeIntersection(std::move(types), restoredTelematicsSignalTypes));

                auto signalqEventToTagName = NDrive::NSignalq::GetEventToTagName();
                for (auto&& [type, name] : signalqEventToTagName) {
                    if (types.contains(type)) {
                        MutableTypes()->insert(type);
                    }
                }
            }
            return *this;
        }

        auto& ParsePriority(TString&& priority) {
            if (priority) {
                Priority.ConstructInPlace();
                if (!TryFromString(priority, *Priority)) {
                    ClearPriority();
                }
            }
            return *this;
        }

        bool Apply(const TSignalDescription& signalDescription) const {
            if ((!HasTypes() || GetTypesRef().contains(signalDescription.GetType())) &&
                (!HasSources() || GetSourcesRef().contains(signalDescription.GetSource())) &&
                (!HasPriority() || GetPriorityRef() == signalDescription.GetPriority())
            ) {
                return true;
            }
            return false;
        }
    };

    struct TUserInfo {
    public:
        NJson::TJsonValue Value;

    public:
        TUserInfo(NJson::TJsonValue&& value):
            Value(value)
        {
        }
    };

    class TSignalEvent {
    public:
        class TEntity {
        public:
            using TObjectDetails = std::variant<std::monostate, TMinimalCompiledRidingConstRef, TDriveCarInfoConstRef, TEventsSessionPtr, TUserInfo>;

        public:
            R_FIELD(NEntityTagsManager::EEntityType, EntityType);
            R_FIELD(TString, ObjectId);
            R_FIELD(TObjectDetails, ObjectDetails);

        public:
            DECLARE_FIELDS(
                Field(NJson::Stringify(EntityType), "entity_type"),
                Field(ObjectId, "object_id")
            );
        };

    public:
        R_FIELD(TString, Name);
        R_FIELD(TVector<TEntity>, LinkedEntities);
        R_FIELD(TInstant, Since, TInstant::Zero());
        R_FIELD(bool, IsActual, false);
        R_FIELD(TString, Resolution);
        R_FIELD(TString, TagId);
        R_OPTIONAL(TInstant, Until);
        R_OPTIONAL(TTagHistoryEventId, EventId);

        R_FIELD(NEntityTagsManager::EEntityType, EntityType);
        R_OPTIONAL(NJson::TJsonValue, Details);

    public:
        bool operator<(const TSignalEvent& rhs) const {
            return Since < rhs.Since;
        }

    public:
        DECLARE_FIELDS(
            Field(Name, "name"),
            Field(LinkedEntities, "linked_entities"),
            Field(NJson::Seconds(Since), "since"),
            Field(IsActual, "is_actual"),
            Field(NJson::Nullable(Resolution), "resolution"),
            Field(TagId, "tag_id"),
            Field(EventId, "event_id"),
            Field(NJson::Seconds(Until), "until")
        );
    };

    TMaybe<NJson::TJsonValue> BuildSignalEventDetails(
            const TConstDBTag& dbTag,
            const TMaybe<TSerialNumberEventIdToEventPresignedUrlsMap>& signalqPresignedUrlsMap) {
        if (auto tag = dbTag.GetTagAs<TScoringTraceTag>()) {
            NJson::TJsonValue result = NJson::TMapBuilder("score", tag->GetValue());
            return result;
        }
        if (auto tag = dbTag.GetTagAs<TSignalqEventTraceTag>()) {
            NJson::TJsonValue result;
            NJson::InsertNonNull(result, "resolution", tag->GetResolution());

            if (signalqPresignedUrlsMap) {
                const auto serialNumberEventId = tag->MakeSerialNumberEventId();
                if (auto it = signalqPresignedUrlsMap->find(serialNumberEventId);
                        it != signalqPresignedUrlsMap->end()) {
                    NJson::InsertField(result, "media", it->second.BuildReportJson());
                }
            }
            return result;
        }
        return Nothing();
    }

    TInstant CalculateSignalStart(std::string_view name, TInstant timestamp, const TMap<TSignalName, TSignalDescription>& signalNameToDescription) {
        auto it = signalNameToDescription.find(name);
        if (it != signalNameToDescription.end()) {
            return timestamp - it->second.OptionalActivationThreshold().GetOrElse(TDuration::Zero());
        }
        return timestamp;
    }

    void ApplyHistoryPatch(TSignalDescription& result, const NJson::TJsonValue& param, TInstant since = TInstant::Zero()) {
        if (!param.GetMap().empty()) {
            if (!result.HasParams()) {
                result.OptionalParams().ConstructInPlace();
            }
            result.MutableParams()->Add(param, since);
        }
    }

    TSignalDescription BuildSignalDescription(NDrivematics::ISignalConfiguration::TPtr configurationPtr) {
        R_ENSURE(configurationPtr, HTTP_INTERNAL_SERVER_ERROR, "configuration");
        const auto& configuration = *configurationPtr;
        TSignalDescription result;
        result.SetName(configuration.GetSignalName());
        result.SetType(configuration.GetType());
        if (!TryFromString(configuration.GetSource(), result.MutableSource())) {
            NDrive::TEventLog::Log("BuildSignalDescription", NJson::TMapBuilder
                ("description", "cannot parse signal source")
                ("value", configuration.GetSource())
            );
        }
        result.SetPriority(configuration.GetSignalPriority());
        result.SetDisplayName(configuration.GetDisplayName());
        if (configuration.HasCarsFilter()) {
            result.SetCarsFilter(configuration.GetCarsFilterRef());
        }
        result.SetIsEnabled(configuration.GetIsEnabled());
        result.SetIsEditable(true);
        result.OptionalActivationThreshold() = configurationPtr->OptionalActivationThreshold();
        return result;
    }

    TMap<TSignalName, TSignalDescription> GetObservableSignalsDescriptions(TUserPermissions::TPtr permissions, const TSignalFilter* filter, const NDrive::IServer& server, bool isSupportMode) {
        TVector<TSignalDescription> signalsDescriptions;
        {
            signalsDescriptions = NDrive::NSignalq::GetSignalDescriptions();
        }
        {
            TVector<TSignalDescription> tmpSignalsDescription;
            const auto maybeSignalsNamesStr = permissions->GetSetting<TString>(TGetSignalsProcessor::SignalsDescriptionSettingName);
            if (maybeSignalsNamesStr) {
                NJson::TJsonValue value;
                R_ENSURE(NJson::ReadJsonFastTree(*maybeSignalsNamesStr, &value), HTTP_INTERNAL_SERVER_ERROR, "can't parse signals names");
                R_ENSURE(NJson::ParseField(value["descriptions"], tmpSignalsDescription), HTTP_INTERNAL_SERVER_ERROR, "can't convert json into signals descriptions");
                for (auto& elem : tmpSignalsDescription) {
                    signalsDescriptions.push_back(std::move(elem));
                }
            }
        }
        if (isSupportMode) {
            R_ENSURE(
                permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Tag),
                HTTP_FORBIDDEN,
                "no permission to Modify::Tag"
            );
            TSignalDescription supportTagDescription;
            supportTagDescription
                .SetName(NDrive::NSignalq::GetSupportPreapprovedTagName())
                .SetSource(TSignalDescription::ESource::SignalQ)
                .SetPriority(NDrivematics::ISignalConfiguration::ESignalPriority::Normal)
                .SetIsPermanent(true);
            signalsDescriptions.push_back(std::move(supportTagDescription));
        }

        const auto observableTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
        TMap<TSignalName, TSignalDescription> result;
        for (auto& signalDesc : signalsDescriptions) {
            if (observableTagsNames.contains(signalDesc.GetName())) {
                // manually created signals are not editable in any way and are always active
                signalDesc.SetIsEnabled(true);
                signalDesc.SetIsEditable(false);
                if (!filter || filter->Apply(signalDesc)) {
                    result[signalDesc.GetName()] = signalDesc;
                }
            }
        }
        auto tx = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        TSet<NDrivematics::TSignalConfigurationDB::TId> signalsConfIds;
        {
            auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, server);
            auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), server);
            signalsConfIds = aclCompany->GetEntityObjects(TAdministrativeAction::EEntity::SignalConfiguration, permissions, /*force=*/true);
        }
        TVector<TObjectEvent<TSignalConfigurationDB>> signalConfigurationsHistoryEvents;
        TSignalsConfigurationsHistoryManager::TQueryOptions signalsHistoryQuery;
        if (!signalsConfIds.empty()) {
            signalsHistoryQuery
                .SetSignalIds(signalsConfIds)
                .SetActions({ EObjectHistoryAction::Add, EObjectHistoryAction::UpdateData })
                .SetDescending(true);

            auto optionalEvents = server.GetDriveAPI()->GetSignalsConfigurationsDB()->GetHistoryManager().GetEvents({}, {}, tx, std::move(signalsHistoryQuery));
            R_ENSURE(optionalEvents, {}, "can't get events", tx);
            signalConfigurationsHistoryEvents = std::move(*optionalEvents);

            for (auto signalConfigurationDB : signalConfigurationsHistoryEvents) {
                const auto& signalName = Yensured(signalConfigurationDB.GetData())->GetSignalName();
                if (!result.contains(signalName)) {
                    auto signalDescription = BuildSignalDescription(signalConfigurationDB.GetData());
                    if (!filter || filter->Apply(signalDescription)) {
                        result[signalName] = std::move(signalDescription);
                    } else {
                        continue;
                    }
                }
                ApplyHistoryPatch(result[signalName], signalConfigurationDB->SerializeParamsToJson(), signalConfigurationDB.GetHistoryTimestamp());
            }
        }
        return result;
    }
}
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::TSignalEvent& object) {
    NJson::TJsonValue result = NJson::FieldsToJson(object.GetFields());
    if (object.HasDetails()) {
        result["details"] = object.GetDetailsRef();
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::TSignalEvent::TEntity& object) {
    using namespace NDrive;
    auto result = NJson::FieldsToJson(object.GetFields());
    auto& details = result["details"];
    if (std::holds_alternative<TMinimalCompiledRidingConstRef>(object.GetObjectDetails())) {
        const auto& compiledRiding = std::get<TMinimalCompiledRidingConstRef>(object.GetObjectDetails()).get();
        details["since"] = compiledRiding.GetStartInstant().Seconds();
        details["until"] = compiledRiding.GetFinishInstant().Seconds();
    } else if (std::holds_alternative<TDriveCarInfoConstRef>(object.GetObjectDetails())) {
        const auto& carInfo = std::get<TDriveCarInfoConstRef>(object.GetObjectDetails()).get();
        details["number"] = carInfo.GetNumber();
        details["model"] = carInfo.GetModel();
    } else if (std::holds_alternative<TEventsSessionPtr>(object.GetObjectDetails())) {
        const auto actualRiding = std::get<TEventsSessionPtr>(object.GetObjectDetails());
        details["since"] = actualRiding->GetStartTS().Seconds();
    } else if (std::holds_alternative<TUserInfo>(object.GetObjectDetails())) {
        const auto userData = std::get<TUserInfo>(object.GetObjectDetails());
        details = userData.Value;
    }
    return result;
}

void NDrive::TGetSignalsDescriptionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    auto locale = GetLocale();

    TSignalFilter filter;
    filter
        .ParseTypes(MakeSet(GetStrings(Context->GetCgiParameters(), "type", false)))
        .ParseSources(MakeSet(GetStrings(Context->GetCgiParameters(), "source", false)))
        .ParsePriority(GetString(Context->GetCgiParameters(), "priority", false));

    auto isSupportMode = GetValue<bool>(Context->GetCgiParameters(), "is_support_mode", false).GetOrElse(false);
    auto signalNameToDescription = GetObservableSignalsDescriptions(permissions, &filter, *Server, isSupportMode);
    {
        NJson::TJsonValue signalDescriptions;
        for (auto&& [name, description] : signalNameToDescription) {
            signalDescriptions.InsertValue(name, description.GetReport(locale, *Server));
        }
        g.AddReportElement("signals_descriptions", std::move(signalDescriptions));
    }
    g.SetCode(HTTP_OK);
}

void NDrive::TGetSignalsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    auto process2 = GetHandlerSetting<bool>("get_signals_process2").GetOrElse(false);
    if (!process2) {
        Process1(g, permissions);
    } else {
        Process2(g, permissions);
    }
}

namespace {
    TVector<TString>&& ToCaseVector(TVector<TString>&& containerStr, bool toLower = true) {
        std::function<void(TString&)> func;
        if (toLower) {
            func = [](TString& str) -> void {
                str = ToLowerUTF8(str);
            };
        } else {
            func = [](TString& str) -> void {
                str = ToUpperUTF8(str);
            };
        }
        for (auto& valueStr : containerStr) {
            func(valueStr);
        }
        return std::move(containerStr);
    }
}

void NDrive::TGetSignalsProcessor::Process1(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const auto& api = *Yensured(Server->GetDriveAPI());
    const auto& server = *Yensured(Server);

    auto tx = BuildTx<NSQL::ReadOnly>();

    const auto& cgi = Context->GetCgiParameters();
    auto now = Now();
    auto defaultDepth = TDuration::Days(7);
    auto signalId = GetString(cgi, "signal_id", false);
    auto actualOnly = GetValue<bool>(cgi, "actual_only", false);
    auto signalsExt = MakeSet(GetStrings(cgi, "signals", false));
    auto since = GetTimestamp(cgi, "since", now - defaultDepth);
    auto until = GetTimestamp(cgi, "until", now);
    auto carTagsHistoryCursor = GetValue<ui64>(cgi, "cars_cursor", false);
    auto userTagsHistoryCursor = GetValue<ui64>(cgi, "user_cursor", false);
    auto sessionTagsHistoryCursor = GetValue<ui64>(cgi, "sessions_cursor", false);
    auto pageSize = GetValue<ui64>(cgi, "page_size", false);

    TRange<TInstant> timeRange;
    if (!signalId) {
        timeRange.From = since;
        timeRange.To = until;
    }

    auto isSupportMode = GetValue<bool>(cgi, "is_support_mode", false).GetOrElse(false);

    TSet<TString> carsIdsExt;
    if (cgi.Has("cars_ids")) {
        carsIdsExt = MakeSet(GetStrings(cgi, "cars_ids", false));
    } else if (cgi.Has("numbers")) {
        carsIdsExt = MakeSet(api.GetCarIdByNumbers(ToCaseVector(GetStrings(cgi, "numbers", false))));
    } else if (cgi.Has("vins")) {
        auto vins = MakeSet(ToCaseVector(GetStrings(cgi, "vins", false), false));
        auto cars = DriveApi->GetCarManager().FetchCarsByVIN(vins, tx);
        carsIdsExt.merge(MakeSet<TString>(IterateKeys(cars)));
    }
    auto carsModels = MakeSet(GetStrings(cgi, "models", false));

    TSignalFilter filter;
    filter
        .ParseTypes(MakeSet(GetStrings(Context->GetCgiParameters(), "type", false)))
        .ParseSources(MakeSet(GetStrings(Context->GetCgiParameters(), "source", false)))
        .ParsePriority(GetString(Context->GetCgiParameters(), "priority", false));

    auto locale = GetLocale();

    auto visibleCarIdToTaggedObject = [&]() -> TMap<TCarId, TTaggedObject> {
        TTaggedObjectsSnapshot carsDevicesSnapshot;
        {
            auto eg = g.BuildEventGuard("restore_cars");
            R_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds(carsIdsExt, carsDevicesSnapshot, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can't restore cars from cache");
        }
        TMap<TCarId, TTaggedObject> result;
        {
            auto eg = g.BuildEventGuard("get_visibility");
            for (const auto& [carId, taggedObject] : carsDevicesSnapshot) {
                if (permissions->GetVisibility(taggedObject, NEntityTagsManager::EEntityType::Car) == TUserPermissions::EVisibility::Visible) {
                    result[carId] = taggedObject;
                }
            }
        }
        return result;
    }();
    auto visibleCarIds = MakeVector<TString>(IterateKeys(visibleCarIdToTaggedObject));

    TMap<TString, TDriveCarInfo> visibleCarIdToCarInfo;
    {
        auto eg = g.BuildEventGuard("get_car_info");
        auto action = [&visibleCarIdToCarInfo, &carsModels, &visibleCarIdToTaggedObject](const TString& carId, const TDriveCarInfo& carInfo) -> void {
            if (carsModels.empty() || carsModels.contains(carInfo.GetModel())) {
                visibleCarIdToCarInfo[carId] = carInfo;
            } else {
                visibleCarIdToTaggedObject.erase(carId);
            }
        };
        R_ENSURE(
            Yensured(api.GetCarsData())->ForObjectsMap(action, TInstant::Zero(), &visibleCarIds)
            , {}
            , "can't get cars info"
            , tx
        );
    }

    visibleCarIds = MakeVector<TString>(IterateKeys(visibleCarIdToCarInfo));

    auto signalNameToDescription = GetObservableSignalsDescriptions(permissions, &filter, *Server, isSupportMode);
    auto reportedSignalsNames = MakeSet<TString>(IterateKeys(signalNameToDescription));
    if (!signalsExt.empty()) {
        reportedSignalsNames = MakeIntersection(reportedSignalsNames, signalsExt);
    }

    auto visibleSessionIdToCompiledRiding = [&]() -> TMap<TString, TObjectEvent<TMinimalCompiledRiding>> {
        auto eg = g.BuildEventGuard("get_compiled_sessions");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>("get_signals_processor");
        auto maybeCompiledSessions = server.GetDriveDatabase().GetCompiledSessionManager().GetObject<TMinimalCompiledRiding>(visibleCarIds, tx, ydbTx, timeRange);
        R_ENSURE(maybeCompiledSessions, HTTP_INTERNAL_SERVER_ERROR, "can't fetch compiled sessions", tx);
        TMap<TString, TObjectEvent<TMinimalCompiledRiding>> result;
        for (auto& compiledRiding : *maybeCompiledSessions) {
            result[compiledRiding.GetSessionId()] = std::move(compiledRiding);
        }
        return result;
    }();
    auto visibleSessionIds = MakeVector<TString>(IterateKeys(visibleSessionIdToCompiledRiding));

    TMap<TString, TCommonTagSessionManager::TSessionConstPtr> visibleSessionIdToActualRiding;
    if (GetHandlerSettingDef<bool>("enable_actual_sessions_signals", false)) {
        auto actualSessions = DriveApi->GetSessionManager().GetObjectsSessions(visibleCarIds, tx);
        R_ENSURE(actualSessions, HTTP_INTERNAL_SERVER_ERROR, "can't fetch actual sessions", tx);
        visibleSessionIds.reserve(visibleSessionIds.size() + actualSessions->size());
        for (auto&& riding: *actualSessions) {
            if (riding && riding->GetSessionId()) {
                visibleSessionIds.push_back(riding->GetSessionId());
                visibleSessionIdToActualRiding[riding->GetSessionId()] = riding;
            }
        }
    }

    TVector<TObjectEvent<TConstDBTag>> carTagsHistoryEvents;
    {
        TTagEventsManager::TQueryOptions carTagsHistoryQuery;
        if (signalId) {
            carTagsHistoryQuery
                .SetTagIds({signalId})
                .SetTags(std::cref(reportedSignalsNames))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleCarIds));
        } else if (actualOnly.GetOrElse(false)) {
            TVector<TTagId> visibleCarsSignalTagIds;
            for (const auto& [_, taggedObject] : visibleCarIdToTaggedObject) {
                for (const auto& tag : taggedObject.GetTags()) {
                    if (reportedSignalsNames.contains(tag->GetName())) {
                        visibleCarsSignalTagIds.push_back(tag.GetTagId());
                    }
                }
            }
            carTagsHistoryQuery
                .SetTagIds(std::move(visibleCarsSignalTagIds))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleCarIds))
                .SetDescending(true);
        } else {
            carTagsHistoryQuery
                .SetTags(std::cref(reportedSignalsNames))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleCarIds))
                .SetDescending(true);
        }

        if (pageSize) {
            carTagsHistoryQuery.SetLimit(*pageSize + 1);
        }

        {
            auto eg = g.BuildEventGuard("scan_car_tags_history");
            auto optionalEvents = api.GetTagsManager().GetDeviceTags().GetEvents({ 0, carTagsHistoryCursor }, timeRange, tx, carTagsHistoryQuery);
            R_ENSURE(optionalEvents, {}, "can't get events", tx);
            carTagsHistoryEvents = std::move(*optionalEvents);
        }
    }

    auto visibleUserIdToTags = [&]() -> TMap<TUserId, TVector<TConstDBTag>> {
        if (!carsIdsExt.empty() || !carsModels.empty()) {
            return {};
        }
        auto company = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, *Server, tx);
        TTagsSearchRequest req({company->GetName()}, {}, {}, 1000);

        auto search = DriveApi->GetTagsSearch(NEntityTagsManager::EEntityType::User);
        R_ENSURE(search, ConfigHttpStatus.ServiceUnavailable, "can't find users", NDrive::MakeError("component_unavailable"), tx);
        auto result = search->Search(req, tx);
        auto userToTags = result.GetAssociatedTags();
        for (auto&& userId : result.GetMatchedIds()) {
            if (!CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId)) {
                userToTags.erase(userId);
            }
        }
        return userToTags;
    }();

    auto visibleUserIds = MakeVector<TString>(IterateKeys(visibleUserIdToTags));
    auto usersInfo = Yensured(api.GetUsersData())->FetchInfo(visibleUserIds, tx);

    TVector<TObjectEvent<TConstDBTag>> userTagsHistoryEvents;
    {

        TTagEventsManager::TQueryOptions userTagsHistoryQuery;
        if (signalId) {
            userTagsHistoryQuery
                .SetTagIds({signalId})
                .SetTags(std::cref(reportedSignalsNames))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleUserIds));
        } else if (actualOnly.GetOrElse(false)) {
            TVector<TTagId> visibleUsersSignalTagIds;
            for (const auto& [_, tags] : visibleUserIdToTags) {
                for (const auto& tag : tags) {
                    if (reportedSignalsNames.contains(tag->GetName())) {
                        visibleUsersSignalTagIds.push_back(tag.GetTagId());
                    }
                }
            }
            userTagsHistoryQuery
                .SetTagIds(std::move(visibleUsersSignalTagIds))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleUserIds))
                .SetDescending(true);
        } else {
            userTagsHistoryQuery
                .SetTags(std::cref(reportedSignalsNames))
                .SetActions({ EObjectHistoryAction::Add })
                .SetObjectIds(std::move(visibleUserIds))
                .SetDescending(true);
        }

        {
            auto eg = g.BuildEventGuard("scan_user_tags_history");
            auto optionalEvents = api.GetTagsManager().GetUserTags().GetEvents({ 0, userTagsHistoryCursor }, timeRange, tx, userTagsHistoryQuery);
            R_ENSURE(optionalEvents, {}, "can't get events", tx);
            userTagsHistoryEvents = std::move(*optionalEvents);
        }
    }

    const TStringBuf traceTagsSettingsGvarsPath = isSupportMode
        ? "signalq.signalq_events.support_trace_tags_query_settings"
        : "signalq.signalq_events.trace_tags_query_settings";
    auto maybeTraceSignalsQuerySettingsStr = server.GetSettings().GetValue<TString>(traceTagsSettingsGvarsPath);
    TTraceSignalsQuerySettings traceSignalsQuerySettings;
    if (maybeTraceSignalsQuerySettingsStr) {
        NJson::TJsonValue value;
        R_ENSURE(NJson::ReadJsonFastTree(*maybeTraceSignalsQuerySettingsStr, &value), HTTP_INTERNAL_SERVER_ERROR, "can't parse signalq trace signals settings");
        R_ENSURE(NJson::ParseField(value, traceSignalsQuerySettings), HTTP_INTERNAL_SERVER_ERROR, "can't convert signalq trace signals settings to json");
    }

    TSet<TString> signalqSignalsNames = [&signalNameToDescription]() {
        TSet<TString> result;
        for (const auto& elem : signalNameToDescription) {
            if (elem.second.GetSource() == TSignalDescription::ESource::SignalQ) {
                result.insert(elem.first);
            }
        }
        return result;
    }();

    TTraceSignalsQueryParams traceSignalsParams;
    traceSignalsParams
        .SetPageSize(pageSize)
        .SetSessionTagHistoryCursor(sessionTagsHistoryCursor)
        .SetReportedSignalsNames(std::cref(reportedSignalsNames))
        .SetSignalqSignalsNames(std::move(signalqSignalsNames))
        .SetVisibleSessionIds(std::move(visibleSessionIds))
        .SetPeriodRange(timeRange)
        .SetMaxQueriesCount(traceSignalsQuerySettings.GetMaxQueriesCount())
        .SetLimitPerQuery(traceSignalsQuerySettings.GetLimitPerQuery())
        .SetSupportMode(isSupportMode)
    ;

    if (signalId) {
        traceSignalsParams.SetSignalId(signalId);
    }

    auto sessionTagsHistoryEventsData = GetTraceSignals(g, tx, api.GetTagsManager().GetTraceTags(), std::move(traceSignalsParams));
    auto sessionTagsHistoryEvents = std::move(sessionTagsHistoryEventsData.GetTraceSignals());
    std::reverse(sessionTagsHistoryEvents.begin(), sessionTagsHistoryEvents.end());
    std::reverse(carTagsHistoryEvents.begin(), carTagsHistoryEvents.end());
    std::reverse(userTagsHistoryEvents.begin(), userTagsHistoryEvents.end());

    TVector<TString> carTagIds;
    for (auto&& event : carTagsHistoryEvents) {
        carTagIds.push_back(event.GetTagId());
    }

    TVector<TString> userTagIds;
    for (auto&& event : userTagsHistoryEvents) {
        userTagIds.push_back(event.GetTagId());
    }

    if (signalId && carTagIds.empty() && userTagIds.empty()) {
        bool valid = false;
        for (const auto& event : sessionTagsHistoryEvents){
            if (event.GetTagId() == signalId) {
                valid = true;
                break;
            }
        }
        if (!valid || sessionTagsHistoryEvents.size() > 1) {
            g.SetCode(HTTP_NOT_FOUND);
            return;
        }
    }

    TVector<TString> sessionTagIds;
    TVector<NSignalq::TV1EventWithMedia> signalqEventsWithMedia;
    const auto isSignalqMediaEnabled = server.GetSettings().GetValue<bool>("signalq.signals_media.enabled").GetOrElse(false);
    for (auto&& event : sessionTagsHistoryEvents) {
        sessionTagIds.push_back(event.GetTagId());
        if (!isSignalqMediaEnabled) {
            continue;
        }

        if (const auto tag = event.GetTagAs<TSignalqEventTraceTag>()) {
            const auto& tagEvent = tag->GetEvent();

            NSignalq::TV1EventWithMedia signalqEventWithMedia;
            signalqEventWithMedia.SetSerialNumberEventId(tag->MakeSerialNumberEventId());
            signalqEventWithMedia.OptionalS3VideoPath() = tagEvent.OptionalS3VideoPath();
            signalqEventWithMedia.OptionalS3ExternalVideoPath() = tagEvent.OptionalS3ExternalVideoPath();
            signalqEventWithMedia.OptionalS3PhotoPath() = tagEvent.OptionalS3PhotoPath();
            signalqEventWithMedia.OptionalS3ExternalPhotoPath() = tagEvent.OptionalS3ExternalPhotoPath();

            signalqEventsWithMedia.push_back(std::move(signalqEventWithMedia));
        }
    }

    auto signalqPresignedUrlsFuture = NThreading::MakeFuture(TSerialNumberEventIdToEventPresignedUrlsMap());
    if (!signalqEventsWithMedia.empty()) {
        const auto client = server.GetTaxiSignalqDrivematicsApiClient();
        if (!client) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "NoSignalqDrivematicsClient")
                ("error", "Can not fetch signalq presigned urls: no client"));
        } else {
            static const TDuration defaultTtlDuration = TDuration::Days(1);
            const auto signalqSignalsMediaPresignedUrlsTtl = server.GetSettings().GetValue<TDuration>("signalq.signals_media.presigned_urls_ttl").GetOrElse(defaultTtlDuration);
            const auto signalqSignalsMediaOnlyUploaded = server.GetSettings().GetValue<bool>("signalq.signals_media.only_uploaded").GetOrElse(false);

            const auto start = Now();
            NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest requestParams;
            const auto linksExpiresAt = start + signalqSignalsMediaPresignedUrlsTtl;
            requestParams.SetLinksExpiresAt(linksExpiresAt);
            requestParams.SetOnlyUploaded(signalqSignalsMediaOnlyUploaded);
            requestParams.SetEvents(std::move(signalqEventsWithMedia));

            const auto requestEventsCount = signalqEventsWithMedia.size();
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventsPresignedUrls")
                ("request_events_count", requestEventsCount));
            const auto signalqPresignedUrlsResponse = client->GetEventsMediaPresignedUrls(requestParams);
            signalqPresignedUrlsFuture = signalqPresignedUrlsResponse.Apply([start, requestEventsCount, linksExpiresAt](const NThreading::TFuture<NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse>& response) {
                if (response.HasValue()) {
                    const auto finish = Now();
                    const auto duration = finish - start;
                    INFO_LOG << "Fetched signalq events presigned urls, duration: " << duration << Endl;
                    const auto eventsCount = response.GetValue().GetEvents().size();
                    if (eventsCount != requestEventsCount) {
                        ERROR_LOG << "Fetched signalq events in count: " << eventsCount << " differs than expected: " << requestEventsCount << Endl;
                    } else {
                        INFO_LOG << "Fetched events with presigned urls in count: " << eventsCount << Endl;
                    }
                    return response.GetValue().BuildSerialNumberEventIdToEventPresignedUrlsMap(linksExpiresAt);
                } else {
                    ythrow yexception() << "Cant fetch signalq presigned urls: no response, exp: " << NThreading::GetExceptionMessage(response) << Endl;
                }
            });
        }
    }

    TMap<TString, TInstant> removeTagIdTimestamps;
    {
        auto eg = g.BuildEventGuard("scan_car_tags_history_removes");
        TTagEventsManager::TQueryOptions options;
        options
            .SetActions({ EObjectHistoryAction::Remove })
            .SetTagIds(std::cref(carTagIds));
        if (signalId) {
            options.SetTagIds({signalId});
        }
        auto optionalEvents = api.GetTagsManager().GetDeviceTags().GetEvents({}, timeRange.From, tx, options);
        R_ENSURE(optionalEvents, {}, "cannot get DeviceTagEvents", tx);
        for (auto&& event : *optionalEvents) {
            removeTagIdTimestamps[event.GetTagId()] = event.GetHistoryTimestamp();
        }
    }
    {
        auto eg = g.BuildEventGuard("scan_user_tags_history_removes");
        TTagEventsManager::TQueryOptions options;
        options
            .SetActions({ EObjectHistoryAction::Remove })
            .SetTagIds(std::cref(userTagIds));
        if (signalId) {
            options.SetTagIds({signalId});
        }
        auto optionalEvents = api.GetTagsManager().GetUserTags().GetEvents({}, timeRange.From, tx, options);
        R_ENSURE(optionalEvents, {}, "cannot get UserTagEvents", tx);
        for (auto&& event : *optionalEvents) {
            removeTagIdTimestamps[event.GetTagId()] = event.GetHistoryTimestamp();
        }
    }
    {
        auto eg = g.BuildEventGuard("scan_trace_tags_history_removes");
        TTagEventsManager::TQueryOptions options;
        options
            .SetActions({ EObjectHistoryAction::Remove })
            .SetTagIds(std::cref(sessionTagIds));
        if (signalId) {
            options.SetTagIds({signalId});
        }
        auto optionalEvents = api.GetTagsManager().GetTraceTags().GetEvents({}, timeRange.From, tx, options);
        R_ENSURE(optionalEvents, {}, "cannot get TraceTagEvents", tx);
        for (auto&& event : *optionalEvents) {
            removeTagIdTimestamps[event.GetTagId()] = event.GetHistoryTimestamp();
        }
    }

    auto buildSignalEvents = [&removeTagIdTimestamps, &signalNameToDescription] (
        std::deque<TSignalEvent>& signalEvents,
        NEntityTagsManager::EEntityType entityType,
        const auto& events,
        const auto& getLinkedEntitiesActor,
        const auto& checkIsActualActor,
        const TMaybe<TSerialNumberEventIdToEventPresignedUrlsMap>& signalqPresignedUrlsMap,
        const TMaybe<TString>& signalqSupportPreapprovedTag
    ) -> void {
        for (const auto& ev : events) {
            const auto& objectId = ev.GetObjectId();
            const auto& signalName = ev->GetName();
            const auto& tagId = ev.GetTagId();
            auto historyAction = ev.GetHistoryAction();
            switch (historyAction) {
                case EObjectHistoryAction::UpdateData:
                case EObjectHistoryAction::TagEvolve:
                case EObjectHistoryAction::Add: {
                    auto& signalEvent = signalEvents.emplace_back();
                    if (signalqSupportPreapprovedTag && signalName == *signalqSupportPreapprovedTag) {
                        auto tag = *Yensured(ev.template GetTagAs<TSignalqEventTraceTag>());
                        signalEvent.SetName(tag.GetEvent().GetType());
                    } else {
                        signalEvent.SetName(signalName);
                    }
                    signalEvent.OptionalDetails() = BuildSignalEventDetails(ev, signalqPresignedUrlsMap);
                    signalEvent.SetLinkedEntities(getLinkedEntitiesActor(objectId));
                    signalEvent.SetSince(CalculateSignalStart(ev->GetName(), ev.GetHistoryInstant(), signalNameToDescription));
                    signalEvent.SetEntityType(entityType);
                    signalEvent.SetEventId(ev.GetHistoryEventId());
                    signalEvent.SetTagId(tagId);
                    if (checkIsActualActor(objectId, tagId)) {
                        signalEvent.SetIsActual(true);
                    }
                    auto i = removeTagIdTimestamps.find(tagId);
                    if (i != removeTagIdTimestamps.end()) {
                        signalEvent.SetUntil(i->second);
                    }
                    break;
                }
                default:
                    break;
            }
        }
    };

    auto getCarEntity = [&visibleCarIdToCarInfo](const TString& objectId) -> TSignalEvent::TEntity {
        TSignalEvent::TEntity result;
        result.SetObjectId(objectId);
        result.SetEntityType(NEntityTagsManager::EEntityType::Car);
        if (auto it = visibleCarIdToCarInfo.find(objectId); it != visibleCarIdToCarInfo.end()) {
            result.SetObjectDetails(std::cref(it->second));
        }
        return result;
    };

    std::deque<TSignalEvent> carsSignalEvents;
    {
        buildSignalEvents(
            carsSignalEvents,
            NEntityTagsManager::EEntityType::Car,
            carTagsHistoryEvents,
            [&getCarEntity] (const TString& objectId) { // getLinkedEntitiesActor
                TVector<TSignalEvent::TEntity> result;
                result.push_back(getCarEntity(objectId));
                return result;
            },
            [&visibleCarIdToTaggedObject] (const TString& objectId, const TString& tagId) { // checkIsActualActor
                const auto it = visibleCarIdToTaggedObject.find(objectId);
                if (it == visibleCarIdToTaggedObject.end()) {
                    return false;
                }
                const auto& objectTags = it->second.GetTags();
                auto tagsIt = std::find_if(objectTags.begin(), objectTags.end(), [&tagId](const TDBTag& dbTag) {
                    return dbTag.GetTagId() == tagId;
                });
                return tagsIt != objectTags.end();
            },
            Nothing(),
            Nothing()
        );
    }

    auto getUserEntity = [&usersInfo, &permissions](const TString& objectId) -> TSignalEvent::TEntity {
        TSignalEvent::TEntity result;
        result.SetObjectId(objectId);
        result.SetEntityType(NEntityTagsManager::EEntityType::User);
        if (auto u = usersInfo.find(objectId); u != usersInfo.end()) {
            result.SetObjectDetails(
                TUserInfo(u->second.GetReport(NUserReport::ReportUserSignal & permissions->GetUserReportTraits(), true))
            );
        }
        return result;
    };

    std::deque<TSignalEvent> usersSignalEvents;
    {
        buildSignalEvents(
            usersSignalEvents,
            NEntityTagsManager::EEntityType::User,
            userTagsHistoryEvents,
            [&getUserEntity] (const TString& objectId) { // getLinkedEntitiesActor
                TVector<TSignalEvent::TEntity> result;
                result.push_back(getUserEntity(objectId));
                return result;
            },
            [&visibleUserIdToTags] (const TString& objectId, const TString& tagId) { // checkIsActualActor
                const auto it = visibleUserIdToTags.find(objectId);
                if (it == visibleUserIdToTags.end()) {
                    return false;
                }
                const auto& objectTags = it->second;
                auto tagsIt = std::find_if(objectTags.begin(), objectTags.end(), [&tagId](const TConstDBTag& dbTag) {
                    return dbTag.GetTagId() == tagId;
                });
                return tagsIt != objectTags.end();
            },
            Nothing(),
            Nothing()
        );
    }

    signalqPresignedUrlsFuture.Wait();
    TMaybe<TSerialNumberEventIdToEventPresignedUrlsMap> signalqPresignedUrlsMap;
    if (signalqPresignedUrlsFuture.HasValue()) {
        signalqPresignedUrlsMap = signalqPresignedUrlsFuture.ExtractValue();
    } else {
        ERROR_LOG << "Failed to fetch signalq statuses" << NThreading::GetExceptionMessage(signalqPresignedUrlsFuture) << Endl;
    }

    std::deque<TSignalEvent> sessionsSignalEvents;
    {
        const auto signalqSupportPreapprovedTag = NDrive::NSignalq::GetSupportPreapprovedTagName();
        buildSignalEvents(
            sessionsSignalEvents,
            NEntityTagsManager::EEntityType::Trace,
            sessionTagsHistoryEvents,
            [&visibleSessionIdToCompiledRiding, &visibleSessionIdToActualRiding, &getCarEntity](const TString& objectId) { // getLinkedEntitiesActor
                TVector<TSignalEvent::TEntity> result;
                result.emplace_back().SetObjectId(objectId).SetEntityType(NEntityTagsManager::EEntityType::Trace);
                auto it = visibleSessionIdToCompiledRiding.find(objectId);
                if (it != visibleSessionIdToCompiledRiding.end()) {
                    result.back().SetObjectDetails(std::cref(static_cast<const TMinimalCompiledRiding&>(it->second)));
                    result.push_back(getCarEntity(it->second.GetObjectId()));
                } else if (auto it2 = visibleSessionIdToActualRiding.find(objectId); it2 != visibleSessionIdToActualRiding.end()) {
                    result.back().SetObjectDetails(it2->second);
                    result.push_back(getCarEntity(it2->second->GetObjectId()));
                }
                return result;
            },
            [](const TString& /* objectId */, const TString& /* tagId */) { // checkIsActualActor
                return true;
            },
            signalqPresignedUrlsMap,
            signalqSupportPreapprovedTag
        );
    }
    {
        TVector<TSignalEvent> resultSignalEvents;
        if (!signalId) {
            resultSignalEvents.reserve(carsSignalEvents.size() + sessionsSignalEvents.size() + usersSignalEvents.size());
            std::merge(carsSignalEvents.begin(), carsSignalEvents.end(),
                        sessionsSignalEvents.begin(), sessionsSignalEvents.end(),
                        std::back_inserter(resultSignalEvents));
            std::move(usersSignalEvents.begin(), usersSignalEvents.end(), std::back_inserter(resultSignalEvents));
            std::reverse(resultSignalEvents.begin(), resultSignalEvents.end());
            if (pageSize) {
                ui32 carSignalsEventsCount = 0;
                ui32 sessionSignalsEventsCount = 0;
                ui32 userSignalsEventsCount = 0;
                for (ui32 i = 0; i < *pageSize && i < resultSignalEvents.size(); ++i) {
                    switch (resultSignalEvents[i].GetEntityType()) {
                        case NEntityTagsManager::EEntityType::Car: {
                            ++carSignalsEventsCount;
                            break;
                        }
                        case NEntityTagsManager::EEntityType::Trace: {
                            ++sessionSignalsEventsCount;
                            break;
                        }
                        case NEntityTagsManager::EEntityType::User: {
                            ++userSignalsEventsCount;
                            break;
                        }
                        default:
                            break;
                    }
                }
                Y_ENSURE(carSignalsEventsCount <= carTagsHistoryEvents.size());
                Y_ENSURE(userSignalsEventsCount <= userTagsHistoryEvents.size());
                Y_ENSURE(sessionSignalsEventsCount <= sessionTagsHistoryEvents.size());
                TMaybe<ui64> nextCarTagsHistoryCursor = carSignalsEventsCount ? MakeMaybe(carTagsHistoryEvents[carTagsHistoryEvents.size() - carSignalsEventsCount].GetHistoryEventId()) : carTagsHistoryCursor;
                TMaybe<ui64> nextUserTagsHistoryCursor = userSignalsEventsCount ? MakeMaybe(userTagsHistoryEvents[userTagsHistoryEvents.size() - userSignalsEventsCount].GetHistoryEventId()) : userTagsHistoryCursor;
                TMaybe<ui64> nextSessionTagsHistoryCursor = sessionTagsHistoryEventsData.OptionalLastEventsCursor();
                if (sessionSignalsEventsCount && sessionSignalsEventsCount < sessionTagsHistoryEvents.size()) {
                    nextSessionTagsHistoryCursor = sessionTagsHistoryEvents[sessionTagsHistoryEvents.size() - sessionSignalsEventsCount].GetHistoryEventId();
                }

                bool canGetMorePages = resultSignalEvents.size() > *pageSize;
                if (nextCarTagsHistoryCursor) {
                    g.MutableReport().AddReportElement("next_cars_cursor", *nextCarTagsHistoryCursor);
                }
                if (nextUserTagsHistoryCursor) {
                    g.MutableReport().AddReportElement("next_users_cursor", *nextUserTagsHistoryCursor);
                }
                if (nextSessionTagsHistoryCursor) {
                    g.MutableReport().AddReportElement("next_sessions_cursor", *nextSessionTagsHistoryCursor);
                }
                g.MutableReport().AddReportElement("can_get_more_pages", canGetMorePages);
                if (canGetMorePages) {
                    resultSignalEvents.resize(*pageSize);
                }
            }
        } else {
            std::merge(carsSignalEvents.begin(), carsSignalEvents.end(),
                sessionsSignalEvents.begin(), sessionsSignalEvents.end(),
                std::back_inserter(resultSignalEvents)
            );
            std::move(usersSignalEvents.begin(), usersSignalEvents.end(), std::back_inserter(resultSignalEvents));
            Y_ENSURE(resultSignalEvents.size() < 2);
        }
        g.MutableReport().AddReportElement("signals", NJson::ToJson(resultSignalEvents));
    }
    {
        NJson::TJsonValue signalDescriptions;
        for (auto&& [name, description] : signalNameToDescription) {
            signalDescriptions.InsertValue(name, description.GetReport(locale, *Server));
        }
        g.AddReportElement("signals_descriptions", std::move(signalDescriptions));
    }
    g.SetCode(HTTP_OK);
}

void NDrive::TGetSignalsProcessor::Process2(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions) {
    const auto& cgi = Context->GetCgiParameters();
    auto actualOnly = GetValue<bool>(cgi, "actual_only", false);
    auto requiredCarIds = MakeSet(GetStrings(cgi, "cars_ids", false));
    auto requiredSignals = MakeSet(GetStrings(cgi, "signals", false));
    auto requiredSignalIds = MakeSet(GetStrings(cgi, "signal_id", false));

    auto requiredNumbers = GetStrings(cgi, "numbers", false);
    if (!requiredNumbers.empty()) {
        auto numbers = requiredNumbers;
        auto ids = MakeSet(DriveApi->GetCarIdByNumbers(ToCaseVector(std::move(numbers))));
        requiredCarIds.insert(ids.begin(), ids.end());
    }

    auto requiredVins = GetStrings(cgi, "vins", false);
    if (!requiredVins.empty()) {
        ythrow yexception() << "vins unimplemented";
    }

    auto deadline = Context->GetRequestDeadline();
    auto start = Context->GetRequestStartTime();
    auto cursor = GetValue<ui64>(cgi, "cars_cursor", false);
    auto since = GetTimestamp(cgi, "since", start - TDuration::Days(7));
    auto until = GetTimestamp(cgi, "until", false);
    auto pageSize = GetValue<ui64>(cgi, "page_size", false);
    auto isSupportMode = GetValue<bool>(cgi, "is_support_mode", false).GetOrElse(false);

    const auto& settings = GetSettings();
    const auto signalqMediaEnabled = settings.GetValue<bool>("signalq.signals_media.enabled").GetOrElse(false);
    const auto signalqSignalsMediaPresignedUrlsTtl = settings.GetValue<TDuration>("signalq.signals_media.presigned_urls_ttl").GetOrElse(TDuration::Days(1));
    const auto signalqSignalsMediaOnlyUploaded = settings.GetValue<bool>("signalq.signals_media.only_uploaded").GetOrElse(false);

    auto effectiveUntil = until;
    if (cursor && effectiveUntil) {
        effectiveUntil = std::min(TInstant::Seconds(*cursor), *effectiveUntil);
    }
    if (cursor && !effectiveUntil) {
        effectiveUntil = TInstant::Seconds(*cursor);
    }
    auto locale = GetLocale();

    TSignalFilter filter;
    filter
        .ParseTypes(MakeSet(GetStrings(cgi, "type", false)))
        .ParseSources(MakeSet(GetStrings(cgi, "source", false)))
        .ParsePriority(GetString(cgi, "priority", false));

    auto tx = BuildTx<NSQL::ReadOnly | NSQL::Deferred | NSQL::RepeatableRead>();
    auto visibleSignals = GetObservableSignalsDescriptions(permissions, &filter, *Server, isSupportMode);
    auto visibleSignalNames = MakeSet<TString>(IterateKeys(visibleSignals));
    auto fetchedSignalNames = requiredSignals.empty() ? visibleSignalNames : MakeIntersection(visibleSignalNames, requiredSignals);

    const auto& tagManager = Server->GetDriveDatabase().GetTagsManager();
    auto visibleObjects = [&] {
        auto eg = g.BuildEventGuard("get_visible_objects");
        TVector<TTaggedObject> taggedObjects;
        R_ENSURE(tagManager.GetDeviceTags().GetObjectsFromCacheByIds(requiredCarIds, taggedObjects), HTTP_INTERNAL_SERVER_ERROR, "cannot GetObjectsFromCacheByIds", tx);

        TMap<TString, TTaggedObject> result;
        for (auto&& taggedObject : taggedObjects) {
            auto visibility = permissions->GetVisibility(taggedObject, NEntityTagsManager::EEntityType::Car);
            if (visibility == TUserPermissions::EVisibility::Visible) {
                result[taggedObject.GetId()] = std::move(taggedObject);
            }
        }
        return result;
    }();
    auto visibleObjectsIds = MakeSet<TString>(NContainer::Keys(visibleObjects));

    auto visibleUserIdToTags = [&]() -> TMap<TUserId, TVector<TConstDBTag>> {
        if (!requiredCarIds.empty()) {
            return {};
        }
        auto company = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, *Server, tx);
        TTagsSearchRequest req({company->GetName()}, {}, {}, 1000);

        auto search = DriveApi->GetTagsSearch(NEntityTagsManager::EEntityType::User);
        R_ENSURE(search, ConfigHttpStatus.ServiceUnavailable, "can't find users", NDrive::MakeError("component_unavailable"), tx);
        auto result = search->Search(req, tx);
        auto userToTags = result.GetAssociatedTags();
        for (auto&& userId : result.GetMatchedIds()) {
            if (!CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId)) {
                userToTags.erase(userId);
            }
        }
        return userToTags;
    }();
    auto visibleUserIds = MakeSet<TString>(IterateKeys(visibleUserIdToTags));

    NDrivematics::TSignalManager::TQueryOptions queryOptions;
    queryOptions.SetSince(since);
    queryOptions.SetUntil(effectiveUntil);
    queryOptions.SetActual(actualOnly);
    queryOptions.SetNames(fetchedSignalNames);

    queryOptions.SetObjectIdsOr(visibleObjectsIds);
    queryOptions.SetUserIdsOr(visibleUserIds);

    queryOptions.SetDescending(true);
    if (!requiredSignalIds.empty()) {
        queryOptions.SetIds(requiredSignalIds);
    }
    if (pageSize) {
        queryOptions.SetLimit(*pageSize + 1);
    }
    if (!isSupportMode) {
        queryOptions.SetVisible(true);
    }
    NDrivematics::TSignalManager manager;
    auto optionalSignals = manager.Fetch(tx, std::move(queryOptions));
    R_ENSURE(optionalSignals, {}, "cannot Fetch signals", tx);

    TVector<TString> objectIds;
    TVector<TString> userIds;
    TVector<TString> sessionIds;
    TSet<TString> objectTagIds;
    TSet<TString> userTagIds;
    TSet<TString> sessionTagIds;
    TVector<NSignalq::TV1EventWithMedia> signalqEventsWithMedia;
    {
        objectIds.reserve(optionalSignals->size());
        userIds.reserve(optionalSignals->size() / 2);
        sessionIds.reserve(optionalSignals->size());
        for (auto&& signal : *optionalSignals) {
            if (signal.ObjectId) {
                objectIds.push_back(signal.ObjectId);
            }
            if (signal.SessionId) {
                sessionIds.push_back(signal.SessionId);
            }
            if (signal.UserId) {
                userIds.push_back(signal.UserId);
            }
            if (signal.Type == NEntityTagsManager::EEntityType::User) {
                userTagIds.insert(signal.Id);
                continue;
            }
            if (signal.Type == NEntityTagsManager::EEntityType::Car) {
                objectTagIds.insert(signal.Id);
                continue;
            }
            if (signal.Type == NEntityTagsManager::EEntityType::Trace) {
                sessionTagIds.insert(signal.Id);
                continue;
            }

        }
    }
    auto objectTags = [&] {
        auto eg = g.BuildEventGuard("fetch_object_tags");
        auto optionalTags = Server->GetDriveDatabase().GetTagsManager().GetDeviceTags().RestoreTags(objectTagIds, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreObjectTags", tx);
        auto result = TMap<TString, TAtomicSharedPtr<NDrivematics::TSignalTagTraits>>();
        for (auto&& tag : *optionalTags) {
            auto signal = std::dynamic_pointer_cast<NDrivematics::TSignalTagTraits>(tag.GetData());
            if (!signal) {
                continue;
            }
            result.emplace(tag.GetTagId(), std::move(signal));
        }
        return result;
    }();

    auto userTags = [&] {
        auto eg = g.BuildEventGuard("fetch_user_tags");
        auto optionalTags = Server->GetDriveDatabase().GetTagsManager().GetUserTags().RestoreTags(userTagIds, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreUserTags", tx);
        auto result = TMap<TString, TAtomicSharedPtr<NDrivematics::TSignalTagTraits>>();
        for (auto&& tag : *optionalTags) {
            auto signal = std::dynamic_pointer_cast<NDrivematics::TSignalTagTraits>(tag.GetData());
            if (!signal) {
                continue;
            }
            result.emplace(tag.GetTagId(), std::move(signal));
        }
        return result;
    }();

    auto sessionTags = [&] {
        auto eg = g.BuildEventGuard("fetch_session_tags");
        auto optionalTags = Server->GetDriveDatabase().GetTagsManager().GetTraceTags().RestoreTags(sessionTagIds, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreSessionTags", tx);
        auto result = TMap<TString, TAtomicSharedPtr<NDrivematics::TSignalTagTraits>>();
        for (auto&& tag : *optionalTags) {
            auto signal = std::dynamic_pointer_cast<NDrivematics::TSignalTagTraits>(tag.GetData());
            if (!signal) {
                continue;
            }
            if (signalqMediaEnabled) {
                auto signalqTag = std::dynamic_pointer_cast<TSignalqEventTraceTag>(signal);
                if (signalqTag) {
                    const auto& tagEvent = signalqTag->GetEvent();

                    NSignalq::TV1EventWithMedia signalqEventWithMedia;
                    signalqEventWithMedia.SetS3ExternalPhotoPath(tagEvent.OptionalS3ExternalPhotoPath());
                    signalqEventWithMedia.SetS3ExternalVideoPath(tagEvent.OptionalS3ExternalVideoPath());
                    signalqEventWithMedia.SetS3PhotoPath(tagEvent.OptionalS3PhotoPath());
                    signalqEventWithMedia.SetS3VideoPath(tagEvent.OptionalS3VideoPath());
                    signalqEventWithMedia.SetSerialNumberEventId(signalqTag->MakeSerialNumberEventId());

                    signalqEventsWithMedia.push_back(std::move(signalqEventWithMedia));
                }
            }
            result.emplace(tag.GetTagId(), std::move(signal));
        }
        return result;
    }();

    auto asyncSignalqPresignedUrls = [&] {
        if (signalqEventsWithMedia.empty()) {
            return NThreading::MakeFuture<TSerialNumberEventIdToEventPresignedUrlsMap>();
        }
        auto client = Server->GetTaxiSignalqDrivematicsApiClient();
        if (!client) {
            g.AddEvent("NoSignalqDrivematicsClient");
            return NThreading::MakeFuture<TSerialNumberEventIdToEventPresignedUrlsMap>();
        }

        const auto start = Now();
        const auto linksExpiresAt = start + signalqSignalsMediaPresignedUrlsTtl;
        NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest requestParams;
        requestParams.SetLinksExpiresAt(linksExpiresAt);
        requestParams.SetOnlyUploaded(signalqSignalsMediaOnlyUploaded);
        requestParams.SetEvents(std::move(signalqEventsWithMedia));
        g.AddEvent(NJson::TMapBuilder
            ("event", "SignalqEventsPresignedUrls")
            ("request_events_count", requestParams.GetEvents().size())
        );

        auto signalqPresignedUrlsResponse = client->GetEventsMediaPresignedUrls(requestParams);
        auto state = NDrive::TEventLog::CaptureState();
        return signalqPresignedUrlsResponse.Apply([linksExpiresAt, state = std::move(state)](auto&& r) {
            auto stateGuard = NDrive::TEventLog::Guard(state);
            if (state.EventLogger) {
                state.EventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "SignalqEventsPresignedUrlResponse")
                    ("response_events_count", r.GetValue().GetEvents().size())
                );
            }
            return r.GetValue().BuildSerialNumberEventIdToEventPresignedUrlsMap(linksExpiresAt);
        });
    }();

    auto objects = [&] {
        auto eg = g.BuildEventGuard("fetch_objects");
        return Server->GetDriveDatabase().GetCarManager().FetchInfo(objectIds, tx);
    }();

    auto usersInfo = [&] {
        auto eg = g.BuildEventGuard("fetch_users");
        return Yensured(DriveApi->GetUsersData())->FetchInfo(userIds, tx);
    }();

    auto compiledSessions = [&] {
        auto eg = g.BuildEventGuard("fetch_compiled_sessions");
        auto ydbTx = BuildYdbTx<NSQL::ReadOnly>(GetTypeName());
        auto optionalSessions = Server->GetDriveDatabase().GetCompiledSessionManager().Get<TMinimalCompiledRiding>(sessionIds, tx, ydbTx);
        R_ENSURE(optionalSessions, {}, "cannot GetCompiledSessions", tx);

        auto result = TMap<TString, TMinimalCompiledRiding>();
        for (auto&& session : *optionalSessions) {
            auto sessionId = session.GetSessionId();
            result.emplace(std::move(sessionId), std::move(session));
        }
        return result;
    }();
    auto sessions = [&] {
        auto eg = g.BuildEventGuard("fetch_sessions");
        auto result = TMap<TString, TSessionManager::TSessionConstPtr>();
        for (auto&& sessionId : sessionIds) {
            if (compiledSessions.contains(sessionId)) {
                continue;
            }
            auto optionalSession = Server->GetDriveDatabase().GetSessionManager().GetSession(sessionId, tx);
            R_ENSURE(optionalSession, {}, "cannot GetSession", tx);
            auto session = std::move(*optionalSession);
            if (!session) {
                continue;
            }
            result.emplace(sessionId, std::move(session));
        }
        return result;
    }();

    auto signalqPresignedUrls = [&]() -> TSerialNumberEventIdToEventPresignedUrlsMap {
        auto eg = g.BuildEventGuard("wait_signalq_presigned_urls");
        if (!asyncSignalqPresignedUrls.Wait(deadline)) {
            g.AddEvent("SignalqEventsPresignedUrlsTimeout");
            return {};
        }
        if (!asyncSignalqPresignedUrls.HasValue()) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "SignalqEventsPresignedUrlsError")
                ("error", NThreading::GetExceptionInfo(asyncSignalqPresignedUrls))
            );
            return {};
        }
        return asyncSignalqPresignedUrls.GetValue();
    }();

    {
        TVector<TSignalEvent> signalEvents;
        for (auto&& signal : *optionalSignals) {
            cursor = signal.Start.Seconds();

            TSignalEvent signalEvent;
            signalEvent.SetTagId(signal.Id);
            signalEvent.SetEntityType(signal.Type);
            signalEvent.SetName(signal.Name);
            signalEvent.SetResolution(signal.Resolution);
            signalEvent.SetSince(signal.Start);
            if (signal.Start) {
                auto since = CalculateSignalStart(signal.Name, signal.Start, visibleSignals);
                signalEvent.SetSince(since);
            }
            if (signal.Finish) {
                signalEvent.SetUntil(signal.Finish);
            }
            if (signal.Type == NEntityTagsManager::EEntityType::Car) {
                signalEvent.SetIsActual(signal.Finish == TInstant::Zero());
            }
            if (signal.Type == NEntityTagsManager::EEntityType::User) {
                signalEvent.SetIsActual(signal.Finish == TInstant::Zero());
            }
            if (signal.Type == NEntityTagsManager::EEntityType::Trace) {
                signalEvent.SetIsActual(true);
            }

            TAtomicSharedPtr<NDrivematics::TSignalTagTraits> tag;
            if (!tag) {
                auto objectTag = objectTags.find(signalEvent.GetTagId());
                if (objectTag != objectTags.end()) {
                    tag = objectTag->second;
                }
            }
            if (!tag) {
                auto userTag = userTags.find(signalEvent.GetTagId());
                if (userTag != userTags.end()) {
                    tag = userTag->second;
                }
            }
            if (!tag) {
                auto sessionTag = sessionTags.find(signalEvent.GetTagId());
                if (sessionTag != sessionTags.end()) {
                    tag = sessionTag->second;
                }
            }
            if (tag) {
                auto details = tag->GetDetailReport();
                auto signalqTag = std::dynamic_pointer_cast<TSignalqEventTraceTag>(tag);
                if (signalqTag) {
                    auto serialNumberEventId = signalqTag->MakeSerialNumberEventId();
                    auto p = signalqPresignedUrls.find(serialNumberEventId);
                    if (p != signalqPresignedUrls.end()) {
                        NJson::InsertField(details, "media", p->second.BuildReportJson());
                    }
                }
                if (details.IsDefined()) {
                    signalEvent.SetDetails(std::move(details));
                }

                auto nameOverride = tag->GetSignalNameOverride();
                if (nameOverride) {
                    signalEvent.SetName(nameOverride);
                }
            }

            if (signal.Type == NEntityTagsManager::EEntityType::Car) {
                auto objectId = ToString(signal.ObjectId);
                auto object = objects.GetResultPtr(objectId);
                if (object) {
                    signalEvent.MutableLinkedEntities().emplace_back()
                        .SetEntityType(NEntityTagsManager::EEntityType::Car)
                        .SetObjectId(objectId)
                        .SetObjectDetails(std::cref(*object))
                    ;
                }
            } else if (signal.Type == NEntityTagsManager::EEntityType::User) {
                auto userId = ToString(signal.UserId);
                auto user = usersInfo.GetResultPtr(userId);
                if (user) {
                    signalEvent.MutableLinkedEntities().emplace_back()
                        .SetEntityType(NEntityTagsManager::EEntityType::User)
                        .SetObjectId(userId)
                        .SetObjectDetails(
                            TUserInfo(user->GetReport(NUserReport::ReportUserSignal & permissions->GetUserReportTraits(), true))
                        )
                    ;
                }
            } else if (signal.Type == NEntityTagsManager::EEntityType::Trace) {
                auto sessionId = ToString(signal.SessionId);
                auto compiledSession = compiledSessions.find(sessionId);
                if (compiledSession != compiledSessions.end()) {
                    signalEvent.MutableLinkedEntities().emplace_back()
                        .SetEntityType(NEntityTagsManager::EEntityType::Trace)
                        .SetObjectId(sessionId)
                        .SetObjectDetails(std::cref(compiledSession->second))
                    ;
                }
                auto session = sessions.find(sessionId);
                if (session != sessions.end()) {
                    signalEvent.MutableLinkedEntities().emplace_back()
                        .SetEntityType(NEntityTagsManager::EEntityType::Trace)
                        .SetObjectId(sessionId)
                        .SetObjectDetails(session->second)
                    ;
                }
            }
            signalEvents.push_back(std::move(signalEvent));
            if (signalEvents.size() == pageSize) {
                break;
            }
        }
        g.AddReportElement("signals", NJson::ToJson(signalEvents));
    }
    {
        NJson::TJsonValue signalDescriptions;
        for (auto&& [name, description] : visibleSignals) {
            signalDescriptions.InsertValue(name, description.GetReport(locale, *Server));
        }
        g.AddReportElement("signals_descriptions", std::move(signalDescriptions));
    }
    if (cursor) {
        g.AddReportElement("next_cars_cursor", *cursor);
    }
    if (pageSize) {
        g.AddReportElement("can_get_more_pages", optionalSignals->size() > pageSize);
    }
    g.SetCode(HTTP_OK);
}
