#include "list_organization_members.h"

#include "common.h"

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/scoring/scoring.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/enum.h>

#include <library/cpp/http/misc/httpcodes.h>

class TMemberInfo {
public:
    R_FIELD(TString, Id);
    R_FIELD(TString, FirstName);
    R_FIELD(TString, PName);
    R_FIELD(TString, LastName);
    R_FIELD(TString, Email);
    R_FIELD(TString, Status);
    R_FIELD(TString, PhoneNumber);
    R_FIELD(NDrivematics::TUserOrganizationAffiliationTag::TRoles, Roles);

    R_FIELD(TAtomicSharedPtr<const TScoringUserTag>, Score);
};

template<>
NJson::TJsonValue NJson::ToJson(const TMemberInfo& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result["id"] = object.GetId();
    result["first_name"] = object.GetFirstName();
    result["pn"] = object.GetPName();
    result["last_name"] = object.GetLastName();
    result["email"] = object.GetEmail();
    result["phone_number"] = object.GetPhoneNumber();
    result["status"] = object.GetStatus();
    result["roles"] = NJson::ToJson(object.GetRoles());
    if (object.GetScore()) {
        result["user_score"] = object.GetScore()->SerializeToJson();
    }
    return result;
}

namespace NDrivematics {
    void TListOrganizationMembersProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
        const auto& cgi = Context->GetCgiParameters();
        auto usersIdsExt = MakeSet(GetStrings(requestData, "users_ids", false));
        bool includeUserScore = GetValue<bool>(cgi, "user_score", false).GetOrElse(false);
        auto tx = BuildTx<NSQL::ReadOnly>();
        const auto& api = *Yensured(Server->GetDriveAPI());
        auto affiliatedCompanyTagDesc = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, *Server, tx);
        auto dbTags = api.GetTagsManager().GetUserTags().RestoreTagsRobust({}, { affiliatedCompanyTagDesc->GetName() }, tx);

        R_ENSURE(dbTags, {}, "can't restore tags", tx);
        TSet<TString> usersIds;
        for (const auto& dbTag : *dbTags) {
            usersIds.insert(dbTag.GetObjectId());
        }
        if (!usersIdsExt.empty()) {
            usersIds = MakeIntersection(usersIds, usersIdsExt);
        }
        R_ENSURE(!usersIds.empty(), HTTP_BAD_REQUEST, "users ids set is empty");
        auto usersInfo = api.GetUsersData()->FetchInfo(usersIds, tx);
        R_ENSURE(usersInfo, {}, "can't restore users", tx);
        TVector<TMemberInfo> membersInfo;
        membersInfo.reserve(dbTags->size());
        for(const auto& dbTag : *dbTags) {
            const auto& userId = dbTag.GetObjectId();
            auto tag = dbTag.GetTagAs<TUserOrganizationAffiliationTag>();
            R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag as user organization affiliation tag");
            auto it = usersInfo.GetResult().find(userId);
            if (it != usersInfo.GetResult().end()) {
                const auto& userInfo = it->second;
                if (userInfo.GetStatus() == NDrive::UserStatusDeleted) {
                    continue;
                }
                auto& element = membersInfo.emplace_back();
                element.SetFirstName(userInfo.GetFirstName())
                    .SetPName(userInfo.GetPName())
                    .SetLastName(userInfo.GetLastName())
                    .SetEmail(userInfo.GetEmail())
                    .SetPhoneNumber(userInfo.GetPhone())
                    .SetStatus(userInfo.GetStatus())
                    .SetRoles(tag->GetRoles())
                    .SetId(userId);

                if (includeUserScore) {
                    auto taggedUser = api.GetTagsManager().GetUserTags().RestoreObject(userId, tx);
                    if (taggedUser) {
                        TDBTag scoreTag = taggedUser->GetFirstTagByClass<TScoringUserTag>();
                        auto score = scoreTag.GetTagAs<TScoringUserTag>();
                        if (score) {
                            element.MutableScore() = std::dynamic_pointer_cast<const TScoringUserTag>(scoreTag.GetData());
                        }

                    }
                }
            }
        }
        g.MutableReport().AddReportElement("members", ::NJson::ToJson(membersInfo));
        g.SetCode(HTTP_OK);
    }
}
