#include "notifiers_company.h"

#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/notifications/manager.h>
#include <drive/backend/notifications/telegram/telegram.h>
#include <drive/backend/processors/leasing/common.h>


namespace {
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersCompanyAddProcessor::THandlerConfig> Registrator1(TNotifiersCompanyAddProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersCompanyRemoveProcessor::THandlerConfig> Registrator2(TNotifiersCompanyRemoveProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersCompanyListProcessor::THandlerConfig> Registrator3(TNotifiersCompanyListProcessor::GetTypeName() + "~");
    IRequestProcessorConfig::TFactory::TRegistrator<TNotifiersCompanyModifyProcessor::THandlerConfig> Registrator4(TNotifiersCompanyModifyProcessor::GetTypeName() + "~");

    const int DefaultMessageLengthLimit = 4096;
    const bool UseEventLog = false;

    using namespace NDrivematics;

    TString GenerateName(TAtomicSharedPtr<TUserOrganizationAffiliationTag::TDescription> affiliatedCompanyTagDescription, const TString& type) {
        return ToString(affiliatedCompanyTagDescription->GetUid()) + "-" + type + "-" + CreateGuidAsString();
    }
}

void TNotifiersCompanyAddProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    R_ENSURE(requestData["object"].IsMap(), HTTP_BAD_REQUEST, "'objects' is not map", NDrive::MakeError("notifiers.incorrect_request"), tx);

    auto objectJson = requestData["object"];

    TUserOrganizationAffiliationTag::ESupportNotifierType type;
    R_ENSURE(
        TryFromString(objectJson["type"].GetString(), type)
        , HTTP_BAD_REQUEST
        , "unsupport type: " + requestData["object"]["type"].GetString()
        , NDrive::MakeError("notifiers.incorrect_request")
        , tx
    );

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    auto affiliatedCompanyTagDescription =
        std::dynamic_pointer_cast<TUserOrganizationAffiliationTag::TDescription>(
            TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx)
            ->Clone()
        )
    ;

    const auto notifiersManager = Server->GetNotifiersManager();
    const auto notifiersManagerImpl = dynamic_cast<const TNotifiersManager*>(notifiersManager);
    R_ENSURE(notifiersManager && notifiersManagerImpl, HTTP_NOT_IMPLEMENTED, "not configured", NDrive::MakeError("notifiers.not_configured"), tx);

    TNotifierContainer newNotifier;
    switch (type) {
        case TUserOrganizationAffiliationTag::ESupportNotifierType::TelegramCompany:
        {

            auto notifierName =
                GenerateName(affiliatedCompanyTagDescription, objectJson["type"].GetString());
            objectJson["name"] = notifierName;
            objectJson["meta"]["bot"]["owner"] = affiliatedCompanyTagDescription->GetName();
            objectJson["message_length_limit"] = DefaultMessageLengthLimit;
            objectJson["use_event_log"] = UseEventLog;

            TMessagesCollector errors;
            R_ENSURE(
                TBaseDecoder::DeserializeFromJsonVerbose(newNotifier, objectJson, errors)
                , HTTP_BAD_REQUEST
                , "incorrect object json: " + errors.GetStringReport()
                , NDrive::MakeError("notifiers.incorrect_request")
                , tx
            );

            auto newNotifierImpl = newNotifier.GetAs<TTelegramNotificationsCompanyConfig>();
            auto notifiersCompanyIds = aclCompany->GetEntityObjects(GetEntityType(), permissions);

            TVector<TNotifierContainer> restoredNotifiers;
            R_ENSURE(
                RestoreCompanyNotifier(restoredNotifiers, notifiersCompanyIds, ToString(type), notifiersManagerImpl)
                , {}
                , "notifiers restore error"
                , NDrive::MakeError("notifiers.restore")
                , tx
            );
            ForEach(restoredNotifiers.begin(), restoredNotifiers.end(), [&](const TNotifierContainer& notifier) -> void {
                R_ENSURE(!(notifier.GetDisplayName() == newNotifier.GetDisplayName()), HTTP_BAD_REQUEST, "notifier already exist: " << newNotifier.GetDisplayName(), NDrive::MakeError("notifiers.already_exist.display_name"), tx);
            });

            TString requestBotId;
            {
                if (objectJson["meta"]["bot"].Has("bot_id") && objectJson["meta"]["bot"]["bot_id"].IsString()) {
                    requestBotId = objectJson["meta"]["bot"]["bot_id"].GetString();
                }
                R_ENSURE(requestBotId, HTTP_BAD_REQUEST, "bot_id required", NDrive::MakeError("notifiers.incorrect_request"), tx);
            }
            auto& telegramCompany = affiliatedCompanyTagDescription->MutableTelegramNotifier();
            R_ENSURE(
                  !telegramCompany.GetBotId(newNotifierImpl->GetChat().GetChatId())
                  && telegramCompany.Add(requestBotId, &newNotifierImpl->GetChat().GetChatId())
                , {}
                , "chat_id alerady use by another bot_id"
                , NDrive::MakeError("notifiers.already_exist.chat_id")
                , tx
            );
            R_ENSURE(
                  notifiersManager->UpsertObject(newNotifier, permissions->GetUserId(), tx)
                , HTTP_INTERNAL_SERVER_ERROR
                , "cannot add notifier"
                , NDrive::MakeError("notifiers.add")
                , tx
            );
            break;
        }
        case TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport:
        default:
            g.SetCode(HTTP_BAD_REQUEST);
            return;
    }

    R_ENSURE(
        Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescription, permissions->GetUserId(), tx)
        , {}
        , "cannot update UserOrganizationAffiliationTag tag"
        , NDrive::MakeError("notifiers.company.update")
        , tx
    );
    {
        TMessagesCollector errors;
        R_ENSURE(
            aclCompany->AddEntityObject(GetEntityType(), newNotifier.GetInternalId(), permissions, errors)
            , HTTP_BAD_REQUEST
            , "cannot add object to company " << errors.GetStringReport()
            , NDrive::MakeError("acl.add")
            , tx
        );
        R_ENSURE(
            Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx)
            , {}
            , "cannot update acl tag"
            , NDrive::MakeError("acl.update")
            , tx
        );
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);
    g.SetCode(HTTP_OK);
}

void TNotifiersCompanyListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /* requestData */) {
    const TSet<typename TNotifierContainer::TId> ids = MakeSet(TBase::template GetValues<typename TNotifierContainer::TId>(Context->GetCgiParameters(), "ids", false));
    const auto requestType = GetValue<TUserOrganizationAffiliationTag::ESupportNotifierType>(Context->GetCgiParameters(), "type", false);

    auto tx = BuildTx<NSQL::ReadOnly>();

    R_ENSURE(
          !requestType || requestType != TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport
        , HTTP_NOT_IMPLEMENTED
        , "not configured"
        , NDrive::MakeError("notifiers.incorrect_request")
        , tx
    );

    const auto notifiersManager = Server->GetNotifiersManager();
    const auto notifiersManagerImpl = static_cast<const TNotifiersManager*>(notifiersManager);
    R_ENSURE(notifiersManager && notifiersManagerImpl, HTTP_NOT_IMPLEMENTED, "not configured", NDrive::MakeError("notifiers.not_configured"), tx);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    auto notifiersCompanyIds = aclCompany->GetEntityObjects(GetEntityType(), permissions);

    if (!ids.empty()) {
        notifiersCompanyIds = MakeIntersection(notifiersCompanyIds, ids);
    }

    TSet<TNotifierContainer> restoredNotifiers;
    R_ENSURE(
        RestoreCompanyNotifier(restoredNotifiers, notifiersCompanyIds, requestType ? ToString(*requestType) : Default<TString>(), notifiersManagerImpl)
        , {}
        , "notifiers restore error"
        , NDrive::MakeError("notifiers.restore")
        , tx
    );

    NJson::TJsonValue reportJson(NJson::JSON_ARRAY);
    for (auto&& notifier : restoredNotifiers) {
        reportJson.AppendValue(notifier.GetUserReport(*Server));
    }
    auto affiliatedCompanyTagDescription =
        std::dynamic_pointer_cast<TUserOrganizationAffiliationTag::TDescription>(
            TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx)
            ->Clone()
        )
    ;
    g.MutableReport().AddReportElement("objects", std::move(reportJson));
    g.MutableReport().AddReportElement("bot_ids_orphans",  NJson::ToJson(affiliatedCompanyTagDescription->MutableTelegramNotifier().GetOrphanBotId()));
    g.SetCode(HTTP_OK);
}

void TNotifiersCompanyRemoveProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto tx = BuildTx<NSQL::Writable>();

    const auto requestType = GetValue<TUserOrganizationAffiliationTag::ESupportNotifierType>(requestData, "type", true);
    R_ENSURE(
          !requestType || requestType != TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport
        , HTTP_NOT_IMPLEMENTED
        , "not configured"
        , NDrive::MakeError("notifiers.incorrect_request")
        , tx
    );

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    const auto notifiersManager = Server->GetNotifiersManager();
    const auto notifiersManagerImpl = dynamic_cast<const TNotifiersManager*>(notifiersManager);
    R_ENSURE(notifiersManager && notifiersManagerImpl, HTTP_NOT_IMPLEMENTED, "not configured", NDrive::MakeError("notifiers.not_configured"), tx);

    switch (*requestType) {
        case TUserOrganizationAffiliationTag::ESupportNotifierType::TelegramCompany:
        {
            TSet<typename TNotifierContainer::TId> ids;
            TSet<TUserOrganizationAffiliationTag::TNotifierIdsReference::TBotId> botIds;
            if (requestData.Has("notifier_ids")) {
                R_ENSURE(requestData["notifier_ids"].IsArray(), HTTP_BAD_REQUEST, "'notifier_ids' is not array", NDrive::MakeError("notifiers.incorrect_request"), tx);
                TJsonProcessor::ReadContainer(requestData, "notifier_ids", ids);
            } else if (requestData.Has("bot_ids")) {
                R_ENSURE(requestData["bot_ids"].IsArray(), HTTP_BAD_REQUEST, "'bot_ids' is not array", NDrive::MakeError("notifiers.incorrect_request"), tx);
                TJsonProcessor::ReadContainer(requestData, "bot_ids", botIds);
            } else {
                R_ENSURE(
                      false
                    , HTTP_BAD_REQUEST
                    , "notifier_ids or bot_ids not present"
                    , NDrive::MakeError("notifiers.incorrect_request")
                    , tx
                );
            }

            if (!ids.empty()) {
                auto affiliatedCompanyTagDescription =
                    std::dynamic_pointer_cast<TUserOrganizationAffiliationTag::TDescription>(
                        TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx)
                        ->Clone()
                    )
                ;
                auto& telegramCompany = affiliatedCompanyTagDescription->MutableTelegramNotifier();

                auto notifiersCompanyIds = aclCompany->GetEntityObjects(GetEntityType(), permissions);
                auto intersection = MakeIntersection(ids, notifiersCompanyIds);
                R_ENSURE(
                      intersection.size() == ids.size()
                    , HTTP_BAD_REQUEST
                    , "not all notifier_ids existed"
                    , NDrive::MakeError("notifiers.incorrect_request")
                    , tx
                );
                R_ENSURE(aclCompany->RemoveEntityObjects(GetEntityType(), ids, permissions, Server, tx), {}, "cannot remove objects", NDrive::MakeError("acl.remove"), tx);
                R_ENSURE(
                      Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(aclCompany, permissions->GetUserId(), tx)
                    , {}
                    , "cannot update acl tag"
                    , NDrive::MakeError("acl.update")
                    , tx
                );
                TVector<TNotifierContainer> restoredNotifiers;
                {
                    R_ENSURE(
                        RestoreCompanyNotifier(restoredNotifiers, ids, ToString(*requestType), notifiersManagerImpl)
                        , {}
                        , "notifiers restore error"
                        , NDrive::MakeError("notifiers.restore")
                        , tx
                    );
                    R_ENSURE(
                          restoredNotifiers.size() == ids.size()
                        , HTTP_BAD_REQUEST
                        , "not all notifier_ids existed"
                        , NDrive::MakeError("notifiers.incorrect_request")
                        , tx
                    );
                }
                for (const auto& notifier : restoredNotifiers) {
                    if (auto notifierImpl = notifier.GetAs<TTelegramNotificationsCompanyConfig>()) {
                        telegramCompany.RemoveChatId(notifierImpl->GetChat().GetChatId());
                    }
                }
                R_ENSURE(
                      Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescription, permissions->GetUserId(), tx)
                    , {}
                    , "cannot update acl tag"
                    , NDrive::MakeError("notifiers.company.update")
                    , tx
                );
            }
            if (!botIds.empty()) {
                auto affiliatedCompanyTagDescription =
                    std::dynamic_pointer_cast<TUserOrganizationAffiliationTag::TDescription>(
                        TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx)
                        ->Clone()
                    )
                ;
                auto& telegramCompany = affiliatedCompanyTagDescription->MutableTelegramNotifier();
                auto intersection = MakeIntersection(botIds, telegramCompany.GetOrphanBotId());
                R_ENSURE(
                      intersection.size() == botIds.size()
                    , HTTP_BAD_REQUEST
                    , "not all bot_ids existed"
                    , NDrive::MakeError("notifiers.incorrect_request")
                    , tx
                );
                ForEach(botIds.begin(), botIds.end(), [&telegramCompany](const auto& botId){
                    telegramCompany.MutableOrphanBotId().erase(botId);
                });

                R_ENSURE(
                      Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescription, permissions->GetUserId(), tx)
                    , {}
                    , "cannot update acl tag"
                    , NDrive::MakeError("notifiers.company.update")
                    , tx
                );
            }
            break;
        }
        case TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport:
        default:
            g.SetCode(HTTP_BAD_REQUEST);
            return;
    }
    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);
    g.SetCode(HTTP_OK);
}

void TNotifiersCompanyModifyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Notifier);
    auto tx = BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    auto requestType = GetValue<TUserOrganizationAffiliationTag::ESupportNotifierType>(requestData, "type", false);
    requestType = !requestType ? GetValue<TUserOrganizationAffiliationTag::ESupportNotifierType>(requestData["object"], "type", true) : requestType;
    R_ENSURE(
          !requestType || requestType != TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport
        , HTTP_NOT_IMPLEMENTED
        , "not configured"
        , NDrive::MakeError("notifiers.incorrect_request")
        , tx
    );

    const auto notifiersManager = Server->GetNotifiersManager();
    const auto notifiersManagerImpl = static_cast<const TNotifiersManager*>(notifiersManager);
    R_ENSURE(notifiersManager && notifiersManagerImpl, HTTP_NOT_IMPLEMENTED, "not configured", NDrive::MakeError("notifiers.not_configured"), tx);

    auto acl = NDrivematics::TACLTag::GetACLTag(permissions, tx, *Server);
    auto aclCompany = acl->GetCompanyTagDescription(permissions->GetUserId(), *Server);

    switch (*requestType) {
        case TUserOrganizationAffiliationTag::ESupportNotifierType::TelegramCompany:
        {
            auto affiliatedCompanyTagDescription =
                std::dynamic_pointer_cast<TUserOrganizationAffiliationTag::TDescription>(
                    TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), *Server, tx)
                    ->Clone()
                )
            ;
            auto& telegramCompany = affiliatedCompanyTagDescription->MutableTelegramNotifier();

            if (requestData.Has("bot_id") && requestData.Has("new_bot_id")) {
                R_ENSURE(requestData["bot_id"].IsString() && requestData["new_bot_id"].IsString(), HTTP_BAD_REQUEST, "bot_id or new_bot_id is not present", NDrive::MakeError("notifiers.incorrect_request"), tx);
                R_ENSURE(
                      !telegramCompany.GetBotIds().contains(requestData["new_bot_id"].GetString())
                    , HTTP_BAD_REQUEST
                    , "new_bot_id already exist"
                    , NDrive::MakeError("notifiers.incorrect_request.new_bot_id")
                    , tx
                );
                R_ENSURE(
                      telegramCompany.ChangeGlobalBotId(requestData["bot_id"].GetString(), requestData["new_bot_id"].GetString())
                    , HTTP_BAD_REQUEST
                    , "bot_id not found"
                    , NDrive::MakeError("notifiers.incorrect_request.bot_id")
                    , tx
                );
                R_ENSURE(
                      Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescription, permissions->GetUserId(), tx)
                    , {}
                    , "cannot update acl tag"
                    , NDrive::MakeError("notifiers.company.update")
                    , tx
                );
            } else {
                auto oldNotifierId = requestData["object"]["name"].GetString();
                auto chatId = requestData["object"]["meta"]["chat"]["chat_id"].GetString();
                auto requestBotId = requestData["object"]["meta"]["bot"]["bot_id"].GetString();
                auto requestDisplayName = requestData["object"]["display_name"].GetString();

                auto notifiersCompanyIds = aclCompany->GetEntityObjects(GetEntityType(), permissions);
                R_ENSURE(
                      notifiersCompanyIds.contains(oldNotifierId)
                    , HTTP_BAD_REQUEST
                    , "display_name not found"
                    , NDrive::MakeError("notifiers.incorrect_request.name")
                    , tx
                );
                { // patch notifier data
                    TVector<TNotifierContainer> restoredNotifiers;
                    R_ENSURE(
                          RestoreCompanyNotifier(restoredNotifiers, notifiersCompanyIds, ToString(*requestType), notifiersManagerImpl)
                            && restoredNotifiers.size() == notifiersCompanyIds.size()
                        , {}
                        , "notifiers restore error"
                        , NDrive::MakeError("notifiers.restore")
                        , tx
                    );
                    auto findIt = std::find_if(restoredNotifiers.begin(), restoredNotifiers.end(), [&oldNotifierId](const TNotifierContainer& notifier){
                        return notifier.GetInternalId() == oldNotifierId;
                    });

                    auto& restoredNotifier = *findIt;
                    auto restoredDisplayName = restoredNotifier.GetDisplayName();
                    TMessagesCollector errors;
                    R_ENSURE(
                          restoredNotifier.UserPatch<TTelegramNotificationsCompanyConfig>(requestData["object"], errors)
                        , {}
                        , "unsupport changes"
                        , NDrive::MakeError("notifiers.patch")
                        , tx
                    );
                    if (requestDisplayName != restoredDisplayName) {
                        ForEach(restoredNotifiers.begin(), restoredNotifiers.end(), [&](const TNotifierContainer& notifier) -> void {
                            R_ENSURE(
                                  !(notifier.GetDisplayName() == requestDisplayName && notifier.GetInternalId() != oldNotifierId)
                                , HTTP_BAD_REQUEST, "notifier already exist: " << requestDisplayName
                                , NDrive::MakeError("notifiers.already_exist.display_name")
                                , tx
                            );
                        });
                        R_ENSURE(
                              notifiersManager->UpsertObject(restoredNotifier, permissions->GetUserId(), tx)
                            , HTTP_INTERNAL_SERVER_ERROR
                            , "cannot add notifier"
                            , NDrive::MakeError("notifiers.update")
                            , tx
                        );
                    }
                }
                { // change bot_id
                    auto existedBotId = telegramCompany.GetBotId(chatId);
                    R_ENSURE(
                          existedBotId
                        , HTTP_BAD_REQUEST
                        , "chat_id not found"
                        , NDrive::MakeError("notifiers.incorrect_request.chat_id")
                        , tx
                    );
                    if (*existedBotId != requestBotId) {
                        R_ENSURE(
                              telegramCompany.ChangeLocalBotId(chatId, requestBotId)
                            , HTTP_BAD_REQUEST
                            , "chat_id not found"
                            , NDrive::MakeError("notifiers.incorrect_request.chat_id")
                            , tx
                        );
                        R_ENSURE(
                              Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescription, permissions->GetUserId(), tx)
                            , {}
                            , "cannot update acl tag"
                            , NDrive::MakeError("notifiers.company.update")
                            , tx
                        );
                    }
                }
            }
            break;
        }
        case TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport:
        default:
            g.SetCode(HTTP_BAD_REQUEST);
            return;
    }

    R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

    g.SetCode(HTTP_OK);
}
