#include "portfolio.h"
#include "common.h"

#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <rtline/util/algorithm/tuple.h>

void TLeasingPortfolioProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto daySince = GetValue<ui32>(requestData, "dates.since", false);
    const auto dayUntil = GetValue<ui32>(requestData, "dates.until", false);
    const auto oldDaySince = GetValue<ui32>(requestData, "old_dates.since", false);
    const auto oldDayUntil = GetValue<ui32>(requestData, "old_dates.until", false);

    auto session = BuildTx<NSQL::ReadOnly>();
    const auto& api = *Yensured(Server->GetDriveAPI());

    auto [firstObservableTagName, leasingTagType] = NDrivematics::GetFirstObservableTagName(permissions, api);
    if (leasingTagType == NDrivematics::ELeasingTagType::LeasingCompany) {
        auto leasingCompanyTagDescription = std::dynamic_pointer_cast<const NDrivematics::TLeasingCompanyTag::TDescription>(api.GetTagsManager().GetTagsMeta().GetDescriptionByName(firstObservableTagName));
        R_ENSURE(leasingCompanyTagDescription, HTTP_INTERNAL_SERVER_ERROR, "can't get leasing tag description");
        const auto observeTagsNames = permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe);
        if (observeTagsNames.contains(leasingCompanyTagDescription->GetName())) {
            g.MutableReport().AddReportElement("score", leasingCompanyTagDescription->GetScore());
        }
    }

    TVector<TDBTag> dbTags;
    R_ENSURE(api.GetTagsManager().GetDeviceTags().RestoreTags({}, { firstObservableTagName }, dbTags, session), {}, "can not restore tags", session);
    TSet<TString> carsIds;
    for (const auto& dbTag : dbTags) {
        carsIds.insert(dbTag.GetObjectId());
    }

    TTaggedObjectsSnapshot devicesSnapshot;
    {
        auto eg = g.BuildEventGuard("restore_cars");
        R_ENSURE(api.GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds(carsIds, devicesSnapshot, TInstant::Zero()), HTTP_INTERNAL_SERVER_ERROR, "can't restore cars from cache");
    }


    auto daysCount = Server->GetSettings().GetValueDef<ui32>("leasing_cabinet.aggregation.days_count", 60);
    auto noSHThreshold = Server->GetSettings().GetValueDef<ui32>("leasing_cabinet.supply_hours_threshold", 320);

    NJson::TJsonValue portfolioStats;
    {
        auto allCarsStats = NDrivematics::CalculateStats(
            carsIds,
            devicesSnapshot,
            daySince,
            dayUntil,
            daysCount,
            noSHThreshold,
            NDrivematics::TLeasingStatsTag::CalculatePortfolioStats,
            [](const TTaggedDevice& /* taggedDevice */) {
                return true;
            }
        );
        auto telematicsCarsStats = NDrivematics::CalculateStats(
            carsIds,
            devicesSnapshot,
            daySince,
            dayUntil,
            daysCount,
            noSHThreshold,
            NDrivematics::TLeasingStatsTag::CalculatePortfolioStats,
            [](const TTaggedDevice& taggedDevice) {
                return taggedDevice.GetTag(NDrivematics::THasTelematicsTag::TypeName);
            }
        );

        Zip(allCarsStats.GetFields(), telematicsCarsStats.GetFields(), [&portfolioStats] (auto&& allCarsField, auto&& telematicsCarsField) {
            NJson::TJsonValue stats;
            if (!(allCarsField.GetTraits() & NDrivematics::ReportInPortfolio)) {
                return;
            }
            stats["name"] = ToString(allCarsField.GetName());
            stats["display_name"] = allCarsField.GetDisplayName();
            stats["type"] = ToString(allCarsField.GetStatsType());
            stats["all_cars"] = NJson::TMapBuilder("value", std::move(allCarsField.Value));
            stats["telematics_cars"] = NJson::TMapBuilder("value", std::move(telematicsCarsField.Value));
            portfolioStats.AppendValue(std::move(stats));
        });
    }

    if (oldDaySince && oldDayUntil && daySince && dayUntil) {
        auto oldAllCarsStats = NDrivematics::CalculateStats(
            carsIds,
            devicesSnapshot,
            oldDaySince,
            oldDayUntil,
            daysCount,
            noSHThreshold,
            NDrivematics::TLeasingStatsTag::CalculatePortfolioStats,
            [](const TTaggedDevice& /* taggedDevice */) {
                return true;
            }
        );
        auto oldTelematicsCarsStats = NDrivematics::CalculateStats(
            carsIds,
            devicesSnapshot,
            oldDaySince,
            oldDayUntil,
            daysCount,
            noSHThreshold,
            NDrivematics::TLeasingStatsTag::CalculatePortfolioStats,
            [](const TTaggedDevice& taggedDevice) {
                return taggedDevice.GetTag(NDrivematics::THasTelematicsTag::TypeName);
            }
        );

        int fieldIdx = -1;
        Zip(oldAllCarsStats.GetFields(), oldTelematicsCarsStats.GetFields(), [&fieldIdx, &portfolioStats] (auto&& allCarsField, auto&& telematicsCarsField) {
            ++fieldIdx;
            if (!(allCarsField.GetTraits() & NDrivematics::ReportInPortfolio)) {
                return;
            }
            NJson::TJsonValue& stats = portfolioStats[fieldIdx];
            stats["all_cars"]["old_value"] = std::move(allCarsField.Value);
            stats["telematics_cars"]["old_value"] = std::move(telematicsCarsField.Value);
        });
    }

    g.MutableReport().AddReportElement("portfolio", std::move(portfolioStats));
    g.SetCode(HTTP_OK);
}
