#include "score.h"

#include <drive/backend/data/leasing/leasing.h>

void TAddLeasingScoreProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TMaybe<TString> leasingCompany = GetValue<TString>(requestData, "leasing_company", false);
    const TMaybe<ui64> taxiCompanyTin = GetValue<ui64>(requestData, "taxi_company_tin", false);
    double score = GetValue<double>(requestData, "score", true).GetOrElse(0);
    TMaybe<double> telematicsCarsScore = GetValue<double>(requestData, "telematics_cars_score", false);
    const auto& api = *Yensured(Yensured(Server)->GetDriveAPI());
    R_ENSURE(leasingCompany || taxiCompanyTin, HTTP_BAD_REQUEST, "'leasing_company' or 'taxi_company_tin' parameter should be present");
    auto session = BuildTx<NSQL::Writable>();
    if (leasingCompany) {
        ITagsMeta::TTagDescriptionsByName tags = api.GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { NDrivematics::TLeasingCompanyTag::TypeName });
        auto it = std::find_if(tags.begin(), tags.end(), [&leasingCompany](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const NDrivematics::TLeasingCompanyTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag description");
            return tagDesc->GetLeasingCompanyName() == *leasingCompany;
        });
        R_ENSURE(it != tags.end(), HTTP_BAD_REQUEST, "can't find leasing company with name " << *leasingCompany);
        auto tagDesc = std::dynamic_pointer_cast<NDrivematics::TLeasingCompanyTag::TDescription>(it->second->Clone());
        tagDesc->SetScore(score);
        R_ENSURE(api.GetTagsManager().GetTagsMeta().RegisterTag(tagDesc, permissions->GetUserId(), session), HTTP_INTERNAL_SERVER_ERROR, "cannot update tag description", session);
    } else if (taxiCompanyTin) {
        ITagsMeta::TTagDescriptionsByName tags = api.GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car, { NDrivematics::TTaxiCompanyTag::TypeName });
        auto it = std::find_if(tags.begin(), tags.end(), [&taxiCompanyTin](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const NDrivematics::TTaxiCompanyTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag description");
            return tagDesc->GetTin() == *taxiCompanyTin;
        });
        R_ENSURE(it != tags.end(), HTTP_BAD_REQUEST, "can't find taxi company with tin " << *taxiCompanyTin);
        auto tagDesc = std::dynamic_pointer_cast<NDrivematics::TTaxiCompanyTag::TDescription>(it->second->Clone());
        tagDesc->SetScore(score);
        if (telematicsCarsScore) {
            tagDesc->SetTelematicsCarsScore(*telematicsCarsScore);
        }
        R_ENSURE(api.GetTagsManager().GetTagsMeta().RegisterTag(tagDesc, permissions->GetUserId(), session), HTTP_INTERNAL_SERVER_ERROR, "cannot update tag description", session);
    }
    R_ENSURE(session.Commit(), {}, "cannot commit session", session);
    g.SetCode(HTTP_OK);
}
